﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/atk_SoundControlSession.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/fnd/basis/atkfnd_Memory.h>
#include <nn/atk/viewer/atk_SoundObjectController.h>
#include <nn/atk/viewer/detail/atk_IErrorProvider.h>
#include <nn/atk/viewer/detail/hio/atk_HioAsyncChannel.h>
#include <nn/atk/viewer/detail/protocol/atk_PlaySoundPacket.h>
#include <nn/atk/viewer/detail/protocol/atk_StopSoundPacket.h>
#include <nn/atk/viewer/detail/protocol/atk_PauseSoundPacket.h>
#include <nn/atk/viewer/detail/protocol/atk_UpdateSoundInfoPacket.h>

#if !defined(NN_SDK_BUILD_RELEASE)
//#define NN_ATK_ENABLE_COM_DEBUG
#endif

namespace nn {
namespace atk {
namespace viewer {

NN_DEFINE_STATIC_CONSTANT( const size_t SoundControlSession::DefaultChannelStreamBufferSize );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundControlSession::DefaultSyncTimeout );
NN_DEFINE_STATIC_CONSTANT( const int SoundControlSession::DefaultMaxItemName );

//----------------------------------------------------------
SoundControlSession::SoundControlSession() NN_NOEXCEPT :
m_State(State_NotInitialized),
m_Port(0),
m_SyncTimeout(0),
m_SoundObjectController(NULL)
{
}

//----------------------------------------------------------
SoundControlSession::~SoundControlSession() NN_NOEXCEPT
{
    // Finalize() 済みであることを確認します。
    NN_SDK_ASSERT(!IsInitialized(), "SoundControlSession is not finalized.\n");
}

//----------------------------------------------------------
void
SoundControlSession::Finalize() NN_NOEXCEPT
{
    Close();

    UnregisterSoundObjectController(NULL);

    FinalizeChannel();

    m_HioManager.Finalize();

    m_SyncTimeout = 0;
    m_State = State_NotInitialized;
}

//----------------------------------------------------------
viewer::Result
SoundControlSession::RegisterSoundObjectController(SoundObjectController* soundObjectController) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundControlSession is not initialized.\n");
    NN_SDK_ASSERT(m_SoundObjectController == NULL, "SoundObjectController is already registered.\n");

    if(soundObjectController == NULL ||
        !soundObjectController->IsInitialized())
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return viewer::Result(viewer::ResultType_Failed);
    }

    m_SoundObjectController = soundObjectController;

    m_PlaySoundHandler.SetSoundObjectController(soundObjectController);
    m_StopSoundHandler.SetSoundObjectController(soundObjectController);
    m_PauseSoundHandler.SetSoundObjectController(soundObjectController);

    return viewer::Result(viewer::ResultType_True);
}

//----------------------------------------------------------
void
SoundControlSession::UnregisterSoundObjectController(SoundObjectController* soundObjectController) NN_NOEXCEPT
{
    // TODO : 複数サウンドアーカイブに対応したら利用します。
    (void)soundObjectController;

    NN_SDK_ASSERT(IsInitialized(), "SoundControlSession is not initialized.\n");
    m_SoundObjectController = NULL;

    m_PlaySoundHandler.SetSoundObjectController(NULL);
    m_StopSoundHandler.SetSoundObjectController(NULL);
    m_PauseSoundHandler.SetSoundObjectController(NULL);
}

//----------------------------------------------------------
void
SoundControlSession::Open() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundControlSession is not initialized.\n");

    if(IsOpened())
    {
        return;
    }

    ClearBuffer();
    m_State = State_Opened;

    m_SyncStopWatch.Start();
}

//----------------------------------------------------------
void
SoundControlSession::Close() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundControlSession is not initialized.\n");

    m_SyncStopWatch.Stop();

    m_State = State_Initialized;
}

//----------------------------------------------------------
void
SoundControlSession::Update() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundControlSession is not initialized.\n");

    if(!IsOpened())
    {
        return;
    }

    if(!m_Channel.IsOpened())
    {
        if(!m_Channel.Open(GetChannelInfo(viewer::detail::HioChannelType_SndeditCtrl)))
        {
            return;
        }
    }

    m_HioManager.Update();

    // SYNC タイムアウトを経過したら、同期します。
    if(m_SyncStopWatch.GetElapsedTime().ToMilliSeconds() >= m_SyncTimeout)
    {
        m_SyncStopWatch.Reset();
        Sync();
    }
}

//----------------------------------------------------------
viewer::Result
SoundControlSession::InitializeChannel(
    atk::detail::fnd::FrameHeap& allocator,
    size_t recvStreamBufferSize,
    size_t recvPacketBufferSize) NN_NOEXCEPT
{
    void* recvStreamBuffer = allocator.Alloc(recvStreamBufferSize);
    void* recvPacketBuffer = allocator.Alloc(recvPacketBufferSize);

    void* workBuffer = allocator.Alloc(GetRequiredWorkBufferSize());

    if(recvStreamBuffer == NULL || recvPacketBuffer == NULL)
    {
        return viewer::Result(viewer::ResultType_OutOfMemory);
    }

    viewer::detail::HioResult result = m_Channel.Initialize(
        recvStreamBuffer,
        recvStreamBufferSize,
        recvPacketBuffer,
        recvPacketBufferSize,
        workBuffer,
        GetRequiredWorkBufferSize()
    );

    if(result.IsFailed())
    {
        return viewer::Result(viewer::ResultType(result));
    }

    m_HioManager.RegisterChannel(m_Channel);

    InitializeHandlers();

    return viewer::Result(viewer::ResultType_True);
}

//----------------------------------------------------------
void
SoundControlSession::FinalizeChannel() NN_NOEXCEPT
{
    m_HioManager.UnregisterChannel(m_Channel);

    FinalizeHandlers();

    m_Channel.Close();
    m_Channel.Finalize();
}

//----------------------------------------------------------
void
SoundControlSession::InitializeHandlers() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Channel.IsInitialized());

    m_Channel.RegisterMessageHandler(m_PlaySoundHandler);
    m_Channel.RegisterMessageHandler(m_StopSoundHandler);
    m_Channel.RegisterMessageHandler(m_PauseSoundHandler);
}

//----------------------------------------------------------
void
SoundControlSession::FinalizeHandlers() NN_NOEXCEPT
{
    if(m_Channel.IsInitialized())
    {
        m_Channel.UnregisterMessageHandler(m_PlaySoundHandler);
        m_Channel.UnregisterMessageHandler(m_StopSoundHandler);
        m_Channel.UnregisterMessageHandler(m_PauseSoundHandler);
    }
}

//----------------------------------------------------------
size_t
SoundControlSession::GetChannelRecvPacketBufferSize(int maxItemName) const NN_NOEXCEPT
{
    size_t result = 0;

    // TODO : チャンネルで扱うパケットの種類が増えたら、ここに追加します。
    // TODO : チャンネルで扱うパケットのサイズが変わったら、ここを編集します。
    result = std::max(result, detail::PlaySoundPacket::GetRequiredSize(maxItemName));
    result = std::max(result, detail::StopSoundPacket::GetRequiredSize());
    result = std::max(result, detail::PauseSoundPacket::GetRequiredSize());
    result = std::max(result, detail::UpdateSoundInfoPacket::GetRequiredSize());

    return nn::util::align_up(result, atk::detail::fnd::MemoryTraits::DefaultAlignment);
}

//----------------------------------------------------------
void
SoundControlSession::ClearBuffer() NN_NOEXCEPT
{
    m_Channel.ClearBuffer();
}

//----------------------------------------------------------
void
SoundControlSession::Sync() NN_NOEXCEPT
{
    detail::UpdateSoundInfoPacket packet;

    if(m_SoundObjectController != NULL)
    {
        for(uint32_t index = 0; index < SoundObjectController::SoundControllerCount; ++index)
        {
            detail::UpdateSoundInfoPacket::SoundState state =
                static_cast<detail::UpdateSoundInfoPacket::SoundState>(
                m_SoundObjectController->GetSoundController(index)->GetState()
                );

            packet.GetBody().SetSoundState(index, state);
        }
    }

    m_Channel.Send(packet);
}

//----------------------------------------------------------
viewer::Result
SoundControlSession::Initialize(
    void* buffer,
    size_t bufferLength,
    const Configs& configs) NN_NOEXCEPT
{
    if(buffer == NULL ||
        bufferLength == 0 ||
        configs.channelStreamBufferSize == 0 ||
        configs.syncTimeout == 0 ||
        configs.maxItemName == 0)
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return viewer::Result(viewer::ResultType_Failed);
    }

    //----------------------------------------------------------
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // バッファサイズをチェックします。
    if(bufferLength < GetRequiredMemorySize(configs))
    {
        NN_SDK_ASSERT(false, "bufferLength too small.\n");
        return viewer::Result(viewer::ResultType_Failed);
    }
#endif // defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    //----------------------------------------------------------

    if(IsInitialized())
    {
        return viewer::Result(viewer::ResultType_False);
    }

    m_Port = configs.port;

    if(m_Channel.IsInitialized() ||
        m_Channel.IsOpened())
    {
        return viewer::Result(viewer::ResultType_Failed);
    }

    if(!m_HioManager.Initialize())
    {
        return viewer::Result(viewer::ResultType_Failed);
    }

    atk::detail::fnd::FrameHeap allocator;
    allocator.Initialize(buffer, bufferLength);

    viewer::Result result = InitializeChannel(
        allocator,
        configs.channelStreamBufferSize,
        GetChannelRecvPacketBufferSize(configs.maxItemName));

    if(result.IsFailed())
    {
        Finalize();
        return result;
    }

    m_SyncTimeout = configs.syncTimeout;
    m_State = State_Initialized;

    return viewer::Result(viewer::ResultType_True);
}

//----------------------------------------------------------
size_t
SoundControlSession::GetRequiredMemorySize(const Configs& configs) const NN_NOEXCEPT
{
    // チャンネルの受信用ストリームバッファサイズ
    size_t result = nn::util::align_up(
        configs.channelStreamBufferSize, atk::detail::fnd::MemoryTraits::DefaultAlignment );

    // チャンネルの受信用パケットバッファサイズ
    result += GetChannelRecvPacketBufferSize(configs.maxItemName);

    result += viewer::detail::HioChannel::DefaultWorkBufferLength;

    return result;
}

//----------------------------------------------------------
size_t
SoundControlSession::GetRequiredWorkBufferSize() const NN_NOEXCEPT
{
    return viewer::detail::HioChannel::DefaultWorkBufferLength;
}

//----------------------------------------------------------
viewer::detail::HioStream::ChannelType
SoundControlSession::GetChannelInfo(viewer::detail::HioChannelType channel) const NN_NOEXCEPT
{
    switch(channel)
    {
    case viewer::detail::HioChannelType_SndeditCtrl:
        return "NN_ATK_CTRL";
    default:
        NN_SDK_ASSERT(false);
        return "INVALID_CH";
    }
}


} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
