﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundSystem.h>
#include <nn/atk/atk_DeviceOutRecorder.h>
#include <nn/atk/detail/atk_Macro.h>

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const uint32_t DeviceOutRecorder::RecordingBufferSize );
NN_DEFINE_STATIC_CONSTANT( const uint32_t DeviceOutRecorder::DefaultWriteBlockPerSamples );
NN_DEFINE_STATIC_CONSTANT( const size_t DeviceOutRecorder::RequiredThreadStackSize );
#ifdef NN_ATK_DEBUG_REC_BUFFER
NN_DEFINE_STATIC_CONSTANT( const uint32_t DeviceOutRecorder::RecorderBuffer::g_PrintInterval );
#endif

//---------------------------------------------------------------------------
DeviceOutRecorder::RecorderBuffer::RecorderBuffer(const char* deviceName/*=NULL*/) NN_NOEXCEPT :
m_SampleBuffer(NULL),
m_MaxBufferSamples(0),
m_MaxSamples(0),
m_ValidSamples(0),
m_ReadPosition(0),
m_WritePosition(0),
m_ReadBlockSamples(1),
m_DeviceName(deviceName == NULL ? "" : deviceName)
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    m_IsWarnedDropSamples = false;
#endif

#ifdef NN_ATK_DEBUG_REC_BUFFER
    m_PushCount = 0;
#endif
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::Initialize(
    int16_t* sampleBuffer,
    uint32_t maxSamples) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(sampleBuffer);
    NN_SDK_ASSERT(maxSamples > 0);

    m_SampleBuffer = sampleBuffer;
    m_MaxBufferSamples = maxSamples;

    m_WriteState.channelIndex = 0;
    m_WriteState.writtenSampleCount = 0;

    UpdateMaxSamples();
    Clear();
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::Finalize() NN_NOEXCEPT
{
    m_SampleBuffer = NULL;
    m_MaxSamples = 0;
    m_ReadBlockSamples = 1;
}

//---------------------------------------------------------------------------
uint32_t
DeviceOutRecorder::RecorderBuffer::Push(const int16_t* sampleBuffer, uint32_t samples) NN_NOEXCEPT
{
    if(samples == 0)
    {
        return 0;
    }

#ifdef NN_ATK_DEBUG_REC_BUFFER
    if(m_PushCount++ == g_PrintInterval)
    {
        NN_DETAIL_ATK_INFO("[DeviceOutRecorder:%4s] buffer : %8d / %8d samples\n", m_DeviceName, static_cast<uint32_t>( m_ValidSamples ), m_MaxSamples);
        m_PushCount = 0;
    }
#endif // NN_ATK_DEBUG_REC_BUFFER

    NN_SDK_ASSERT_NOT_NULL(sampleBuffer);

    // 書き込みポインタ⇔バッファ終端     への書き込み、または
    // 書き込みポインタ⇔読み込みポインタ への書き込み。
    uint32_t copyCount = std::min(samples, GetWritableCount());

    if(copyCount > 0)
    {
        Write(sampleBuffer, copyCount);
    }

    if(copyCount == samples)
    {
        return samples;
    }

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    if(!m_IsWarnedDropSamples)
    {
        m_IsWarnedDropSamples = true;
        NN_ATK_WARNING("[DeviceOutRecorder:%4s] buffer is full(%dsamples). dropped samples.", m_DeviceName, static_cast<uint32_t>( m_ValidSamples ));
    }
#endif

    return copyCount;
}

//---------------------------------------------------------------------------
uint32_t
DeviceOutRecorder::RecorderBuffer::Pop(uint32_t samples) NN_NOEXCEPT
{
    if(samples == 0)
    {
        return 0;
    }

    // 読み込みポインタを後方へ移動
    uint32_t skipCount = std::min(samples, GetContiguousReadableCount());
    Skip(skipCount);

    if(skipCount == samples)
    {
        return skipCount;
    }

    // 読み込みポインタをバッファ先頭に戻して移動
    NN_SDK_ASSERT(m_ReadPosition == 0);

    uint32_t restCount = std::min(samples - skipCount, GetContiguousReadableCount());
    Skip(restCount);

    return skipCount + restCount;
}

//---------------------------------------------------------------------------
int16_t*
DeviceOutRecorder::RecorderBuffer::Peek() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_SampleBuffer);
    return GetReadableCount() > 0 ? m_SampleBuffer + m_ReadPosition : NULL;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::SetReadBlockSamples(uint32_t value) NN_NOEXCEPT
{
    m_ReadBlockSamples = std::max<uint32_t>(1, value);
    UpdateMaxSamples();
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::Clear() NN_NOEXCEPT
{
    m_ValidSamples = 0;
    m_ReadPosition = 0;
    m_WritePosition = 0;

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    m_IsWarnedDropSamples = false;
#endif
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::UpdateMaxSamples() NN_NOEXCEPT
{
    if(m_ReadBlockSamples == 0)
    {
        m_MaxSamples = m_MaxBufferSamples;
    }
    else
    {
        // アライメント調整のため、ブロック単位で読み込むので、
        // ここで有効サンプル数を調整します。
        m_MaxSamples = m_MaxBufferSamples / m_ReadBlockSamples * m_ReadBlockSamples;
        NN_SDK_ASSERT(m_ReadBlockSamples <= m_MaxSamples);
    }
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::Skip(uint32_t samples) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_ValidSamples >= samples);

    m_ReadPosition = IncrementPosition(m_ReadPosition, samples);
    m_ValidSamples.fetch_sub( samples );
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecorderBuffer::Write(const int16_t* sampleBuffer, uint32_t samples) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_SampleBuffer);
    NN_SDK_ASSERT(m_ValidSamples + samples <= m_MaxSamples);

    // バッファには常に 6ch 分コピーする
    uint32_t channels = detail::driver::HardwareManager::GetInstance().GetChannelCountMax();
    uint32_t sampleCountPerChannel = detail::driver::HardwareManager::GetInstance().GetRendererSampleCount();

    for(uint32_t i = 0; i < samples; ++i)
    {
        uint32_t writeTo = IncrementPosition( m_WritePosition, m_WriteState.channelIndex + m_WriteState.writtenSampleCount * channels );
        m_SampleBuffer[writeTo] = sampleBuffer[i];

        if( ++m_WriteState.writtenSampleCount >= sampleCountPerChannel )
        {
            m_WriteState.writtenSampleCount = 0;

            if( ++m_WriteState.channelIndex >= channels )
            {
                m_WriteState.channelIndex = 0;

                uint32_t readSamples = sampleCountPerChannel * channels;
                m_WritePosition = IncrementPosition( m_WritePosition, readSamples );
                m_ValidSamples.fetch_add( readSamples );
            }
        }
    }
}

//---------------------------------------------------------------------------
uint32_t
DeviceOutRecorder::RecorderBuffer::GetWritableCount() const NN_NOEXCEPT
{
    uint32_t channels = detail::driver::HardwareManager::GetInstance().GetChannelCountMax();
    uint32_t sampleCountPerChannel = detail::driver::HardwareManager::GetInstance().GetRendererSampleCount();
    uint32_t unWritableSampleCount = m_ValidSamples + channels * sampleCountPerChannel;

    return m_MaxSamples >= unWritableSampleCount ? m_MaxSamples - unWritableSampleCount : 0;
}

//---------------------------------------------------------------------------
uint32_t
DeviceOutRecorder::RecorderBuffer::GetContiguousReadableCount() const NN_NOEXCEPT
{
    volatile uint32_t validSamples = m_ValidSamples;
    uint32_t samples = 0;

    if(m_ReadPosition <= m_WritePosition)
    {
        samples = validSamples;
    }
    else
    {
        samples = m_MaxSamples - m_ReadPosition;
    }

    if(m_ReadBlockSamples == 0)
    {
        return samples;
    }
    else
    {
        return samples / m_ReadBlockSamples * m_ReadBlockSamples;
    }
}

//---------------------------------------------------------------------------
uint32_t
DeviceOutRecorder::RecorderBuffer::IncrementPosition(uint32_t position, uint32_t length) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(length <= m_MaxSamples);

    uint32_t result = position + length;
    return result >= m_MaxSamples ? result - m_MaxSamples : result;
}

//---------------------------------------------------------------------------
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(push)
#pragma warning(disable:4355) // warning: used in base member initializer list
#endif
DeviceOutRecorder::DeviceOutRecorder(const char* deviceName/*=NULL*/) NN_NOEXCEPT :
m_State(State_NotInitialized),
m_Channels(0),
m_OutputMode(OutputMode_Stereo),
m_IsLeadSilenceTrimming(false),
m_MaxSamples(0),
m_WrittenSamples(0),
m_Thread(),
m_MessageQueue(&m_Message, 1),
m_MessageResult(0),
m_MessageDoneEvent(nn::os::EventClearMode_ManualClear),
m_RecordingBuffer(deviceName),
m_WorkBuffer(nullptr),
m_WriteBlockPerSamples(DefaultWriteBlockPerSamples)
{
}
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(pop) // disable:4355
#endif

//---------------------------------------------------------------------------
DeviceOutRecorder::~DeviceOutRecorder() NN_NOEXCEPT
{
    Finalize();
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::Initialize(void* recordingBuffer, size_t recordingBufferSize, void* pThreadStack, size_t threadStackSize) NN_NOEXCEPT
{
    InitializationOptions param;
    return Initialize(recordingBuffer, recordingBufferSize, pThreadStack, threadStackSize, param);
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::Initialize(void* recordingBuffer, size_t recordingBufferSize, void* pThreadStack, size_t threadStackSize, const InitializationOptions& options) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(recordingBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(recordingBufferSize, GetRequiredMemorySizeForRecording());
    NN_SDK_REQUIRES_NOT_NULL(pThreadStack);
    NN_SDK_REQUIRES_ALIGNED(pThreadStack, nn::os::ThreadStackAlignment);
    NN_SDK_REQUIRES_GREATER_EQUAL(threadStackSize, RequiredThreadStackSize);
    NN_UNUSED( threadStackSize );

    if(IsInitialized())
    {
        return false;
    }

    // 録音用バッファを maxFrameLength で割り切れるサイズに丸めます。
    const uint32_t maxFrameLength = GetMaxFrameLength();
    const size_t bufferLength = recordingBufferSize / maxFrameLength * maxFrameLength;
    m_RecordingBuffer.Initialize( reinterpret_cast<int16_t*>( recordingBuffer ), static_cast<uint32_t>( bufferLength / sizeof(int16_t) ) );

    m_ThreadStack = pThreadStack;
    if(!StartThread(options.GetPriority(), options.GetIdealCoreNumber()))
    {
        return false;
    }

    if( !detail::driver::HardwareManager::GetInstance().RegisterRecorder( this ) )
    {
        return false;
    }

    m_State = State_Initialized;

    return true;
}

//---------------------------------------------------------------------------
size_t
DeviceOutRecorder::GetRequiredMemorySizeForRecording() NN_NOEXCEPT
{
    return RecordingBufferSize;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::Finalize() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return;
    }

    detail::driver::HardwareManager::GetInstance().UnregisterRecorder( this );

    if(m_State == State_Recording || m_State == State_Recorded)
    {
        Stop(true);
    }
    StopThread();

    m_WorkBuffer = nn::util::BytePtr(nullptr);
    m_RecordingBuffer.Clear();
    m_RecordingBuffer.Finalize();

    m_State = State_NotInitialized;

    m_ThreadStack = nullptr;
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::Start(detail::fnd::FileStream& fileStream, const RecordingOptions& options) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return false;
    }

    if(m_State == State_Recording)
    {
        return true;
    }

    m_OutputMode = SoundSystem::GetOutputMode();
    m_Channels = GetValidChannels(options.GetChannels());
    m_Stream = &fileStream;

    m_IsLeadSilenceTrimming = options.IsLeadSilenceTrimmingEnabled();
    m_MaxSamples = options.GetMaxFrames() * m_Channels;
    m_WriteBlockPerSamples = options.GetWriteBlockPerSamples();
    m_WrittenSamples = 0;

    m_RecordingBuffer.SetReadBlockSamples(GetReadBlockSamples(m_Channels));
    m_RecordingBuffer.Clear();

    if(Prepare() < 0)
    {
        Stop(false);
        return false;
    }

    OnStart();

    return true;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::Stop(bool isBlocking) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return;
    }

    State state = m_State;

    if(state == State_Recording || state == State_Recorded)
    {
        OnStop();
    }

    if(isBlocking)
    {
        m_MessageDoneEvent.Clear();
    }

    // メッセージキューを空にします。
    uintptr_t message;
    while (m_MessageQueue.TryReceive(&message))
    {
    }
    SendMessage(Message_RequestStop);

    if(isBlocking)
    {
        m_MessageDoneEvent.Wait();
    }

    m_Stream = NULL;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::RecordSamples(const int16_t* sampleBuffer, uint32_t samples) NN_NOEXCEPT
{
    if(!IsInitialized() || m_State != State_Recording)
    {
        return;
    }

    NN_SDK_ASSERT_NOT_NULL(sampleBuffer);
    m_RecordingBuffer.Push(sampleBuffer, samples);

    PostMessage(Message_WriteSamples);
}

//---------------------------------------------------------------------------
uint32_t
DeviceOutRecorder::Run(void* param) NN_NOEXCEPT
{
    NN_UNUSED(param);
    bool shouldExit = false;

    while(!shouldExit)
    {
        uintptr_t msgBuf;
        m_MessageQueue.Receive(&msgBuf);

        switch(msgBuf)
        {
        case Message_Prepare:
            m_MessageResult = OnPrepare();

            if(m_MessageResult >= 0)
            {
                m_State = State_Recording;
            }
            else
            {
                NN_ATK_WARNING("[DeviceOutRecorder] Prepare Failed.\n");
            }

            m_MessageDoneEvent.Signal();
            break;

        case Message_RequestStop:
            OnWriteSamples(true);
            OnRequestStop();
            m_State = State_Initialized;
            m_MessageDoneEvent.Signal();
            break;

        case Message_Exit:
            shouldExit = true;
            break;

        case Message_WriteSamples:
            if(!OnWriteSamples(false))
            {
                m_State = State_Recorded;
            }
            break;

        default:
            continue;
        }
    }

    NN_SDK_ASSERT(IsInitialized());
    OnExit();

    return 0;
}

//---------------------------------------------------------------------------
uint32_t
DeviceOutRecorder::GetReadBlockSamples(uint32_t channels) const NN_NOEXCEPT
{
    return channels * sizeof(int16_t);
}

//---------------------------------------------------------------------------
uint32_t
DeviceOutRecorder::GetLeadSilenceSamples(const int16_t* sampleBuffer, uint32_t samples, uint32_t channels) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(sampleBuffer);
    NN_SDK_ASSERT(channels > 0);

    uint32_t result = 0;

    for(uint32_t frameLeadIndex = 0; frameLeadIndex < samples; frameLeadIndex += channels)
    {
        bool isSilentFrame = true;

        // 全チャンネルが無音かどうかを調べます。
        for(uint32_t sampleOffset = 0; sampleOffset < channels; ++sampleOffset)
        {
            if(sampleBuffer[frameLeadIndex + sampleOffset] != 0)
            {
                isSilentFrame = false;
                break;
            }
        }

        if(!isSilentFrame)
        {
            break;
        }

        // 無音部分のサンプル数をフレーム単位で加算します。
        result += channels;
    }

    return result;
}

//---------------------------------------------------------------------------
uint32_t
DeviceOutRecorder::GetWritableSamples(uint32_t samples) const NN_NOEXCEPT
{
    if(m_MaxSamples == 0)
    {
        return samples;
    }

    if(IsNoMoreSamples())
    {
        return 0;
    }

    return std::min(samples, m_MaxSamples - m_WrittenSamples);
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::IsNoMoreSamples() const NN_NOEXCEPT
{
    return m_MaxSamples > 0 && m_MaxSamples <= m_WrittenSamples;
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::StartThread(uint32_t priority, int idealCoreNumber) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Thread.GetState() == detail::fnd::Thread::State_NotRun || m_Thread.GetState() == detail::fnd::Thread::State_Released);

    // メッセージキューを空にします。
    uintptr_t message;
    while(m_MessageQueue.TryReceive(&message))
    {
    }

    m_MessageDoneEvent.Clear();

    nn::atk::detail::fnd::Thread::RunArgs args;
    args.name = "nn::atk::DeviceOutRecorder";
    args.stack = m_ThreadStack;
    args.stackSize = RequiredThreadStackSize;
    args.idealCoreNumber = idealCoreNumber;
    args.priority = priority;
    args.param = nullptr;
    args.handler = this;

    if(!m_Thread.Run(args))
    {
        NN_ATK_WARNING("[DeviceOutRecorder] failed to StartThread.");
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::StopThread() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return;
    }

    // メッセージキューを空にします。
    uintptr_t message;
    while(m_MessageQueue.TryReceive(&message))
    {
    }
    SendMessage(Message_Exit);

    m_Thread.WaitForExit();
    NN_SDK_ASSERT(m_Thread.GetState() == detail::fnd::Thread::State_Exited);
    m_Thread.Release();
}

//---------------------------------------------------------------------------
int32_t
DeviceOutRecorder::Prepare() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return -1;
    }

    m_MessageDoneEvent.Clear();

    m_MessageQueue.Send(Message_Prepare);

    m_MessageDoneEvent.Wait();

    return m_MessageResult;
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::SendMessage(Message message) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return false;
    }

    m_MessageQueue.Send(message);

    return true;
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::PostMessage(Message message) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return false;
    }

    return m_MessageQueue.TrySend(message);
}

//---------------------------------------------------------------------------
int32_t
DeviceOutRecorder::OnPrepare() NN_NOEXCEPT
{
    if(m_Stream == NULL ||
        !m_Stream->IsOpened())
    {
        NN_ATK_WARNING("[DeviceOutRecorder] m_Stream is not opened.");
        return -1;
    }

    if(!m_WavOutStream.Open(*m_Stream, m_Channels, GetSamplesPerSec()))
    {
        NN_ATK_WARNING("[DeviceOutRecorder] Failed To Open WavOutStream.");
        return -1;
    }

    if(!m_WavOutStream.IsAvailable())
    {
        NN_ATK_WARNING("[DeviceOutRecorder] WavOutStream is not available.");
        return -1;
    }

    return 0;
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::OnRequestStop() NN_NOEXCEPT
{
    m_WavOutStream.Close();
}

//---------------------------------------------------------------------------
void
DeviceOutRecorder::OnExit() NN_NOEXCEPT
{
}

//---------------------------------------------------------------------------
bool
DeviceOutRecorder::OnWriteSamples(bool isForceWriteMode) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return false;
    }

    while(m_State == State_Recording)
    {
        uint32_t samples = m_RecordingBuffer.GetContiguousReadableCount();

        // IsRecording() の場合、
        // バッファ終端でなく、m_WriteBlockPerSamples に満たない場合は、
        // サンプルが十分に溜まるまで書き込みません。
        if(m_State == State_Recording &&
            samples == m_RecordingBuffer.GetReadableCount() &&
            samples < m_WriteBlockPerSamples &&
            !isForceWriteMode)
        {
            break;
        }
        int16_t* sampleBuffer = m_RecordingBuffer.Peek();

        if(sampleBuffer == NULL)
        {
            break;
        }

        uint32_t validSamples = samples;

        // 先頭の無音部分をトリミングします。
        if(m_IsLeadSilenceTrimming)
        {
            uint32_t silenceSamples = GetLeadSilenceSamples(sampleBuffer, samples, m_Channels);

            sampleBuffer += silenceSamples;
            validSamples -= silenceSamples;

            // 発音を確認したら以降はトリミングを行わないようにします。
            if(silenceSamples < samples)
            {
                m_IsLeadSilenceTrimming = false;
            }
        }

        if(validSamples > 0 && !IsNoMoreSamples())
        {
            // DMA バッファの内容を波形出力用に加工します。
            validSamples = OnProcessSamples(sampleBuffer, validSamples);

            // 書き込み可能なサンプル数を求めます。
            validSamples = GetWritableSamples(validSamples);

            if(!m_WavOutStream.IsAvailable())
            {
                return false;
            }

            // 加工した波形データを出力します。
            uint32_t writeLength = validSamples * sizeof(int16_t);
            size_t result = m_WavOutStream.Write(sampleBuffer, writeLength);

            if(result != static_cast<int32_t>(writeLength))
            {
                NN_ATK_WARNING("[DeviceOutRecorder] failed to WriteFile. : %d", result);
            }

            m_WrittenSamples += validSamples;
        }

        m_RecordingBuffer.Pop(samples);
    }

    return !IsNoMoreSamples();
}

} // atk
} // nn
