﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/io/atkfnd_StreamCache.h>

#include <algorithm>
#include <nn/atk/fnd/io/atkfnd_FileStream.h>
#include <nn/util/util_BytePtr.h>

#if !defined(NN_SDK_BUILD_RELEASE)
//#define NN_ATK_FND_STREAMIO_DEBUG
#endif

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

NN_DEFINE_STATIC_CONSTANT( const position_t StreamCache::InvalidPosition );

//---------------------------------------------------------------------------
StreamCache::StreamCache() NN_NOEXCEPT :
m_Stream(NULL),
m_CurrentPosition(0),
m_CacheBuffer(NULL),
m_CacheBufferLength(0),
m_CachePosition(InvalidPosition),
m_CachedLength(0),
m_CacheState(CacheState_None)
{

}

//---------------------------------------------------------------------------
StreamCache::StreamCache(Stream* sourceStream, void* buffer, size_t length) NN_NOEXCEPT :
m_Stream(sourceStream),
m_CurrentPosition(0),
m_CacheBuffer(buffer),
m_CacheBufferLength(length),
m_CachePosition(InvalidPosition),
m_CachedLength(0),
m_CacheState(CacheState_None)
{
    NN_SDK_ASSERT_NOT_NULL(sourceStream);
}

//---------------------------------------------------------------------------
void
StreamCache::Initialize(Stream* sourceStream, void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(sourceStream);
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT(length > 0);

    m_Stream = sourceStream;
    m_CurrentPosition = 0;
    m_CacheBuffer = buffer;
    m_CacheBufferLength = length;
    m_CachePosition = InvalidPosition;
    m_CachedLength = 0;
    m_CacheState = CacheState_None;
}

//---------------------------------------------------------------------------
void
StreamCache::Finalize() NN_NOEXCEPT
{
    m_Stream = NULL;
    m_CurrentPosition = 0;
    m_CacheBuffer = NULL;
    m_CacheBufferLength = 0;
    m_CachePosition = InvalidPosition;
    m_CachedLength = 0;
    m_CacheState = CacheState_None;
}

//---------------------------------------------------------------------------
size_t
StreamCache::Read(void* buf, size_t length, FndResult* result /*= NULL*/, FsAccessLog* log  /*= nullptr*/, void* pFileStream  /*= nullptr*/) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "StreamCache must be initialized.\n");
        if( log != nullptr )
        {
            log->BeginRead(pFileStream);
        }

        size_t readSize = m_Stream->Read(buf, length, result);

        if( log != nullptr )
        {
            log->EndRead(pFileStream);
        }

        return readSize;
    }

    // 読み込みにキャッシュを利用するので、
    // 書き込みキャッシュがあれば、フラッシュしておきます。
    FndResult flushResult = FlushWriteCache();

    if(flushResult.IsFailed())
    {
        if(result != NULL)
        {
            *result = flushResult;
        }

        return 0;
    }

    // キャッシュから読み込む
    size_t cacheHitLength = GetReadCacheHitLength(length);

    if (cacheHitLength > 0)
    {
#if defined(NN_ATK_FND_STREAMIO_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[%s][%08x] cache hit  : cache readBegin=%08x, length=%d.\n",
            __FUNCTION__, this, m_CurrentPosition, cacheHitLength);
#endif
        std::memcpy(
            buf,
            util::BytePtr(m_CacheBuffer, m_CurrentPosition - m_CachePosition).Get(),
            cacheHitLength);

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[%s][%08x] current pos : 0x%08x -> 0x%08x.\n",
            __FUNCTION__, this, m_CurrentPosition, m_CurrentPosition + cacheHitLength);
#endif

        m_CurrentPosition += cacheHitLength;
    }

    // キャッシュから読みきれた場合は、終了
    if(cacheHitLength >= length)
    {
        return length;
    }

    size_t restLength = length - cacheHitLength;

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[%s][%08x] cache miss : readBegin=0x%08x, restLength=%d\n",
        __FUNCTION__, this, m_CurrentPosition, restLength);
#endif

    // StreamCache の現在位置と対象 Stream の現在位置が違う場合はシークする
    FndResult syncResult = SyncStreamCurrentPosition(m_CurrentPosition);
    if(syncResult.IsFailed())
    {
        if(result != NULL)
        {
            *result = syncResult;
        }

        return 0;
    }

    size_t readLength = 0;

    // キャッシュバッファに収まる場合、キャッシュしてバッファにコピー
    if (restLength <= m_CacheBufferLength)
    {
        if( log != nullptr )
        {
            log->BeginRead(pFileStream);
        }

        FndResult readResult;
        readLength = m_Stream->Read(m_CacheBuffer, m_CacheBufferLength, &readResult);

        if( log != nullptr )
        {
            log->EndRead(pFileStream);
        }

        if(result != NULL)
        {
            *result = readResult;
        }

        if (readResult.IsFailed())
        {
            ClearCache();
            return cacheHitLength;
        }

        m_CacheState = CacheState_ForRead;
        m_CachePosition = m_CurrentPosition;
        m_CachedLength = readLength;

        std::memcpy(
            util::BytePtr(buf, cacheHitLength).Get(),
            m_CacheBuffer,
            std::min(restLength, readLength));

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[%s][%08x] cache : begin=0x%08x, length=%d.\n",
            __FUNCTION__, this, m_CachePosition, m_CachedLength);
#endif
    }
    // キャッシュに収まらない場合、直接バッファに読み込んでから、可能な限りキャッシュ
    else
    {
        if( log != nullptr )
        {
            log->BeginRead(pFileStream);
        }

        FndResult readResult;
        readLength = m_Stream->Read(buf, restLength, &readResult);

        if( log != nullptr )
        {
            log->EndRead(pFileStream);
        }

        if(result != NULL)
        {
            *result = readResult;
        }

        if (readResult.IsFailed())
        {
            ClearCache();
            return cacheHitLength;
        }

        m_CachedLength = std::min(readLength, m_CacheBufferLength);
        m_CachePosition = m_CurrentPosition - m_CachedLength;

        // int16_t の範囲を超えるキャッシュサイズは考慮していません。
        std::memcpy(
            m_CacheBuffer,
            util::BytePtr(buf, -static_cast<int16_t>(m_CachedLength)).Get(),
            m_CacheBufferLength);

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[%s][%08x] cache parts : begin=0x%08x, length=%d.\n",
            __FUNCTION__, this, m_CachePosition, m_CachedLength);
#endif
    }

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[%s][%08x] current pos : 0x%08x -> 0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, m_CurrentPosition + restLength);
#endif

    m_CurrentPosition += restLength;

    if(cacheHitLength + readLength < length)
    {
        return cacheHitLength + readLength;
    }

    // m_Stream をキャッシュし切った場合も、
    // 残りキャッシュがあれば、TRUE を返す。
    if(result != NULL)
    {
        *result = FndResult(FndResultType_True);
    }

    return length;
} // NOLINT(impl/function_size)

//---------------------------------------------------------------------------
size_t
StreamCache::Write(const void* buf, size_t length, FndResult* result /*= NULL*/) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "StreamCache must be initialized.\n");
        return m_Stream->Write(buf, length, result);
    }

    // キャッシュバッファサイズを超える場合は、直接書き込みます。
    // キャッシュに書き込みデータがあれば先に処理します。
    if(length > m_CacheBufferLength)
    {
        FndResult flushResult = FlushWriteCache();

        if(flushResult.IsFailed())
        {
            if(result != NULL)
            {
                *result = flushResult;
            }

            return 0;
        }

        return m_Stream->Write(buf, length, result);
    }

    // キャッシュバッファにコピーし書き込みを遅延します。
    // キャッシュの空き領域に収まらない場合は、先にフラッシュします。
    if(GetWritableCacheLength(length) < length)
    {
        FndResult flushResult = FlushWriteCache();

        if(flushResult.IsFailed())
        {
            if(result != NULL)
            {
                *result = flushResult;
            }

            return 0;
        }
    }

    std::memcpy(
        util::BytePtr(m_CacheBuffer, m_CachedLength).Get(),
        buf,
        length);

    if(m_CacheState == CacheState_ForWrite)
    {
        m_CachedLength += length;

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[%s][%08x] add write cache : cachePos=0x%08x, length=%d.\n",
            __FUNCTION__, this, m_CachePosition, m_CachedLength);
#endif
    }
    else
    {
        NN_SDK_ASSERT(m_CachedLength == 0);

        m_CacheState = CacheState_ForWrite;
        m_CachePosition = m_CurrentPosition;
        m_CachedLength = length;

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[%s][%08x] new write cache : cachePos=0x%08x, length=%d.\n",
            __FUNCTION__, this, m_CachePosition, m_CachedLength);
#endif
    }

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[%s][%08x] current pos : 0x%08x -> 0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, m_CurrentPosition + length);
#endif

    m_CurrentPosition += length;

    // キャッシュバッファが満タンならフラッシュ
    if(m_CachedLength == m_CacheBufferLength)
    {
        FndResult flushResult = FlushWriteCache();

        if(flushResult.IsFailed())
        {
            if(result != NULL)
            {
                *result = flushResult;
            }

            return length;
        }
    }

    if(result != NULL)
    {
        *result = FndResult(FndResultType_True);
    }

    return length;
}

//---------------------------------------------------------------------------
FndResult
StreamCache::Seek(position_t offset, Stream::SeekOrigin origin) NN_NOEXCEPT
{
    position_t fileEndPosition = static_cast<position_t>(m_Stream->GetSize());

    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "StreamCache must be initialized.\n");
        return m_Stream->Seek(offset, origin);
    }

    if(!m_Stream->CanSeek())
    {
        return FndResult(FndResultType_Failed);
    }

    // ファイルサイズを超えてシークできないようにしている
    switch (origin)
    {
    case Stream::SeekOrigin_Begin:
        if(offset < 0 || fileEndPosition <= offset)
        {
            return FndResult(FndResultType_Failed);
        }

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[%s][%08x] current pos : 0x%08x -> 0x%08x (SeekOrigin_Begin).\n",
            __FUNCTION__, this, m_CurrentPosition, offset);
#endif

        m_CurrentPosition = offset;
        break;

    case Stream::SeekOrigin_End:
        if(fileEndPosition < offset)
        {
            return FndResult(FndResultType_Failed);
        }

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[%s][%08x] current pos : 0x%08x -> 0x%08x (SeekOrigin_End).\n",
            __FUNCTION__, this, m_CurrentPosition, fileEndPosition - offset);
#endif

        m_CurrentPosition = fileEndPosition - offset;
        break;

    case Stream::SeekOrigin_Current:
        if(fileEndPosition <= m_CurrentPosition + offset)
        {
            return FndResult(FndResultType_Failed);
        }

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[%s][%08x] current pos : 0x%08x -> 0x%08x (SeekOrigin_Current).\n",
            __FUNCTION__, this, m_CurrentPosition, m_CurrentPosition + offset);
#endif

        m_CurrentPosition += offset;
        break;

    default:
        NN_SDK_ASSERT(false, "invalid seek origin.\n");
        return FndResult(FndResultType_Failed);
    }

    return FndResult(FndResultType_True);
}

//---------------------------------------------------------------------------
void
StreamCache::ClearCache() NN_NOEXCEPT
{
    FlushWriteCache();

    m_CachePosition = InvalidPosition;
    m_CachedLength = 0;
}

//---------------------------------------------------------------------------
size_t
StreamCache::GetReadCacheHitLength(size_t length) const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return 0;
    }

    if (m_CacheState != CacheState_ForRead)
    {
        return 0;
    }

    if (m_CachePosition == InvalidPosition)
    {
        return 0;
    }

    if (m_CurrentPosition < m_CachePosition)
    {
        return 0;
    }

    size_t offset = m_CurrentPosition - m_CachePosition;

    if (m_CachedLength < offset + length)
    {
        return m_CachedLength > offset ? m_CachedLength - offset : 0;
    }

    return length;
}

//---------------------------------------------------------------------------
size_t
StreamCache::GetWritableCacheLength(size_t length) const NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return 0;
    }

    // 読み込みにキャッシュを利用している場合、
    // キャッシュバッファ全体を利用可能とします。
    if (m_CacheState == CacheState_ForRead)
    {
        return std::min(length, m_CacheBufferLength);
    }

    return std::min(length, m_CacheBufferLength - m_CachedLength);
}

//---------------------------------------------------------------------------
FndResult
StreamCache::SyncStreamCurrentPosition(position_t position) NN_NOEXCEPT
{
    if(position == m_Stream->GetCurrentPosition())
    {
        return FndResult(FndResultType_False);
    }

    // 現在位置がずれるのは StreamCache::Seek() した場合のみを想定
    // シーク不可能な場合は、内部エラー扱いとし、0を返す
    if(!m_Stream->CanSeek())
    {
        return FndResult(FndResultType_Failed);
    }

    FndResult result = m_Stream->Seek(position, Stream::SeekOrigin_Begin);

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[%s][%08x] seek : curPos=0x%08x, cachePos=0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, m_CachePosition);
#endif

    return result;
}

//---------------------------------------------------------------------------
FndResult
StreamCache::FlushWriteCache() NN_NOEXCEPT
{
    if(m_CachedLength == 0 || m_CacheState != CacheState_ForWrite)
    {
        return FndResult(FndResultType_False);
    }

    // StreamCache の現在位置と対象 Stream の現在位置が違う場合はシークする
    FndResult result = SyncStreamCurrentPosition(m_CachePosition);
    if(result.IsFailed())
    {
        return result;
    }

#if defined(NN_ATK_FND_STREAMIO_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[%s][%08x] flush write cache : pos=0x%08x, length=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, m_CachedLength);
#endif

    result = FndResult(FndResultType_True);
    m_Stream->Write(m_CacheBuffer, m_CachedLength, &result);

    m_CacheState = CacheState_None;
    m_CachePosition = InvalidPosition;
    m_CachedLength = 0;

    return result;
}

} // namespace nn::atk::detail::fnd
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
