﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/io/atkfnd_FileStreamImpl.h>

#include <nn/util/util_BytePtr.h>

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

//---------------------------------------------------------------------------
size_t
FileStreamImpl::Read(void* buf, size_t length, FndResult* result /*= NULL*/) NN_NOEXCEPT
{
    if(IsCacheEnabled())
    {
        return m_StreamCache.Read(buf, length, result, m_pAccessLog, this);
    }

    return ReadDirect(buf, length, result);
}

//---------------------------------------------------------------------------
size_t
FileStreamImpl::Write(const void* buf, size_t length, FndResult* result /*= NULL*/) NN_NOEXCEPT
{
    if(IsCacheEnabled())
    {
        return m_StreamCache.Write(buf, length, result);
    }

    return WriteDirect(buf, length, result);
}

//---------------------------------------------------------------------------
FndResult
FileStreamImpl::Seek(position_t offset, SeekOrigin origin) NN_NOEXCEPT
{
    if(IsCacheEnabled())
    {
        return m_StreamCache.Seek(offset, origin);
    }

    return SeekDirect(offset, origin);
}

//---------------------------------------------------------------------------
bool
FileStreamImpl::CanRead() const NN_NOEXCEPT
{
    return IsOpened();
}

//---------------------------------------------------------------------------
bool
FileStreamImpl::CanWrite() const NN_NOEXCEPT
{
    return IsOpened();
}

//---------------------------------------------------------------------------
bool
FileStreamImpl::CanSeek() const NN_NOEXCEPT
{
    return IsOpened();
}

//---------------------------------------------------------------------------
void
FileStreamImpl::EnableCache(void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT(length > 0);

    if(m_StreamCache.IsInitialized())
    {
        m_StreamCache.Finalize();
    }

    void* alignedBuffer = util::BytePtr(buffer).AlignUp(GetIoBufferAlignment()).Get();

    m_StreamCache.Initialize(
        &m_DirectStream,
        alignedBuffer,
        length - util::BytePtr(buffer).Distance(alignedBuffer));
}

//---------------------------------------------------------------------------
void
FileStreamImpl::DisableCache() NN_NOEXCEPT
{
    m_StreamCache.Finalize();
}

//---------------------------------------------------------------------------
void
FileStreamImpl::ValidateAlignment(const void* buf) const NN_NOEXCEPT
{
    if(util::ConstBytePtr(buf).AlignUp(GetIoBufferAlignment()).Get() != buf)
    {
        NN_SDK_ASSERT(false,
            "invalid alignment : buf(0x%08x) must be aligned by %d.\n",
            buf,
            GetIoBufferAlignment());
    }
}

} // namespace nn::atk::detail::fnd
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
