﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_WaveSound.h>

#include <nn/atk/atk_SoundInstanceManager.h>
#include <nn/atk/atk_WaveFileReader.h>
#include <nn/atk/atk_DriverCommand.h>

namespace nn {
namespace atk {
namespace detail {

/* ========================================================================
        member function
   ======================================================================== */

/*--------------------------------------------------------------------------------*
  Name:         WaveSound

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
WaveSound::WaveSound( WaveSoundInstanceManager& manager ) NN_NOEXCEPT
: m_Manager( manager )
, m_InitializeFlag( false )
, m_IsCalledPrepare( false )
, m_ChannelCount( 0 )
{
}

bool WaveSound::Initialize(OutputReceiver* pOutputReceiver) NN_NOEXCEPT
{
    bool result = BasicSound::Initialize( pOutputReceiver );
    if( !result )
    {
        return false;
    }

    m_pTempSpecialHandle = NULL;
    m_pWaveFile = NULL;
    m_WaveType = WaveType_Invalid;

    m_IsCalledPrepare = false;
    m_InitializeFlag = true;
    m_ChannelCount = 0;

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         Shutdown

  Description:  サウンドの終了

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void WaveSound::Finalize() NN_NOEXCEPT
{
    if ( ! m_InitializeFlag )
    {
        return;
    }

    m_InitializeFlag = false;
    m_IsCalledPrepare = false;
    m_ChannelCount = 0;

    BasicSound::Finalize();
    m_Manager.Free( this );
}

/*--------------------------------------------------------------------------------*
  Name:         Prepare

  Description:  サウンド開始

  Arguments:    seqDataBase   - シーケンスデータベースアドレス
                seqDataOffset - シーケンスデータのオフセット
                bank          - バンクデータ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void WaveSound::Prepare(
        const void* wsdFile,
        const void* waveFile,
        const driver::WaveSoundPlayer::StartInfo& startInfo,
        int8_t waveType) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( wsdFile );
    NN_SDK_ASSERT( m_InitializeFlag );

    // コマンド
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandWaveSoundPrepare* command =
            cmdmgr.AllocCommand<DriverCommandWaveSoundPrepare>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_WsdPrepare;
        command->player = &m_PlayerInstance;
        command->startInfo = startInfo; // 構造体コピー
        command->arg.wsdFile = wsdFile;
        command->arg.waveFile = waveFile;
        command->arg.waveType = waveType;
        cmdmgr.PushCommand(command);
    }

    m_WaveType = waveType;
    m_pWaveFile = waveFile;
    m_IsCalledPrepare = true;

    WaveFileReader reader( m_pWaveFile, m_WaveType );
    WaveInfo waveInfo;
    if ( reader.ReadWaveInfo( &waveInfo ) )
    {
        m_ChannelCount = static_cast<uint32_t>( std::min( waveInfo.channelCount, 2 ) );
    }
}

void WaveSound::RegisterDataLoadTask(
        const driver::WaveSoundLoader::LoadInfo& loadInfo,
        const driver::WaveSoundPlayer::StartInfo& startInfo ) NN_NOEXCEPT
{
    // コマンド
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandWaveSoundLoad* command =
            cmdmgr.AllocCommand<DriverCommandWaveSoundLoad>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_WsdLoad;
        command->player = &m_PlayerInstance;
        command->startInfo = startInfo; // 構造体コピー
        command->arg.soundDataManager = loadInfo.soundDataManager;
        command->arg.soundArchive = loadInfo.soundArchive;
        command->arg.soundPlayer = loadInfo.soundPlayer;
        command->arg.loadInfoWsd = *(loadInfo.loadInfoWsd); // 構造体コピー
        command->arg.index = startInfo.index;

        cmdmgr.PushCommand(command);
    }
}

void WaveSound::SetChannelPriority( int priority ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( priority >= nn::atk::ChannelPriorityMin && priority <= nn::atk::ChannelPriorityMax );
    uint8_t uint8_t_prio = static_cast<uint8_t>(priority);
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandWaveSoundChannelPrio* command =
            cmdmgr.AllocCommand<DriverCommandWaveSoundChannelPrio>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_WsdChannelPrio;
        command->player = &m_PlayerInstance;
        command->priority = uint8_t_prio;
        cmdmgr.PushCommand(command);
    }
}

void WaveSound::InitializeChannelParam(int priority, bool isReleasePriorityFix) NN_NOEXCEPT
{
    NN_SDK_ASSERT( priority >= nn::atk::ChannelPriorityMin && priority <= nn::atk::ChannelPriorityMax );
    uint8_t uint8_t_prio = static_cast<uint8_t>(priority);

    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandWaveSoundChannelParam* command =
            cmdmgr.AllocCommand<DriverCommandWaveSoundChannelParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_WsdChannelParam;
        command->player = &m_PlayerInstance;
        command->priority = uint8_t_prio;
        command->isReleasePriorityFix = isReleasePriorityFix;
        cmdmgr.PushCommand(command);
    }
}

/*--------------------------------------------------------------------------------*
  Name:         OnUpdatePlayerPriority

  Description:  プレイヤープライオリティを変更

  Arguments:

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void WaveSound::OnUpdatePlayerPriority() NN_NOEXCEPT
{
    int priority = CalcCurrentPlayerPriority();
    m_Manager.UpdatePriority( this, priority );
}

bool WaveSound::IsAttachedTempSpecialHandle() NN_NOEXCEPT
{
    return m_pTempSpecialHandle != NULL;
}

void WaveSound::DetachTempSpecialHandle() NN_NOEXCEPT
{
    // TODO: m_pTempSpecialHandle->DetachSound();
}

bool WaveSound::ReadWaveSoundDataInfo( WaveSoundDataInfo* info ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( info );

    const void* waveFile = m_pWaveFile;
    if (waveFile == NULL)
    {
        if (IsPlayerAvailable() == true)
        {
            waveFile = m_PlayerInstance.GetWaveFile();
        }
    }

    if (waveFile == NULL)
    {
        return false;
    }

    if (m_WaveType != WaveType_Nwwav)
    {
        return false;
    }

    WaveFileReader reader( m_pWaveFile );
    WaveInfo waveInfo;
    if ( ! reader.ReadWaveInfo( &waveInfo ) )
    {
        return false;
    }
    info->loopFlag = ( waveInfo.loopFlag != 0 );
    info->sampleRate = static_cast<int>( waveInfo.sampleRate );
    info->loopStart = waveInfo.originalLoopStartFrame;
    info->loopEnd = waveInfo.loopEndFrame - (waveInfo.loopStartFrame - waveInfo.originalLoopStartFrame);
    info->compatibleLoopStart = waveInfo.loopStartFrame;
    info->compatibleLoopEnd = waveInfo.loopEndFrame;
    info->channelCount = std::min( waveInfo.channelCount, 2 );

    return true;
}

position_t WaveSound::GetPlaySamplePosition(bool isOriginalSamplePosition) const NN_NOEXCEPT
{
    if ( ! IsPlayerAvailable() ) return 0;

    // TODO: ロック区間の短縮を検討する
    driver::AtkStateAndParameterUpdateLock lock;
    return m_PlayerInstance.GetPlaySamplePosition(isOriginalSamplePosition);
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

