﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_VoiceCommand.h>
#include <nn/atk/atk_Config.h>
#include <nn/atk/atk_Voice.h>
#include <nn/atk/atk_HardwareManager.h>

namespace nn {
namespace atk {
namespace detail {

//===========================================================================
//
// LowLevelVoiceCommand
//
//===========================================================================
LowLevelVoiceCommand& LowLevelVoiceCommand::GetInstance() NN_NOEXCEPT
{
    static LowLevelVoiceCommand instance;
    return instance;
}

void LowLevelVoiceCommand::Initialize( void* buffer, size_t bufferSize, size_t commandBufferSize, int queueCount, void* waveBuffer, size_t waveBufferSize, int waveBufferPacketCount ) NN_NOEXCEPT
{
    NN_UNUSED( waveBufferSize );
    NN_SDK_REQUIRES_NOT_NULL( waveBuffer );
    NN_SDK_REQUIRES_GREATER_EQUAL( waveBufferSize, GetRequiredWaveBufferMemSize(waveBufferPacketCount) );
    NN_SDK_ASSERT_GREATER_EQUAL( waveBufferPacketCount, 0 );

    //  WaveBufferPacket の AdpcmContext を nn::audio::BufferAlignSize アラインします
    NN_SDK_REQUIRES_ALIGNED( waveBuffer, nn::audio::BufferAlignSize );
    NN_STATIC_ASSERT( sizeof( WaveBufferPacket ) % nn::audio::BufferAlignSize == 0 );

    m_WaveBufferPacketCount = waveBufferPacketCount;
    m_pWaveBufferPacket = reinterpret_cast<WaveBufferPacket*>( waveBuffer );
    for ( auto i = 0; i < m_WaveBufferPacketCount; i++ )
    {
        m_pWaveBufferPacket[i].adpcmContext = AdpcmContext();
        m_pWaveBufferPacket[i].waveBuffer.Initialize();
    }
    CommandManager::Initialize( buffer, bufferSize, commandBufferSize, queueCount, ProcessCommandList );
}

size_t LowLevelVoiceCommand::GetRequiredWaveBufferMemSize(int waveBufferPacketCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL( waveBufferPacketCount, 0 );
    return waveBufferPacketCount * sizeof( WaveBufferPacket );
}

void LowLevelVoiceCommand::LowLevelVoiceDisposeCallback( LowLevelVoice* /* voice */, void* arg ) NN_NOEXCEPT
{
    // TODO: キャストは clang ビルドエラー回避のための暫定対応
    uint32_t voiceId = static_cast<uint32_t>(reinterpret_cast<uint64_t>(arg));
    VirtualVoiceManager::GetInstance().SetLowLevelVoice( voiceId, nullptr );
}

LowLevelVoiceCommand::WaveBufferPacket* LowLevelVoiceCommand::GetFreeWaveBuffer() NN_NOEXCEPT
{
    for ( int i = 0; i < m_WaveBufferPacketCount; i++ )
    {
        WaveBufferPacket* packet = &m_pWaveBufferPacket[i];
        if ( packet->waveBuffer.status == WaveBuffer::Status_Free ||
             packet->waveBuffer.status == WaveBuffer::Status_Done )
        {
            return packet;
        }
    }
    return nullptr;
}

void LowLevelVoiceCommand::ProcessCommandList( Command* commandList ) NN_NOEXCEPT
{
    Command* command = commandList;

    while( command != nullptr )
    {
        switch( command->id )
        {
        case VoiceCommandId_AllocVoice:
        {
            VoiceCommandAlloc* c = reinterpret_cast<VoiceCommandAlloc*>(command);
            LowLevelVoice* lowLevelVoice = driver::HardwareManager::GetInstance().GetLowLevelVoiceAllocator().AllocVoice();
            if ( lowLevelVoice == nullptr ) {
                break;
            }

            lowLevelVoice->SetPriority( c->priority );
            lowLevelVoice->SetVoice( reinterpret_cast<Voice*>(c->userId) );

            VirtualVoiceManager::GetInstance().SetLowLevelVoice( c->voiceId, lowLevelVoice );
            break;
        }
        case VoiceCommandId_FreeVoice:
        {
            VoiceCommandFree* c = reinterpret_cast<VoiceCommandFree*>(command);
            LowLevelVoice* lowLevelVoice = VirtualVoiceManager::GetInstance().GetLowLevelVoice( c->voiceId );
            if ( lowLevelVoice == nullptr ) break;

            driver::HardwareManager::GetInstance().GetLowLevelVoiceAllocator().FreeVoice(lowLevelVoice);
            VirtualVoiceManager::GetInstance().SetLowLevelVoice( c->voiceId, nullptr );
            break;
        }
        case VoiceCommandId_SetPriority:
        {
            VoiceCommandPriority* c = reinterpret_cast<VoiceCommandPriority*>(command);
            LowLevelVoice* lowLevelVoice = VirtualVoiceManager::GetInstance().GetLowLevelVoice( c->voiceId );
            if ( lowLevelVoice == nullptr ) break;

            lowLevelVoice->SetPriority( c->priority );
            break;
        }
        case VoiceCommandId_Play:
        {
            VoiceCommandPlay* c = reinterpret_cast<VoiceCommandPlay*>(command);
            LowLevelVoice* lowLevelVoice = VirtualVoiceManager::GetInstance().GetLowLevelVoice( c->voiceId );
            if ( lowLevelVoice == nullptr ) break;

            lowLevelVoice->SetSampleRate( c->sampleRate );
            lowLevelVoice->SetSampleFormat( c->sampleFormat );
            lowLevelVoice->SetAdpcmParam( c->adpcmParam );
            lowLevelVoice->SetOutputReceiver( c->pOutputReceiver );
            lowLevelVoice->SetState( VoiceState_Play );
            break;
        }
        case VoiceCommandId_Pause:
        {
            VoiceCommandPause* c = reinterpret_cast<VoiceCommandPause*>(command);
            LowLevelVoice* lowLevelVoice = VirtualVoiceManager::GetInstance().GetLowLevelVoice( c->voiceId );
            if ( lowLevelVoice == nullptr ) break;

            lowLevelVoice->SetState( VoiceState_Pause );
            break;
        }
        case VoiceCommandId_Stop:
        {
            VoiceCommandFree* c = reinterpret_cast<VoiceCommandFree*>(command);
            LowLevelVoice* lowLevelVoice = VirtualVoiceManager::GetInstance().GetLowLevelVoice( c->voiceId );
            if ( lowLevelVoice == nullptr ) break;

            lowLevelVoice->FreeAllWaveBuffer();
            lowLevelVoice->SetState( VoiceState_Stop );
            break;
        }
        case VoiceCommandId_AppendWaveBuffer:
        {
            VoiceCommandAppendWaveBuffer* c = reinterpret_cast<VoiceCommandAppendWaveBuffer*>(command);
            LowLevelVoice* lowLevelVoice = VirtualVoiceManager::GetInstance().GetLowLevelVoice( c->voiceId );
            if ( lowLevelVoice == nullptr ) break;

            WaveBufferPacket* packet = GetInstance().GetFreeWaveBuffer();
            if ( packet == nullptr ) break;

            WaveBuffer* waveBuffer = &(packet->waveBuffer);
            waveBuffer->bufferAddress = c->bufferAddress;
            waveBuffer->bufferSize = c->bufferSize;
            waveBuffer->sampleLength = c->sampleLength;
            waveBuffer->sampleOffset = c->sampleOffset;
            if ( c->adpcmContextEnable )
            {
                packet->adpcmContext = c->adpcmContext;
                waveBuffer->pAdpcmContext = &(packet->adpcmContext);
            }
            else
            {
                waveBuffer->pAdpcmContext = nullptr;
            }
            waveBuffer->loopFlag = c->loopFlag;
            waveBuffer->userParam = c->tag;
            lowLevelVoice->AppendWaveBuffer( waveBuffer );
            break;
        }
        case VoiceCommandId_UpdateParam:
        {
            VoiceCommandParam* c = reinterpret_cast<VoiceCommandParam*>(command);
            LowLevelVoice* lowLevelVoice = VirtualVoiceManager::GetInstance().GetLowLevelVoice( c->voiceId );
            if ( lowLevelVoice == nullptr ) break;

            lowLevelVoice->SetVoiceParam( c->voiceParam );
            break;
        }
        default:
            break;

        } // switch

        command = command->next;
    } // while

} // NOLINT(impl/function_size)

//===========================================================================
//
// VoiceReplyCommand
//
//===========================================================================

VoiceReplyCommand& VoiceReplyCommand::GetInstance() NN_NOEXCEPT
{
    static VoiceReplyCommand instance;
    return instance;
}

void VoiceReplyCommand::ProcessCommandList( Command* commandList ) NN_NOEXCEPT
{
    Command* command = commandList;

    while( command != NULL )
    {
        switch( command->id )
        {
        case VoiceReplyCommandId_WaveBufferUpdate:
        {
            break;
        }
        default:
            break;
        } // switch

        command = command->next;
    } // while

}


}}}
