﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_WavOutFileStream.h>
#include <nn/atk/atk_WavBinary.h>
#include <nn/atk/detail/atk_Macro.h>

namespace nn {
namespace atk {
namespace detail {

NN_DEFINE_STATIC_CONSTANT( const int WavOutFileStream::FileIoBufferAlignment );

//---------------------------------------------------------------------------
WavOutFileStream::WavOutFileStream() NN_NOEXCEPT :
m_pFileStream(NULL),
m_WaveDataSize(0),
m_IsWaveDataSizeCalculating(false),
m_Buffer(NULL),
m_BufferLength(0),
m_ValidBufferLength(0)
{
}

//---------------------------------------------------------------------------
WavOutFileStream::~WavOutFileStream() NN_NOEXCEPT
{
}

//---------------------------------------------------------------------------
bool
WavOutFileStream::Open(fnd::FileStream& stream, int channels, size_t samplesPerSec) NN_NOEXCEPT
{
    NN_SDK_ASSERT(stream.IsOpened());

    m_pFileStream = &stream;

    if(!WriteHeader(channels, samplesPerSec))
    {
        return false;
    }

    // data チャンクのサイズ計算を開始します。
    m_WaveDataSize = 0;
    m_IsWaveDataSizeCalculating = true;

    return true;
}

//---------------------------------------------------------------------------
void
WavOutFileStream::Close() NN_NOEXCEPT
{
    if(!IsAvailable())
    {
        return;
    }

    // 閉じる前にバッファの残りを書き出します。
    FlushBuffer();

    m_IsWaveDataSizeCalculating = false;

    // 正確な出力サイズを取得するために、Flush() します。
    FlushBuffer();

    // ヘッダサイズ分を出力できていない場合は、ヘッダの更新を中止します。
    if(m_pFileStream->GetSize() < CalcRiffChunkSize(0))
    {
        m_pFileStream = NULL;
        return;
    }

    if(!UpdateRiffChunkSize())
    {
        m_pFileStream = NULL;
        return;
    }

    if(!UpdateDataChunkSize())
    {
        m_pFileStream = NULL;
        return;
    }

    m_pFileStream = NULL;
    m_WaveDataSize = 0;
}

//---------------------------------------------------------------------------
size_t
WavOutFileStream::Write(const void* buf, size_t length) NN_NOEXCEPT
{
    if(!IsAvailable())
    {
        return 0;
    }

    size_t writtenLength = 0;

    while(writtenLength < length)
    {
        const size_t restLength = length - writtenLength;

        if(restLength > m_BufferLength)
        {
            fnd::FndResult result;
            size_t writtenBytes = WriteDirect(buf, restLength, &result);

            if (result.IsFailed())
            {
                return 0;
            }

            writtenLength += writtenBytes;
            continue;
        }

        size_t copyLength = std::min(restLength, m_BufferLength - m_ValidBufferLength);

        memcpy(
            util::BytePtr(m_Buffer, m_ValidBufferLength).Get(),
            util::ConstBytePtr(buf, length - restLength).Get(),
            copyLength);

        m_ValidBufferLength += copyLength;
        writtenLength += copyLength;

        if(m_ValidBufferLength == m_BufferLength)
        {
            FlushBuffer();
        }
    }

    if(m_IsWaveDataSizeCalculating)
    {
        m_WaveDataSize += length;
    }

    return length;
}

//---------------------------------------------------------------------------
bool
WavOutFileStream::Seek(position_t offset, fnd::Stream::SeekOrigin origin) NN_NOEXCEPT
{
    if(!m_pFileStream->CanSeek())
    {
        return false;
    }

    // シーク前にバッファの残りを書き出します。
    FlushBuffer();

    return m_pFileStream->Seek(offset, origin).IsSucceeded();
}

//---------------------------------------------------------------------------
void
WavOutFileStream::SetCacheBuffer(char* buf, size_t length) NN_NOEXCEPT
{
    m_Buffer = buf;
    m_BufferLength = length;
}

//---------------------------------------------------------------------------
bool
WavOutFileStream::WriteHeader(int channels, size_t samplesPerSec) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsAvailable());

    char buffer[sizeof(WaveBinaryHeader) + FileIoBufferAlignment];
    void* alignedBuffer = util::BytePtr(buffer).AlignUp(FileIoBufferAlignment).Get();

    WaveBinaryHeader* header = new(alignedBuffer) WaveBinaryHeader();

    header->riffChunk.header.size = static_cast<fnd::PcBinU32>(CalcRiffChunkSize(0));

    header->fmtChunk.formatTag      = FmtChunk::FormatPcm;
    header->fmtChunk.channels       = static_cast<fnd::PcBinU16>(channels);
    header->fmtChunk.samplesPerSec  = static_cast<fnd::PcBinU32>(samplesPerSec);
    header->fmtChunk.bitsPerSample  = 16;
    header->fmtChunk.blockAlign     = static_cast<uint16_t>(header->fmtChunk.channels * (header->fmtChunk.bitsPerSample / 8));
    header->fmtChunk.avgBytesPerSec = header->fmtChunk.samplesPerSec * header->fmtChunk.blockAlign;

    size_t result = Write(alignedBuffer, sizeof(WaveBinaryHeader));
    return result == sizeof(WaveBinaryHeader);
}

//---------------------------------------------------------------------------
bool
WavOutFileStream::UpdateRiffChunkSize() NN_NOEXCEPT
{
    if(!IsAvailable())
    {
        return false;
    }

    static const position_t riffChunkSizePosition = 4;

    if(!Seek(riffChunkSizePosition, fnd::Stream::SeekOrigin::SeekOrigin_Begin))
    {
        NN_ATK_WARNING("[WavOutFileStream] failed to update wave header.");
        return false;
    }

    char buffer[sizeof(fnd::PcBinU32) + FileIoBufferAlignment];
    void* alignedBuffer = util::BytePtr(buffer).AlignUp(FileIoBufferAlignment).Get();

    fnd::PcBinU32& size = *new(alignedBuffer) fnd::PcBinU32();

    size = static_cast<fnd::PcBinU32>(CalcRiffChunkSize(m_WaveDataSize));

    fnd::FndResult result;
    WriteDirect(&size, sizeof(size), &result);
    if (result.IsFailed())
    {
        NN_ATK_WARNING("[WavOutFileStream] failed to update wave header.");
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
bool
WavOutFileStream::UpdateDataChunkSize() NN_NOEXCEPT
{
    if(!IsAvailable())
    {
        return false;
    }

    static const int32_t dataChunkSizePosition = sizeof(RiffChunk) + sizeof(FmtChunk) + 4;

    if(!Seek(dataChunkSizePosition, fnd::Stream::SeekOrigin::SeekOrigin_Begin))
    {
        NN_ATK_WARNING("[WavOutFileStream] failed to update wave header.");
        return false;
    }

    uint8_t buffer[sizeof(fnd::PcBinU32) + FileIoBufferAlignment];
    void* alignedBuffer = util::BytePtr(buffer).AlignUp(FileIoBufferAlignment).Get();

    fnd::PcBinU32& size = *new(alignedBuffer) fnd::PcBinU32();

    size = static_cast<fnd::PcBinU32>(m_WaveDataSize);

    fnd::FndResult result;
    WriteDirect(&size, sizeof(size), &result);

    if (result.IsFailed())
    {
        NN_ATK_WARNING("[WavOutFileStream] failed to update wave header.");
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
size_t
WavOutFileStream::CalcRiffChunkSize(size_t dataSize) NN_NOEXCEPT
{
    return
        sizeof(RiffChunk)   // RiffChunk サイズ
        - 8                 // RiffChunk の chunkID, chunkSize 分
        + sizeof(FmtChunk)  // FmtChunk サイズ
        + sizeof(DataChunk) // DataChunk サイズ
        + dataSize;         // データサイズ
}

//---------------------------------------------------------------------------
size_t
WavOutFileStream::WriteDirect(const void* buf, size_t length, fnd::FndResult* pResult) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pResult);

    if(!IsAvailable())
    {
        // IOStream の実装次第でエラーコードが変わるので、とりあえず -1 を返しています。
        return 0;
    }

    // キャッシュは存在しないはず
    NN_SDK_ASSERT(m_ValidBufferLength == 0);

    const void* alignedBuffer = util::ConstBytePtr(buf).AlignUp(FileIoBufferAlignment).Get();

    ptrdiff_t unAlignedLength = util::ConstBytePtr(buf).Distance(alignedBuffer);
    size_t alignedBufferLength = length - unAlignedLength;

    size_t writtenBytes = 0;
    if(unAlignedLength > 0)
    {
        // アライメント調整されていない場合は、２回に分けて出力します。
        char bufferForAlignment[FileIoBufferAlignment + FileIoBufferAlignment];
        char* alignedBufferForAlignment =
            static_cast<char*>(util::BytePtr(bufferForAlignment).AlignUp(FileIoBufferAlignment).Get());

        memcpy(alignedBufferForAlignment, buf, unAlignedLength);

        writtenBytes = m_pFileStream->Write(alignedBufferForAlignment, unAlignedLength, pResult);

        if (pResult->IsFailed())
        {
            return 0;
        }

        if( writtenBytes != static_cast<size_t>(unAlignedLength) )
        {
            return 0;
        }
    }

    writtenBytes = m_pFileStream->Write(alignedBuffer, alignedBufferLength, pResult);

    if (pResult->IsFailed())
    {
        return 0;
    }

    return writtenBytes + unAlignedLength;
}

//---------------------------------------------------------------------------
size_t
WavOutFileStream::FlushBuffer() NN_NOEXCEPT
{
    size_t length = m_ValidBufferLength;
    m_ValidBufferLength = 0;

    if(length == 0)
    {
        return 0;
    }

    fnd::FndResult result;
    return WriteDirect(m_Buffer, length, &result);
}

} // detail
} // atk
} // nn
