﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_TaskManager.h>

#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>

namespace nn {
namespace atk {
namespace detail {

NN_DEFINE_STATIC_CONSTANT( const int TaskManager::PriorityCount );
NN_DEFINE_STATIC_CONSTANT( const int TaskManager::ThreadMessageBufferSize );

/*--------------------------------------------------------------------------------*
  Name:         GetInstance

  Description:  シングルトンのインスタンスを取得する

  Arguments:    なし

  Returns:      インスタンス
 *--------------------------------------------------------------------------------*/
TaskManager& TaskManager::GetInstance() NN_NOEXCEPT
{
    static TaskManager instance;
    return instance;
}

/*--------------------------------------------------------------------------------*
  Name:         TaskManager

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
TaskManager::TaskManager() NN_NOEXCEPT
: m_IsWaitTaskCancel( false )
, m_BlockingQueue(m_MsgBuffer,ThreadMessageBufferSize)
{
}

void TaskManager::Initialize(bool isEnableProfiling) NN_NOEXCEPT
{
    m_TaskProfileLogger.SetProfilingEnabled( isEnableProfiling );
}

void TaskManager::Finalize() NN_NOEXCEPT
{
    m_TaskProfileLogger.SetProfilingEnabled( false );
    m_TaskProfileLogger.Finalize();
}

/*--------------------------------------------------------------------------------*
  Name:         AppendTask

  Description:  タスクリストにタスクを登録

  Arguments:    task - 登録するタスク
                priority - タスクを登録するリストのプライオリティ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void TaskManager::AppendTask( Task* task, TaskPriority priority ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( priority, 0, PriorityCount );


    task->m_Event.Clear();
    task->m_Status = Task::Status_Append;

    {
        fnd::ScopedLock<fnd::CriticalSection> scopeLock(m_CriticalSection);
        m_TaskList[ priority ].push_back( *task );
    }

    m_BlockingQueue.TrySend(Message_Append);
}

/*--------------------------------------------------------------------------------*
  Name:         GetNextTask

  Description:  次のタスクを取得する

  Arguments:    priority - タスクを抜き出すリストのプライオリティ
                doRemove - リストから削除するかどうか

  Returns:      取得したタスク
 *--------------------------------------------------------------------------------*/
Task* TaskManager::GetNextTask( TaskPriority priority, bool doRemove ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( priority, 0, PriorityCount );

    fnd::ScopedLock<fnd::CriticalSection> scopeLock(m_CriticalSection);

    if ( m_TaskList[ priority ].empty() ) return nullptr;

    Task* task = &m_TaskList[ priority ].front();
    if ( doRemove ) {
        m_TaskList[ priority ].pop_front();
    }
    return task;
}

/*--------------------------------------------------------------------------------*
  Name:         PopTask

  Description:  最優先のタスクを取り出す

  Arguments:    なし

  Returns:      取り出したタスク
 *--------------------------------------------------------------------------------*/
Task* TaskManager::PopTask() NN_NOEXCEPT
{
    fnd::ScopedLock<fnd::CriticalSection> scopeLock(m_CriticalSection);

    Task* task = GetNextTask( TaskPriority_High, true );
    if ( task != nullptr ) return task;

    task = GetNextTask( TaskPriority_Middle, true );
    if ( task != nullptr ) return task;

    task = GetNextTask( TaskPriority_Low, true );
    if ( task != nullptr ) return task;

    return nullptr;
}

/*--------------------------------------------------------------------------------*
  Name:         PeekTask

  Description:  最優先のタスクを取得する（リストから削除しない）

  Arguments:    なし

  Returns:      取得したタスク
 *--------------------------------------------------------------------------------*/
Task* TaskManager::PeekTask() NN_NOEXCEPT
{
    Task* task = GetNextTask( TaskPriority_High, false );
    if ( task != nullptr ) return task;

    task = GetNextTask( TaskPriority_Middle, false );
    if ( task != nullptr ) return task;

    task = GetNextTask( TaskPriority_Low, false );
    if ( task != nullptr ) return task;

    return nullptr;
}

/*--------------------------------------------------------------------------------*
  Name:         ExecuteTask

  Description:  全てのタスクを実行する

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void TaskManager::ExecuteTask() NN_NOEXCEPT
{
    for ( ;; )
    {
        Task* task = PopTask();
        if ( task == nullptr ) break;

        task->Execute( m_TaskProfileLogger );
        task->m_Status = Task::Status_Done;
        task->m_Event.Signal();
    }
}

/*--------------------------------------------------------------------------------*
  Name:         CancelTask

  Description:  タスクをキャンセルする

  Arguments:    task - キャンセルするタスク

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void TaskManager::CancelTask( Task* task ) NN_NOEXCEPT
{
    if ( TryRemoveTask(task) ) {
        return;
    }

    task->m_Event.Wait();
}

/*--------------------------------------------------------------------------------*
  Name:         CancelTaskById

  Description:  ID指定で、タスクをキャンセルする

  Arguments:    id - キャンセルするタスクのID

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void TaskManager::CancelTaskById( uint32_t id ) NN_NOEXCEPT
{
    RemoveTaskById(id);
}

/*--------------------------------------------------------------------------------*
  Name:         CancelAllTask

  Description:  全タスクをキャンセルする

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void TaskManager::CancelAllTask() NN_NOEXCEPT
{
    // TODO
}

/*--------------------------------------------------------------------------------*
  Name:         RemoveTask

  Description:  タスクを削除する

  Arguments:    task - 削除するタスク

  Returns:      削除できたかどうか
 *--------------------------------------------------------------------------------*/
bool TaskManager::TryRemoveTask( Task* task ) NN_NOEXCEPT
{
    fnd::ScopedLock<fnd::CriticalSection> scopeLock(m_CriticalSection);

    for ( int i = 0; i < PriorityCount; i++ )
    {
        TaskPriority priority = static_cast<TaskPriority>( i );
        TaskList::iterator itr = m_TaskList[ priority ].begin();
        while ( itr != m_TaskList[ priority ].end() )
        {
            TaskList::iterator curItr = itr++;
            if ( &*curItr == task ) {
                m_TaskList[ priority ].erase( curItr );
                curItr->m_Status = Task::Status_Cancel;
                curItr->m_Event.Signal();
                return true;
            }
        }
    }

    return false;
}


/*--------------------------------------------------------------------------------*
  Name:         RemoveTaskById

  Description:  タスクを削除する

  Arguments:    id - 削除するタスクのID

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void TaskManager::RemoveTaskById( uint32_t id ) NN_NOEXCEPT
{
    fnd::ScopedLock<fnd::CriticalSection> scopeLock(m_CriticalSection);

    for ( int i = 0; i < PriorityCount; i++ )
    {
        TaskPriority priority = static_cast<TaskPriority>( i );
        TaskList::iterator itr = m_TaskList[ priority ].begin();
        while ( itr != m_TaskList[ priority ].end() )
        {
            TaskList::iterator curItr = itr++;
            if ( curItr->m_Id == id ) {
                m_TaskList[ priority ].erase( curItr );
                curItr->m_Status = Task::Status_Cancel;
                curItr->m_Event.Signal();
            }
        }
    }
}


/*--------------------------------------------------------------------------------*
  Name:         WaitTask

  Description:  タスクが登録されるまで待つ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void TaskManager::WaitTask() NN_NOEXCEPT
{
    m_IsWaitTaskCancel = false;

    while( PeekTask() == nullptr && ! m_IsWaitTaskCancel )
    {
        uintptr_t msg;
        m_BlockingQueue.Receive( &msg );
        if ( msg == Message_Append ) {
            break;
        }
    }
}

/*--------------------------------------------------------------------------------*
  Name:         CancelWaitTask

  Description:  WaitTaskの処理をキャンセルする

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void TaskManager::CancelWaitTask() NN_NOEXCEPT
{
    m_IsWaitTaskCancel = true;

    m_BlockingQueue.Send(Message_Append);
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

