﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundThread.h>

#include <nn/atk/atk_AudioRendererPerformanceReader.h>
#include <nn/atk/atk_HardwareManager.h>
#include <nn/atk/atk_MultiVoiceManager.h>
#include <nn/atk/atk_ChannelManager.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/atk_Config.h>
#include <nn/atk/atk_Voice.h>
#include <nn/atk/atk_VoiceCommand.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>

namespace
{
inline void GetTick(nn::os::Tick* pVariable)
{
    *pVariable = nn::os::GetSystemTick();
}

inline void DoNothing(nn::os::Tick* pVariable)
{
    NN_UNUSED(pVariable);
}
}

namespace nn {
namespace atk {
namespace detail {
namespace driver {

NN_DEFINE_STATIC_CONSTANT( const int SoundThread::ThreadMessageBufferSize );
NN_DEFINE_STATIC_CONSTANT( const int SoundThread::RendererEventWaitTimeoutMilliSeconds );

/* ========================================================================
        member function
   ======================================================================== */

/*--------------------------------------------------------------------------------*
  Name:         SoundThread

  Description:  コンストラクタ

  Arguments:    なし

  Returns:      なし
 *--------------------------------------------------------------------------------*/
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(push)
#pragma warning(disable:4355) // warning: used in base member initializer list
#endif
SoundThread::SoundThread() NN_NOEXCEPT
: m_Thread()
, m_BlockingQueue(m_MsgBuffer, ThreadMessageBufferSize)
, m_CreateFlag(false)
, m_PauseFlag(false)
, m_CurrentPerformanceFrameBufferIndex(0)
, m_IsProfilingEnabled(false)
, m_pSoundThreadProfileFunc(nullptr)
, m_IsUserThreadRenderingEnabled(false)
, m_pAudioRendererPerformanceReader(nullptr)
, m_RendererEventWaitTimeMilliSeconds(0)
{
}
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(pop) // disable:4355
#endif

/*--------------------------------------------------------------------------------*
  Name:         GetInstance

  Description:  シングルトンのインスタンスを取得します

  Arguments:    無し

  Returns:      インスタンスへの参照
 *--------------------------------------------------------------------------------*/
SoundThread& SoundThread::GetInstance() NN_NOEXCEPT
{
    static SoundThread instance;
    return instance;
}

bool SoundThread::CreateSoundThread(
        int32_t threadPriority,
        void* stackBase,
        size_t stackSize,
        int idealCoreNumber,
        uint32_t affinityMask ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( HardwareManager::GetInstance().IsInitialized(), "not initialized nn::atk::detail::HardwareManager.\n" );

    m_SoundThreadAffinityMask = affinityMask;

    nn::atk::detail::fnd::Thread::RunArgs args;
    args.name = "nn::atk::detail::driver::SoundThread";
    args.stack = stackBase;
    args.stackSize = stackSize;
    args.idealCoreNumber = idealCoreNumber;
    args.affinityMask = static_cast<nn::atk::detail::fnd::Thread::AffinityMask>(affinityMask);
    // TODO: priority値変換
    args.priority = threadPriority;
    args.param = nullptr;
    args.handler = this;

    m_CreateFlag = m_Thread.Run( args );
    return m_CreateFlag;
}

void SoundThread::Initialize( void* performanceFrameBuffer, size_t performanceFrameBufferSize, bool isProfilingEnabled, bool isDetailSoundThreadProfilerEnabled, bool isUserThreadRenderingEnabled ) NN_NOEXCEPT
{
    uintptr_t ptr = reinterpret_cast<uintptr_t>(performanceFrameBuffer);
    NN_SDK_ASSERT(nn::util::is_aligned(ptr, nn::audio::BufferAlignSize));
    NN_SDK_ASSERT(nn::util::is_aligned(performanceFrameBufferSize, nn::audio::BufferAlignSize));

    m_pAudioRendererPerformanceReader = nullptr;
    m_CurrentPerformanceFrameBufferIndex = 0;
    m_IsProfilingEnabled = isProfilingEnabled;
    if(isDetailSoundThreadProfilerEnabled)
    {
        m_pSoundThreadProfileFunc = &GetTick;
    }
    else
    {
        m_pSoundThreadProfileFunc = &DoNothing;
    }
    m_IsUserThreadRenderingEnabled = isUserThreadRenderingEnabled;
    m_RendererEventWaitTimeMilliSeconds = 0;
#ifndef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    m_PrevAudioFrameCount = 0;
    m_CurrentAudioFrameCount = 0;
#endif
    const size_t bufferSize = performanceFrameBufferSize / MaxPerformanceBufferCount;
    m_PerformanceFrameUpdateBufferSize = bufferSize;
    for ( auto i = 0; i < MaxPerformanceBufferCount; i++ )
    {
        m_pPerformanceFrameUpdateBuffer[i] = reinterpret_cast<void*>(ptr);
        ptr += bufferSize;
    }
#if 0
    // TODO: 詳細なパフォーマンス情報を取得する設定
    nn::audio::SetPerformanceDetailTarget(
        &detail::driver::HardwareManager::GetInstance().GetAudioRendererConfig(),
        &detail::driver::HardwareManager::GetInstance().GetSubMix() );
#endif

    NN_SDK_ASSERT(m_SoundFrameCallbackList.empty(), "Registered SoundFrameCallback exists. Register the callback with SoundThread::RegisterSoundFrameCallback() after SoundSystem::Initialize().");

#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    detail::VirtualVoiceManager::GetInstance().Initialize();
#endif
}

void SoundThread::Destroy() NN_NOEXCEPT
{
    if ( ! m_CreateFlag ) return;

    // サウンドスレッド終了メッセージ送信
    m_BlockingQueue.Jam( Message_Shutdown );

    // ユーザが ExecuteAudioRendererRendering() を呼ばない場合、スレッドの終了待機で無限に待ってしまうので手動で呼び出す
    if(m_IsUserThreadRenderingEnabled)
    {
        HardwareManager::GetInstance().ExecuteAudioRendererRendering();
    }

    // スレッドの終了を待つ
    m_Thread.WaitForExit();
    m_Thread.Release();

    m_SoundThreadAffinityMask = 0;
    m_CreateFlag = false;
}

/*--------------------------------------------------------------------------------*
  Name:         Shutdown

  Description:  サウンドスレッドを終了する

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundThread::Finalize() NN_NOEXCEPT
{
    // プレイヤーを全て停止する
    for ( PlayerCallbackList::iterator itr = m_PlayerCallbackList.begin();
            itr != m_PlayerCallbackList.end();
        )
    {
        PlayerCallbackList::iterator curItr = itr++;
        curItr->OnShutdownSoundThread();
    }

    NN_SDK_ASSERT(m_SoundFrameCallbackList.empty(), "Registered SoundFrameCallback still exists. Unregister the callback with SoundThread::UnregisterSoundFrameCallback() before SoundSystem::Finalize().");

    for ( auto i = 0; i < MaxPerformanceBufferCount; i++ )
    {
        m_pPerformanceFrameUpdateBuffer[i] = nullptr;
    }
    m_PerformanceFrameUpdateBufferSize = 0;

    m_CurrentPerformanceFrameBufferIndex = 0;
    m_pAudioRendererPerformanceReader = nullptr;
}

#ifndef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
void SoundThread::UpdateElapsedFrameCount() NN_NOEXCEPT
{
    m_PrevAudioFrameCount = m_CurrentAudioFrameCount;
    m_CurrentAudioFrameCount = HardwareManager::GetInstance().GetElapsedAudioFrameCount();
}
#endif

/*--------------------------------------------------------------------------------*
  Name:         UpdateLowLevelVoices

  Description:  物理ボイスの更新処理

  Arguments:

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundThread::UpdateLowLevelVoices() NN_NOEXCEPT
{
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    LowLevelVoiceCommand::GetInstance().ProcessCommand();
#endif

    OutputMode outputMode = HardwareManager::GetInstance().GetOutputMode(OutputDevice_Main);
    HardwareManager::GetInstance().GetLowLevelVoiceAllocator().UpdateAllVoiceState(outputMode);

#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    VirtualVoiceManager::GetInstance().UpdateVoiceInfo();
#endif
}

void SoundThread::ForceWakeup() NN_NOEXCEPT
{
#ifndef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    m_BlockingQueue.TrySend( Message_ForceWakeup );
#endif
}

void SoundThread::RegisterSoundFrameUserCallback( SoundFrameUserCallback callback, uintptr_t arg ) NN_NOEXCEPT
{
    SoundThreadLock lock;
    m_UserCallbackArg = arg;
    m_UserCallback = callback;
}

void SoundThread::ClearSoundFrameUserCallback() NN_NOEXCEPT
{
    SoundThreadLock lock;
    m_UserCallback = nullptr;
    m_UserCallbackArg = 0;
}

void SoundThread::RegisterThreadBeginUserCallback(SoundFrameUserCallback callback, uintptr_t arg) NN_NOEXCEPT
{
    SoundThreadLock lock;
    m_ThreadBeginUserCallbackArg = arg;
    m_ThreadBeginUserCallback = callback;
}

void SoundThread::ClearThreadBeginUserCallback() NN_NOEXCEPT
{
    SoundThreadLock lock;
    m_ThreadBeginUserCallback = nullptr;
    m_ThreadBeginUserCallbackArg = 0;
}

void SoundThread::RegisterThreadEndUserCallback(SoundFrameUserCallback callback, uintptr_t arg) NN_NOEXCEPT
{
    SoundThreadLock lock;
    m_ThreadEndUserCallbackArg = arg;
    m_ThreadEndUserCallback = callback;
}

void SoundThread::ClearThreadEndUserCallback() NN_NOEXCEPT
{
    SoundThreadLock lock;
    m_ThreadEndUserCallback = nullptr;
    m_ThreadEndUserCallbackArg = 0;
}

void SoundThread::RegisterSoundFrameCallback( SoundFrameCallback* callback ) NN_NOEXCEPT
{
    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_CriticalSection );
    m_SoundFrameCallbackList.push_back( *callback );
}

void SoundThread::UnregisterSoundFrameCallback( SoundFrameCallback* callback ) NN_NOEXCEPT
{
    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_CriticalSection );
    m_SoundFrameCallbackList.erase( m_SoundFrameCallbackList.iterator_to( *callback ) );
}

void SoundThread::RegisterPlayerCallback( PlayerCallback* callback ) NN_NOEXCEPT
{
    m_PlayerCallbackList.push_back( *callback );
}

void SoundThread::UnregisterPlayerCallback( PlayerCallback* callback ) NN_NOEXCEPT
{
    m_PlayerCallbackList.erase( m_PlayerCallbackList.iterator_to( *callback ) );
}

void SoundThread::LockAtkStateAndParameterUpdate() NN_NOEXCEPT
{
    m_UpdateAtkStateAndParameterSection.Lock();
}

void SoundThread::UnlockAtkStateAndParameterUpdate() NN_NOEXCEPT
{
    m_UpdateAtkStateAndParameterSection.Unlock();
}

void SoundThread::RegisterAudioRendererPerformanceReader( AudioRendererPerformanceReader& performanceReader ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_pAudioRendererPerformanceReader == nullptr, "You can register only one instance." );
    m_pAudioRendererPerformanceReader = &performanceReader;
}

void SoundThread::RegisterSoundThreadUpdateProfileReader( SoundThreadUpdateProfileReader& profileReader ) NN_NOEXCEPT
{
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_LockUpdateProfile );
    m_UpdateProfileReaderList.push_back(profileReader);
}

void SoundThread::UnregisterSoundThreadUpdateProfileReader( SoundThreadUpdateProfileReader& profileReader ) NN_NOEXCEPT
{
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_LockUpdateProfile );
    m_UpdateProfileReaderList.erase(m_UpdateProfileReaderList.iterator_to(profileReader));
}

void SoundThread::RegisterSoundThreadInfoRecorder(SoundThreadInfoRecorder& recorder) NN_NOEXCEPT
{
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_LockRecordInfo );
    m_InfoRecorderList.push_back( recorder );
}

void SoundThread::UnregisterSoundThreadInfoRecorder(SoundThreadInfoRecorder& recorder) NN_NOEXCEPT
{
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_LockRecordInfo );
    m_InfoRecorderList.erase(m_InfoRecorderList.iterator_to(recorder));
}

/*--------------------------------------------------------------------------------*
  Name:         FrameProcess

  Description:  サウンドフレーム処理

  Arguments:    None.

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void SoundThread::FrameProcess( UpdateType updateType ) NN_NOEXCEPT
{
    m_CriticalSection.Lock();

    nn::audio::AudioRendererConfig& config = detail::driver::HardwareManager::GetInstance().GetAudioRendererConfig();
    void* performanceFrameBuffer = nullptr;
    if ( m_IsProfilingEnabled )
    {
        {
            detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
            performanceFrameBuffer = nn::audio::SetPerformanceFrameBuffer( &config, m_pPerformanceFrameUpdateBuffer[m_CurrentPerformanceFrameBufferIndex], m_PerformanceFrameUpdateBufferSize );
        }

        m_CurrentPerformanceFrameBufferIndex++;
        if ( m_CurrentPerformanceFrameBufferIndex >= MaxPerformanceBufferCount )
        {
            m_CurrentPerformanceFrameBufferIndex = 0;
        }
    }

    const nn::os::Tick beginTick = nn::os::GetSystemTick();

    // TODO: Lock を機能的に分離する件について検討する
    m_UpdateAtkStateAndParameterSection.Lock();

    // サウンドフレーム開始コールバック
    {
        for ( SoundFrameCallbackList::iterator itr = m_SoundFrameCallbackList.begin();
              itr != m_SoundFrameCallbackList.end();
        )
        {
            SoundFrameCallbackList::iterator curItr = itr++;
            curItr->OnBeginSoundFrame();
        }
    }

    uint32_t nwVoiceCount = 0;

    // サウンドスレッドのメイン処理
    {
        // SDK処理結果をボイスに反映
        //   MultiVoice::UpdateVoiceStatus
        //    波形停止チェック
        //    ボイスドロップチェック
        if ( updateType == UpdateType_AudioFrame )
        {
            MultiVoiceManager::GetInstance().UpdateAudioFrameVoiceStatus();
        }
        else
        {
            MultiVoiceManager::GetInstance().UpdateAllVoiceStatus();
        }

        // コマンド処理
        {
            while( DriverCommand::GetInstanceForTaskThread().ProcessCommand() ) {}
            while( DriverCommand::GetInstance().ProcessCommand() ) {}
        }

        // プレイヤー更新
        {
            int updateFrameCount;
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
            updateFrameCount = 1;
#else
            updateFrameCount = static_cast<int>(m_CurrentAudioFrameCount - m_PrevAudioFrameCount);
            // SoundSystem::ResumeAudioRenderer 直後に経過フレーム数が巻き戻るが、
            // その時に負の値が updateFrameCount に入らないようにする
            updateFrameCount = updateFrameCount < 0 ? 0 : updateFrameCount;
#endif

            for ( PlayerCallbackList::iterator itr = m_PlayerCallbackList.begin();
                  itr != m_PlayerCallbackList.end();
            )
            {
                PlayerCallbackList::iterator curItr = itr++;
                if ( updateType == UpdateType_AudioFrame )
                {
                    curItr->OnUpdateFrameSoundThreadWithAudioFrameFrequency(updateFrameCount);
                }
                else
                {
                    curItr->OnUpdateFrameSoundThread(updateFrameCount);
                }
            }
        }

        // チャンネルパラメータの計算
        //   Channel::Update
        //     パラメータ計算
        //     エンベロープ・LFO
        //     MultiVoiceへのパラメータ反映
        if ( updateType == UpdateType_AudioFrame )
        {
            ChannelManager::GetInstance().UpdateAudioFrameChannel();
        }
        else
        {
            ChannelManager::GetInstance().UpdateAllChannel();
        }

        // ボイスパラメータの計算
        //   MultiVoice::Calc
        //     パラメータ計算
        //     Voiceへのパラメータ反映
        //   MultiVoice::Update
        //     一部パラメータ計算
        //     ボイスステータス更新
        //     Voiceへのパラメータ反映
        if ( updateType == UpdateType_AudioFrame )
        {
            MultiVoiceManager::GetInstance().UpdateAudioFrameVoices();
        }
        else
        {
            MultiVoiceManager::GetInstance().UpdateAllVoices();
        }

        m_UpdateAtkStateAndParameterSection.Unlock();

        // HardwareManager 更新
        HardwareManager::GetInstance().Update();

        // 乱数列更新
        (void)Util::CalcRandom();
    }

    m_UpdateAtkStateAndParameterSection.Lock();

    // サウンドフレーム終了コールバック
    {
        for ( SoundFrameCallbackList::iterator itr = m_SoundFrameCallbackList.begin();
              itr != m_SoundFrameCallbackList.end();
        )
        {
            SoundFrameCallbackList::iterator curItr = itr++;
            curItr->OnEndSoundFrame();
        }
    }

    m_UpdateAtkStateAndParameterSection.Unlock();

    // ユーザーコールバック呼び出し
    if ( m_UserCallback )
    {
        m_UserCallback( m_UserCallbackArg );
    }

    // Recorder の更新
    HardwareManager::GetInstance().UpdateRecorder();

    m_CriticalSection.Unlock();

    nn::os::Tick endTick = nn::os::GetSystemTick();

    // プロファイル処理
    if ( m_IsProfilingEnabled && performanceFrameBuffer != nullptr )
    {
        nn::audio::PerformanceInfo performanceInfo;
        if ( performanceInfo.SetBuffer( performanceFrameBuffer, m_PerformanceFrameUpdateBufferSize ) )
        {
            do
            {
                RecordPerformanceInfo( performanceInfo, m_LastPerformanceFrameBegin, m_LastPerformanceFrameEnd, nwVoiceCount );
            } while ( performanceInfo.MoveToNextFrame() );
        }

        //  nn::audio パフォーマンス情報を伝える
        if ( m_pAudioRendererPerformanceReader != nullptr )
        {
            m_pAudioRendererPerformanceReader->Record( performanceFrameBuffer, m_PerformanceFrameUpdateBufferSize, beginTick );
        }
    }

    m_LastPerformanceFrameBegin = beginTick;
    m_LastPerformanceFrameEnd = endTick;
} // NOLINT(impl/function_size)

#ifndef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
void SoundThread::UpdateLowLevelVoicesParam() NN_NOEXCEPT
{
    OutputMode outputMode = HardwareManager::GetInstance().GetOutputMode(OutputDevice_Main);
    HardwareManager::GetInstance().GetLowLevelVoiceAllocator().UpdateAllVoiceParam(outputMode);
}
#endif

void SoundThread::EffectFrameProcess() NN_NOEXCEPT
{
    HardwareManager::GetInstance().UpdateEffect();
}

void SoundThread::RecordPerformanceInfo( nn::audio::PerformanceInfo& src, nn::os::Tick beginTick,  nn::os::Tick endTick, uint32_t nwVoiceCount ) NN_NOEXCEPT
{
    if ( m_ProfileReaderList.empty() )
    {
        return;
    }

    SoundProfile dst;
    auto numVoices = 0;
    int totalVoiceProcessingTime = 0;

    int performanceEntryCount = 0;
    const nn::audio::PerformanceEntry* performanceEntries = src.GetEntries(&performanceEntryCount);

    dst.rendererFrameProcess.begin = nn::os::Tick(0);
    dst.rendererFrameProcess.end = nn::os::Tick(nn::TimeSpan::FromMicroSeconds(src.GetTotalProcessingTime()));
    dst.voiceProcess.begin = nn::os::Tick(0);
    for ( auto i = 0; i < performanceEntryCount; ++i )
    {
        const nn::audio::PerformanceEntry& performanceEntry = performanceEntries[i];

        TimeRange time;
        time.begin = nn::os::Tick(nn::TimeSpan::FromMicroSeconds(performanceEntry.startTime));
        time.end   = nn::os::Tick(nn::TimeSpan::FromMicroSeconds(performanceEntry.startTime + performanceEntry.processingTime));

        if ( performanceEntry.entryType == nn::audio::PerformanceEntryType_Voice )
        {
            totalVoiceProcessingTime += performanceEntry.processingTime;

            dst._voiceIdTable[numVoices] = performanceEntry.id;
            dst._voiceProcessTable[numVoices] = time;
            numVoices++;
        }
        else if ( performanceEntry.entryType == nn::audio::PerformanceEntryType_SubMix )
        {
            const auto& hardwareManager = nn::atk::detail::driver::HardwareManager::GetInstance();

#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
            detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
#endif
            if (hardwareManager.IsPresetSubMixEnabled())
            {
                if( nn::audio::GetSubMixNodeId( hardwareManager.GetSubMix( 0 ).GetAudioSubMixInstance() ) == performanceEntry.id )
                {
                    dst.mainMixProcess = time;
                }
                else if( hardwareManager.IsAdditionalEffectEnabled() && nn::audio::GetSubMixNodeId( hardwareManager.GetSubMix( 1 ).GetAudioSubMixInstance() ) == performanceEntry.id )
                {
                    dst._additionalSubMixProcess = time;
                }
            }
            else
            {
                if( nn::audio::GetSubMixNodeId( hardwareManager.GetSubMix( 0 ).GetAudioSubMixInstance() ) == performanceEntry.id )
                {
                    dst.mainMixProcess = time;
                }
                else if( hardwareManager.GetSubMixCount() > 1 && nn::audio::GetSubMixNodeId( hardwareManager.GetSubMix( 1 ).GetAudioSubMixInstance() ) == performanceEntry.id )
                {
                    dst._additionalSubMixProcess = time;
                }
            }
        }
        else if ( performanceEntry.entryType == nn::audio::PerformanceEntryType_FinalMix )
        {
            dst.finalMixProcess = time;
        }
        else if ( performanceEntry.entryType == nn::audio::PerformanceEntryType_Sink )
        {
            const auto& hardwareManager = nn::atk::detail::driver::HardwareManager::GetInstance();

#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
            detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
#endif
            if( !m_IsUserThreadRenderingEnabled && nn::audio::GetSinkNodeId( &hardwareManager.GetDeviceSink() ) == performanceEntry.id )
            {
                dst.sinkProcess = time;
            }
            else if( hardwareManager.GetUserCircularBufferSinkState() != CircularBufferSinkState_Invalid && nn::audio::GetSinkNodeId( &hardwareManager.GetUserCircularBufferSink() ) == performanceEntry.id )
            {
                dst.circularBufferSinkProcess = time;
            }
        }
    }
    dst.nwFrameProcess.begin = beginTick;
    dst.nwFrameProcess.end   = endTick;
    dst.voiceProcess.end = nn::os::Tick(nn::TimeSpan::FromMicroSeconds(totalVoiceProcessingTime));
#if 0 // 参考用 : NW4F 実装
    dst.nwVoiceParamUpdate.begin = nn::os::Tick(src.userCallbackStart);
    dst.nwVoiceParamUpdate.end   = nn::os::Tick(src.userCallbackEnd);
    dst.voiceRendering.begin = nn::os::Tick(src.axPPCStart);
    dst.voiceRendering.end = nn::os::Tick(src.axPPCEnd);
    dst.syncVoiceParam.begin = nn::os::Tick(src.axFrameStart);
    dst.syncVoiceParam.end   = nn::os::Tick(src.axPPCStart);
    dst.outputFormatProcess.begin = nn::os::Tick(src.userFinalMixEnd);
    dst.outputFormatProcess.end   = nn::os::Tick(src.axFrameEnd);
    dst.auxProcess.begin = nn::os::Tick(src.userAuxProcStart);
    dst.auxProcess.end   = nn::os::Tick(src.userAuxProcEnd);
    dst.mainMixProcess.begin = nn::os::Tick(src.axPPCPostStart);
    dst.mainMixProcess.end   = nn::os::Tick(src.axPPCPostEnd);
    dst.axFrameProcess.begin = nn::os::Tick(src.axFrameStart);
    dst.axFrameProcess.end   = nn::os::Tick(src.axFrameEnd);
    dst.dspFrameProcess.begin = nn::os::Tick(src.axDSPStart);
    dst.dspFrameProcess.end   = nn::os::Tick(src.axDSPEnd);
#endif

    // ボイス数
    dst.rendererVoiceCount = numVoices;
    dst.nwVoiceCount = nwVoiceCount;
    dst.totalVoiceCount = dst.rendererVoiceCount + dst.nwVoiceCount;

#if 0 // 参考用 : NW4F 実装
    // フレームティック
    dst.axFrameProcessTick = ( src.userCallbackEnd - src.axFrameStart ) + ( src.axFrameEnd - src.axPPCPostStart );
#endif

    dst.nwFrameProcessTick = ( dst.nwFrameProcess.end - dst.nwFrameProcess.begin ).GetInt64Value();

#if 0 // 参考用 : NW4F 実装
    if ( dst.nwFrameProcess.begin.GetInt64Value() <= dst.axFrameProcess.begin.GetInt64Value() && dst.axFrameProcess.begin.GetInt64Value() <= dst.nwFrameProcess.end.GetInt64Value() )
    {
        dst.nwFrameProcessTick -= ( src.userCallbackEnd - src.axFrameStart );
    }

    nn::os::Tick axPPCPostStart = nn::os::Tick(src.axPPCPostStart);

    if ( dst.nwFrameProcess.begin.GetInt64Value() <= axPPCPostStart.GetInt64Value() && axPPCPostStart.GetInt64Value() <= dst.nwFrameProcess.end.GetInt64Value() )
    {
        dst.nwFrameProcessTick -= ( src.axFrameEnd - src.axPPCPostStart );
    }
#endif

    // ProfileReaderへの反映
    for (ProfileReaderList::iterator it = m_ProfileReaderList.begin(); it != m_ProfileReaderList.end(); ++it)
    {
        it->Record(dst);
    }
} // NOLINT(impl/function_size)

void SoundThread::RecordUpdateProfile(const SoundThreadUpdateProfile& updateProfile) NN_NOEXCEPT
{
    if( !m_UpdateProfileReaderList.empty() )
    {
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_LockUpdateProfile );
        for( auto itr = m_UpdateProfileReaderList.begin(); itr != m_UpdateProfileReaderList.end(); ++itr )
        {
            itr->Record( updateProfile );
        }
    }
}

/*--------------------------------------------------------------------------------*
  Name:         SoundThreadProc

  Description:  サウンドスレッドプロシージャ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
uint32_t SoundThread::Run(void* param) NN_NOEXCEPT
{
    NN_UNUSED(param);

    for ( ;; )
    {
        if(m_PauseFlag)
        {
            // CPU 負荷を考慮し、オーディオフレーム間隔のスリープを挟む
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(detail::driver::HardwareManager::SoundFrameIntervalMsec));
            continue;
        }

        // 開始ユーザーコールバック呼び出し
        if (m_ThreadBeginUserCallback)
        {
            m_ThreadBeginUserCallback(m_ThreadBeginUserCallbackArg);
        }

        m_LastUpdateProfile.soundThreadProcess.begin = nn::os::GetSystemTick();

        uintptr_t msgBuf;
        bool result = m_BlockingQueue.TryReceive( &msgBuf );
        Message msg = static_cast<Message>( result ? msgBuf & 0xf0000000 : 0x0 );

#ifndef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if ( msg == Message_ForceWakeup )
        {
            // コマンド処理だけ行う
            while(DriverCommand::GetInstance().ProcessCommand()){}
            while(DriverCommand::GetInstanceForTaskThread().ProcessCommand()){}
        }
        else if ( msg == Message_Shutdown )
        {
            break;
        }
        else
        {
            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._updateElapsedFrameCountProcess.begin);
            UpdateElapsedFrameCount();
            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._updateElapsedFrameCountProcess.end);

            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._updateLowLevelVoiceProcess.begin);
            UpdateLowLevelVoices();
            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._updateLowLevelVoiceProcess.end);

            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._frameProcess.begin);
            FrameProcess();
            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._frameProcess.end);

            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._updateLowLevelVoiceParamProcess.begin);
            UpdateLowLevelVoicesParam();
            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._updateLowLevelVoiceParamProcess.end);

            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._userEffectFrameProcess.begin);
            EffectFrameProcess();
            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._userEffectFrameProcess.end);
        }
#else
        if ( msg == Message_Shutdown )
        {
            break;
        }
        else
        {
            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._updateLowLevelVoiceProcess.begin);
            UpdateLowLevelVoices();
            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._updateLowLevelVoiceProcess.end);

            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._userEffectFrameProcess.begin);
            EffectFrameProcess();
            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._userEffectFrameProcess.end);
        }
#endif

        {
            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._updateRendererProcess.begin);
            NN_ABORT_UNLESS_RESULT_SUCCESS(HardwareManager::GetInstance().RequestUpdateAudioRenderer());
            m_pSoundThreadProfileFunc(&m_LastUpdateProfile._updateRendererProcess.end);
        }

        m_LastUpdateProfile.soundThreadProcess.end = nn::os::GetSystemTick();

        // 終了ユーザーコールバック呼び出し
        if (m_ThreadEndUserCallback)
        {
            m_ThreadEndUserCallback(m_ThreadEndUserCallbackArg);
        }

        // AudioRenderer のイベント待ち
        m_pSoundThreadProfileFunc(&m_LastUpdateProfile._waitRendererEventProcess.begin);
        while (!HardwareManager::GetInstance().TimedWaitAudioRendererEvent(nn::TimeSpan::FromMilliSeconds(RendererEventWaitTimeoutMilliSeconds)))
        {
            m_RendererEventWaitTimeMilliSeconds += RendererEventWaitTimeoutMilliSeconds;
        }
        m_RendererEventWaitTimeMilliSeconds = 0;
        m_pSoundThreadProfileFunc(&m_LastUpdateProfile._waitRendererEventProcess.end);

        RecordUpdateProfile(m_LastUpdateProfile);
    }

    return 0;
}

} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

