﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_StreamSoundFileLoader.h>

#include <nn/atk/atk_StreamSoundFileReader.h>
#include <nn/atk/fnd/io/atkfnd_FileStream.h>

namespace nn {
namespace atk {
namespace detail {

/* ========================================================================
        StreamSoundFileLoader
   ======================================================================== */

/*--------------------------------------------------------------------------------*
  Name:         LoadFileHeader

  Description:  ストリームデータのファイルヘッダとHEADブロックをロードし、
                StreamSoundFileReaderに渡す

  Arguments:    buffer - ロードに使用するバッファ
                size - バッファサイズ

  Returns:      AdpcmInfo構造体
 *--------------------------------------------------------------------------------*/
bool StreamSoundFileLoader::LoadFileHeader(
        StreamSoundFileReader* reader, void* buffer, unsigned long size ) NN_NOEXCEPT
{
    const size_t HeaderSize = NN_ATK_ROUND_UP_32B( sizeof( StreamSoundFile::FileHeader ) );
    const size_t AlignSize = 64;
        // NOTE: FSAGetVolumeInfo → FSAVolumeInfo.buf_align_r が 32 になっていたが、
        //       FSAReadFile で 32 バイト境界のバッファを渡すと FSA_STATUS_INVALID_ALIGNMENT
        //       を返すため、AlignSize を 64 とする。(CafeSDK-1.3)

    char buffer2[ HeaderSize + AlignSize ];

    // ファイルヘッダロード
    m_pStream->Seek( 0, fnd::Stream::SeekOrigin::SeekOrigin_Begin );
    size_t readSize = m_pStream->Read( util::BytePtr( buffer2 ).AlignUp( AlignSize ).Get(), HeaderSize );
    if ( readSize != HeaderSize )
    {
        return false;
    }

    StreamSoundFile::FileHeader* header =
        reinterpret_cast<StreamSoundFile::FileHeader*>( util::BytePtr( buffer2 ).AlignUp( AlignSize ).Get() );

    if ( ! reader->IsValidFileHeader( header ) )
    {
        return false;
    }

    // バッファサイズがファイルヘッダーと INFO ブロックをロードできるサイズかチェック
    // (必ず、ヘッダーと INFO ブロックが並んで格納されているものとする)
    uint32_t loadSize = header->GetInfoBlockOffset() + header->GetInfoBlockSize();
    if ( loadSize > size )
    {
        return false;
    }

    // ファイルヘッダーと INFO ブロックをロード
    m_pStream->Seek( 0, fnd::Stream::SeekOrigin::SeekOrigin_Begin );
    readSize = m_pStream->Read( buffer, loadSize );
    if ( readSize != static_cast<int32_t>(loadSize) )
    {
        return false;
    }

    reader->Initialize( buffer );

    m_SeekBlockOffset = reader->GetSeekBlockOffset();
    m_RegionDataOffset = reader->GetRegionDataOffset();
    m_RegionInfoBytes = static_cast<uint16_t>(reader->GetRegionInfoBytes());

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         ReadSeekBlockData

  Description:  SEEK ブロックの指定されたデータを読み取る

  Arguments:    yn1, yn2 - 読み取ったデータを格納する配列
                blockIndex - 読み取るデータのブロック番号
                channelCount - チャンネル数(yn1,yn2の配列サイズ)

  Returns:      読み取れたら true, 読み取れなかったら false
 *--------------------------------------------------------------------------------*/
bool StreamSoundFileLoader::ReadSeekBlockData(
    uint16_t* yn1, uint16_t* yn2, int blockIndex, int channelCount ) NN_NOEXCEPT
{
    static const size_t SeekInfoMaxSize = StreamChannelCountPerTrack * StreamTrackCount * sizeof(uint16_t) * 2;  // 2ch * 8 Track * 2byte [yn1, yn2 のサイズ] * 2個 [ yn1, yn2 ]
    size_t readDataSize = sizeof(uint16_t) * 2 * channelCount;

    position_t readOffset = static_cast<position_t>(
        m_SeekBlockOffset +
        sizeof(BinaryBlockHeader) +
        readDataSize * blockIndex
    );
    m_pStream->Seek( readOffset, fnd::Stream::SeekOrigin::SeekOrigin_Begin );

    // 以下は 16 チャンネル以上で動作できないコードなのでASSERT
    NN_SDK_ASSERT( readDataSize <= SeekInfoMaxSize );
    if ( readDataSize > SeekInfoMaxSize )
    {
        return false;
    }

    const int Align = PpcIoBufferAlign;
    uint16_t bufferBase[SeekInfoMaxSize + Align];
    uint16_t* buffer = reinterpret_cast<uint16_t*>(util::BytePtr( bufferBase ).AlignUp( Align ).Get());
    size_t readSize = m_pStream->Read( buffer, readDataSize );
    if ( readSize != readDataSize )
    {
        return false;
    }

    for ( int i = 0; i < channelCount; i++ )
    {
        yn1[i] = buffer[i * 2];
        yn2[i] = buffer[i * 2 + 1];
        // NN_DETAIL_ATK_INFO("[%d] yn1(%6d) yn2(%6d)\n",
        //         i, static_cast<int16_t>(yn1[i]), static_cast<int16_t>(yn2[i]));
    }

    return true;
}

bool StreamSoundFileLoader::ReadRegionInfo(
        StreamSoundFile::RegionInfo* pInfo, uint32_t regionIndex ) const NN_NOEXCEPT
{
    if ( m_RegionDataOffset == 0 || m_RegionInfoBytes == 0 )
    {
        return false;
    }

    position_t offset = m_RegionDataOffset + (static_cast<position_t>(m_RegionInfoBytes) * regionIndex);
    m_pStream->Seek( offset, fnd::Stream::SeekOrigin::SeekOrigin_Begin );

    const int Align = PpcIoBufferAlign;
    const size_t ReadSize = sizeof(StreamSoundFile::RegionInfo);
    uint8_t bufferBase[ReadSize + (Align / 2) * 2];
    uint8_t* buffer = reinterpret_cast<uint8_t*>(util::BytePtr( bufferBase ).AlignUp( Align ).Get());
    size_t readSize = m_pStream->Read( buffer, ReadSize );
    if ( readSize != ReadSize )
    {
        return false;
    }

    *pInfo = *reinterpret_cast<StreamSoundFile::RegionInfo*>(buffer);
    return true;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

