﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SequenceSoundFileReader.h>

namespace nn {
namespace atk {
namespace detail {

namespace
{

const uint32_t SignatureDataBlockSeq  = NN_UTIL_CREATE_SIGNATURE_4( 'D', 'A', 'T', 'A' );
const uint32_t SignatureLabelBlockSeq = NN_UTIL_CREATE_SIGNATURE_4( 'L', 'A', 'B', 'L' );

const uint32_t SupportedFileVersionSeq = 0x00010000;
const uint32_t CurrentFileVersionSeq   = 0x00020000;

bool IsValidFileHeaderSeq( const void* sequenceFile ) NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader* header =
        reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( sequenceFile );

    // シグニチャ確認
    NN_SDK_ASSERT( header->signature == SequenceSoundFileReader::SignatureFile,
            "invalid file signature. sequence file is not available." );
    if ( header->signature != SequenceSoundFileReader::SignatureFile )
    {
        return false;
    }

    // バイトオーダーマーク確認
    NN_SDK_ASSERT( header->byteOrder == BinaryFileHeader::ValidByteOrderMark,
            "invalid file byte order mark. [expected:0x%04x][this:0x%04x]", BinaryFileHeader::ValidByteOrderMark, header->byteOrder );
    if ( header->byteOrder != BinaryFileHeader::ValidByteOrderMark )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SupportedFileVersionSeq <= header->version) &&
                                   (header->version <= CurrentFileVersionSeq) )
    {
        isSupportedVersion = true;

    }
    NN_SDK_ASSERT( isSupportedVersion,
            "bfseq file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SupportedFileVersionSeq, CurrentFileVersionSeq, header->version
    );
    return isSupportedVersion;
}

} // anonymous namespace

NN_DEFINE_STATIC_CONSTANT( const uint32_t SequenceSoundFileReader::SignatureFile );

SequenceSoundFileReader::SequenceSoundFileReader( const void* sequenceFile ) NN_NOEXCEPT
: m_pHeader( NULL )
, m_pDataBlockBody( NULL )
, m_pLabelBlockBody( NULL )
{
    NN_SDK_ASSERT_NOT_NULL( sequenceFile );

    if ( ! IsValidFileHeaderSeq( sequenceFile ) ) return;

    m_pHeader = reinterpret_cast<const SequenceSoundFile::FileHeader*>( sequenceFile );

    const SequenceSoundFile::DataBlock* dataBlock = m_pHeader->GetDataBlock();
    NN_SDK_ASSERT_NOT_NULL( dataBlock );
    NN_SDK_ASSERT( dataBlock->header.kind == SignatureDataBlockSeq );
    if ( dataBlock->header.kind != SignatureDataBlockSeq )
    {
        return;
    }

    const SequenceSoundFile::LabelBlock* labelBlock = m_pHeader->GetLabelBlock();
    NN_SDK_ASSERT_NOT_NULL( labelBlock );
    NN_SDK_ASSERT( labelBlock->header.kind == SignatureLabelBlockSeq );
    if ( labelBlock->header.kind != SignatureLabelBlockSeq )
    {
        return;
    }

    m_pDataBlockBody = &dataBlock->body;
    m_pLabelBlockBody = &labelBlock->body;
}

const void* SequenceSoundFileReader::GetSequenceData() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pDataBlockBody );
    return m_pDataBlockBody->GetSequenceData();
}

bool SequenceSoundFileReader::GetOffsetByLabel( const char* label, uint32_t* offsetPtr ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pLabelBlockBody );
    return m_pLabelBlockBody->GetOffsetByLabel( label, offsetPtr );
}

const char* SequenceSoundFileReader::GetLabelByOffset( uint32_t offset ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pLabelBlockBody );
    return m_pLabelBlockBody->GetLabelByOffset( offset );
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

