﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_HardwareManager.h>
#include <nn/nn_Abort.h>
#include <nn/atk/atk_DeviceOutRecorder.h>
#include <nn/atk/atk_MultiVoiceManager.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_Config.h>
#include <nn/atk/atk_Result.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>
#include <cstring>


namespace {
    //  nn::atk::ChannelIndex の順番と nn::audio::ChannelMapping との対応関係
    const int8_t AudioChannelIndexSettings[nn::atk::ChannelIndex_Count] =
    {
        nn::audio::ChannelMapping_FrontLeft, nn::audio::ChannelMapping_FrontRight,
        nn::audio::ChannelMapping_RearLeft, nn::audio::ChannelMapping_RearRight,
        nn::audio::ChannelMapping_FrontCenter, nn::audio::ChannelMapping_LowFrequency,
    };
    //  nn::atk::ChannelIndex の順番
    const int8_t AtkChannelIndexSettings[] =
    {
        nn::atk::ChannelIndex_FrontLeft, nn::atk::ChannelIndex_FrontRight,
        nn::atk::ChannelIndex_RearLeft, nn::atk::ChannelIndex_RearRight,
        nn::atk::ChannelIndex_FrontCenter, nn::atk::ChannelIndex_Lfe,
    };
    NN_STATIC_ASSERT( sizeof(AtkChannelIndexSettings) / sizeof(AtkChannelIndexSettings[0]) == nn::atk::ChannelIndex_Count );

    //  SubMix の MainBus の Index です
    const int SubMixMainBusIndex = 0;
    //  FinalMix のバス数です
    const int FinalMixBusCount = 1;
    //  AdditionalSubMix のバス数です
    const int AdditionalSubMixBusCount = 1;
    //  AuxBus の Index を SubMix の実際の Bus の Index に変換します
    inline int ConvertAuxBusIndexToSubMixBusIndex(nn::atk::AuxBus bus)
    {
        return bus + 1;
    }
}

namespace nn {
namespace atk {

namespace {

//
// AuxDeviceSet
//

enum AuxDeviceSet
{
    AuxDeviceSet_MainAuxA,
    AuxDeviceSet_MainAuxB,
    AuxDeviceSet_MainAuxC,
    AuxDeviceSet_DrcAuxA,
    AuxDeviceSet_DrcAuxB,
    AuxDeviceSet_DrcAuxC,
    AuxDeviceSet_Drc0AuxA = AuxDeviceSet_DrcAuxA,
    AuxDeviceSet_Drc0AuxB = AuxDeviceSet_DrcAuxB,
    AuxDeviceSet_Drc0AuxC = AuxDeviceSet_DrcAuxC,
    AuxDeviceSet_Drc1AuxA,
    AuxDeviceSet_Drc1AuxB,
    AuxDeviceSet_Drc1AuxC,
    AuxDeviceSet_Count,
    AuxDeviceSet_Invalid = -1
};

#if 0 // 参考用 : NW4F 実装
AuxDeviceSet GetAuxDeviceSet( AuxBus bus, OutputDevice device )
{
    NN_SDK_ASSERT_RANGE(bus, 0, AuxBus_Count);
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);

    AuxDeviceSet result = AuxDeviceSet_Invalid;
    switch ( device )
    {
    case OutputDevice_Main:
        result = static_cast<AuxDeviceSet>(AuxDeviceSet_MainAuxA + bus);
        break;
    default:
        break;
    }
    return result;
}

AuxBus GetAuxBusFromAuxDeviceSet( AuxDeviceSet set )
{
    AuxBus bus = static_cast<AuxBus>(set % AuxBus_Count);
    return bus;
}

OutputDevice GetDeviceFromAuxDeviceSet( AuxDeviceSet set )
{
    OutputDevice device = static_cast<OutputDevice>(set / AuxBus_Count);
    return device;
}

//
// GetAxAuxId
//
uint32_t GetAxAuxId( AuxBus bus )
{
    return static_cast<uint32_t>(bus);
}
#endif

void SetupDownMixParameter(nn::audio::DeviceSinkType::DownMixParameter* pOutValue)
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    for (auto i = 0; i < nn::audio::DeviceSinkType::DownMixParameter::CoeffCount; ++i)
    {
        // frontLeft/Right(i == 0) と rearLeft/Right(i == 3) の場合のみ 1.0f とする
        if (i == 0 || i == 3)
        {
            pOutValue->coeff[i] = 1.0f;
        }
        else
        {
            pOutValue->coeff[i] = 0.0f;
        }
    }
}

} // namespace

namespace detail {
namespace driver {

const BiquadFilterLpf     BiquadFilterInstanceLpf;
const BiquadFilterHpf     BiquadFilterInstanceHpf;
const BiquadFilterBpf512  BiquadFilterInstanceBpf512;
const BiquadFilterBpf1024 BiquadFilterInstanceBpf1024;
const BiquadFilterBpf2048 BiquadFilterInstanceBpf2048;
const BiquadFilterLpfNw4fCompatible48k     BiquadFilterInstanceLpfNw4fCompatible48k;
const BiquadFilterHpfNw4fCompatible48k     BiquadFilterInstanceHpfNw4fCompatible48k;
const BiquadFilterBpf512Nw4fCompatible48k  BiquadFilterInstanceBpf512Nw4fCompatible48k;
const BiquadFilterBpf1024Nw4fCompatible48k BiquadFilterInstanceBpf1024Nw4fCompatible48k;
const BiquadFilterBpf2048Nw4fCompatible48k BiquadFilterInstanceBpf2048Nw4fCompatible48k;

NN_DEFINE_STATIC_CONSTANT( const uint32_t HardwareManager::SoundFrameIntervalMsec );
NN_DEFINE_STATIC_CONSTANT( const uint32_t HardwareManager::SoundFrameIntervalUsec );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::DefaultRendererSampleRate );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::DefaultRendererUserEffectCount );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::DefaultRendererVoiceCountMax );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::DefaultRecordingAudioFrameCount );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::AtkVoiceCountMax );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::MixerCount );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::ChannelCountMax );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::BusCount );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::DefaultRendererSampleCount );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::DefaultRendererMixBufferCount );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::DefaultRendererSubMixCount );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::DefaultRendererSinkCount );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::DefaultRendererPerformanceFrameCount );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::DefaultRendererSystemEffectCount );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::SubMixCountMax );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::SubMixCountForAdditionalEffect );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::ChannelCountForAdditionalEffect );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::AuxBusCountForAdditionalEffect );
NN_DEFINE_STATIC_CONSTANT( const int HardwareManager::MixBufferCountForAdditionalEffect );
NN_DEFINE_STATIC_CONSTANT( const bool HardwareManager::DefaultRendererIsVoiceDropEnabled );

HardwareManager::EffectAuxListScopedLock::EffectAuxListScopedLock() NN_NOEXCEPT
{
    HardwareManager::GetInstance().LockEffectAuxList();
}
HardwareManager::EffectAuxListScopedLock::~EffectAuxListScopedLock() NN_NOEXCEPT
{
    HardwareManager::GetInstance().UnlockEffectAuxList();
}
HardwareManager::EffectAuxListForFinalMixScopedLock::EffectAuxListForFinalMixScopedLock() NN_NOEXCEPT
{
    HardwareManager::GetInstance().LockEffectAuxListForFinalMix();
}
HardwareManager::EffectAuxListForFinalMixScopedLock::~EffectAuxListForFinalMixScopedLock() NN_NOEXCEPT
{
    HardwareManager::GetInstance().UnlockEffectAuxListForFinalMix();
}
HardwareManager::EffectAuxListForAdditionalSubMixScopedLock::EffectAuxListForAdditionalSubMixScopedLock() NN_NOEXCEPT
{
    HardwareManager::GetInstance().LockEffectAuxListForAdditionalSubMix();
}
HardwareManager::EffectAuxListForAdditionalSubMixScopedLock::~EffectAuxListForAdditionalSubMixScopedLock() NN_NOEXCEPT
{
    HardwareManager::GetInstance().UnlockEffectAuxListForAdditionalSubMix();
}
HardwareManager::SubMixListScopedLock::SubMixListScopedLock() NN_NOEXCEPT
{
    HardwareManager::GetInstance().LockSubMixList();
}
HardwareManager::SubMixListScopedLock::~SubMixListScopedLock() NN_NOEXCEPT
{
    HardwareManager::GetInstance().UnlockSubMixList();
}

void HardwareManager::HardwareManagerParameter::SetSubMixParameter(bool isStereoModeEnabled, bool isEffectEnabled, bool isSubMixEnabled, bool isAdditionalEffectBusEnabled, bool isAdditionalSubMixEnabled, bool isCustomSubMixEnabled, int customSubMixCount, int customMixTotalChannelCount) NN_NOEXCEPT
{
    //  デフォルト設定のときのみカスタムサブミックスを有効にします。
    m_IsCustomSubMixEnabled =
        isCustomSubMixEnabled &&
        isSubMixEnabled &&
        isAdditionalEffectBusEnabled == false &&
        isAdditionalSubMixEnabled == false;
    m_IsPresetSubMixEnabled = isSubMixEnabled && !m_IsCustomSubMixEnabled;
    m_IsAdditionalEffectBusEnabled = isAdditionalEffectBusEnabled && !m_IsCustomSubMixEnabled;
    m_IsAdditionalSubMixEnabled = isAdditionalSubMixEnabled && !m_IsCustomSubMixEnabled;

    int subMixCount = DefaultRendererSubMixCount;
    int mixBufferCount = DefaultRendererMixBufferCount;
    const int channelCountMax = isStereoModeEnabled ? 2 : ChannelCountMax;

    if( m_IsCustomSubMixEnabled )
    {
        subMixCount = customSubMixCount;
        mixBufferCount = customMixTotalChannelCount + channelCountMax;    //  SubMix の分 + FinalMix の分
    }
    else
    {
        if ( !isEffectEnabled )
        {
            mixBufferCount = 2 * channelCountMax; // FinalMix と SubMix
        }

        if( isAdditionalEffectBusEnabled )
        {
            mixBufferCount += isEffectEnabled ? MixBufferCountForAdditionalEffect : ChannelCountForAdditionalEffect;
            subMixCount    += SubMixCountForAdditionalEffect;
        }

        if( isAdditionalSubMixEnabled )
        {
            mixBufferCount += isEffectEnabled ? channelCountMax : 0;
            subMixCount    += 1;
        }

        if( !isSubMixEnabled )
        {
            mixBufferCount = channelCountMax;
            subMixCount = 0;
        }
    }

    m_SubMixCount = subMixCount;
    m_MixBufferCount = mixBufferCount;
}


/*--------------------------------------------------------------------------------*
    Name:           HardwareManager

    Description:    コンストラクタ

    Arguments:      なし

    Returns:        なし
 *--------------------------------------------------------------------------------*/
HardwareManager::HardwareManager() NN_NOEXCEPT
: m_IsInitialized( false )
, m_IsInitializedEffect( false )
, m_IsPresetSubMixEnabled( true )
, m_IsAdditionalEffectEnabled( false )
, m_IsAdditionalSubMixEnabled( false )
, m_IsStereoModeEnabled( false )
, m_IsInitializedSoundThread( true )
, m_IsPreviousSdkVersionLowPassFilterCompatible( false )
, m_IsMemoryPoolAttachCheckEnabled( false )
, m_RecordingCircularBufferSinkState( CircularBufferSinkState_Invalid )
, m_IsRecordingCircularBufferSinkAllocated( false )
, m_pRecorder( nullptr )
, m_UserCircularBufferSinkState( CircularBufferSinkState_Invalid )
, m_IsUserThreadRenderingEnabled( false )
{
    ResetParameters();
}

void HardwareManager::ResetParameters() NN_NOEXCEPT
{
    m_SrcType = SampleRateConverterType_4Tap;
    m_MasterVolume.InitValue( 1.0f );
    m_VolumeForReset.InitValue( 1.0f );
    for ( int i = 0; i < OutputDevice_Count; i++ )
    {
        // 不正な値にしておく。Initialize で OutputMode は Stereo に、EndUserOutputMode は Surround にセットされる
        m_OutputMode[i] = OutputMode_Count;
        m_EndUserOutputMode[i] = OutputMode_Count;
    }

    // レンダラのパラメータを初期化
    nn::audio::InitializeAudioRendererParameter(&m_AudioRendererParameter);
    m_AudioRendererParameter.sampleRate            = DefaultRendererSampleRate;
    m_AudioRendererParameter.sampleCount           = DefaultRendererSampleCount;
    m_AudioRendererParameter.mixBufferCount        = DefaultRendererMixBufferCount;
    m_AudioRendererParameter.subMixCount           = DefaultRendererSubMixCount;
    m_AudioRendererParameter.voiceCount            = DefaultRendererVoiceCountMax;
    m_AudioRendererParameter.sinkCount             = DefaultRendererSinkCount;
    m_AudioRendererParameter.effectCount           = DefaultRendererSystemEffectCount + DefaultRendererUserEffectCount;
    m_AudioRendererParameter.performanceFrameCount = DefaultRendererPerformanceFrameCount;
    m_AudioRendererParameter.isVoiceDropEnabled    = DefaultRendererIsVoiceDropEnabled;

    // レンダラの初期パラメータが不正でないか確認
    NN_ABORT_UNLESS(nn::audio::IsValidAudioRendererParameter(m_AudioRendererParameter), "Invalid renderer parameters specified." );
}

nn::audio::MemoryPoolType::State HardwareManager::GetMemoryPoolState(nn::audio::MemoryPoolType * pPool) NN_NOEXCEPT
{
    UpdateAudioRendererScopedLock lock;
    return nn::audio::GetMemoryPoolState( pPool );
}

void HardwareManager::SetupAudioRendererParameter(nn::audio::AudioRendererParameter* pOutValue, const HardwareManagerParameter& parameter) const NN_NOEXCEPT
{
    nn::audio::InitializeAudioRendererParameter(pOutValue);
    pOutValue->sampleRate            = parameter.GetRendererSampleRate();
    pOutValue->sampleCount           = parameter.GetRendererSampleRate() * SoundFrameIntervalMsec / 1000;
    pOutValue->mixBufferCount        = parameter.GetMixBufferCount();
    pOutValue->subMixCount           = parameter.GetSubMixCount();
    pOutValue->voiceCount            = parameter.GetVoiceCount();
    pOutValue->sinkCount             = DefaultRendererSinkCount;
    pOutValue->effectCount           = DefaultRendererSystemEffectCount + parameter.GetUserEffectCount();
    pOutValue->performanceFrameCount = DefaultRendererPerformanceFrameCount;
    pOutValue->isVoiceDropEnabled    = parameter.IsVoiceDropEnabled();

    if ( parameter.IsProfilerEnabled() )
    {
        pOutValue->performanceFrameCount = MaxRendererPerformanceFrameCount;
    }

    if ( !parameter.IsEffectEnabled() )
    {
        pOutValue->effectCount = 0;
    }

    if ( parameter.IsRecordingEnabled() )
    {
        //  録音用 CircularBufferSink のために 1 つ増やします
        pOutValue->sinkCount++;
    }

    if ( parameter.IsUserCircularBufferSinkEnabled() )
    {
        pOutValue->sinkCount++;
    }

    if( parameter.IsUserThreadRenderingEnabled() )
    {
        pOutValue->renderingDevice = nn::audio::AudioRendererRenderingDevice_Cpu;
        pOutValue->executionMode = nn::audio::AudioRendererExecutionMode_ManualExecution;
    }
}

size_t HardwareManager::GetRequiredMemSize(const HardwareManagerParameter& hardwareManagerParameter) const NN_NOEXCEPT
{
    const bool isEffectEnabled = hardwareManagerParameter.IsEffectEnabled();
    nn::audio::AudioRendererParameter param;
    SetupAudioRendererParameter(&param, hardwareManagerParameter);

    size_t result = 0;
    result += nn::audio::GetAudioRendererWorkBufferSize(param);
    result += nn::audio::GetAudioRendererConfigWorkBufferSize(param);
    result += FinalMix::GetRequiredMemorySize( isEffectEnabled );

    if( hardwareManagerParameter.IsPresetSubMixEnabled() )
    {
        const int busCount = hardwareManagerParameter.IsEffectEnabled() ? MixerCount + 1 : 1;
        const int channelCountMax = hardwareManagerParameter.IsStereoModeEnabled() ? 2 : ChannelCountMax;

        if( hardwareManagerParameter.IsAdditionalSubMixEnabled() )
        {
            result += m_SubMix[0].GetRequiredMemorySize( busCount, channelCountMax, AdditionalSubMixBusCount, channelCountMax, isEffectEnabled, true );
            // アライメント調整用
            result += nn::atk::SubMix::SubMixAlignment;
            result += m_AdditionalSubMix.GetRequiredMemorySize( AdditionalSubMixBusCount, channelCountMax, FinalMixBusCount, channelCountMax, isEffectEnabled, true );
        }
        else
        {
            result += m_SubMix[0].GetRequiredMemorySize( busCount, channelCountMax, FinalMixBusCount, channelCountMax, isEffectEnabled, true );
        }

        if( hardwareManagerParameter.IsAdditionalEffectBusEnabled() )
        {
            // アライメント調整用
            result += nn::atk::SubMix::SubMixAlignment;
            result += m_SubMix[1].GetRequiredMemorySize( AuxBusCountForAdditionalEffect + 1, ChannelCountForAdditionalEffect, FinalMixBusCount, channelCountMax, isEffectEnabled, true );
        }
    }

    return result;
}

size_t HardwareManager::GetRequiredMemSizeForMemoryPool(int voiceCount) const NN_NOEXCEPT
{
    size_t voiceBufferSize  = LowLevelVoiceAllocator::GetRequiredMemSize(voiceCount);

    return voiceBufferSize;
}

size_t HardwareManager::GetRequiredRecorderWorkBufferSize(const HardwareManagerParameter& hardwareManagerParameter) const NN_NOEXCEPT
{
    size_t result = 0;

    result += GetRequiredCircularBufferSinkWithMemoryPoolBufferSize(hardwareManagerParameter);  //  CircularBufferSink に割り当てるメモリの分
    result += GetRequiredCircularBufferSinkBufferSize(hardwareManagerParameter);                //  CircularBufferSink から波形を取り出すバッファの分

    return result;
}

size_t HardwareManager::GetRequiredCircularBufferSinkWithMemoryPoolBufferSize(const HardwareManagerParameter& hardwareManagerParameter) const NN_NOEXCEPT
{
    size_t result = GetRequiredCircularBufferSinkBufferSize(hardwareManagerParameter);

    // メモリプールにアタッチするため
    result = nn::util::align_up( result, nn::audio::MemoryPoolType::SizeGranularity );
    result += nn::audio::MemoryPoolType::AddressAlignment;

    return result;
}

size_t HardwareManager::GetRequiredCircularBufferSinkBufferSize(const HardwareManagerParameter& hardwareManagerParameter) const NN_NOEXCEPT
{
    nn::audio::AudioRendererParameter param;
    SetupAudioRendererParameter(&param, hardwareManagerParameter);
    return nn::audio::GetRequiredBufferSizeForCircularBufferSink(&param, GetChannelCountMax(), hardwareManagerParameter.GetRecordingAudioFrameCount(), nn::audio::SampleFormat_PcmInt16);
}

bool HardwareManager::RegisterRecorder(nn::atk::DeviceOutRecorder* pRecorder) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pRecorder );

    //  SoundThread から呼ばれる HardwareManager::UpdateRecorder() と排他制御します
    nn::atk::detail::driver::SoundThreadLock lock;

    if ( m_pRecorder != nullptr )
    {
        NN_ATK_WARNING("Recorder is already registered.");
        return false;
    }

    m_pRecorder = pRecorder;
    return true;
}

void HardwareManager::UnregisterRecorder(nn::atk::DeviceOutRecorder* pRecorder) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pRecorder );
    NN_UNUSED( pRecorder );

    //  SoundThread から呼ばれる HardwareManager::UpdateRecorder() と排他制御します
    nn::atk::detail::driver::SoundThreadLock lock;

    // 登録した Recorder と同じものであるかを確認します
    NN_SDK_ASSERT_EQUAL( m_pRecorder, pRecorder );
    m_pRecorder = nullptr;
}

void HardwareManager::UpdateRecorder() NN_NOEXCEPT
{
    if( m_pRecorder != nullptr && m_RecordingCircularBufferSinkState == CircularBufferSinkState_Started && !m_IsRecordingCircularBufferSinkAllocated )
    {
        // ・レコーダが登録されている
        // ・録音が有効である
        // ・録音用 Sink が HardwareManager 以外で使われていない
        // 以上の 3 つを満たすとき、レコーダへ録音します
        int16_t* buffer = reinterpret_cast<int16_t*>( m_RecordingBuffer );
        size_t size = 0;
        size = ReadRecordingCircularBufferSink( buffer, m_RecordingBufferSize );

        if( size != 0 ){
            m_pRecorder->RecordSamples( buffer, static_cast<uint32_t>( size / sizeof(int16_t) ) );
        }
    }
}

nn::audio::CircularBufferSinkType* HardwareManager::AllocateRecordingCircularBufferSink() NN_NOEXCEPT
{
    if ( m_RecordingCircularBufferSinkState == CircularBufferSinkState_Invalid )
    {
        NN_ATK_WARNING("Recording is not enabled.");
        return nullptr;
    }

    if ( m_IsRecordingCircularBufferSinkAllocated )
    {
        return nullptr;
    }

    // 今は録音用の CircularBufferSink を 1 つしか用意していないため、
    // フラグを用いてインスタンスの Alloc/Free を管理します。
    m_IsRecordingCircularBufferSinkAllocated = true;
    return &m_RecordingCircularBufferSink;
}
void HardwareManager::FreeRecordingCircularBufferSink(nn::audio::CircularBufferSinkType* pSink) NN_NOEXCEPT
{
    // &m_RecordingCircularBufferSink != nullptr ですので、以下の ASSERT は pSink の nullptr 確認も含みます
    NN_SDK_ASSERT_EQUAL( &m_RecordingCircularBufferSink, pSink );
    NN_UNUSED( pSink );

    m_IsRecordingCircularBufferSinkAllocated = false;
}

void HardwareManager::StartRecordingCircularBufferSink() NN_NOEXCEPT
{
    {
        UpdateAudioRendererScopedLock lock;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddCircularBufferSink(&m_Config, &m_RecordingCircularBufferSink, m_FinalMix.GetAudioFinalMixInstance(), AtkChannelIndexSettings, GetChannelCountMax(), m_RecordingCircularBufferSinkBuffer, m_RecordingBufferSize, nn::audio::SampleFormat_PcmInt16));
    }

    m_RecordingCircularBufferSinkState = CircularBufferSinkState_Started;
}

size_t HardwareManager::ReadRecordingCircularBufferSink(void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
{
    if( m_RecordingCircularBufferSinkState != CircularBufferSinkState_Started )
    {
        return 0;
    }

    UpdateAudioRendererScopedLock lock;
    return nn::audio::ReadCircularBufferSink(&m_RecordingCircularBufferSink, pOutBuffer, bufferSize);
}

void HardwareManager::StopUserCircularBufferSink() NN_NOEXCEPT
{
    if( m_UserCircularBufferSinkState != CircularBufferSinkState_Started )
    {
        return;
    }

    {
        UpdateAudioRendererScopedLock lock;
        nn::audio::RemoveCircularBufferSink(&m_Config, &m_UserCircularBufferSink, m_FinalMix.GetAudioFinalMixInstance());
    }

    m_UserCircularBufferSinkState = CircularBufferSinkState_Stopped;
}

void HardwareManager::StartUserCircularBufferSink(bool isForceStartMode) NN_NOEXCEPT
{
    if( !isForceStartMode && m_UserCircularBufferSinkState != CircularBufferSinkState_Stopped )
    {
        return;
    }

    {
        UpdateAudioRendererScopedLock lock;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddCircularBufferSink(&m_Config, &m_UserCircularBufferSink, m_FinalMix.GetAudioFinalMixInstance(), AtkChannelIndexSettings, GetChannelCountMax(), m_UserCircularBufferSinkBuffer, m_UserCircularBufferSinkBufferSize, nn::audio::SampleFormat_PcmInt16));
    }

    m_UserCircularBufferSinkState = CircularBufferSinkState_Started;
}

size_t HardwareManager::ReadUserCircularBufferSink(void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
{
    if( m_UserCircularBufferSinkState != CircularBufferSinkState_Started )
    {
        return 0;
    }

    UpdateAudioRendererScopedLock lock;
    return nn::audio::ReadCircularBufferSink(&detail::driver::HardwareManager::GetInstance().GetUserCircularBufferSink(), pOutBuffer, bufferSize);
}

void HardwareManager::AttachMemoryPool( nn::audio::MemoryPoolType* pPool, void* buffer, size_t bufferSize, bool isSoundThreadEnabled ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( buffer );
    NN_SDK_ASSERT_NOT_NULL( pPool );
    {
        UpdateAudioRendererScopedLock lock;

        bool resultForPool = nn::audio::AcquireMemoryPool( &m_Config, pPool, buffer, bufferSize );
        NN_SDK_ASSERT( resultForPool );
        resultForPool = nn::audio::RequestAttachMemoryPool( pPool );
        NN_SDK_ASSERT( resultForPool );
    }

    // MemoryPool がアタッチされるまで待つ
    if ( isSoundThreadEnabled && m_IsInitializedSoundThread && !m_IsUserThreadRenderingEnabled )
    {
        while ( GetMemoryPoolState( pPool ) != nn::audio::MemoryPoolType::State_Attached )
        {
            // サウンドスレッドが有効な場合、オーディオフレームだけ待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(SoundFrameIntervalMsec));
        }
    }
    else
    {
        // サウンドスレッドが有効でない場合、手動でオーディオレンダラの更新を行う
        NN_ABORT_UNLESS_RESULT_SUCCESS( RequestUpdateAudioRenderer() );
    }
}

void HardwareManager::DetachMemoryPool( nn::audio::MemoryPoolType* pPool, bool isSoundThreadEnabled ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pPool );
    {
        UpdateAudioRendererScopedLock lock;
        bool resultForPool = nn::audio::RequestDetachMemoryPool( pPool );
        NN_UNUSED( resultForPool );
        NN_SDK_ASSERT( resultForPool );
    }

    // MemoryPool がデタッチされるまで待つ
    if ( isSoundThreadEnabled && m_IsInitializedSoundThread && !m_IsUserThreadRenderingEnabled )
    {
        // SoundThread を立て、 CPU レンダリングを使用しない通常のケース
        while ( GetMemoryPoolState( pPool ) != nn::audio::MemoryPoolType::State_Detached )
        {
            // サウンドスレッドが有効な場合、オーディオフレームだけ待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(SoundFrameIntervalMsec));
        }
    }
    else
    {
        if( m_IsUserThreadRenderingEnabled )
        {
            // CPU レンダリングを使用するケース
            //
            // nn::atk::SoundSystem::ExecuteRendering をユーザーが呼ぶ場合、
            // SoundThread コールバック経由で呼ぶ場合のどちらにも対応するため、
            // SoundThread の処理を一旦止めて処理を行う
            SoundThreadLock threadLock;
            UpdateAudioRendererScopedLock updateLock;
            while ( GetMemoryPoolState( pPool ) != nn::audio::MemoryPoolType::State_Detached )
            {
                // サウンドスレッドをロックして、手動でオーディオレンダラの更新を行う
                NN_ABORT_UNLESS_RESULT_SUCCESS( RequestUpdateAudioRenderer() );
                // ユーザが ExecuteAudioRendererRendering() を呼ばない場合、
                // オーディオレンダラのシステムイベント待機で無限に待ってしまうので手動で呼び出す
                ExecuteAudioRendererRendering();
                WaitAudioRendererEvent();
            }
        }
        else
        {
            // CPU レンダリングを使用せず、 SoundThread を立てないケース
            while ( GetMemoryPoolState( pPool ) != nn::audio::MemoryPoolType::State_Detached )
            {
                // サウンドスレッドが有効でない場合、手動でオーディオレンダラの更新を行う
                NN_ABORT_UNLESS_RESULT_SUCCESS( RequestUpdateAudioRenderer() );
                WaitAudioRendererEvent();
            }
        }
    }

    {
        UpdateAudioRendererScopedLock updateLock;
        nn::audio::ReleaseMemoryPool( &m_Config, pPool );

        if(m_IsUserThreadRenderingEnabled)
        {
            // CPU レンダリングを使用するケース
            //
            // MemoryPool がデタッチされるまで待つ処理では、 WaitAudioRendererEvent が最後に呼ばれており、
            // このままイベント待ちを行うと無限待ちに陥るため、手動で呼び出す
            SoundThreadLock threadLock;
            ExecuteAudioRendererRendering();
        }
    }
}

void HardwareManager::LockEffectAuxList() NN_NOEXCEPT
{
    m_EffectAuxListLock.Lock();
}
void HardwareManager::UnlockEffectAuxList() NN_NOEXCEPT
{
    m_EffectAuxListLock.Unlock();
}
void HardwareManager::LockEffectAuxListForFinalMix() NN_NOEXCEPT
{
    m_EffectAuxListForFinalMixLock.Lock();
}
void HardwareManager::UnlockEffectAuxListForFinalMix() NN_NOEXCEPT
{
    m_EffectAuxListForFinalMixLock.Unlock();
}
void HardwareManager::LockEffectAuxListForAdditionalSubMix() NN_NOEXCEPT
{
    m_EffectAuxListForAdditionalSubMixLock.Lock();
}
void HardwareManager::UnlockEffectAuxListForAdditionalSubMix() NN_NOEXCEPT
{
    m_EffectAuxListForAdditionalSubMixLock.Unlock();
}
void HardwareManager::LockSubMixList() NN_NOEXCEPT
{
    m_SubMixListLock.Lock();
}
void HardwareManager::UnlockSubMixList() NN_NOEXCEPT
{
    m_SubMixListLock.Unlock();
}

int HardwareManager::GetDroppedLowLevelVoiceCount() const NN_NOEXCEPT
{
    return m_LowLevelVoiceAllocator.GetDroppedVoiceCount();
}

int64_t HardwareManager::GetElapsedAudioFrameCount() const NN_NOEXCEPT
{
    return nn::audio::GetAudioRendererElapsedFrameCount(&m_Config);
}

size_t HardwareManager::GetRequiredPerformanceFramesBufferSize(const HardwareManagerParameter& hardwareManagerParameter) const NN_NOEXCEPT
{
    nn::audio::AudioRendererParameter param;
    SetupAudioRendererParameter(&param, hardwareManagerParameter);
    size_t bufferSize = nn::audio::GetRequiredBufferSizeForPerformanceFrames(param);
    return nn::util::align_up(bufferSize, nn::audio::BufferAlignSize);
}

nn::Result HardwareManager::Initialize(
    void* buffer,
    size_t bufferSize,
    void* memoryPoolBuffer,
    size_t memoryPoolBufferSize,
    void* circularBufferSinkBuffer,
    size_t circularBufferSinkBufferSize,
    const HardwareManagerParameter& parameter) NN_NOEXCEPT
{
    // TODO: 各種初期化失敗時に巻き戻して false を返す

    NN_UNUSED( bufferSize );
    NN_UNUSED( memoryPoolBufferSize );
    NN_UNUSED( circularBufferSinkBufferSize );

    if ( m_IsInitialized )
    {
        // SoundSystem で二重初期化のチェックを行っているため、通常ここには到達しない。
        NN_SDK_ASSERT(false);
        NN_DETAIL_ATK_INFO("HardwareManager is already initialized\n");
        return nn::atk::ResultUnknown();
    }

    m_IsInitializedEffect = parameter.IsEffectEnabled();
    m_IsPresetSubMixEnabled = parameter.IsPresetSubMixEnabled();
    m_IsAdditionalEffectEnabled = parameter.IsAdditionalEffectBusEnabled();
    m_IsAdditionalSubMixEnabled = parameter.IsAdditionalSubMixEnabled();
    m_IsStereoModeEnabled = parameter.IsStereoModeEnabled();
    m_IsMemoryPoolAttachCheckEnabled = parameter.IsMemoryPoolAttachCheckEnabled();

    SetupAudioRendererParameter(&m_AudioRendererParameter, parameter);
    // レンダラのパラメータが不正でないか確認
    NN_ABORT_UNLESS(nn::audio::IsValidAudioRendererParameter(m_AudioRendererParameter), "Invalid renderer parameters specified." );

    nn::util::BytePtr ptrWorkBuffer( buffer );
    nn::util::BytePtr ptrMemoryPool( memoryPoolBuffer );
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(m_AudioRendererParameter);
    m_pAudioRendererWorkBuffer = ptrWorkBuffer.Get();
    ptrWorkBuffer.Advance( workBufferSize );

    // レンダラを初期化します。
    nn::Result result = nn::audio::OpenAudioRenderer(&m_RendererHandle, &m_SystemEvent, m_AudioRendererParameter, m_pAudioRendererWorkBuffer, workBufferSize);

    if( result.IsFailure() )
    {
        NN_DETAIL_ATK_INFO("Failed to open AudioRenderer\n");
        m_pAudioRendererWorkBuffer = nullptr;
        return result;
    }
    m_AudioRendererUpdateCount = 0;

    // AudioRendererConfig を初期化
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(m_AudioRendererParameter);
    m_pAudioRendererConfigWorkBuffer = ptrWorkBuffer.Get();
    ptrWorkBuffer.Advance( configBufferSize );
    nn::audio::InitializeAudioRendererConfig(&m_Config, m_AudioRendererParameter, m_pAudioRendererConfigWorkBuffer, configBufferSize);

    // レンダラの出力先を用意
    const bool isEffectEnabled = m_IsInitializedEffect;
    const size_t finalMixBufferSize = FinalMix::GetRequiredMemorySize( isEffectEnabled );
    m_FinalMix.Initialize( &m_Config, GetChannelCountMax(), isEffectEnabled, ptrWorkBuffer.Get(), finalMixBufferSize );
    ptrWorkBuffer.Advance( finalMixBufferSize );
    if( !parameter.IsUserThreadRenderingEnabled() )
    {
        nn::audio::AddDeviceSink(&m_Config, &m_Sink, m_FinalMix.GetAudioFinalMixInstance(), AudioChannelIndexSettings, GetChannelCountMax(), "MainAudioOut");

        // ダウンミックスの設定
        if (parameter.IsCompatibleDownMixSettingEnabled())
        {
            nn::audio::DeviceSinkType::DownMixParameter downMixParameter;
            SetupDownMixParameter(&downMixParameter);
            nn::audio::SetDownMixParameter(&m_Sink, &downMixParameter);
            nn::audio::SetDownMixParameterEnabled(&m_Sink, true);
        }
        else
        {
            nn::audio::SetDownMixParameterEnabled(&m_Sink, false);
        }
    }

    // 録音用の CircularBufferSink を追加します
    if ( parameter.IsRecordingEnabled() )
    {
        // 録音用のメモリプールのアタッチを行います
        ptrWorkBuffer.AlignUp( nn::audio::MemoryPoolType::AddressAlignment );
        m_RecordingBufferSize = nn::audio::GetRequiredBufferSizeForCircularBufferSink( &m_AudioRendererParameter, GetChannelCountMax(), parameter.GetRecordingAudioFrameCount(), nn::audio::SampleFormat_PcmInt16 );
        const size_t granularitySize = nn::util::align_up( m_RecordingBufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        m_RecordingCircularBufferSinkBuffer = ptrWorkBuffer.Get();
        AttachMemoryPool(&m_RecordingCircularBufferSinkMemoryPool, m_RecordingCircularBufferSinkBuffer, granularitySize, false);

        StartRecordingCircularBufferSink();
        ptrWorkBuffer.Advance( granularitySize );

        // CircularBufferSink から波形を受け取る用のバッファサイズは
        // CircularBufferSink に割り当てた分と同じだけ確保します
        m_RecordingBuffer = ptrWorkBuffer.Get();
        ptrWorkBuffer.Advance( m_RecordingBufferSize );
    }

    if ( parameter.IsUserCircularBufferSinkEnabled() )
    {
        NN_SDK_ASSERT_NOT_NULL(circularBufferSinkBuffer);

        nn::util::BytePtr ptrCircularBufferSinkBuffer( circularBufferSinkBuffer );

        // ユーザーが利用するメモリプールのアタッチを行います
        ptrCircularBufferSinkBuffer.AlignUp( nn::audio::MemoryPoolType::AddressAlignment );
        m_UserCircularBufferSinkBufferSize = nn::audio::GetRequiredBufferSizeForCircularBufferSink( &m_AudioRendererParameter, GetChannelCountMax(), parameter.GetRecordingAudioFrameCount(), nn::audio::SampleFormat_PcmInt16 );
        const size_t granularitySize = nn::util::align_up( m_UserCircularBufferSinkBufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        m_UserCircularBufferSinkBuffer = ptrCircularBufferSinkBuffer.Get();
        AttachMemoryPool(&m_UserCircularBufferSinkMemoryPool, m_UserCircularBufferSinkBuffer, granularitySize, false);

        StartUserCircularBufferSink(true);
    }

    m_IsPreviousSdkVersionLowPassFilterCompatible = parameter.IsPreviousSdkVersionLowPassFilterCompatible();
    m_IsInitializedSoundThread = parameter.IsSoundThreadEnabled();
    m_IsCompatibleBusVolumeEnabled = parameter.IsCompatibleBusVolumeEnabled();
    m_IsUserThreadRenderingEnabled = parameter.IsUserThreadRenderingEnabled();

    // 設定したパラメータをレンダラに反映させます。
    NN_ABORT_UNLESS_RESULT_SUCCESS(RequestUpdateAudioRenderer());

    // ボイスを初期化します
    const size_t voiceBufferSize = LowLevelVoiceAllocator::GetRequiredMemSize( parameter.GetVoiceCount() );
    m_LowLevelVoiceAllocator.Initialize( parameter.GetVoiceCount(), ptrMemoryPool.Get(), voiceBufferSize );
    ptrMemoryPool.Advance( voiceBufferSize );

    // レンダリングを開始します。
    nn::audio::StartAudioRenderer(m_RendererHandle);
    m_AudioRendererSuspendCount = 0;

    // biquadフィルタテーブル初期化
    for ( int i = BiquadFilterType_DataMin; i < BiquadFilterType_Max + 1; i++ )
    {
        m_BiquadFilterCallbackTable[i] = NULL;
    }
    SetBiquadFilterCallback( BiquadFilterType_LowPassFilter,      &BiquadFilterInstanceLpf );
    SetBiquadFilterCallback( BiquadFilterType_HighPassFilter,     &BiquadFilterInstanceHpf );
    SetBiquadFilterCallback( BiquadFilterType_BandPassFilter512,  &BiquadFilterInstanceBpf512 );
    SetBiquadFilterCallback( BiquadFilterType_BandPassFilter1024, &BiquadFilterInstanceBpf1024 );
    SetBiquadFilterCallback( BiquadFilterType_BandPassFilter2048, &BiquadFilterInstanceBpf2048 );
    SetBiquadFilterCallback( BiquadFilterType_LowPassFilterNw4fCompatible48k,      &BiquadFilterInstanceLpfNw4fCompatible48k );
    SetBiquadFilterCallback( BiquadFilterType_HighPassFilterNw4fCompatible48k,     &BiquadFilterInstanceHpfNw4fCompatible48k );
    SetBiquadFilterCallback( BiquadFilterType_BandPassFilter512Nw4fCompatible48k,  &BiquadFilterInstanceBpf512Nw4fCompatible48k);
    SetBiquadFilterCallback( BiquadFilterType_BandPassFilter1024Nw4fCompatible48k, &BiquadFilterInstanceBpf1024Nw4fCompatible48k);
    SetBiquadFilterCallback( BiquadFilterType_BandPassFilter2048Nw4fCompatible48k, &BiquadFilterInstanceBpf2048Nw4fCompatible48k);

    // マスターボリューム設定
#if 0 // TODO
    nn::snd::SetMasterVolume( 1.0f );
#endif

#if 0 // 参考用 : NW4F 実装
    // DeviceFinalMixCallbackコールバック設定
    if ( ::AXGetDeviceFinalMixCallback(AX_DEVICE_TV, &m_LastMainFinalMixCallback) == AXPB_ERROR_NONE )
    {
        bool isSuccess = ::AXRegisterDeviceFinalMixCallback(AX_DEVICE_TV, MainFinalMixCallbackFunc) == AXPB_ERROR_NONE;
        if (!isSuccess)
        {
            NN_ATK_WARNING("unexpected result.");
        }
    }

    if ( ::AXGetDeviceFinalMixCallback(AX_DEVICE_DRC, &m_LastDrcFinalMixCallback) == AXPB_ERROR_NONE )
    {
        bool isSuccess = ::AXRegisterDeviceFinalMixCallback(AX_DEVICE_DRC, DrcFinalMixCallbackFunc) == AXPB_ERROR_NONE;
        if (!isSuccess)
        {
            NN_ATK_WARNING("unexpected result.");
        }
    }
#else
    // TODO: FinalMix の対応
#endif

    // TODO: 本体設定の値をもとに m_EndUserOutputMode を初期化するように変更
    m_EndUserOutputMode[OutputDevice_Main] = OutputMode_Surround;
    SetOutputMode( OutputMode_Surround, OutputDevice_Main );

    // 出力先デバイス
    for ( int i = 0; i < OutputLineIndex_Max; i++ )
    {
        m_OutputDeviceFlag[i] = ( i < OutputLineIndex_ReservedMax ) ? 1 << i : 0;
    }

    m_IsInitialized = true;

    // サブミックスの設定
    if ( IsPresetSubMixEnabled() )
    {
        const bool isUnusedEffectChannelMutingEnabled = parameter.IsUnusedEffectChannelMutingEnabled();
        const int busCount = parameter.IsEffectEnabled() ? MixerCount + 1 : 1;
        const int channelCount = GetChannelCountMax();
        const int FinalMixBusIndex = 0;
        NN_SDK_ASSERT_ALIGNED(ptrWorkBuffer.Get(), nn::atk::SubMix::SubMixAlignment);

        if ( !parameter.IsAdditionalSubMixEnabled() )
        {
            // SubMix[0] を FinalMix へルーティングする
            const size_t subMixBufferSize = m_SubMix[0].GetRequiredMemorySize( busCount, channelCount, FinalMixBusCount, channelCount, isEffectEnabled, true );
            m_SubMix[0].Initialize( busCount, channelCount, FinalMixBusCount, channelCount, isEffectEnabled, true, ptrWorkBuffer.Get(), subMixBufferSize );
            m_SubMix[0].SetMuteUnusedEffectChannel( isUnusedEffectChannelMutingEnabled );
            ptrWorkBuffer.Advance( subMixBufferSize );

            m_SubMix[0].SetDestination( &m_FinalMix );
            for(int bus = 0; bus < busCount; bus++)
            {
                m_SubMix[0].SetSend( bus, FinalMixBusIndex, 1.0f );
            }
        }
        else
        {
            // SubMix[0] を AdditionalSubMix へルーティングし、AdditionalSubMix を FinalMix へルーティングする
            const size_t subMixBufferSize = m_SubMix[0].GetRequiredMemorySize( busCount, channelCount, FinalMixBusCount, channelCount, isEffectEnabled, true );
            m_SubMix[0].Initialize( busCount, channelCount, FinalMixBusCount, channelCount, isEffectEnabled, true, ptrWorkBuffer.Get(), subMixBufferSize );
            m_SubMix[0].SetMuteUnusedEffectChannel( isUnusedEffectChannelMutingEnabled );
            ptrWorkBuffer.Advance( subMixBufferSize );

            ptrWorkBuffer.AlignUp( nn::atk::SubMix::SubMixAlignment );
            const size_t additionalSubMixBufferSize = m_AdditionalSubMix.GetRequiredMemorySize( AdditionalSubMixBusCount, channelCount, FinalMixBusCount, channelCount, isEffectEnabled, true );
            m_AdditionalSubMix.Initialize( AdditionalSubMixBusCount, channelCount, FinalMixBusCount, channelCount, isEffectEnabled, true, ptrWorkBuffer.Get(), additionalSubMixBufferSize );
            m_AdditionalSubMix.SetMuteUnusedEffectChannel( isUnusedEffectChannelMutingEnabled );
            ptrWorkBuffer.Advance( additionalSubMixBufferSize );

            m_SubMix[0].SetDestination( &m_AdditionalSubMix );
            m_AdditionalSubMix.SetDestination( &m_FinalMix );

            const int AdditionalSubMixIndex = 0;
            for(int bus = 0; bus < busCount; bus++)
            {
                m_SubMix[0].SetSend( bus, AdditionalSubMixIndex , 1.0f );
            }
            m_AdditionalSubMix.SetSend( AdditionalSubMixIndex, FinalMixBusIndex, 1.0f );
        }

        if( parameter.IsAdditionalEffectBusEnabled() )
        {
            // 追加のエフェクトバスが有効な場合の、追加サブミックスの設定
            ptrWorkBuffer.AlignUp( nn::atk::SubMix::SubMixAlignment );
            const int additionalBusCount = AuxBusCountForAdditionalEffect + 1;
            const size_t effectSubMixBufferSize = m_SubMix[1].GetRequiredMemorySize( additionalBusCount, ChannelCountForAdditionalEffect, FinalMixBusCount, channelCount, isEffectEnabled, true );
            m_SubMix[1].Initialize( additionalBusCount, ChannelCountForAdditionalEffect, FinalMixBusCount, channelCount, isEffectEnabled, true, ptrWorkBuffer.Get(), effectSubMixBufferSize );
            m_SubMix[1].SetMuteUnusedEffectChannel( isUnusedEffectChannelMutingEnabled );
            ptrWorkBuffer.Advance( effectSubMixBufferSize );

            m_SubMix[1].SetDestination( &m_FinalMix );
            for(int bus = 0; bus < additionalBusCount; bus++)
            {
                m_SubMix[1].SetSend( bus, FinalMixBusIndex, 1.0f );
            }
        }
    }

    // 使用メモリ確認。ワークメモリの方はアラインアップがあるために、厳密に確認できません。
    NN_SDK_ASSERT( -ptrWorkBuffer.Distance( buffer ) <= static_cast<ptrdiff_t>( bufferSize ) );
    NN_SDK_ASSERT( -ptrMemoryPool.Distance( memoryPoolBuffer ) == static_cast<ptrdiff_t>( memoryPoolBufferSize ) );

    return nn::ResultSuccess();
} // NOLINT(impl/function_size)

void HardwareManager::SetEndUserOutputMode(OutputMode mode) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(mode, 0, OutputMode_Count);
    m_EndUserOutputMode[OutputDevice_Main] = mode;
}

void HardwareManager::UpdateEndUserOutputMode() NN_NOEXCEPT
{
    // TODO: 本体設定の値をもとに m_EndUserOutputMode を更新する処理を追加
}

void HardwareManager::Finalize() NN_NOEXCEPT
{
    if ( !m_IsInitialized ) return;

    m_LowLevelVoiceAllocator.Finalize();

    nn::audio::StopAudioRenderer(m_RendererHandle);

    // エフェクトの終了処理
    if ( m_IsInitializedEffect )
    {
        if( m_IsPresetSubMixEnabled )
        {
            for(int bus = 0; bus < m_SubMix[0].GetBusCount(); bus++)
            {
                m_SubMix[0].ClearEffectImpl( bus );
            }
            if( m_IsAdditionalEffectEnabled )
            {
                for(int bus = 0; bus < m_SubMix[1].GetBusCount(); bus++)
                {
                    m_SubMix[1].ClearEffectImpl( bus );
                }
            }
            if( m_IsAdditionalSubMixEnabled )
            {
                m_AdditionalSubMix.ClearEffectImpl( 0 );
            }

            m_IsInitializedEffect = false;
        }
        m_FinalMix.ClearEffectImpl( 0 );
    }

    m_SubMix[0].Finalize();
    m_SubMix[1].Finalize();
    m_AdditionalSubMix.Finalize();
    m_SubMixList.clear();

    NN_ABORT_UNLESS_RESULT_SUCCESS(RequestUpdateAudioRenderer());
    m_FinalMix.Finalize( &m_Config );

    nn::audio::CloseAudioRenderer(m_RendererHandle);
    nn::os::DestroySystemEvent(m_SystemEvent.GetBase());

#if 0 // 参考用 : NW4F 実装
    {
        bool isSuccess = ::AXRegisterDeviceFinalMixCallback(AX_DEVICE_TV, m_LastMainFinalMixCallback) == AXPB_ERROR_NONE;
        if (!isSuccess)
        {
            NN_ATK_WARNING("unexpected result.");
        }
        m_LastMainFinalMixCallback = NULL;
    }

    {
        bool isSuccess = ::AXRegisterDeviceFinalMixCallback(AX_DEVICE_DRC, m_LastDrcFinalMixCallback) == AXPB_ERROR_NONE;
        if (!isSuccess)
        {
            NN_ATK_WARNING("unexpected result.");
        }
        m_LastDrcFinalMixCallback = NULL;
    }
#else
    // TODO: FinalMix の対応
#endif

    // 録音用 CircularBufferSink の終了処理
    if ( m_RecordingCircularBufferSinkState != CircularBufferSinkState_Invalid )
    {
        NN_SDK_ASSERT( m_pRecorder == nullptr, "Recording is still running.\n" );
        m_RecordingCircularBufferSinkState =  CircularBufferSinkState_Invalid;

        // m_RecordingCircularBufferSink, m_RecordingCircularBufferSinkMemoryPool は
        // nn::audio::CloseAudioRenderer のときに AudioRenderer との関係が切られるため、
        // 解放処理を行わなくても大丈夫なはずです。
    }

    // メンバ変数をコンストラクタでの初期状態まで戻します
    ResetParameters();
    m_UserCircularBufferSinkState = CircularBufferSinkState_Invalid;
    m_IsInitializedSoundThread = true;
    m_IsInitialized = false;
    m_IsPresetSubMixEnabled = true;
    m_IsAdditionalEffectEnabled = false;
    m_IsAdditionalSubMixEnabled = false;
    m_IsStereoModeEnabled = false;
    m_IsPreviousSdkVersionLowPassFilterCompatible = false;
    m_IsCompatibleBusVolumeEnabled = false;
    m_IsUserThreadRenderingEnabled = false;
    m_IsMemoryPoolAttachCheckEnabled = false;
}

void HardwareManager::Update() NN_NOEXCEPT
{
    fnd::ScopedLock<fnd::CriticalSection> lock(m_UpdateHardwareManagerLock);
    if (m_IsInitializedEffect && IsPresetSubMixEnabled() )
    {
        if( m_IsCompatibleBusVolumeEnabled )
        {
            // NW4F 互換フラグが有効な場合、エフェクト未使用のバスはミュートする
            for (int bus = 0; bus < AuxBus_Count; bus++)
            {
                const AuxBus auxBus = static_cast<AuxBus>(bus);
                const int busIdx = ConvertAuxBusIndexToSubMixBusIndex( auxBus );
                m_SubMix[0].SetBusMute( busIdx, !m_SubMix[0].HasEffect( busIdx ) );
            }

            if (m_IsAdditionalEffectEnabled)
            {
                for (int bus = 0; bus < AuxBusCountForAdditionalEffect; bus++)
                {
                    const AuxBus auxBus = static_cast<AuxBus>(bus);
                    const int busIdx = ConvertAuxBusIndexToSubMixBusIndex( auxBus );
                    m_SubMix[1].SetBusMute( busIdx, !m_SubMix[1].HasEffect( busIdx ) );
                }
            }
        }
    }

    //  SubMix の更新
    {
        SubMixListScopedLock subMixListlock;
        for(auto itr = m_SubMixList.begin(); itr != m_SubMixList.end(); ++itr)
        {
            itr->Update();
        }
    }

    // 音量のアップデート
    if ( ! m_MasterVolume.IsFinished() ) {
        m_MasterVolume.Update();
        MultiVoiceManager::GetInstance().UpdateAllVoicesSync( MultiVoice::UpdateVe );
    }

    if ( ! m_VolumeForReset.IsFinished() ) {
        m_VolumeForReset.Update();
        MultiVoiceManager::GetInstance().UpdateAllVoicesSync( MultiVoice::UpdateVe );
    }
}

void HardwareManager::UpdateEffect() NN_NOEXCEPT
{
    if ( !m_IsInitializedEffect )
    {
        return;
    }

    {
        SubMixListScopedLock lock;
        for(auto itr = m_SubMixList.begin(); itr != m_SubMixList.end(); ++itr)
        {
            itr->UpdateEffectAux();
        }
    }
    m_FinalMix.UpdateEffectAux();
}

void HardwareManager::SuspendAudioRenderer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized, "AudioRenderer is not initialized.");
    UpdateAudioRendererScopedLock lock;
    if (m_AudioRendererSuspendCount == 0)
    {
        nn::audio::StopAudioRenderer(m_RendererHandle);
    }
    m_AudioRendererSuspendCount++;
}

void HardwareManager::ResumeAudioRenderer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized, "AudioRenderer is not initialized.");
    if (m_AudioRendererSuspendCount == 0)
    {
        return;
    }

    UpdateAudioRendererScopedLock lock;
    m_AudioRendererSuspendCount--;
    if (m_AudioRendererSuspendCount == 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(m_RendererHandle));
    }
}

void HardwareManager::ExecuteAudioRendererRendering() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized, "AudioRenderer is not initialized.");

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ExecuteAudioRendererRendering(m_RendererHandle));
}

nn::Result HardwareManager::RequestUpdateAudioRenderer() NN_NOEXCEPT
{
    UpdateAudioRendererScopedLock lock;
    nn::Result result = nn::audio::RequestUpdateAudioRenderer(m_RendererHandle, &m_Config);
    m_AudioRendererUpdateCount.fetch_add(1);
    if (result.IsFailure())
    {
        if (nn::audio::ResultNoMemoryPoolEntry::Includes(result))
        {
            if (m_IsMemoryPoolAttachCheckEnabled)
            {
                NN_ABORT("Failed to request UpdateRenderer. Some of audio buffers (waveforms or effect work buffers)  are not managed by any nn::audio::MemoryPool.");
            }
            return nn::ResultSuccess();
        }
        else if (nn::audio::ResultInvalidUpdateInfo::Includes(result))
        {
            NN_ABORT("Failed to request UpdateRenderer. Invalid parameters are found.");
        }
    }
    return result;
}

void HardwareManager::WaitAudioRendererEvent() NN_NOEXCEPT
{
    m_SystemEvent.Wait();
}

bool HardwareManager::TimedWaitAudioRendererEvent(nn::TimeSpan timeout) NN_NOEXCEPT
{
    return m_SystemEvent.TimedWait(timeout);
}

nn::Result HardwareManager::SetAudioRendererRenderingTimeLimit(int limitPercent) NN_NOEXCEPT
{
    UpdateAudioRendererScopedLock lock;
    return nn::audio::SetAudioRendererRenderingTimeLimit(m_RendererHandle, limitPercent);
}

int HardwareManager::GetAudioRendererRenderingTimeLimit() NN_NOEXCEPT
{
    UpdateAudioRendererScopedLock lock;
    return nn::audio::GetAudioRendererRenderingTimeLimit(m_RendererHandle);
}

/* ========================================================================
        リセット前準備
   ======================================================================== */

void HardwareManager::PrepareReset() NN_NOEXCEPT
{
    // if ( m_OldAidCallback != NULL ) return;

    m_VolumeForReset.SetTarget( 0.0f, 3 ); // ３オーディオフレームかけてフェードアウト
    // m_ResetReadyCounter = -1;
    // m_OldAidCallback = ::AI_RegisterDMACallback( AiDmaCallbackFunc );
}

bool HardwareManager::IsResetReady() const NN_NOEXCEPT
{
    return true; // m_ResetReadyCounter == 0 ? true : false;
}

void HardwareManager::AddSubMix(SubMix* pSubMix) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pSubMix );
    SubMixListScopedLock lock;
    m_SubMixList.push_back( *pSubMix );
}

void HardwareManager::RemoveSubMix(SubMix* pSubMix) NN_NOEXCEPT
{
    SubMixListScopedLock lock;
    for(auto itr = m_SubMixList.begin(); itr != m_SubMixList.end(); ++itr)
    {
        if( pSubMix == &( *itr ) )
        {
            m_SubMixList.erase( itr );
            break;
        }
    }
}

SubMix& HardwareManager::GetSubMix(int subMixNumber) NN_NOEXCEPT
{
    if (!IsPresetSubMixEnabled())
    {
        NN_SDK_ASSERT_RANGE(subMixNumber, 0, m_SubMixList.size());

        SubMixListScopedLock lock;
        auto index = 0;
        for(auto itr = m_SubMixList.begin(); itr != m_SubMixList.end(); ++itr)
        {
            if (index == subMixNumber)
            {
                return *itr;
            }
            ++index;
        }
    }

    NN_SDK_ASSERT(IsPresetSubMixEnabled(), "SubMix is not initialized\n");
    NN_SDK_ASSERT_MINMAX(subMixNumber, 0, SubMixCountMax);
    return m_SubMix[subMixNumber];
}

const SubMix& HardwareManager::GetSubMix(int subMixNumber) const NN_NOEXCEPT
{
    if (!IsPresetSubMixEnabled())
    {
        NN_SDK_ASSERT_RANGE(subMixNumber, 0, GetSubMixCount());

        SubMixListScopedLock lock;
        auto index = 0;
        for(auto itr = m_SubMixList.begin(); itr != m_SubMixList.end(); ++itr)
        {
            if (index == subMixNumber)
            {
                return *itr;
            }
            ++index;
        }
    }

    NN_SDK_ASSERT(IsPresetSubMixEnabled(), "SubMix is not initialized\n");
    NN_SDK_ASSERT_MINMAX(subMixNumber, 0, SubMixCountMax);
    return m_SubMix[subMixNumber];
}

int HardwareManager::GetSubMixCount() const NN_NOEXCEPT
{
    return m_SubMixList.size();
}

int HardwareManager::GetChannelCount() const NN_NOEXCEPT
{
    switch(nn::atk::detail::driver::HardwareManager::GetInstance().GetOutputMode())
    {
        case nn::atk::OutputMode_Monaural:
            return 1;
        case nn::atk::OutputMode_Stereo:
            return 2;
        case nn::atk::OutputMode_Surround:
            if (IsStereoModeEnabled())
            {
                return 2;
            }
            else
            {
                return 6;
            }
        default:
            break;
    }

    return 2;
}

int HardwareManager::GetChannelCountMax() const NN_NOEXCEPT
{
    if (IsStereoModeEnabled())
    {
        return 2;
    }
    else
    {
        return ChannelCountMax;
    }
}

void HardwareManager::SetOutputMode( OutputMode mode, OutputDevice device ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(device, 0, OutputDevice_Count);

    if ( m_OutputMode[device] == mode )
    {
        return;
    }

    m_OutputMode[device] = mode;

    // コマンドを投げる
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandAllVoicesSync* command =
            cmdmgr.AllocCommand<DriverCommandAllVoicesSync>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_AllVoicesSync;
        command->syncFlag = MultiVoice::UpdateMix;
        cmdmgr.PushCommand(command);
    }

    // Aux エフェクトの出力モード変更処理
    if( m_IsInitializedEffect )
    {
        {
            SubMixListScopedLock lock;
            for(auto itr = m_SubMixList.begin(); itr != m_SubMixList.end(); ++itr)
            {
                itr->OnChangeOutputMode();
            }
        }
        m_FinalMix.OnChangeOutputMode();
    }
}

float HardwareManager::GetOutputVolume() const NN_NOEXCEPT
{
    float volume = m_MasterVolume.GetValue();
    volume *= m_VolumeForReset.GetValue();
    return volume;
}

void HardwareManager::SetOutputDeviceFlag( uint32_t outputLineIndex, uint8_t outputDeviceFlag ) NN_NOEXCEPT
{
    if ( outputLineIndex >= OutputLineIndex_ReservedMax && outputLineIndex < OutputLineIndex_Max )
    {
        if ( m_OutputDeviceFlag[outputLineIndex] == outputDeviceFlag )
        {
            return;
        }
        m_OutputDeviceFlag[outputLineIndex] = outputDeviceFlag;

        // コマンドを投げる
        {
            DriverCommand& cmdmgr = DriverCommand::GetInstance();
            DriverCommandAllVoicesSync* command =
                cmdmgr.AllocCommand<DriverCommandAllVoicesSync>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
            if (command == nullptr)
            {
                return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
            }
#else
            NN_SDK_ASSERT_NOT_NULL(command);
#endif
            command->id = DriverCommandId_AllVoicesSync;
            command->syncFlag = MultiVoice::UpdateMix;
            cmdmgr.PushCommand(command);
        }
    }
}

void HardwareManager::SetMasterVolume( float volume, int fadeTimes ) NN_NOEXCEPT
{
    if ( volume < 0.0f ) volume = 0.0f;
    m_MasterVolume.SetTarget(
        volume,
        ( fadeTimes + HardwareManager::SoundFrameIntervalMsec - 1 )
        / SoundFrameIntervalMsec
    );

    if ( fadeTimes == 0 )
    {
        // コマンドを投げる
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandAllVoicesSync* command =
            cmdmgr.AllocCommand<DriverCommandAllVoicesSync>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_AllVoicesSync;
        command->syncFlag = MultiVoice::UpdateVe;
        cmdmgr.PushCommand(command);
    }
}

/* ========================================================================
        SRC タイプ
   ======================================================================== */

/*--------------------------------------------------------------------------------*
    Name:           SetSrcType

    Description:    SRC タイプを変更する

    Arguments:      type: SRC タイプ

    Returns:        なし
 *--------------------------------------------------------------------------------*/
void HardwareManager::SetSrcType( SampleRateConverterType type ) NN_NOEXCEPT
{
    if ( m_SrcType == type ) return;

    m_SrcType = type;

    // コマンドを投げる
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandAllVoicesSync* command =
            cmdmgr.AllocCommand<DriverCommandAllVoicesSync>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_AllVoicesSync;
        command->syncFlag = MultiVoice::UpdateSrc;
        cmdmgr.PushCommand(command);
    }
}


/* ========================================================================
        エフェクト
   ======================================================================== */

size_t HardwareManager::GetRequiredEffectAuxBufferSize(const EffectAux* pEffect) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );
    NN_SDK_ASSERT_NOT_NULL(pEffect);
    return pEffect->GetRequiredMemSize( m_AudioRendererParameter );
}

void HardwareManager::SetAuxBusVolume(AuxBus bus, float volume, int fadeFrames, int subMixIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(bus, AuxBus_A, AuxBus_Count);
    NN_SDK_ASSERT_RANGE(subMixIndex, 0, SubMixCountMax);
    NN_SDK_ASSERT(m_IsInitializedEffect, "Effect is not initialized\n");

    float clampedVolume = detail::fnd::Clamp(volume, nn::audio::SubMixType::GetVolumeMin(), nn::audio::SubMixType::GetVolumeMax());

    if (subMixIndex == 0) // Default AuxBus
    {
        m_SubMix[0].SetBusVolume( ConvertAuxBusIndexToSubMixBusIndex( bus ), clampedVolume, fadeFrames );
    }
    else // Additional AuxBus
    {
        NN_SDK_REQUIRES( m_IsAdditionalEffectEnabled );
        NN_SDK_ASSERT(m_IsAdditionalEffectEnabled, "Additional Effect bus is not initialized\n");
        NN_SDK_ASSERT_RANGE(bus, AuxBus_A, AuxBusCountForAdditionalEffect);
        m_SubMix[1].SetBusVolume( ConvertAuxBusIndexToSubMixBusIndex( bus ), clampedVolume, fadeFrames );
    }

    // fadeFrames == 0 の場合は Update() での更新処理が行われないため、このタイミングで AuxBus のボリューム値を変更する
    if (fadeFrames == 0)
    {
        m_SubMix[subMixIndex].UpdateBusMixVolume( ConvertAuxBusIndexToSubMixBusIndex( bus ) );
    }
}

float HardwareManager::GetAuxBusVolume(AuxBus bus, int subMixIndex) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( subMixIndex, 0, SubMixCountMax );
    NN_SDK_ASSERT(m_IsInitializedEffect, "Effect is not initialized\n");

    fnd::ScopedLock<fnd::CriticalSection> lock(m_UpdateHardwareManagerLock);

    if(subMixIndex == 0) // Default AuxBus
    {
        NN_SDK_ASSERT_RANGE( bus, AuxBus_A, AuxBus_Count );
        return m_SubMix[0].GetBusVolume( ConvertAuxBusIndexToSubMixBusIndex( bus ) );
    }
    else // Additional AuxBus
    {
        NN_SDK_REQUIRES( m_IsAdditionalEffectEnabled );
        NN_SDK_ASSERT(m_IsAdditionalEffectEnabled, "Additional Effect bus is not initialized\n");
        NN_SDK_ASSERT_RANGE( bus, AuxBus_A, AuxBusCountForAdditionalEffect );
        return m_SubMix[1].GetBusVolume( ConvertAuxBusIndexToSubMixBusIndex( bus ) );
    }
}

void HardwareManager::SetMainBusChannelVolumeForAdditionalEffect(float volume, int srcChannelIndex, int destChannelIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(srcChannelIndex, 0, ChannelCountForAdditionalEffect);
    NN_SDK_ASSERT_RANGE(destChannelIndex, 0, ChannelCountMax);
    NN_SDK_ASSERT(m_IsInitializedEffect, "Effect is not initialized\n");
    NN_SDK_ASSERT(m_IsAdditionalEffectEnabled, "Additional Effect bus is not initialized\n");

    float clampedVolume = detail::fnd::Clamp(volume, nn::audio::SubMixType::GetVolumeMin(), nn::audio::SubMixType::GetVolumeMax());
    m_SubMix[1].SetSendImpl( SubMixMainBusIndex, srcChannelIndex, SubMixMainBusIndex, destChannelIndex, clampedVolume );
}

float HardwareManager::GetMainBusChannelVolumeForAdditionalEffect(int srcChannelIndex, int destChannelIndex) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(srcChannelIndex, 0, ChannelCountForAdditionalEffect);
    NN_SDK_ASSERT_RANGE(destChannelIndex, 0, ChannelCountMax);
    NN_SDK_ASSERT(m_IsInitializedEffect, "Effect is not initialized\n");
    NN_SDK_ASSERT(m_IsAdditionalEffectEnabled, "Additional Effect bus is not initialized\n");
    return m_SubMix[1].GetSendImpl( SubMixMainBusIndex, srcChannelIndex, SubMixMainBusIndex, destChannelIndex );
}

void HardwareManager::SetAuxBusChannelVolumeForAdditionalEffect(AuxBus bus, float volume, int srcChannelIndex, int destChannelIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(bus, AuxBus_A, AuxBusCountForAdditionalEffect);
    NN_SDK_ASSERT_RANGE(srcChannelIndex, 0, ChannelCountForAdditionalEffect);
    NN_SDK_ASSERT_RANGE(destChannelIndex, 0, ChannelCountMax);
    NN_SDK_ASSERT(m_IsInitializedEffect, "Effect is not initialized\n");
    NN_SDK_ASSERT(m_IsAdditionalEffectEnabled, "Additional Effect bus is not initialized\n");

    float clampedVolume = detail::fnd::Clamp(volume, nn::audio::SubMixType::GetVolumeMin(), nn::audio::SubMixType::GetVolumeMax());
    m_SubMix[1].SetSendImpl( ConvertAuxBusIndexToSubMixBusIndex( bus ), srcChannelIndex, SubMixMainBusIndex, destChannelIndex, clampedVolume );
}

float HardwareManager::GetAuxBusChannelVolumeForAdditionalEffect(AuxBus bus, int srcChannelIndex, int destChannelIndex) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(bus, AuxBus_A, AuxBusCountForAdditionalEffect);
    NN_SDK_ASSERT_RANGE(srcChannelIndex, 0, ChannelCountForAdditionalEffect);
    NN_SDK_ASSERT_RANGE(destChannelIndex, 0, ChannelCountMax);
    NN_SDK_ASSERT(m_IsInitializedEffect, "Effect is not initialized\n");
    NN_SDK_ASSERT(m_IsAdditionalEffectEnabled, "Additional Effect bus is not initialized\n");
    return m_SubMix[1].GetSendImpl( ConvertAuxBusIndexToSubMixBusIndex( bus ), srcChannelIndex, SubMixMainBusIndex, destChannelIndex );
}

// void HardwareManager::AuxCallbackFunc( int32_t** data, void* context, AXAUXCBSTRUCT* info )
// {
//     nn::os::Tick tick = nn::os::GetSystemTick();

//     AuxDeviceSet set = static_cast<AuxDeviceSet>( reinterpret_cast<uint32_t>(context) );

//     AuxBus bus = GetAuxBusFromAuxDeviceSet(set);
//     OutputDevice device = GetDeviceFromAuxDeviceSet(set);
//     NN_SDK_ASSERT_RANGE( bus, 0, AuxBus_Count );
//     NN_SDK_ASSERT_RANGE( device, 0, OutputDevice_Count );
//     NN_SDK_ASSERT_NOT_NULL(info);

//     void* buffer[6] = {NULL};    // 最大 6ch (TODO:あとで int32_t* にしたい)
//     for ( uint32_t ch = 0; ch < info->numChs; ch++ )
//     {
//         buffer[ch] = reinterpret_cast<void*>(data[ch]);
//     }

//     const uint32_t BufferSize = info->numChs * info->numSamples * sizeof(int32_t);
//     const OutputMode OutputMode = GetInstance().GetOutputMode(device);
//     const float FxSampleRateF32 = static_cast<float>(FX_SAMPLE_RATE);

//     for ( FxList::iterator itr = GetInstance().GetEffectList( bus, device ).begin();
//           itr != GetInstance().GetEffectList( bus, device ).end();
//           (void)++itr )
//     {
//         itr->UpdateBuffer(
//                 info->numChs,
//                 buffer,
//                 BufferSize,
//                 FX_SAMPLE_FORMAT,
//                 FxSampleRateF32,
//                 OutputMode );
//     }

//     // 処理時間計測
//     GetInstance().m_EffectProcessTick[ device ][ bus ] = nn::os::GetSystemTick() -  tick;
// }

void HardwareManager::SetBiquadFilterCallback( int type, const BiquadFilterCallback* cb ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( type >= BiquadFilterType_DataMin && type <= BiquadFilterType_Max );

    if ( type <= BiquadFilterType_None /* 0 */ )
    {
        // 0 番は Biquad フィルタ OFF なので、なにもしない
        return;
    }

    m_BiquadFilterCallbackTable[ type ] = cb;
}

void HardwareManager::FlushDataCache( void* address, size_t length ) NN_NOEXCEPT
{
    NN_UNUSED(address);
    NN_UNUSED(length);
}

#if 0 // 参考用 : NW4F 実装
void HardwareManager::MainFinalMixCallbackFunc( AX_FINAL_MIX_CB_STRUCT* info )
{
    HardwareManager::GetInstance().OnMainFinalMixCallback( info );
}

void HardwareManager::OnMainFinalMixCallback( AX_FINAL_MIX_CB_STRUCT* info )
{
    fnd::ScopedLock<fnd::CriticalSection> lock(m_CriticalSection);

    NN_SDK_ASSERT_NOT_NULL(info);
    if (info->numDevices != 1)
    {
        NN_ATK_WARNING("unexpected value.");
    }

    if ( m_LastMainFinalMixCallback != NULL )
    {
        m_LastMainFinalMixCallback( info );
    }

    FinalMixData data;
    if ( info->numChnsIn > 0 )
    {
        data.frontLeft   = info->numChnsIn > 0 ? info->data[0] : NULL;
        data.frontRight  = info->numChnsIn > 1 ? info->data[1] : NULL;
        data.rearLeft    = info->numChnsIn > 2 ? info->data[2] : NULL;
        data.rearRight   = info->numChnsIn > 3 ? info->data[3] : NULL;
        data.frontCenter = info->numChnsIn > 4 ? info->data[4] : NULL;
        data.lowFrequencyEffect = info->numChnsIn > 5 ? info->data[5] : NULL;
    }
    data.sampleCount   = info->numSamples;
    data.channelCount  = info->numChnsIn;

    for ( FinalMixCallbackList::iterator itr = m_FinalMixCallbackList.begin();
          itr != m_FinalMixCallbackList.end();
          ++itr
        )
    {
        itr->OnFinalMix(OutputDevice_Main, &data);
    }

    for ( ReadOnlyFinalMixCallbackList::iterator itr = m_ReadOnlyFinalMixCallbackList.begin();
          itr != m_ReadOnlyFinalMixCallbackList.end();
          ++itr
        )
    {
        itr->OnFinalMix(OutputDevice_Main, &data);
    }
}

void HardwareManager::DrcFinalMixCallbackFunc( AX_FINAL_MIX_CB_STRUCT* info )
{
    HardwareManager::GetInstance().OnDrcFinalMixCallback( info );
}

void HardwareManager::OnDrcFinalMixCallback( AX_FINAL_MIX_CB_STRUCT* info )
{
    fnd::ScopedLock<fnd::CriticalSection> lock(m_CriticalSection);

    const uint32_t DeviceCount = 2;

    NN_SDK_ASSERT_NOT_NULL(info);

    if ( m_LastDrcFinalMixCallback != NULL )
    {
        m_LastDrcFinalMixCallback( info );
    }

    FinalMixData data[DeviceCount];
    const uint16_t InputChannelCount = info->numChnsIn;
    if ( info->numChnsIn > 0 )
    {
        for (int32_t deviceIdx = 0; deviceIdx < info->numDevices; ++deviceIdx)
        {
            data[deviceIdx].frontLeft    = InputChannelCount > 0 ? info->data[0 + (InputChannelCount * deviceIdx)] : NULL;
            data[deviceIdx].frontRight   = InputChannelCount > 1 ? info->data[1 + (InputChannelCount * deviceIdx)] : NULL;
            data[deviceIdx].rearLeft     = InputChannelCount > 2 ? info->data[2 + (InputChannelCount * deviceIdx)] : NULL;
            data[deviceIdx].rearRight    = InputChannelCount > 3 ? info->data[3 + (InputChannelCount * deviceIdx)] : NULL;
            data[deviceIdx].sampleCount  = info->numSamples;
            data[deviceIdx].channelCount = info->numChnsIn;

            OutputDevice outputDevice = static_cast<OutputDevice>(static_cast<int32_t>(OUTPUT_DEVICE_DRC0) + deviceIdx);

            for ( FinalMixCallbackList::iterator itr = m_FinalMixCallbackList.begin();
                  itr != m_FinalMixCallbackList.end();
                  ++itr
                )
            {
                itr->OnFinalMix(outputDevice, &data[deviceIdx]);
            }

            for ( ReadOnlyFinalMixCallbackList::iterator itr = m_ReadOnlyFinalMixCallbackList.begin();
                  itr != m_ReadOnlyFinalMixCallbackList.end();
                  ++itr
                )
            {
                itr->OnFinalMix(outputDevice, &data[deviceIdx]);
            }
        }
    }
}

void HardwareManager::AppendFinalMixCallback( FinalMixCallback* userCallback )
{
    NN_SDK_ASSERT_NOT_NULL( userCallback );
    fnd::ScopedLock<fnd::CriticalSection> lock(m_CriticalSection);
    m_FinalMixCallbackList.push_back(*userCallback);
}

void HardwareManager::PrependFinalMixCallback( FinalMixCallback* userCallback )
{
    NN_SDK_ASSERT_NOT_NULL( userCallback );
    fnd::ScopedLock<fnd::CriticalSection> lock(m_CriticalSection);
    m_FinalMixCallbackList.push_front(*userCallback);
}

void HardwareManager::EraseFinalMixCallback( FinalMixCallback* userCallback )
{
    NN_SDK_ASSERT_NOT_NULL( userCallback );
    fnd::ScopedLock<fnd::CriticalSection> lock(m_CriticalSection);
    m_FinalMixCallbackList.Erase(userCallback);
}

void HardwareManager::AppendReadOnlyFinalMixCallback( ReadOnlyFinalMixCallback* userCallback )
{
    NN_SDK_ASSERT_NOT_NULL( userCallback );
    fnd::ScopedLock<fnd::CriticalSection> lock(m_CriticalSection);
    m_ReadOnlyFinalMixCallbackList.push_back(*userCallback);
}

void HardwareManager::EraseReadOnlyFinalMixCallback( ReadOnlyFinalMixCallback* userCallback )
{
    NN_SDK_ASSERT_NOT_NULL( userCallback );
    fnd::ScopedLock<fnd::CriticalSection> lock(m_CriticalSection);
    m_ReadOnlyFinalMixCallbackList.Erase(userCallback);
}
#else
    // TODO: FinalMix の対応
#endif

} // namespace nn::atk::detail:driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
