﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_WavePlayer.h>

#include <nn/atk/atk_MultiVoiceManager.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_SoundThread.h>

namespace nn {
namespace atk {

bool WavePlayer::Initialize(
    int channelCount,
    /* int voiceOutCount, */
    int priority,
    VoiceCallback callback,
    void* callbackArg ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( channelCount, 0, static_cast<int>(WaveChannelMax) );
    NN_SDK_ASSERT_RANGE( priority, 0, 255 );

    detail::driver::MultiVoice* voice = NULL;
    {
        // ボイスドロップ時に nn::audio の関数が呼ばれる可能性があるため SoundThreadLock を使用する
        detail::driver::SoundThreadLock lock;

        voice = detail::driver::MultiVoiceManager::GetInstance().AllocVoice(
            channelCount,
            priority,
            MultiVoiceCallbackFunc,
            this );
    }

    if ( voice == NULL )
    {
        return false;
    }
    m_pMultiVoice = voice;
    m_Callback = callback;
    m_pCallbackArg = callbackArg;

    return true;
}

void WavePlayer::Finalize() NN_NOEXCEPT
{
#if 0
    detail::driver::AtkStateAndParameterUpdateLock lock;

    if ( m_pMultiVoice )
    {
        m_pMultiVoice->Free();
    }
#else
    Stop();
#endif
}


void WavePlayer::Start() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoicePlay* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoicePlay>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoicePlay;
    cmd->voice = m_pMultiVoice;
    cmd->state = detail::DriverCommandVoicePlay::State_Start;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::Stop() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoicePlay* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoicePlay>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoicePlay;
    cmd->voice = m_pMultiVoice;
    cmd->state = detail::DriverCommandVoicePlay::State_Stop;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::Pause( bool isPauseOn ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoicePlay* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoicePlay>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoicePlay;
    cmd->voice = m_pMultiVoice;
    cmd->state = isPauseOn ? detail::DriverCommandVoicePlay::State_PauseOn :
                             detail::DriverCommandVoicePlay::State_PauseOff;
    cmdmgr.PushCommand( cmd );
}

bool WavePlayer::IsActive() const NN_NOEXCEPT
{
    if ( m_pMultiVoice )
    {
        return m_pMultiVoice->IsActive();
    }
    return false;
}

bool WavePlayer::IsRun() const NN_NOEXCEPT
{
    if ( m_pMultiVoice )
    {
        return m_pMultiVoice->IsRun();
    }
    return false;
}

bool WavePlayer::IsPause() const NN_NOEXCEPT
{
    if ( m_pMultiVoice )
    {
        return m_pMultiVoice->IsPause();
    }
    return false;
}

detail::position_t WavePlayer::GetCurrentPlayingSample() const NN_NOEXCEPT
{
    if ( m_pMultiVoice )
    {
        return m_pMultiVoice->GetCurrentPlayingSample();
    }
    return 0;
}

void WavePlayer::SetWaveInfo( SampleFormat format, int sampleRate, int interpolationType ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoiceWaveInfo* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoiceWaveInfo>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoiceWaveInfo;
    cmd->voice = m_pMultiVoice;
    cmd->format = format;
    cmd->sampleRate = sampleRate;
    cmd->interpolationType = interpolationType;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetAdpcmParam( int channel, /* int voiceOut, */const AdpcmParam& param ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoiceAdpcmParam* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoiceAdpcmParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoiceAdpcmParam;
    cmd->voice = m_pMultiVoice;
    cmd->channel = channel;
    // cmd->voiceOut = voiceOut;
    cmd->param = &param;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::AppendWaveBuffer(
    int channel,
    /* int voiceOut, */
    WaveBuffer* pBuffer,
    bool lastFlag ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

#if 0
    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoiceAppendWaveBuffer* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoiceAppendWaveBuffer>();
    cmd->id = detail::DRIVER_COMMAND_VOICE_APPEND_WAVEBUFFER;
    cmd->voice = m_pMultiVoice;
    cmd->channel = channel;
    // cmd->voiceOut = voiceOut;
    cmd->buffer = pBuffer;
    cmd->lastFlag = lastFlag;
    cmdmgr.PushCommand( cmd );
#else
    detail::driver::AtkStateAndParameterUpdateLock lock;
    m_pMultiVoice->AppendWaveBuffer( channel, pBuffer, lastFlag );
#endif
}

void WavePlayer::SetPriority( int priority ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoicePriority* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoicePriority>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoicePriority;
    cmd->voice = m_pMultiVoice;
    cmd->priority = priority;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetVolume( float volume ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoiceVolume* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoiceVolume>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoiceVolume;
    cmd->voice = m_pMultiVoice;
    cmd->volume = volume;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetPitch( float pitch ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoicePitch* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoicePitch>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoicePitch;
    cmd->voice = m_pMultiVoice;
    cmd->pitch = pitch;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetPanMode( PanMode mode ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoicePanMode* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoicePanMode>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoicePanMode;
    cmd->voice = m_pMultiVoice;
    cmd->mode = mode;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetPanCurve( PanCurve curve ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoicePanCurve* cmd = cmdmgr.AllocCommand<detail::DriverCommandVoicePanCurve>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoicePanCurve;
    cmd->voice = m_pMultiVoice;
    cmd->curve = curve;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetPan( float pan ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoicePan* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoicePan>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoicePan;
    cmd->voice = m_pMultiVoice;
    cmd->pan = pan;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetSurroundPan( float span ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoiceSurroundPan* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoiceSurroundPan>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoiceSpan;
    cmd->voice = m_pMultiVoice;
    cmd->span = span;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetLpfFreq( float lpfFreq ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoiceLpfFreq* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoiceLpfFreq>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoiceLpf;
    cmd->voice = m_pMultiVoice;
    cmd->lpfFreq = lpfFreq;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetBiquadFilter( int type, float value ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoiceBiquadFilter* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoiceBiquadFilter>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoiceBiquad;
    cmd->voice = m_pMultiVoice;
    cmd->type = type;
    cmd->value = value;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetOutputLine( uint32_t lineFlag ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoiceOutputLine* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoiceOutputLine>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoiceOutputLine;
    cmd->voice = m_pMultiVoice;
    cmd->lineFlag = lineFlag;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetMainOutVolume( float volume ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoiceMainOutVolume* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoiceMainOutVolume>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoiceMainOutVolume;
    cmd->voice = m_pMultiVoice;
    cmd->volume = volume;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetMainSend( float send ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoiceMainSend* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoiceMainSend>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoiceMainSend;
    cmd->voice = m_pMultiVoice;
    cmd->send = send;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetFxSend( AuxBus bus, float send ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pMultiVoice );

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandVoiceFxSend* cmd =
        cmdmgr.AllocCommand<detail::DriverCommandVoiceFxSend>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (cmd == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(cmd);
#endif
    cmd->id = detail::DriverCommandId_VoiceFxSend;
    cmd->voice = m_pMultiVoice;
    cmd->bus = bus;
    cmd->send = send;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::MultiVoiceCallbackFunc(
    detail::driver::MultiVoice* driverVoice,
    detail::driver::MultiVoice::VoiceCallbackStatus status,
    void* arg ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( arg );

    switch ( status )
    {
        case detail::driver::MultiVoice::VoiceCallbackStatus_FinishWave:
        case detail::driver::MultiVoice::VoiceCallbackStatus_Cancel:
            driverVoice->Free();    // WaveBuffer との関連付けを切る
            break;
        default:
            break;
    }

    WavePlayer* wavePlayer = static_cast<WavePlayer*>(arg);
    if ( wavePlayer->m_Callback )
    {
        wavePlayer->m_Callback( wavePlayer->m_pCallbackArg );
    }

    wavePlayer->m_pMultiVoice = NULL;
    wavePlayer->m_Callback = NULL;
    wavePlayer->m_pCallbackArg = NULL;
}


} // namespace nn::atk
} // namespace nn
