﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_PlayerHeap.h>
#include <nn/atk/atk_SoundThread.h>
#include <nn/atk/atk_DisposeCallbackManager.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/detail/atk_Macro.h>

namespace nn {
namespace atk {
namespace detail {

/*--------------------------------------------------------------------------------*
  Name:         SoundHeap

  Description:  コンストラクタ

  Arguments:    無し

  Returns:      無し
 *--------------------------------------------------------------------------------*/
PlayerHeap::PlayerHeap() NN_NOEXCEPT
: m_pPlayer( NULL )
, m_pStartAddress( NULL )
, m_pEndAddress( NULL )
, m_pAllocAddress( NULL )
, m_State(State_Constructed)
, m_CallbackList()
{
}

/*--------------------------------------------------------------------------------*
  Name:         ~PlayerHeap

  Description:  デストラクタ

  Arguments:    無し

  Returns:      無し
 *--------------------------------------------------------------------------------*/
PlayerHeap::~PlayerHeap() NN_NOEXCEPT
{
    Destroy();
    m_State = State_Constructed;
}

/*--------------------------------------------------------------------------------*
  Name:         Create

  Description:

  Arguments:    startAddress -
                size -

  Returns:
 *--------------------------------------------------------------------------------*/
bool PlayerHeap::Create( void* startAddress, size_t size ) NN_NOEXCEPT
{
    if (!atk::detail::Util::IsValidMemoryForDsp( startAddress, size ))
    {
        NN_ATK_WARNING(
            "the memory area (0x%08x - 0x%08x %dbyte) provided cross a 512 MB segment.",
            startAddress,
            util::BytePtr( startAddress, size ).Get(),
            size );
    }

    void* endAddress = util::BytePtr( startAddress, size ).Get();
    startAddress = util::BytePtr( startAddress ).AlignUp( nn::audio::MemoryPoolType::AddressAlignment ).Get();
    if ( startAddress > endAddress )
    {
        return false;
    }

    m_pStartAddress = startAddress;
    m_pEndAddress = endAddress;
    m_pAllocAddress = m_pStartAddress;

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         Destroy

  Description:

  Arguments:    なし

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void PlayerHeap::Destroy() NN_NOEXCEPT
{
    Clear();
    m_pAllocAddress = NULL;
}

/*--------------------------------------------------------------------------------*
  Name:         Alloc

  Description:

  Arguments:    size -

  Returns:
 *--------------------------------------------------------------------------------*/
void* PlayerHeap::Allocate( size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( util::is_aligned(reinterpret_cast<uintptr_t>(m_pAllocAddress), nn::audio::MemoryPoolType::AddressAlignment) );

    void* endp = util::BytePtr( m_pAllocAddress, size ).Get();
    if ( endp > m_pEndAddress )
    {
        return NULL;
    }

    void* allocAddress = m_pAllocAddress;
    m_pAllocAddress = util::BytePtr( endp ).AlignUp( nn::audio::MemoryPoolType::AddressAlignment ).Get();

    //NN_SDK_LOG( "Allocate %p - %p\n", allocAddress, m_pAllocAddress );

    return allocAddress;
}

void* PlayerHeap::Allocate( size_t size, SoundMemoryAllocatable::DisposeCallback callback, void* callbackArg ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( util::is_aligned(reinterpret_cast<uintptr_t>(m_pAllocAddress), nn::audio::MemoryPoolType::AddressAlignment) );

    void* callbackNodeBuffer = util::BytePtr( m_pAllocAddress, size ).Get();
    void* endp = util::BytePtr( callbackNodeBuffer, sizeof(CallbackNode) ).Get();
    if ( endp > m_pEndAddress )
    {
        return nullptr;
    }

    void* allocAddress = m_pAllocAddress;
    m_pAllocAddress = util::BytePtr( endp ).AlignUp( nn::audio::MemoryPoolType::AddressAlignment ).Get();

    //NN_SDK_LOG( "Allocate %p - %p\n", allocAddress, m_pAllocAddress );

    CallbackNode* node = new ( callbackNodeBuffer ) CallbackNode();
    node->SetCallback( callback );
    node->SetCallbackArg( callbackArg );
    m_CallbackList.push_back( *node );

    return allocAddress;
}

size_t PlayerHeap::GetAllocateSize(size_t size, bool needMemoryPool) NN_NOEXCEPT
{
    NN_UNUSED(needMemoryPool); // コールバックを使用する場合は sizeof(CallbackNode) を足す必要がある
    return size;
}

/*--------------------------------------------------------------------------------*
  Name:         Clear

  Description:

  Arguments:    なし

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void PlayerHeap::Clear() NN_NOEXCEPT
{
    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstanceForTaskThread();

    detail::DriverCommandInvalidateData* command =
        cmdmgr.AllocCommand<detail::DriverCommandInvalidateData>( false );
    command->id = detail::DriverCommandId_InvalidateData;
    command->mem = m_pStartAddress;
    command->size = static_cast<unsigned long>( util::BytePtr( m_pStartAddress ).Distance( m_pAllocAddress ) );
    cmdmgr.PushCommand(command);
    cmdmgr.FlushCommand(false, false);

    // NN_SDK_LOG( "Clear %p - %p \n", command->mem, nn::util::ConstBytePtr(command->mem, static_cast<ptrdiff_t>(command->size)).Get() );

    m_pAllocAddress = m_pStartAddress;

    for ( CallbackList::iterator itr = m_CallbackList.begin();
        itr != m_CallbackList.end(); ++itr )
    {
        if (itr->GetCallback() != nullptr)
        {
            itr->GetCallback()(itr->GetCallbackArg());
        }
    }

    m_CallbackList.clear();
}

/*--------------------------------------------------------------------------------*
  Name:         GetFreeSize

  Description:

  Arguments:    なし

  Returns:
 *--------------------------------------------------------------------------------*/
size_t PlayerHeap::GetFreeSize() const NN_NOEXCEPT
{
    size_t offset = util::BytePtr( m_pAllocAddress ).Distance( m_pEndAddress );
    return offset;
}


} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

