﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_GroupFileReader.h>

namespace nn {
namespace atk {
namespace detail {

namespace {

const uint32_t SignatureInfoBlockGroup  = NN_UTIL_CREATE_SIGNATURE_4( 'I', 'N', 'F', 'O' );
const uint32_t SignatureFileBlockGroup  = NN_UTIL_CREATE_SIGNATURE_4( 'F', 'I', 'L', 'E' );
const uint32_t SignatureInfoexBlockGroup  = NN_UTIL_CREATE_SIGNATURE_4( 'I', 'N', 'F', 'X' );

const uint32_t SupportedFileVersionGroup = 0x00010000;
const uint32_t CurrentFileVersionGroup   = 0x00010000;

bool IsValidFileHeaderGroup( const void* groupFile ) NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader* header =
        reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( groupFile );

    // シグニチャ確認
    NN_SDK_ASSERT( header->signature == GroupFileReader::SignatureFile,
            "invalid file signature. group file is not available." );
    if ( header->signature != GroupFileReader::SignatureFile )
    {
        return false;
    }

    // バイトオーダーマーク確認
    NN_SDK_ASSERT( header->byteOrder == BinaryFileHeader::ValidByteOrderMark,
            "invalid file byte order mark. [expected:0x%04x][this:0x%04x]", BinaryFileHeader::ValidByteOrderMark, header->byteOrder );
    if ( header->byteOrder != BinaryFileHeader::ValidByteOrderMark )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SupportedFileVersionGroup <= header->version) &&
                                   (header->version <= CurrentFileVersionGroup) )
    {
        isSupportedVersion = true;

    }
    NN_SDK_ASSERT( isSupportedVersion,
            "bfgrp file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SupportedFileVersionGroup, CurrentFileVersionGroup, header->version
    );
    return isSupportedVersion;
}

} // anonymous namespace

NN_DEFINE_STATIC_CONSTANT( const uint32_t GroupFileReader::SignatureFile );

GroupFileReader::GroupFileReader( const void* groupFile ) NN_NOEXCEPT
: m_pInfoBlockBody( NULL )
, m_pFileBlockBody( NULL )
, m_pInfoExBlockBody( NULL )
{
    NN_SDK_ASSERT_NOT_NULL( groupFile );

    if ( ! IsValidFileHeaderGroup( groupFile ) ) return;

    const GroupFile::FileHeader* header =
        static_cast<const GroupFile::FileHeader*>( groupFile );

    const GroupFile::InfoBlock* infoBlock = header->GetInfoBlock();
    const GroupFile::FileBlock* fileBlock = header->GetFileBlock();
    const GroupFile::InfoExBlock* infoExBlock = header->GetInfoExBlock();

    if ( infoBlock == NULL ) return;
    if ( fileBlock == NULL ) return;

    NN_SDK_ASSERT( infoBlock->header.kind == SignatureInfoBlockGroup );
    NN_SDK_ASSERT( fileBlock->header.kind == SignatureFileBlockGroup );

    if ( infoBlock->header.kind != SignatureInfoBlockGroup )
    {
        return;
    }
    if ( fileBlock->header.kind != SignatureFileBlockGroup )
    {
        return;
    }

    // バイナリバージョンが 1.0.0.0 のときは InfoExBlock が含まれない
    if ( infoExBlock != NULL )
    {
        NN_SDK_ASSERT( infoExBlock->header.kind == SignatureInfoexBlockGroup );
        if ( infoExBlock->header.kind != SignatureInfoexBlockGroup )
        {
            return;
        }
        m_pInfoExBlockBody = &(infoExBlock->body);
    }

    m_pInfoBlockBody = &(infoBlock->body);
    m_pFileBlockBody = &(fileBlock->body);
}

bool GroupFileReader::ReadGroupItemLocationInfo(
        GroupItemLocationInfo* out, uint32_t index ) const NN_NOEXCEPT
{
    if ( m_pInfoBlockBody == NULL )
    {
        return false;
    }

    const GroupFile::GroupItemInfo* groupItemInfo = m_pInfoBlockBody->GetGroupItemInfo( index );
    if ( groupItemInfo == NULL )
    {
        return false;
    }

    out->fileId = groupItemInfo->fileId;
    out->address = groupItemInfo->GetFileLocation( m_pFileBlockBody );
    return true;
}

uint32_t GroupFileReader::GetGroupItemExCount() const NN_NOEXCEPT
{
    if ( m_pInfoExBlockBody == NULL )
    {
        return 0;
    }

    return m_pInfoExBlockBody->GetGroupItemInfoExCount();
}

bool GroupFileReader::ReadGroupItemInfoEx( GroupFile::GroupItemInfoEx* out, uint32_t index ) const NN_NOEXCEPT
{
    if ( m_pInfoExBlockBody == NULL )
    {
        return false;
    }

    const GroupFile::GroupItemInfoEx* groupItemInfoEx =
        m_pInfoExBlockBody->GetGroupItemInfoEx( index );
    if ( groupItemInfoEx == NULL )
    {
        return false;
    }

    *out = *groupItemInfoEx; // 構造体のコピー
    return true;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

