﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundPlayer.h>

#include <nn/atk/atk_SoundThread.h>

#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>

namespace nn {
namespace atk {

/* ========================================================================
        public function
   ======================================================================== */

/*--------------------------------------------------------------------------------*
  Name:         SoundPlayer

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
SoundPlayer::SoundPlayer() NN_NOEXCEPT
: m_PlayableCount( 1 )
, m_PlayableLimit( INT_MAX )
, m_PlayerHeapCount( 0 )
, m_Volume( 1.0f )
, m_LpfFreq( 0.0f )
, m_BiquadType( BiquadFilterType_Inherit )
, m_BiquadValue( 0.0f )
, m_OutputLineFlag( OutputLine_Main )
, m_pOutputAdditionalParam( nullptr )
, m_IsFirstComeBased( false )
{
    m_TvParam.Initialize();
}

SoundPlayer::SoundPlayer(detail::OutputAdditionalParam * pParam) NN_NOEXCEPT
: m_PlayableCount( 1 )
, m_PlayableLimit( INT_MAX )
, m_PlayerHeapCount( 0 )
, m_Volume( 1.0f )
, m_LpfFreq( 0.0f )
, m_BiquadType( BiquadFilterType_Inherit )
, m_BiquadValue( 0.0f )
, m_OutputLineFlag( OutputLine_Main )
, m_pOutputAdditionalParam( pParam )
, m_IsFirstComeBased( false )
{
    m_TvParam.Initialize();
    if(m_pOutputAdditionalParam != nullptr)
    {
        m_pOutputAdditionalParam->Reset();
    }
}

/*--------------------------------------------------------------------------------*
  Name:         ~SoundPlayer

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
SoundPlayer::~SoundPlayer() NN_NOEXCEPT
{
    StopAllSound( 0 );
}

/*--------------------------------------------------------------------------------*
  Name:         Update

  Description:  プレイヤーのフレーム処理を更新する

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundPlayer::Update() NN_NOEXCEPT
{
    // 利用済みプレイヤーヒープを m_PlayerHeapFreeList に戻す
    DoFreePlayerHeap();

    // BasicSound::Update
    for ( SoundList::iterator itr = m_SoundList.begin();
          itr != m_SoundList.end();
        )
    {
        SoundList::iterator curItr = itr++;
        curItr->Update();
    }

    // プライオリティリストを整列しなおす
    detail_SortPriorityList();
}

/*--------------------------------------------------------------------------------*
  Name:         StopAllSound

  Description:  全てのサウンドを停止する

  Arguments:    fadeFrames - フェードアウトフレーム数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundPlayer::StopAllSound( int fadeFrames ) NN_NOEXCEPT
{
    for ( SoundList::iterator itr = m_SoundList.begin();
          itr != m_SoundList.end();
        )
    {
        SoundList::iterator curItr = itr++;
        curItr->Stop( fadeFrames );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         PauseAllSound

  Description:  全てのサウンドを一時停止または再開する

  Arguments:    flag       - 一時停止か再開か
                fadeFrames - フェードフレーム数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundPlayer::PauseAllSound( bool flag, int fadeFrames ) NN_NOEXCEPT
{
    for ( SoundList::iterator itr = m_SoundList.begin();
          itr != m_SoundList.end();
        )
    {
        SoundList::iterator curItr = itr++;
        curItr->Pause( flag, fadeFrames );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         PauseAllSound

  Description:  全てのサウンドを一時停止または再開する

  Arguments:    flag       - 一時停止か再開か
                fadeFrames - フェードフレーム数
                pauseMode  - ポーズの設定

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundPlayer::PauseAllSound( bool flag, int fadeFrames, PauseMode pauseMode ) NN_NOEXCEPT
{
    for ( SoundList::iterator itr = m_SoundList.begin();
          itr != m_SoundList.end();
        )
    {
        SoundList::iterator curItr = itr++;
        curItr->Pause( flag, fadeFrames, pauseMode );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         SetVolume

  Description:  音量を変更

  Arguments:    volume - 音量

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundPlayer::SetVolume( float volume ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( volume >= 0.0f );
    if ( volume < 0.0f ) volume = 0.0f;
    m_Volume = volume;
}

/*--------------------------------------------------------------------------------*
  Name:         SetLpfFreq

  Description:  ローパスフィルタの周波数変更

  Arguments:    lpfFreq - ローパスフィルタの周波数値

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void SoundPlayer::SetLowPassFilterFrequency( float lpfFreq ) NN_NOEXCEPT
{
    m_LpfFreq = lpfFreq;
}

/*--------------------------------------------------------------------------------*
  Name:         SetBiquadFilter

  Description:  Biquadフィルタの設定

  Arguments:    type  - フィルタの種類 (-1 - 127)
                value - フィルタの値 (0.0f - 1.0f)

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void SoundPlayer::SetBiquadFilter( int type, float value ) NN_NOEXCEPT
{
    m_BiquadType = type;
    m_BiquadValue = value;
}

void SoundPlayer::SetDefaultOutputLine( uint32_t outputLineFlag ) NN_NOEXCEPT
{
    m_OutputLineFlag = outputLineFlag;
}

void SoundPlayer::SetMainSend( float send ) NN_NOEXCEPT
{
    m_TvParam.mainSend = send;
}

float SoundPlayer::GetMainSend() const NN_NOEXCEPT
{
    return m_TvParam.mainSend;
}

void SoundPlayer::SetEffectSend( AuxBus bus, float send ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE( bus, 0, AuxBus_Count );
    m_TvParam.fxSend[bus] = send;
}

float SoundPlayer::GetEffectSend( AuxBus bus ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE( bus, 0, AuxBus_Count );
    return m_TvParam.fxSend[bus];
}

void SoundPlayer::SetSend(int subMixBus, float send) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE( subMixBus, 0, OutputReceiver::BusCountMax );
    if( subMixBus == 0 )
    {
        SetMainSend(send);
        return;
    }

    if( subMixBus < detail::DefaultBusCount )
    {
        SetEffectSend(static_cast<AuxBus>(subMixBus - 1), send);
        return;
    }

    if(m_pOutputAdditionalParam == nullptr || !m_pOutputAdditionalParam->IsAdditionalSendEnabled())
    {
        // カスタムサブミックス不使用、またはカスタムサブミックス使用で 4 バス以下の場合
        NN_SDK_REQUIRES_RANGE(subMixBus, 0, AuxBus_Count + 1);
    }
    // カスタムサブミックス使用、4バスを超える場合
    NN_SDK_REQUIRES_RANGE(subMixBus, detail::DefaultBusCount, m_pOutputAdditionalParam->GetAdditionalSendAddr()->GetCount() + detail::DefaultBusCount);
    m_pOutputAdditionalParam->TrySetAdditionalSend(subMixBus, send);
}

float SoundPlayer::GetSend(int subMixBus) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE( subMixBus, 0, OutputReceiver::BusCountMax );
    if( subMixBus == 0 )
    {
        return GetMainSend();
    }

    if( subMixBus < detail::DefaultBusCount )
    {
        return GetEffectSend(static_cast<AuxBus>(subMixBus - 1));
    }

    NN_SDK_REQUIRES_NOT_NULL(m_pOutputAdditionalParam);
    NN_SDK_REQUIRES(m_pOutputAdditionalParam->IsAdditionalSendEnabled());
    NN_SDK_REQUIRES_RANGE(subMixBus, detail::DefaultBusCount, m_pOutputAdditionalParam->GetAdditionalSendAddr()->GetCount() + detail::DefaultBusCount);
    return m_pOutputAdditionalParam->TryGetAdditionalSend(subMixBus);
}

void SoundPlayer::SetOutputVolume( OutputDevice device, float volume ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( device, 0, OutputDevice_Count );
    NN_SDK_ASSERT( volume >= 0.0f );
    if ( volume < 0.0f )
    {
        volume = 0.0f;
    }

    switch ( device )
    {
    case nn::atk::OutputDevice_Main:
        m_TvParam.volume = volume;
        break;
    default:
        NN_SDK_ASSERT( false, "Invalid device\n" );
        break;
    }

}

/*--------------------------------------------------------------------------------*
  Name:         RemoveSoundList

  Description:  サウンドをプレイヤーリストから削除する

  Arguments:    pSound - シーケンスサウンド

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundPlayer::RemoveSoundList( detail::BasicSound* pSound ) NN_NOEXCEPT
{
    // 再生リストから削除する
    m_SoundList.erase( m_SoundList.iterator_to( *pSound ) );
    pSound->DetachSoundPlayer( this );
}

/*--------------------------------------------------------------------------------*
  Name:         InsertPriorityList

  Description:  サウンドをプライオリティリストへ追加

  Arguments:    pSound - 追加するサウンド

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void SoundPlayer::InsertPriorityList( detail::BasicSound* pSound ) NN_NOEXCEPT
{
    PriorityList::iterator itr = m_PriorityList.begin();
    while ( itr != m_PriorityList.end() )
    {
        if ( m_IsFirstComeBased )
        {
            if ( pSound->CalcCurrentPlayerPriority() <= itr->CalcCurrentPlayerPriority() ) break;
        }
        else
        {
            if ( pSound->CalcCurrentPlayerPriority() < itr->CalcCurrentPlayerPriority() ) break;
        }
        (void)++itr;
    }
    m_PriorityList.insert( itr, *pSound );
}

/*--------------------------------------------------------------------------------*
  Name:         RemovePriorityList

  Description:  サウンドをプライオリティリストから削除

  Arguments:    pSound - 削除するサウンド

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void SoundPlayer::RemovePriorityList( detail::BasicSound* pSound ) NN_NOEXCEPT
{
    m_PriorityList.erase( m_PriorityList.iterator_to( *pSound ) );
}

/*--------------------------------------------------------------------------------*
  Name:         detail_SortPriorityList

  Description:  指定したサウンドのプライオリティ順序を並び替える

  Arguments:    pSound - 並び替えるサウンド

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void SoundPlayer::detail_SortPriorityList( detail::BasicSound* pSound ) NN_NOEXCEPT
{
    RemovePriorityList( pSound );
    InsertPriorityList( pSound );
}

/*--------------------------------------------------------------------------------*
  Name:         detail_SortPriorityList

  Description:  全サウンドのプライオリティ順序を並び替える

  Arguments:    reverse - 同一プライオリティ内で先着順/後着順を並び替える.

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void SoundPlayer::detail_SortPriorityList( bool reverse ) NN_NOEXCEPT
{
    if ( m_PriorityList.size() < 2 ) return;

    static const int TmpCount = nn::atk::PlayerPriorityMax - nn::atk::PlayerPriorityMin + 1;
    static PriorityList tmplist[ TmpCount ]; // notice: large stack
#ifdef NN_SDK_BUILD_DEBUG
    for ( int i = 0; i < TmpCount; i++ ) NN_SDK_ASSERT( tmplist[i].empty() );
#endif

    while ( !m_PriorityList.empty() )
    {
        detail::BasicSound& front = m_PriorityList.front();
        m_PriorityList.pop_front();
        tmplist[ front.CalcCurrentPlayerPriority() ].push_back( front );
    }
    for ( int i = 0; i < TmpCount; i++ )
    {
        while ( !tmplist[i].empty() )
        {
            if ( reverse )
            {
                detail::BasicSound& back = tmplist[i].back();
                tmplist[i].pop_back();
                m_PriorityList.push_back( back );
            }
            else
            {
                detail::BasicSound& front = tmplist[i].front();
                tmplist[i].pop_front();
                m_PriorityList.push_back( front );
            }
        }
    }
}

/*--------------------------------------------------------------------------------*
  Name:         detail_AppendSound

  Description:  サウンドをサウンドプレイヤーに登録する

  Arguments:    pSound - サウンド

  Returns:      登録できたらtrue
 *--------------------------------------------------------------------------------*/
bool SoundPlayer::detail_AppendSound( detail::BasicSound* pSound ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pSound );

    int allocPriority = pSound->CalcCurrentPlayerPriority();

    // 最大同時再生数のチェック
    if ( GetPlayableSoundCount() == 0 )
    {
        return false;
    }
    while ( GetPlayingSoundCount() >= GetPlayableSoundCount() )
    {
        detail::BasicSound* dropSound = GetLowestPrioritySound();
        if ( dropSound == NULL )
        {
            return false;
        }
        if ( m_IsFirstComeBased )
        {
            if ( allocPriority <= dropSound->CalcCurrentPlayerPriority() ) return false;
        }
        else
        {
            if ( allocPriority < dropSound->CalcCurrentPlayerPriority() ) return false;
        }
        dropSound->Finalize();
    }

    // リストへ登録
    m_SoundList.push_back( *pSound );
    InsertPriorityList( pSound );

    pSound->AttachSoundPlayer( this );

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         detail_RemoveSound

  Description:  サウンドをサウンドプレイヤーから削除する

  Arguments:    pSound - 削除するサウンド

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void SoundPlayer::detail_RemoveSound( detail::BasicSound* pSound ) NN_NOEXCEPT
{
    RemovePriorityList( pSound );
    RemoveSoundList( pSound );
}

/*--------------------------------------------------------------------------------*
  Name:         SetPlayableSoundCount

  Description:  同時再生数を設定

  Arguments:    count - 同時再生数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundPlayer::SetPlayableSoundCount( int count ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( count >= 0 );

    // プレイヤーヒープ使用時の上限
    if (count > m_PlayableLimit)
    {
        NN_ATK_WARNING("playable sound count is over limit.");
    }
    count = nn::atk::detail::fnd::Clamp( count, 0, m_PlayableLimit );

    m_PlayableCount = count;

    // 新しく設定された同時再生数を越えるサウンドを終了する
    while ( GetPlayingSoundCount() > GetPlayableSoundCount() )
    {
        detail::BasicSound* dropSound = GetLowestPrioritySound();
        NN_SDK_ASSERT_NOT_NULL( dropSound );
        dropSound->Finalize();
    }
}

/*--------------------------------------------------------------------------------*
  Name:         detail_SetPlayableSoundLimit

  Description:

  Arguments:    None.

  Returns:      None.
  *--------------------------------------------------------------------------------*/
void SoundPlayer::detail_SetPlayableSoundLimit( int limit ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( limit >= 0 );

    m_PlayableLimit = limit;
}

/*--------------------------------------------------------------------------------*
  Name:         detail_CanPlaySound

  Description:  指定したプライオリティのサウンドを再生できるかどうかを調べる

  Arguments:    startPriority - プライオリティ

  Returns:      再生可能ならtrue
 *--------------------------------------------------------------------------------*/
bool SoundPlayer::detail_CanPlaySound( int startPriority ) NN_NOEXCEPT
{
    // プレイヤー毎の最大同時再生数のチェック
    if ( GetPlayableSoundCount() == 0 ) return false;
    if ( GetPlayingSoundCount() >= GetPlayableSoundCount() )
    {
        detail::BasicSound* dropSound = GetLowestPrioritySound();
        if ( dropSound == NULL ) return false;
        if ( m_IsFirstComeBased )
        {
            if ( startPriority <= dropSound->CalcCurrentPlayerPriority() ) return false;
        }
        else
        {
            if ( startPriority < dropSound->CalcCurrentPlayerPriority() ) return false;
        }
    }

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         detail_AppendPlayerHeap

  Description:  プレイヤーヒープを追加

  Arguments:    heap - 追加するプレイヤーヒープ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundPlayer::detail_AppendPlayerHeap( detail::PlayerHeap* pHeap ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pHeap );

    pHeap->AttachSoundPlayer( this );
    m_PlayerHeapFreeList.push_back( *pHeap );
    m_PlayerHeapCount++;
}

detail::PlayerHeap* SoundPlayer::detail_AllocPlayerHeap() NN_NOEXCEPT
{
    // プレイヤーヒープの確保
    if ( m_PlayerHeapFreeList.empty() )
    {
        return NULL;
    }
    detail::PlayerHeap& playerHeap = m_PlayerHeapFreeList.front();
    m_PlayerHeapFreeList.pop_front();

    // NOTE: プレイヤーヒープ初期化 playerHeap.Clear() は、
    // 各データロードタスクの Execute あたまで呼び出す
    return &playerHeap;
}

void SoundPlayer::detail_FreePlayerHeap(detail::PlayerHeap* pHeap) NN_NOEXCEPT
{
    // プレイヤーヒープの解放
    if (pHeap->GetState() == detail::PlayerHeap::State_TaskFinished)
    {
        m_PlayerHeapFreeList.push_back(*pHeap);
    }
    else
    {
        // タスク未完時は、次の Update で回収を試みる
        m_PlayerHeapFreeReqList.push_back(*pHeap);
    }
}

// 利用済みプレイヤーヒープを m_PlayerHeapFreeList に戻す
void SoundPlayer::DoFreePlayerHeap() NN_NOEXCEPT
{
    for ( PlayerHeapList::iterator itr = m_PlayerHeapFreeReqList.begin();
          itr != m_PlayerHeapFreeReqList.end();
        )
    {
        PlayerHeapList::iterator curItr = itr++;
        if (curItr->GetState() == detail::PlayerHeap::State_TaskFinished)
        {
            m_PlayerHeapFreeReqList.erase(curItr);
            m_PlayerHeapFreeList.push_back(*curItr);
        }
    }
}

} // namespace nn::atk
} // namespace nn

