﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_Sound3DListener.h>
#include <nn/atk/atk_Sound3DManager.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_Matrix.h>

namespace
{

// リスナー行列（カメラでいうところの「ビュー行列」）から位置を逆変換します
void CalculatePositionFromMatrix(nn::util::Vector3fType* position, const nn::util::Matrix4x3fType& matrix) NN_NOEXCEPT
{
    nn::util::Float4x3 tmp;
    nn::util::MatrixStore(&tmp, matrix);
    float x = -( tmp.m[0][0] * tmp.m[3][0] + tmp.m[0][1] * tmp.m[3][1] + tmp.m[0][2] * tmp.m[3][2] );
    float y = -( tmp.m[1][0] * tmp.m[3][0] + tmp.m[1][1] * tmp.m[3][1] + tmp.m[1][2] * tmp.m[3][2] );
    float z = -( tmp.m[2][0] * tmp.m[3][0] + tmp.m[2][1] * tmp.m[3][1] + tmp.m[2][2] * tmp.m[3][2] );
    nn::util::VectorSet(position, x, y, z);
}

}


namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const size_t Sound3DListener::BufferAlignSize );

/*--------------------------------------------------------------------------------*
  Name:         Sound3DListener

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
Sound3DListener::Sound3DListener() NN_NOEXCEPT
: m_InteriorSize( 1.0f )
, m_MaxVolumeDistance( 1.0f )
, m_UnitDistance( 1.0f )
, m_UserParam( 0 )
, m_UnitBiquadFilterValue( 0.5f )
, m_MaxBiquadFilterValue( 1.0f )
, m_OutputTypeFlag( OutputType_Tv )
, m_ResetMatrixFlag( true )
{
    NN_ABORT_UNLESS_ALIGNED( &m_Matrix, Sound3DManager::NnUtilMathTypeAlignSize );
    NN_ABORT_UNLESS_ALIGNED( &m_Position, Sound3DManager::NnUtilMathTypeAlignSize );
    NN_ABORT_UNLESS_ALIGNED( &m_Velocity, Sound3DManager::NnUtilMathTypeAlignSize );

    nn::util::VectorZero(&m_Position);
    nn::util::VectorZero(&m_Velocity);
    nn::util::MatrixZero(&m_Matrix);
}

void Sound3DListener::SetMatrix( const nn::util::Matrix4x3fType& matrix ) NN_NOEXCEPT
{
    nn::util::Vector3fType oldPosition = m_Position;
    CalculatePositionFromMatrix(&m_Position, matrix);

    if ( m_ResetMatrixFlag )
    {
        m_Matrix = matrix;
        m_ResetMatrixFlag = false;
    }
    else
    {
        m_Matrix = matrix;
        nn::util::VectorSubtract( &m_Velocity, m_Position, oldPosition );
    }
}

void Sound3DListener::ResetMatrix() NN_NOEXCEPT
{
    nn::util::MatrixZero(&m_Matrix);
    nn::util::VectorZero(&m_Velocity);
    m_ResetMatrixFlag = true;
}

void Sound3DListener::SetVelocity( const nn::util::Vector3fType& velocity ) NN_NOEXCEPT
{
    m_Velocity = velocity;
}

void Sound3DListener::SetInteriorSize( float interiorSize ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( interiorSize > 0.0f );
    m_InteriorSize = interiorSize;
}

void Sound3DListener::SetMaxVolumeDistance( float maxVolumeDistance ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( maxVolumeDistance >= 0.0f );
    m_MaxVolumeDistance = maxVolumeDistance;
}

void Sound3DListener::SetUnitDistance( float unitDistance ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( unitDistance > 0.0f );
    m_UnitDistance = unitDistance;
}

void Sound3DListener::SetUnitBiquadFilterValue( float value ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( value >= 0.0f && value <= 1.0f );
    m_UnitBiquadFilterValue = value;
}

void Sound3DListener::SetMaxBiquadFilterValue( float value ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( value >= 0.0f && value <= 1.0f );
    m_MaxBiquadFilterValue = value;
}

} // namespace nn::atk
} // namespace nn

