﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/atk2/atk2_IAudioFrameUpdate.h>
#include <nn/atk2/detail/atk2_RendererVoice.h>
#include <nn/atk2/detail/fnd/atk2_CriticalSection.h>

namespace nn { namespace atk2 {

    class AudioEngine;

namespace detail {

    // @brief オーディオレンダラに紐づくボイスを管理するクラスです
    class RendererVoiceManager : public IAudioFrameUpdate
    {
    public:
        // @brief 関数の処理結果を表す列挙体です
        enum Result
        {
            Result_Success = 0,                      //!< 成功です
            Result_InsufficientBufferSize,           //!< バッファサイズが不足しています
            Result_FailedToProcessVoiceCommand,      //!< ボイスコマンドの処理に失敗しました
            Result_FailedToProcessVoiceReplyCommand, //!< ボイスリプライコマンドの胥吏に失敗しました
            Result_ErrorUnknown                      //!< 不明なエラーです
        };

    public:
        // @brief 初期化に使用するパラメータです
        struct InitArg
        {
            int voiceCount;        //!< ボイスの数です
            void* workBuffer;      //!< バッファです
            size_t workBufferSize; //!< ワークバッファのサイズです

            AudioEngine* _pAudioEngine;
        };

        // @brief 初期化に使用するプラットフォーム固有のパラメータです
        struct PlatformInitArg
        {
            void* workBufferForMemoryPool;      //!< メモリプールにアタッチされているバッファです
            size_t workBufferSizeForMemoryPool; //!< メモリプールにアタッチされているバッファのサイズです
        };

    public:
        // @brief 初期化パラメータの初期値を取得します
        void GetDefaultInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
        // @brief 必要になるバッファのサイズを取得します
        size_t GetRequiredBufferSize(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT;
        // @brief 初期化します
        Result Initialize(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
        // @brief 終了処理を行います
        void Finalize() NN_NOEXCEPT;
        // @brief 更新処理を行います
        bool Update() NN_NOEXCEPT NN_OVERRIDE;

        // @brief 更新処理のロックを行います
        void Lock() NN_NOEXCEPT;
        // @brief 更新処理のアンロックを行います
        void Unlock() NN_NOEXCEPT;

        // @brief ボイスを確保します
        RendererVoice* AllocVoice() NN_NOEXCEPT;
        // @brief ボイスを解放します
        void FreeVoice(RendererVoice* pVoice) NN_NOEXCEPT;

        // @brief ボイスの数を取得します
        int GetVoiceCount() const NN_NOEXCEPT;
        // @brief 使用中のボイスの数を取得します
        int GetUsingVoiceCount() const NN_NOEXCEPT;

        // プラットフォーム固有関数
        // @brief プラットフォーム固有の処理に必要なバッファサイズを取得します
        size_t GetRequiredPlatformBufferSize(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT;
        // @brief メモリプールにアタッチされたバッファサイズを取得します
        // RendererManager::BufferAlignSize でアラインされている必要がある
        size_t GetRequiredBufferSizeForMemoryPool(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT;
        // @brief レンダラ更新が必要かどうかを取得します
        bool IsRequiredUpdateRenderer() NN_NOEXCEPT;

    private:
        static const int UnassignedIndex = -1;

    private:
        void UpdateVoiceInfo() NN_NOEXCEPT;

        size_t GetRequiredBufferSizeForVoicePtrTable(int voiceCount) const NN_NOEXCEPT;
        size_t GetRequiredBufferSizeForAssignedTableIndex(int voiceCount) const NN_NOEXCEPT;
        ptrdiff_t GetVoiceArrayIndex(RendererVoice* pVoice) NN_NOEXCEPT;

        // プラットフォーム固有関数
        size_t GetRequiredBufferSizeForVoiceArray(int voiceCount) const NN_NOEXCEPT;
        // RendererManager::BufferAlignSize でアラインされている必要がある
        size_t GetRequiredBufferSizeForAdpcmContextArray(int voiceCount) const NN_NOEXCEPT;

    private:
        // RendererVoice へのポインタのテーブル
        // m_pRendererVoicePtrTable[i] について
        // 0 <= i < m_UsingVoiceCount : 使用中のボイスへのポインタ
        // m_UsingVoiceCount <= i < MaxVoiceCount : 使用していないボイスへのポインタ
        RendererVoice** m_ppVoiceTable;

        // m_pRendererVoiceTable[i] が m_pRendererVoicePtrTable の何番目に割り当てられたかを表します。
        // 未割り当てのときは UnassignedIndex になります。
        int* m_pAssignedTableIndex;

        // 実際の RendererVoice インスタンスの配列
        void* m_pVoiceArray;
        void* m_pAdpcmContextArray;

        int m_VoiceCount;
        int m_UsingVoiceCount;

        fnd::CriticalSection m_UpdateCriticalSection;

        AudioEngine* m_pAudioEngine;
    };

}}}

