﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/audio/audio_AudioRenderer.h>
#include <nn/atk2/atk2_FinalMix.h>
#include <nn/atk2/atk2_IAudioFrameUpdate.h>
#include <nn/atk2/detail/fnd/atk2_CriticalSection.h>

namespace nn { namespace atk2 {

    class AudioEngine;

namespace detail {

    // @brief オーディオレンダラの管理を行います
    class RendererManager : public IAudioFrameUpdate
    {
    public:
        // @brief バッファのアライメントの大きさです
        static const int BufferAlignSize = nn::audio::BufferAlignSize;
        static const int ChannelCount = 6;

    public:
        // @brief 関数の処理結果を表す列挙体です
        enum Result
        {
            Result_Success = 0,                        //!< 成功です
            Result_InvalidParameter,                   //!< 不正なパラメータが使用されました
            Result_FailedOpenRenderer,                 //!< オーディオレンダラのオープンに失敗しました
            Result_FailedToRequestUpdateAudioRenderer, //!< オーディオレンダラの更新要求に失敗しました
            Result_ErrorUnknown                        //!< 不明なエラーが発生しました
        };

    public:
        // @brief 初期化に使用するパラメータです
        struct InitArg
        {
            AudioEngine* _pAudioEngine;
        };

        // プラットフォーム固有
        // @brief 初期化に使用するプラットフォーム固有のパラメータです
        struct PlatformInitArg
        {
            void* workBuffer;                                    //!< ワークバッファです
            size_t workBufferSize;                               //!< ワークバッファのサイズです
            nn::audio::AudioRendererParameter rendererParameter; //!< オーディオレンダラのパラメータです
        };

    public:
        // @brief 初期化パラメータの初期値を取得します
        void GetDefaultInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
        // @brief 初期化を行います
        Result Initialize(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
        // @brief 終了処理を行います
        void Finalize() NN_NOEXCEPT;
        // @brief 更新処理を行います
        bool Update() NN_NOEXCEPT NN_OVERRIDE;

        // @brief 更新処理のロックを行います
        void Lock() NN_NOEXCEPT;
        // @brief 更新処理のアンロックを行います
        void Unlock() NN_NOEXCEPT;

        // プラットフォーム固有
        // @brief 必要なプラットフォーム固有のバッファサイズを取得します
        size_t GetRequiredPlatformBufferSize(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT;
        // @brief 必要なメモリプールにアタッチされたバッファサイズを取得します
        size_t GetRequiredBufferSizeForMemoryPool(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT;
        // @brief ファイナルミックスを取得します
        FinalMix& GetFinalMix() NN_NOEXCEPT;
        // @brief ファイナルミックスを取得します
        const FinalMix& GetFinalMix() const NN_NOEXCEPT;
        // @brief オーディオレンダラの更新イベントがシグナルされるのを待ちます
        void WaitUpdateAudioRendererEvent() NN_NOEXCEPT;
        // @brief オーディオレンダラの更新イベントがシグナルされるのを時限つきで待ちます
        bool TimedWaitUpdateAudioRendererEvent(int64_t waitTimeMsec) NN_NOEXCEPT;
        // @brief シンクやファイナルミックスのチャンネル数を取得します
        int GetChannelCount() const NN_NOEXCEPT;


        nn::audio::AudioRendererConfig& GetAudioRendererConfig() NN_NOEXCEPT;
        nn::audio::AudioRendererParameter& GetAudioRendererParameter() NN_NOEXCEPT;
        const nn::audio::AudioRendererConfig& GetAudioRendererConfig() const NN_NOEXCEPT;
        const nn::audio::AudioRendererParameter& GetAudioRendererParameter() const NN_NOEXCEPT;

    private:
        nn::audio::AudioRendererParameter m_RendererParameter;
        nn::audio::AudioRendererHandle m_RendererHandle;
        nn::audio::AudioRendererConfig m_RendererConfig;
        nn::os::SystemEvent m_RendererEvent;
        nn::audio::DeviceSinkType m_DeviceSinkType;
        void* m_pRendererWorkBuffer;
        void* m_pRendererConfigWorkBuffer;
        bool m_IsNeedRendererUpdate;
        fnd::CriticalSection m_UpdateCriticalSection;
        FinalMix m_FinalMix;

        AudioEngine* m_pAudioEngine;
    };

}}}

