﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nn/atk/atk_SoundArchive.h
 *
 * @file atk_SoundArchive.h
 */

#pragma once

#include <nn/atk/atk_Util.h>
#include <nn/atk/atk_Global.h>
#include <nn/atk/atk_ItemType.h>
#include <nn/atk/atk_Config.h>
#include <nn/atk/fnd/io/atkfnd_FileStream.h>
#include <nn/atk/detail/atk_BinaryTypes.h>

namespace nn {

namespace atk {

namespace detail {
class SoundArchiveFileReader;
class SoundArchiveParametersHook;
} // namespace nn::atk::detail


//---------------------------------------------------------------------------
//! @brief        サウンドアーカイブを扱うクラスです。
//!
//!               ユーザーがこのクラスを直接扱うことはできません。
//!               このクラスを継承した @ref FsSoundArchive クラス、@ref MemorySoundArchive クラス、
//!               または @ref AddonSoundArchive クラスを使用してください。
//!
//! @see FsSoundArchive
//! @see MemorySoundArchive
//! @see AddonSoundArchive
//!
//---------------------------------------------------------------------------
class SoundArchive
{
public:
    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブの中でのアイテム ID を示す型です。
    //!
    //!           サウンド、バンク、プレイヤー、グループ、波形アーカイブに、
    //!           重複しないアイテム ID が割り振られています。
    //!
    //! @see GetItemLabel
    //! @see GetItemId
    //!
    //---------------------------------------------------------------------------
    typedef uint32_t ItemId;

    //---------------------------------------------------------------------------
    //! @brief    無効な ID を表す値です。
    //!
    //---------------------------------------------------------------------------
    static const ItemId InvalidId = 0xffffffff;

    //---------------------------------------------------------------------------
    //! @brief  サウンドのユーザーパラメータの最大数です。
    //!
    //!         @ref ReadSoundUserParam の index として渡せる最大値として利用できます。
    //!
    //! @ref ReadSoundUserParam
    //---------------------------------------------------------------------------
    static const int UserParamIndexMax = 3;
        // snd_SoundArchiveFile.cpp の "UserParamCount - 1" が該当する

    //---------------------------------------------------------------------------
    //! @brief      指定したサウンドの ID が存在しないことを表す値です。
    //!
    //! @see GetSoundUserParam
    //---------------------------------------------------------------------------
    static const uint32_t ResultInvalidSoundId = 0;

    //---------------------------------------------------------------------------
    //! @brief      無効なユーザーパラメータを表す値です。
    //---------------------------------------------------------------------------
    static const uint32_t InvalidUserParam = 0xffffffff;

    //---------------------------------------------------------------------------
    //! @brief    サウンドの種類を表す型です。
    //!
    //! @see GetSoundType
    //!
    //---------------------------------------------------------------------------
    enum SoundType
    {
        SoundType_Invalid = 0,     //!< 不正な種類のサウンドです。

        SoundType_Sequence,             //!< シーケンスサウンドです。
        SoundType_Stream,            //!< ストリームサウンドです。
        SoundType_Wave,            //!< ウェーブサウンドです。
        SoundType_AdvancedWave     //!< 高機能版ウェーブサウンドです。
    };

    //---------------------------------------------------------------------------
    //! @brief    ストリームファイルの種類を表す型です。
    //---------------------------------------------------------------------------
    enum StreamFileType
    {
        StreamFileType_Invalid = 0,       //!< 不正なファイルです。

        StreamFileType_NwStreamBinary,    //!< NintendoWare のストリームバイナリファイルです。

        StreamFileType_Opus = 3           //!< Opus形式 のストリームバイナリファイルです。
    };

    //---------------------------------------------------------------------------
    //! @brief    デコード処理を行う対象を表す型です。
    //---------------------------------------------------------------------------
    enum DecodeMode
    {
        DecodeMode_Default = 0,         //!< デフォルトのデコード処理を行います。 Opus 形式の場合は DecodeMode_Cpu となります。

        DecodeMode_Cpu,                 //!< CPU を利用したデコード処理を行います。

        DecodeMode_Accelerator          //!< ハードウェアアクセラレータを利用したデコード処理を行います。
    };

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブの中でのファイル ID を示す型です。
    //!
    //---------------------------------------------------------------------------
    typedef ItemId FileId;

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブの中での文字列 ID を示す型です。
    //!
    //---------------------------------------------------------------------------
    typedef ItemId StringId;

    //---------------------------------------------------------------------------
    //! @brief  サウンドの設定情報の構造体です。
    //!
    //!         isFrontBypass が true で、@ref SoundSystem::SetOutputMode
    //!         で設定する出力モードが OutputMode_Surround の場合、
    //!         当該サウンドはフロントチャンネルをバイパスします。
    //!
    //! @see ReadSoundInfo
    //! @see FileId
    //! @see ItemId
    //! @see PanMode
    //! @see PanCurve
    //---------------------------------------------------------------------------
    struct SoundInfo
    {
        FileId fileId;      //!< サウンドアーカイブ内でのファイル ID です。
        ItemId playerId;    //!< サウンドを再生するときに使用されるプレイヤーの ID です。

        //! アクターでサウンドを再生するときに使用されるアクタープレイヤーの番号です。
        uint8_t actorPlayerId;
        uint8_t playerPriority;  //!< サウンドを再生するときのプレイヤープライオリティの値です。
        uint8_t volume;          //!< サウンドを再生するときの音量です。
        uint8_t remoteFilter;    //!< リモコンフィルタの値です。
        PanMode panMode;    //!< パンモードの種類です。
        PanCurve panCurve;  //!< パンのカーブの種類です。
        SinglePlayType singlePlayType;        //!< 単一再生の種類です。
        uint16_t singlePlayEffectiveDuration; //!< 単一再生の効果時間です。

        bool isFrontBypass; //!< フロントバイパスするか否かを示すフラグです。
    };

    //---------------------------------------------------------------------------
    //! @brief    サウンドの 3D サウンド設定情報の構造体です。
    //!
    //! @see ReadSound3DInfo
    //!
    //---------------------------------------------------------------------------
    struct Sound3DInfo
    {
        //---------------------------------------------------------------------------
        //! @brief    3D 減衰カーブの形状を示す列挙体です。
        //!
        //---------------------------------------------------------------------------
        enum DecayCurve
        {
            DecayCurve_Log    = 1, //!< 対数カーブです。
            DecayCurve_Linear = 2  //!< リニア (直線) です。
        };

        //---------------------------------------------------------------------------
        //! @brief    3D サウンドで各パラメータを計算するかどうかを示すフラグです。
        //!
        //---------------------------------------------------------------------------
        enum FlagControl
        {
            FlagControl_Volume   = (1 << 0), //!< 音量を計算するかどうかを示すフラグです。
            FlagControl_Priority = (1 << 1), //!< プレイヤープライオリティを計算するかどうかを示すフラグです。
            FlagControl_Pan      = (1 << 2), //!< パンを計算するかどうかを示すフラグです。
            FlagControl_SurroundPan     = (1 << 3), //!< サラウンドパンを計算するかどうかを示すフラグです。
            FlagControl_Filter   = (1 << 4)  //!< Biquad フィルタを計算するかどうかを示すフラグです。
        };

        //---------------------------------------------------------------------------
        //! @brief    3D 計算で処理するパラメータを表現するビットフラグです。
        //!           FlagControl_*** の論理和で表現されます。
        //---------------------------------------------------------------------------
        uint32_t flags;
        float decayRatio;     //!< 3D 減衰率です。
        uint8_t decayCurve;      //!< 3D 減衰カーブの形状です。@ref DecayCurve が格納されます。
        uint8_t dopplerFactor;   //!< 3D ドップラーファクターです。
    };

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドに関連付けられるバンクの最大数です。
    //!
    //---------------------------------------------------------------------------
    static const uint32_t SequenceBankMax = nn::atk::SeqBankMax;

    //---------------------------------------------------------------------------
    //! @brief    シーケンスサウンドの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           シーケンスサウンドの情報を取得するための構造体です。
    //!
    //! @see ReadSequenceSoundInfo
    //---------------------------------------------------------------------------
    struct SequenceSoundInfo
    {
        uint32_t startOffset;              //!< 非公開です。ライブラリ内部で使用されています。
        uint32_t bankIds[ SequenceBankMax ];    //!< シーケンスサウンドで使用されるバンク ID です。

        //---------------------------------------------------------------------------
        //! @brief  シーケンスサウンドに含まれるトラックのビットフラグです。
        //!         下位ビットから順にトラック 0, トラック 1, トラック 2,... を表します。
        //!         有効なビットのトラックを操作することができます。
        //---------------------------------------------------------------------------
        uint32_t allocateTrackFlags;

        //! シーケンスサウンドで発音される音のチャンネルプライオリティの値です。
        uint8_t  channelPriority;

        //---------------------------------------------------------------------------
        //! @brief    シーケンスサウンドで発音される音が、
        //!           リリース時にチャンネルプライオリティを固定されるかどうかのフラグです。
        //!           固定しない場合はリリース時にチャンネルプライオリティが下がります。
        //---------------------------------------------------------------------------
        bool isReleasePriorityFix;

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //---------------------------------------------------------------------------
        SequenceSoundInfo() NN_NOEXCEPT
        : startOffset( 0 ),
          allocateTrackFlags( 0 ),
          channelPriority( 0 ),
          isReleasePriorityFix( false )
        {
            for ( auto i = 0; i < SequenceBankMax; i++ )
            {
                bankIds[ i ] = InvalidId;
            }
        }
    };


    //! @brief  ストリームサウンドの最大トラック数です。
    static const uint32_t StreamTrackCount = detail::StreamTrackCount;

    //! @brief  ストリームサウンドの各トラックの情報です。
    struct StreamTrackInfo
    {
        uint8_t volume;  //!< トラックの音量です。
        uint8_t pan;     //!< トラックのパンです。
        uint8_t surroundPan;    //!< トラックのサラウンドパンです。
        uint8_t flags;   //!< トラックのサラウンドフラグです。
        uint8_t mainSend;            //!< メインバスのセンド量です。
        uint8_t fxSend[AuxBus_Count]; //!< 補助バスのセンド量です。
        uint8_t lowPassFilterFrequency;         //!< ローパスフィルタの値です。
        uint8_t biquadType;          //!< biquadフィルタの種類です。
        uint8_t biquadValue;         //!< biquadフィルタの値です。
        uint8_t channelCount; //!< トラックのチャンネル数です
        int8_t globalChannelIndex[WaveChannelMax]; //!< @internal

        //! @brief コンストラクタです。
        StreamTrackInfo() NN_NOEXCEPT : volume(0), pan(0), surroundPan(0), mainSend(127), lowPassFilterFrequency(64), biquadType(0), biquadValue(0), channelCount(0)
        {
            std::memset(globalChannelIndex, -1, sizeof(int8_t) * WaveChannelMax);
            std::memset(fxSend, 0, sizeof(uint8_t) * AuxBus_Count);
        }
    };

    //---------------------------------------------------------------------------
    //! @brief    ストリームサウンドの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           ストリームサウンドの情報を取得するための構造体です。
    //!
    //! @see ReadStreamSoundInfo
    //---------------------------------------------------------------------------
    struct StreamSoundInfo
    {
        //---------------------------------------------------------------------------
        //! @brief  ストリームサウンドに含まれるトラックのビットフラグです。
        //!         下位ビットから順にトラック 0, トラック 1, トラック 2, … を表します。
        //!         有効なビットのトラックを操作することができます。
        //---------------------------------------------------------------------------
        uint16_t allocateTrackFlags;

        //! ストリームサウンドに含まれるチャンネルの数です。
        uint16_t allocateChannelCount;

        //! ピッチです。
        float pitch;

        //! メインバスのセンド量です。
        uint8_t  mainSend;

        //! 補助バスのセンド量です。
        uint8_t  fxSend[AuxBus_Count];

        //! ストリームサウンドのトラック情報です。
        StreamTrackInfo trackInfo[StreamTrackCount];

        //---------------------------------------------------------------------------
        //! @briefprivate
        //!
        //! @brief  ストリームファイルの種類です。
        //---------------------------------------------------------------------------
        StreamFileType streamFileType;

        //---------------------------------------------------------------------------
        //! @brief  デコード処理を行う対象を指定します。
        //!
        //!         DecodeMode_Default を指定すると、
        //!         NintendoWare のストリームバイナリファイルの場合は設定が無視され、
        //!         Opus 形式の場合は DecodeMode_Cpu となります。
        //---------------------------------------------------------------------------
        DecodeMode decodeMode;

        //---------------------------------------------------------------------------
        //! @briefprivate
        //!
        //! @brief  プリフェッチストリームファイルのサウンドアーカイブ内でのファイル ID です。
        //---------------------------------------------------------------------------
        FileId prefetchFileId;

        //---------------------------------------------------------------------------
        //! @briefprivate
        //!
        //! @brief  ストリームバッファとして使用する外部のバッファプールです。
        //---------------------------------------------------------------------------
        void* streamBufferPool;

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //---------------------------------------------------------------------------
        StreamSoundInfo() NN_NOEXCEPT
        : allocateTrackFlags( 0 ),
          allocateChannelCount( 0 ),
          pitch( 1.0f ),
          mainSend( 127 ),
          streamFileType( StreamFileType_Invalid ),
          decodeMode( DecodeMode_Default ),
          prefetchFileId( InvalidId ),
          streamBufferPool( NULL )
        {
            std::memset(fxSend, 0, sizeof(uint8_t) * AuxBus_Count);
        }

        //---------------------------------------------------------------------------
        //! @brief 内部でトラックに対してチャンネルを割り当てるための計算などを行います。
        //!
        //!        トラック情報のチャンネル数を変更したときには必ず実行する必要があります。
        //!        詳しくは、externalFile デモをご確認ください。
        //---------------------------------------------------------------------------
        void Setup() NN_NOEXCEPT;
    };

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    ストリームサウンド情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           ストリームサウンドの追加情報を取得するための構造体です。
    //!
    //! @see ReadStreamSoundInfo2
    //---------------------------------------------------------------------------
    struct StreamSoundInfo2
    {
        //---------------------------------------------------------------------------
        //! @brief  ループの有無です。
        //---------------------------------------------------------------------------
        bool isLoop;

        //---------------------------------------------------------------------------
        //! @brief  ループスタートフレームです。
        //---------------------------------------------------------------------------
        uint32_t loopStartFrame;

        //---------------------------------------------------------------------------
        //! @brief  ループエンドフレームです。
        //---------------------------------------------------------------------------
        uint32_t loopEndFrame;

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //---------------------------------------------------------------------------
        StreamSoundInfo2() NN_NOEXCEPT
            : isLoop( false ),
            loopStartFrame( 0 ),
            loopEndFrame( 0 )
        {}
    };

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    ウェーブサウンドの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           ウェーブサウンドの情報を取得するための構造体です。
    //!
    //! @see detail_ReadWaveSoundInfo
    //!
    //---------------------------------------------------------------------------
    struct WaveSoundInfo
    {
        uint32_t     index;              // ウェーブサウンドセットファイル (bfwsd) 内で何番目か
        uint32_t     allocateTrackCount; // 現状は 1。今後マルチトラック化した場合、ほかの値が入る
        uint8_t      channelPriority;
        bool    isReleasePriorityFix;

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //---------------------------------------------------------------------------
        WaveSoundInfo() NN_NOEXCEPT
        : allocateTrackCount( 0 ),
          channelPriority( 0 ),
          isReleasePriorityFix( false ) {}
    };

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    高機能版ウェーブサウンドの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           高機能版のウェーブサウンドの情報を取得するための構造体です。
    //!
    //! @see detail_ReadAdvancedWaveSoundInfo
    //!
    //---------------------------------------------------------------------------
    struct AdvancedWaveSoundInfo
    {
        uint32_t waveArchiveId;
    };

    //---------------------------------------------------------------------------
    //! @brief    バンクの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           バンクの情報を取得するための構造体です。
    //!
    //! @see ReadBankInfo
    //! @see FileId
    //!
    //---------------------------------------------------------------------------
    struct BankInfo
    {
        FileId fileId;              //!< サウンドアーカイブ内でのファイル ID です。

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //---------------------------------------------------------------------------
        BankInfo() NN_NOEXCEPT : fileId( InvalidId ) {}
    };

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    波形アーカイブの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           波形アーカイブの情報を取得するための構造体です。
    //!
    //! @see ReadWaveArchiveInfo
    //!
    //---------------------------------------------------------------------------
    struct WaveArchiveInfo
    {
        uint32_t  fileId;
        uint32_t  waveCount;
        bool isLoadIndividual;
        uint8_t padding[3];

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //---------------------------------------------------------------------------
        WaveArchiveInfo() NN_NOEXCEPT : fileId( InvalidId ), isLoadIndividual( false ) {}
    };

    //---------------------------------------------------------------------------
    //! @brief    プレイヤーの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           プレイヤーの情報を取得するための構造体です。
    //!
    //! @see ReadPlayerInfo
    //!
    //---------------------------------------------------------------------------
    struct PlayerInfo
    {
        int playableSoundMax;   //!< プレイヤーで同時に再生できるサウンドの数です。
        uint32_t playerHeapSize;     //!< プレイヤーに割り当てられるプレイヤーヒープのサイズです。

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //---------------------------------------------------------------------------
        PlayerInfo() NN_NOEXCEPT
        : playableSoundMax( 0 ),
          playerHeapSize( 0 ) {}
    };

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    ウェーブサウンドセット・シーケンスサウンドセットの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           ウェーブサウンドセット・シーケンスサウンドセットの情報を
    //!           取得するための構造体です。
    //!
    //! @see detail_ReadSoundGroupInfo
    //---------------------------------------------------------------------------
    struct SoundGroupInfo
    {
        ItemId startId;
        ItemId endId;
        const detail::Util::Table<uint32_t>* fileIdTable;

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //---------------------------------------------------------------------------
        SoundGroupInfo() NN_NOEXCEPT
        : startId( InvalidId ),
          endId( InvalidId ),
          fileIdTable( NULL ) {}
    };

    //---------------------------------------------------------------------------
    //! @brief  グループの設定情報の構造体です。
    //!
    //!         サウンドアーカイブで定義されている、
    //!         グループの情報を取得するための構造体です。
    //!
    //!         groupFileSize には、サウンドアーカイブ内に格納される「グループファイル」
    //!         のファイルサイズが含まれています。
    //!
    //!         SoundMaker 上で設定するグループの出力方式が「埋め込み」の場合は、
    //!         当該グループを SoundDataManager::LoadData
    //!         でロードする際に必要なメモリサイズに相当します。
    //!
    //!         一方、出力方式が「リンク」の場合は、
    //!         グループファイルには「当該グループに含まれるアイテム情報」
    //!         しか含まれていない (＝実データは含まれていない) ため、
    //!         LoadData の際に必要なメモリサイズとは異なります。
    //!
    //! @see ReadGroupInfo
    //---------------------------------------------------------------------------
    struct GroupInfo
    {
        FileId fileId;      //!< サウンドアーカイブ内でのファイル ID です。
        uint32_t groupFileSize;  //!< 上記 fileId のファイルサイズです。

        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //---------------------------------------------------------------------------
        GroupInfo() NN_NOEXCEPT : fileId( InvalidId ), groupFileSize(0) {}
    };

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブプレイヤーの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           サウンドアーカイブプレイヤーの情報を取得するための構造体です。
    //!
    //! @see ReadSoundArchivePlayerInfo
    //!
    //---------------------------------------------------------------------------
    struct SoundArchivePlayerInfo
    {
        int sequenceSoundCount;   //!< 確保するシーケンスサウンドインスタンスの数です。
        int sequenceTrackCount;   //!< 確保するシーケンストラックインスタンスの数です。
        int streamSoundCount;     //!< 確保するストリームサウンドインスタンスの数です。
        int streamTrackCount;     //!< 確保するストリームトラックインスンスの数です。
        int streamChannelCount;   //!< 確保するストリームチャンネルインスタンスの数です。
        int waveSoundCount;       //!< 確保するウェーブサウンドインスタンスの数です。
        int waveTrackCount;       //!< 確保するウェーブサウンドトラックの数です。
        int streamBufferTimes;    //!< 確保するストリームサウンドのバッファサイズの倍率です。
        bool isAdvancedWaveSoundEnabled;  //!< @briefprivate
    };

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    ファイルの設定情報の構造体です。
    //!
    //!           サウンドアーカイブで定義されている、
    //!           ファイルの情報を取得するための構造体です。
    //!
    //! @see detail_ReadFileInfo
    //!
    //---------------------------------------------------------------------------
    struct FileInfo
    {
        static const uint32_t InvalidOffset = detail::BinaryTypes::InvalidOffset;
        static const uint32_t InvalidSize = detail::BinaryTypes::InvalidSize;

        uint32_t fileSize;
        uint32_t offsetFromFileBlockHead;
        const char* externalFilePath;


        //---------------------------------------------------------------------------
        //! @brief コンストラクタです。
        //---------------------------------------------------------------------------
        FileInfo() NN_NOEXCEPT
        : fileSize( InvalidSize ),
          offsetFromFileBlockHead( InvalidOffset ),
          externalFilePath( NULL )
        {}
    };

    //----------------------------------------
    //! @name コンストラクタ / デストラクタ
    //@{
protected:
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //---------------------------------------------------------------------------
    SoundArchive() NN_NOEXCEPT;

public:
    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //---------------------------------------------------------------------------
    virtual ~SoundArchive() NN_NOEXCEPT;

    //@}

public:
    //----------------------------------------
    //! @name 状態取得
    //@{
    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のデータが利用可能な状態か調べます。
    //!
    //! @return   サウンドアーカイブ中のデータが利用可能な状態なら true を、
    //!           そうでなければ false を返します。
    //!
    //---------------------------------------------------------------------------
    bool IsAvailable() const NN_NOEXCEPT;
    //@}

    //----------------------------------------
    //! @name ID / ラベル文字列
    //@{
    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のサウンドの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中のサウンドの個数を返します。
    //!
    //---------------------------------------------------------------------------
    uint32_t GetSoundCount() const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のグループの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中のグループの個数を返します。
    //!
    //---------------------------------------------------------------------------
    uint32_t GetGroupCount() const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のプレイヤーの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中のプレイヤーの個数を返します。
    //!
    //---------------------------------------------------------------------------
    uint32_t GetPlayerCount() const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief    サウンドアーカイブ中のウェーブサウンドセット・
    //!           シーケンスサウンドセットの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中のウェーブサウンドセット・
    //!           シーケンスサウンドセットの個数を返します。
    //!
    //---------------------------------------------------------------------------
    uint32_t GetSoundGroupCount() const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のバンクの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中のバンクの個数を返します。
    //!
    //---------------------------------------------------------------------------
    uint32_t GetBankCount() const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中の波形アーカイブの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中の波形アーカイブの個数を返します。
    //!
    //---------------------------------------------------------------------------
    uint32_t GetWaveArchiveCount() const NN_NOEXCEPT;

    //!@}
    //---------------------------------------------------------------------------
    //! @internal
    //!
    //! @brief    サウンドアーカイブ中のファイルの個数を取得します。
    //!
    //! @return   サウンドアーカイブ中のファイルの個数を返します。
    //!
    //---------------------------------------------------------------------------
    uint32_t detail_GetFileCount() const NN_NOEXCEPT;
    //! @name ID / ラベル文字列
    //@{

    //---------------------------------------------------------------------------
    //! @brief  アイテム ID から、対応するラベル文字列を取得します。
    //!
    //!         指定したアイテム ID が存在しない場合は、NULL を返します。
    //!
    //!         @ref MemorySoundArchive, @ref AddonSoundArchive, @ref FsSoundArchive 問わず、
    //!         文字列情報を利用する場合は、
    //!         サウンドアーカイブファイルに文字列テーブルが含まれている必要があります。
    //!
    //!         サウンドアーカイブファイルに文字列テーブルを含めるには、
    //!         SoundMaker の [プロジェクト設定] - [サウンドアーカイブ] タブ -
    //!         [文字列テーブルを出力する] にチェックをつけておいてください。
    //!
    //!         @ref FsSoundArchive 使用時は、あらかじめ
    //!         @ref FsSoundArchive::LoadLabelStringData でラベル文字列データを
    //!         ロードしておく必要があります。
    //!         文字列データをロードしていない無い場合は、NULL を返します。
    //!
    //!         @ref MemorySoundArchive, @ref AddonSoundArchive に文字列テーブルが含まれていない場合は、
    //!         NULL を返します。
    //!
    //! @param[in]    id      サウンドアーカイブ内のアイテム ID です。
    //!
    //! @return   指定した ID に対応したラベル文字列を返します。
    //!
    //! @see ItemId
    //! @see GetItemId
    //! @see FsSoundArchive::LoadLabelStringData
    //!
    //---------------------------------------------------------------------------
    const char* GetItemLabel( ItemId id ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  ラベル文字列から、対応するアイテム ID に変換します。
    //!
    //!         指定したラベル文字列が存在しない場合は、
    //!         @ref InvalidId を返します。
    //!
    //!         @ref MemorySoundArchive, @ref AddonSoundArchive, @ref FsSoundArchive 問わず、
    //!         文字列情報を利用する場合は、
    //!         サウンドアーカイブファイルに文字列テーブルが含まれている必要があります。
    //!
    //!         サウンドアーカイブファイルに文字列テーブルを含めるには、
    //!         SoundMaker の [プロジェクト設定] - [サウンドアーカイブ] タブ -
    //!         [文字列テーブルを出力する] にチェックをつけておいてください。
    //!
    //!         @ref FsSoundArchive 使用時は、あらかじめ
    //!         @ref FsSoundArchive::LoadLabelStringData でラベル文字列データを
    //!         ロードしておく必要があります。
    //!
    //! @param[in]    pLabel  ラベル文字列です。
    //!
    //! @return   指定したラベル文字列に対応したアイテム ID を返します。
    //!
    //! @see GetItemLabel
    //! @see FsSoundArchive::LoadLabelStringData
    //!
    //---------------------------------------------------------------------------
    ItemId GetItemId( const char* pLabel ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  アイテム ID から、対応するファイル ID を取得します。
    //!
    //!         変換できるアイテム ID の種類は、
    //!         サウンド (STRM / WSD / SEQ) および、
    //!         ウェーブサウンドセット、シーケンスサウンドセット、
    //!         バンク、波形アーカイブ、グループです。
    //!
    //!         ウェーブサウンドセットの ID を指定した場合と、
    //!         当該ウェーブサウンドセットに含まれる任意のウェーブサウンドの ID
    //!         を指定した場合では、同じファイル ID が返ります。
    //!
    //!         シーケンスサウンドセットの ID を指定した場合と、
    //!         当該シーケンスサウンドセットに含まれるシーケンスサウンドのうち、
    //!         一番若い ID のシーケンスサウンドを指定した場合では、
    //!         同じファイル ID が返ります。
    //!
    //!         プレイヤーの ID を指定すると、
    //!         SoundArchive::InvalidId が返ります。
    //!
    //!         本関数で取得したファイル ID は、@ref SoundDataManager::SetFileAddress
    //!         関数で利用することができます。
    //!
    //! @param[in]    id      サウンドアーカイブ内のアイテム ID です。
    //!
    //! @return   指定したアイテム ID に対応したファイル ID を返します。
    //!
    //! @see FileId
    //! @see ItemId
    //! @see SoundDataManager::SetFileAddress
    //!
    //---------------------------------------------------------------------------
    FileId GetItemFileId( ItemId id ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  アイテム ID から、対応するプリフェッチデータのファイル ID を取得します。
    //!
    //!         変換できるアイテム ID の種類は、
    //!         プリフェッチデータを持つストリームサウンドのみです。
    //!
    //!         それ以外のプレイヤーの ID を指定すると、
    //!         SoundArchive::InvalidId が返ります。
    //!
    //!         本関数で取得したファイル ID は、@ref SoundDataManager::SetFileAddress
    //!         関数で利用することができます。
    //!
    //! @param[in]    id      サウンドアーカイブ内のアイテム ID です。
    //!
    //! @return   指定したアイテム ID に対応したプリフェッチデータのファイル ID を返します。
    //!
    //! @see FileId
    //! @see ItemId
    //! @see SoundDataManager::SetFileAddress
    //---------------------------------------------------------------------------
    FileId GetItemPrefetchFileId( ItemId id ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief      サウンドの通し番号をサウンド ID に変換します。
    //!
    //!             index に 0 以上 @ref GetSoundCount 未満の値を渡すと、
    //!             全サウンドの ID を取得することができます。
    //!
    //!             最大値は範囲チェックされませんので、
    //!             取得値を利用する場合は、アプリケーション側で注意してください。
    //!
    //! @param[in]  index   サウンドの通し番号です。
    //!
    //! @return     通し番号に対応したサウンド ID を返します。
    //!
    //! @see ItemId
    //!
    //---------------------------------------------------------------------------
    static ItemId GetSoundIdFromIndex( uint32_t index ) NN_NOEXCEPT
    {
        return detail::Util::GetMaskedItemId( index, detail::ItemType_Sound );
    }
    //---------------------------------------------------------------------------
    //! @briefprivate
    //!
    //! @brief      サウンドグループ (ウェーブサウンドセット、シーケンスサウンドセット)
    //!             の通し番号をサウンド ID に変換します。
    //!
    //!             index に 0 以上 @ref GetSoundGroupCount 未満の値を渡すと、
    //!             全サウンドグループの ID を取得することができます。
    //!
    //!             最大値は範囲チェックされませんので、
    //!             取得値を利用する場合は、アプリケーション側で注意してください。
    //!
    //! @param[in]  index   サウンドグループの通し番号です。
    //!
    //! @return     通し番号に対応したサウンドグループ ID を返します。
    //!
    //! @see ItemId
    //! @see GetSoundGroupCount
    //!
    //---------------------------------------------------------------------------
    static ItemId GetSoundGroupIdFromIndex( uint32_t index ) NN_NOEXCEPT
    {
        return detail::Util::GetMaskedItemId( index, detail::ItemType_SoundGroup );
    }
    //---------------------------------------------------------------------------
    //! @brief      バンクの通し番号をバンク ID に変換します。
    //!
    //!             index に 0 以上 @ref GetBankCount 未満の値を渡すと、
    //!             全バンクの ID を取得することができます。
    //!
    //!             最大値は範囲チェックされませんので、
    //!             取得値を利用する場合は、アプリケーション側で注意してください。
    //!
    //! @param[in]  index   バンクの通し番号です。
    //!
    //! @return     通し番号に対応したバンク ID を返します。
    //!
    //! @see ItemId
    //! @see GetBankCount
    //!
    //---------------------------------------------------------------------------
    static ItemId GetBankIdFromIndex( uint32_t index ) NN_NOEXCEPT
    {
        return detail::Util::GetMaskedItemId( index, detail::ItemType_Bank );
    }
    //---------------------------------------------------------------------------
    //! @brief      プレイヤーの通し番号をプレイヤー ID に変換します。
    //!
    //!             index に 0 以上 @ref GetPlayerCount 未満の値を渡すと、
    //!             全プレイヤーの ID を取得することができます。
    //!
    //!             最大値は範囲チェックされませんので、
    //!             取得値を利用する場合は、アプリケーション側で注意してください。
    //!
    //! @param[in]  index   プレイヤーの通し番号です。
    //!
    //! @return     通し番号に対応したプレイヤー ID を返します。
    //!
    //! @see ItemId
    //! @see GetPlayerCount
    //!
    //---------------------------------------------------------------------------
    static ItemId GetPlayerIdFromIndex( uint32_t index ) NN_NOEXCEPT
    {
        return detail::Util::GetMaskedItemId( index, detail::ItemType_Player );
    }
    //---------------------------------------------------------------------------
    //! @brief      波形アーカイブの通し番号を波形アーカイブ ID に変換します。
    //!
    //!             index に 0 以上 @ref GetWaveArchiveCount 未満の値を渡すと、
    //!             全波形アーカイブの ID を取得することができます。
    //!
    //!             最大値は範囲チェックされませんので、
    //!             取得値を利用する場合は、アプリケーション側で注意してください。
    //!
    //! @param[in]  index   波形アーカイブの通し番号です。
    //!
    //! @return     通し番号に対応した波形アーカイブ ID を返します。
    //!
    //! @see ItemId
    //! @see GetWaveArchiveCount
    //!
    //---------------------------------------------------------------------------
    static ItemId GetWaveArchiveIdFromIndex( uint32_t index ) NN_NOEXCEPT
    {
        return detail::Util::GetMaskedItemId( index, detail::ItemType_WaveArchive );
    }
    //---------------------------------------------------------------------------
    //! @brief      グループの通し番号をグループ ID に変換します。
    //!
    //!             index に 0 以上 @ref GetGroupCount 未満の値を渡すと、
    //!             全グループの ID を取得することができます。
    //!
    //!             最大値は範囲チェックされませんので、
    //!             取得値を利用する場合は、アプリケーション側で注意してください。
    //!
    //! @param[in]  index   グループの通し番号です。
    //!
    //! @return     通し番号に対応したグループ ID を返します。
    //!
    //! @see ItemId
    //! @see GetGroupCount
    //!
    //---------------------------------------------------------------------------
    static ItemId GetGroupIdFromIndex( uint32_t index ) NN_NOEXCEPT
    {
        return detail::Util::GetMaskedItemId( index, detail::ItemType_Group );
    }
    //@}


    //----------------------------------------
    //! @name 情報取得
    //@{
    //---------------------------------------------------------------------------
    //! @brief  指定した ID のサウンドのユーザーパラメータ (無印) を取得します。
    //!
    //!         指定した ID のサウンドが存在しない場合は、
    //!         @ref ResultInvalidSoundId を返します。
    //!
    //!         SoundMaker でユーザーパラメータ (無印) を無効にした場合、
    //!         @ref InvalidUserParam を返します。
    //!
    //!         本関数では、ユーザーパラメータ (無印) のデータしか取得できません。
    //!         ユーザーパラメータ 1 ～ 3 を取得したい場合は、
    //!         @ref ReadSoundUserParam をご利用ください。
    //!
    //! @param[in]    soundId     サウンドアーカイブ内のサウンド ID です。
    //!
    //! @return   サウンドのユーザーパラメータを返します。
    //!
    //! @see    ItemId
    //! @see    ReadSoundUserParam
    //---------------------------------------------------------------------------
    uint32_t GetSoundUserParam( ItemId soundId ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  指定した ID のサウンドのユーザーパラメータを取得します。
    //!
    //!         index は 0 ～ @ref UserParamIndexMax まで指定することができます。
    //!         SoundMaker の [プロジェクトの設定] - [ユーザーパラメータ] タブで有効にした
    //!         ユーザーパラメータの番号と異なる番号を index として渡した場合は、
    //!         取得に失敗し false を返します。
    //!
    //!         SoundMaker 上の「ユーザーパラメータ (無印)」は、index = 0 に相当します。
    //!         「ユーザーパラメータ 1 ～ 3」は、index = 1 ～ 3 に相当します。
    //!
    //!         soundId として 不正なサウンド ID を渡した場合も取得に失敗し、
    //!         false を返します。
    //!
    //!         取得に成功すると、value に取得できたパラメータが格納され、
    //!         true を返します。
    //!
    //!         index に 0 を渡して取得できた値は、
    //!         @ref GetSoundUserParam で得られる値と同じです。
    //!         (@ref GetSoundUserParam は過去との互換性のために残されています)
    //!
    //! @param[out]   pOutValue ユーザーパラメータの格納先です。
    //! @param[in]    soundId   サウンドアーカイブ内のサウンド ID です。
    //! @param[in]    index     ユーザーパラメータの番号です。
    //!
    //! @return   サウンドのユーザーパラメータが取得できた場合は true を、
    //!           取得できなかった場合は false を返します。
    //!
    //! @see    ItemId
    //! @see    UserParamIndexMax
    //! @see    GetSoundUserParam
    //---------------------------------------------------------------------------
    bool ReadSoundUserParam( uint32_t* pOutValue, ItemId soundId, int index ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のサウンドの種類を取得します。
    //!
    //!           指定した ID のサウンドが存在しない場合は、
    //!           @ref SoundType_Invalid を返します。
    //!
    //! @param[in]    soundId     サウンドアーカイブ内のサウンド ID です。
    //!
    //! @return   サウンドの種類を返します。
    //!
    //! @see SoundType
    //! @see ItemId
    //!
    //---------------------------------------------------------------------------
    SoundType GetSoundType( ItemId soundId ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のサウンドの情報を取得します。
    //!
    //!           指定した ID のサウンドが存在しない場合は読み取りは失敗します。
    //!
    //! @param[out]   pOutValue   サウンド情報を格納するための構造体です。
    //! @param[in]    soundId     サウンドアーカイブ内のサウンド ID です。
    //!
    //! @return   サウンド情報の取得に成功したら true を、失敗したら false を返します。
    //!
    //! @see ItemId
    //! @see SoundInfo
    //---------------------------------------------------------------------------
    bool ReadSoundInfo( SoundInfo* pOutValue, ItemId soundId ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のシーケンスサウンドの情報を取得します。
    //!
    //!           指定した ID のシーケンスサウンドが存在しない場合は読み取りは失敗します。
    //!
    //! @param[out]   pOutValue   シーケンスサウンド情報を格納するための構造体です。
    //! @param[in]    soundId     サウンドアーカイブ内のサウンド ID です。
    //!
    //! @return   シーケンスサウンド情報の取得に成功したら true を、失敗したら false を返します。
    //!
    //! @see ItemId
    //! @see SequenceSoundInfo
    //---------------------------------------------------------------------------
    bool ReadSequenceSoundInfo( SequenceSoundInfo* pOutValue, ItemId soundId ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のバンクの情報を取得します。
    //!
    //!           指定した ID のバンクが存在しない場合は読み取りは失敗します。
    //!
    //! @param[out]   pOutValue   バンク情報を格納するための構造体です。
    //! @param[in]    bankId      サウンドアーカイブ内のバンク ID です。
    //!
    //! @return   バンク情報の取得に成功したら true を、失敗したら false を返します。
    //!
    //! @see ItemId
    //! @see PlayerInfo
    //---------------------------------------------------------------------------
    bool ReadBankInfo( BankInfo* pOutValue, ItemId bankId ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のプレイヤーの情報を取得します。
    //!
    //!           指定した ID のプレイヤーが存在しない場合は読み取りは失敗します。
    //!
    //! @param[out]   pOutValue   プレイヤー情報を格納するための構造体です。
    //! @param[in]    playerId    サウンドアーカイブ内のプレイヤー ID です。
    //!
    //! @return   プレイヤー情報の取得に成功したら true を、失敗したら false を返します。
    //!
    //! @see ItemId
    //! @see PlayerInfo
    //---------------------------------------------------------------------------
    bool ReadPlayerInfo( PlayerInfo* pOutValue, ItemId playerId ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブプレイヤーの情報を取得します。
    //!
    //! @param[out]   pOutValue    サウンドアーカイブプレイヤー情報を格納するための構造体です。
    //!
    //! @return   サウンドアーカイブプレイヤー情報の取得に成功したら true を、
    //!           失敗したら false を返します。
    //!
    //! @see SoundArchivePlayerInfo
    //---------------------------------------------------------------------------
    bool ReadSoundArchivePlayerInfo( SoundArchivePlayerInfo* pOutValue ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のストリームサウンドの情報を取得します。
    //!
    //!           指定した ID のストリームサウンドが存在しない場合は読み取りは失敗します。
    //!
    //! @param[out]   pOutValue   ストリームサウンド情報を格納するための構造体です。
    //! @param[in]    soundId     サウンドアーカイブ内のサウンド ID です。
    //!
    //! @return   ストリームサウンド情報の取得に成功したら true を、失敗したら false を返します。
    //!
    //! @see ItemId
    //! @see StreamSoundInfo
    //---------------------------------------------------------------------------
    bool ReadStreamSoundInfo( StreamSoundInfo* pOutValue, ItemId soundId ) const NN_NOEXCEPT;

    //! @}
    //! @internal
    //! @param soundId :private
    //! @param info :private
    //! @return :private
    bool detail_ReadStreamSoundInfo2( ItemId soundId, StreamSoundInfo2* info ) const NN_NOEXCEPT;

    //! @internal
    //! @param soundId :private
    //! @param info :private
    //! @return :private
    bool detail_ReadWaveSoundInfo( ItemId soundId, WaveSoundInfo* info ) const NN_NOEXCEPT;

    //! @internal
    //! @param soundId :private
    //! @param info :private
    //! @return :private
    bool detail_ReadAdvancedWaveSoundInfo( ItemId soundId, AdvancedWaveSoundInfo* info ) const NN_NOEXCEPT;
    //! @name 情報取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief    指定した ID の 3D サウンド情報を取得します。
    //!
    //!           指定した ID のサウンドが存在しない場合は読み取りに失敗します。
    //!
    //! @param[out]   pOutValue サウンドアーカイブプレイヤー情報を格納するための構造体です。
    //! @param[in]    soundId サウンドアーカイブ内のサウンド ID です。
    //!
    //! @return   3D サウンド情報の取得に成功したら true を、
    //!           失敗したら false を返します。
    //!
    //! @see Sound3DInfo
    //---------------------------------------------------------------------------
    bool ReadSound3DInfo( Sound3DInfo* pOutValue, ItemId soundId ) const NN_NOEXCEPT;

    //! @briefprivate
    //! @param warcId :private
    //! @param info :private
    //! @return :private
    bool ReadWaveArchiveInfo( ItemId warcId, WaveArchiveInfo* info ) const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID のグループ情報を取得します。
    //!
    //!           指定した ID のグループが存在しない場合は読み取りに失敗します。
    //!
    //! @param[out]   pOutValue グループ情報を格納するための構造体です。
    //! @param[in]    groupId サウンドアーカイブ内のグループ ID です。
    //!
    //! @return   グループ情報の取得に成功したら true を、
    //!           失敗したら false を返します。
    //!
    //! @see GroupInfo
    //---------------------------------------------------------------------------
    bool ReadGroupInfo( GroupInfo* pOutValue, ItemId groupId ) const NN_NOEXCEPT;

    //! @}
    //! @internal
    //! @param groupId :private
    //! @param info :private
    //! @return :private
    bool detail_ReadSoundGroupInfo( ItemId groupId, SoundGroupInfo* info ) const NN_NOEXCEPT;

    //! @internal
    //! @param fileId :private
    //! @param info :private
    //! @return :private
    bool detail_ReadFileInfo( FileId fileId, FileInfo* info ) const NN_NOEXCEPT;

    // TODO: bool detail_ReadGroupItemInfo( GroupId groupId, unsigned long index, GroupItemInfo* info ) const;

    // TODO: bool detail_ReadFilePos( FileId fileId, unsigned long index, FilePos* filePos ) const;

    //! @internal
    //! @param id :private
    //! @return :private
    const detail::Util::Table<uint32_t>* detail_GetWaveArchiveIdTable( ItemId id ) const NN_NOEXCEPT;

    // ファイルアクセス
    //! @internal
    //! @param fileId :private
    //! @return :private
    virtual const void* detail_GetFileAddress( FileId fileId ) const NN_NOEXCEPT = 0;
    //! @internal
    //! @return :private
    virtual size_t detail_GetRequiredStreamBufferSize() const NN_NOEXCEPT = 0;

    //! @internal
    //! @param fileId :private
    //! @param buffer :private
    //! @param size :private
    //! @param cacheBuffer :private
    //! @param cacheSize :private
    //! @return :private
    detail::fnd::FileStream* detail_OpenFileStream(
            FileId fileId, void* buffer, size_t size, void* cacheBuffer, size_t cacheSize ) const NN_NOEXCEPT;

    //! @internal
    //! @param fileId :private
    //! @return :private
    const detail::Util::Table<uint32_t>*
        detail_GetAttachedGroupTable( FileId fileId ) const NN_NOEXCEPT;

    //! @internal
    //! @return :private
    detail::SoundArchiveParametersHook* detail_GetParametersHook() const NN_NOEXCEPT
    {
        return m_pParametersHook;
    }

    //! @internal
    //! @param parametersHook :private
    void detail_SetParametersHook(detail::SoundArchiveParametersHook* parametersHook) NN_NOEXCEPT
    {
        m_pParametersHook = parametersHook;
    }

    //----------------------------------------
    //! @name ファイルパス
    //@{
    //---------------------------------------------------------------------------
    //! @brief    外部参照ファイルのルートディレクトリを設定します。
    //!
    //!           サウンドアーカイブの外部参照ファイルのファイルパスが、
    //!           相対パスで記述されている場合には、
    //!           この関数で設定したルートディレクトリからの相対パスとして解釈されます。
    //!
    //!           初期状態はルートディレクトリに設定されています。
    //!
    //!           この関数は @ref nn::atk::FsSoundArchive::Open から呼び出されています。
    //!
    //! @param[in] extFileRoot    ルートディレクトリに設定するディレクトリパスです。
    //!
    //! @see nn::atk::FsSoundArchive::Open
    //!
    //---------------------------------------------------------------------------
    void SetExternalFileRoot( const char* extFileRoot ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief    指定した ID が参照するストリームサウンドのファイルパスを取得します。
    //!
    //!           @ref nn::atk::FsSoundArchive を使用している場合は、
    //!           @ref nn::atk::FsSoundArchive::Open 呼出し後に使用できるようになります。
    //!
    //!           @ref nn::atk::MemorySoundArchive, @ref nn::atk::AddonSoundArchive を使用している場合は、
    //!           @ref nn::atk::SoundArchive::SetExternalFileRoot による
    //!           セットアップ完了後に使用できるようになります。
    //!
    //! @param[out] outFilePathBuffer  正常に終了した場合、指定した ID が参照するストリームサウンドのファイルパスがヌル文字を含めて格納されます。
    //! @param[in]  filePathBufferSize  outFilePathBuffer のサイズを指定します。ヌル文字を含めたファイルパスを格納できるサイズを指定してください。
    //! @param[in]  soundId  ファイルパスを取得したいストリームサウンドの ID を指定します。
    //!
    //! @return  ファイルパスの読み込みに成功したかの値を返します。成功した場合には true を返します。
    //!          soundId にストリームサウンド以外の ID を指定した場合や、存在しない ID を指定した場合など
    //!          正しく処理を続行できなかった場合には false を返します。
    //!
    //! @see nn::atk::FsSoundArchive::Open
    //! @see nn::atk::SoundArchive::SetExternalFileRoot
    //!
    //---------------------------------------------------------------------------
    bool ReadStreamSoundFilePath( char* outFilePathBuffer, size_t filePathBufferSize, ItemId soundId ) const NN_NOEXCEPT;

    //! @briefprivate
    virtual void FileAccessBegin() const NN_NOEXCEPT {}

    //! @briefprivate
    virtual void FileAccessEnd() const NN_NOEXCEPT {}

    //! @briefprivate
    //! @brief 追加サウンドアーカイブであるかを取得します。
    //! @return 追加サウンドアーカイブであれば true を返します。
    virtual bool IsAddon() const NN_NOEXCEPT
    {
        return false;
    }
    //@}

    //! @internal
    //! @param path :private
    //! @param buf :private
    //! @param bufSize :private
    //! @return :private
    const char* detail_GetExternalFileFullPath(const char* path, char* buf, size_t bufSize) const NN_NOEXCEPT;

protected:
    static const int FilePathMax  = detail::FilePathMax;

    //! @internal
    //! @param fileReader :private
    void Initialize( detail::SoundArchiveFileReader* fileReader ) NN_NOEXCEPT;

    //! @internal
    void Finalize() NN_NOEXCEPT;

    //! @internal
    //! @param buffer :private
    //! @param size :private
    //! @param begin :private
    //! @param length :private
    //! @return :private
    virtual detail::fnd::FileStream* OpenStream(
        void* buffer,
        size_t size,
        detail::position_t begin,
        size_t length
    ) const NN_NOEXCEPT = 0;

    //! @internal
    //! @param buffer :private
    //! @param size :private
    //! @param extFilePath :private
    //! @param cacheBuffer :private
    //! @param cacheSize :private
    //! @return :private
    virtual detail::fnd::FileStream* OpenExtStream(
        void* buffer,
        size_t size,
        const char* extFilePath,
        void* cacheBuffer,
        size_t cacheSize
    ) const NN_NOEXCEPT = 0;

private:
    detail::fnd::FileStream* OpenExtStreamImpl(
        void* buffer,
        size_t size,
        const char* extFilePath,
        void* cacheBuffer,
        size_t cacheSize
    ) const NN_NOEXCEPT;

    detail::SoundArchiveFileReader* m_pFileReader;
    detail::SoundArchiveParametersHook* m_pParametersHook;

    char m_ExtFileRoot[ FilePathMax ];

    // キャッシュ (頻繁にアクセスする)
    uint32_t m_FileBlockOffset;
};

} // namespace nn::atk
} // namespace nn

