﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Diagnostics;

using LOG = Common.Logger;
using LOG_LEVEL = Common.Logger.Level;

namespace BackupSaveData
{
    public class Execute
    {
        public static bool ExecuteTarget(string executePath, string targetString, string[] args, string successPattern = null, bool verbose = false)
        {
            string runOnTargetPath = GetRunOnTargetCommandPath();
            if (runOnTargetPath == null) return false;

            // RunOnTarget.exe BackupSaveData.nsp --failure-timeout 60
            List<string> argsList1 = new List<string>() { "--target", targetString, "--failure-timeout", "60" };
            if(successPattern != null)
            {
                argsList1.Add("--pattern-not-found-failure=\"" + successPattern + '"');
            }
            if(verbose)
            {
                argsList1.Add("--verbose");
            }
            List<string> argsList2 = new List<string>() { executePath, "--" };
            argsList2.AddRange(args);
            List<string> argsList = new List<string>(argsList1.Concat(argsList2));

            return ExecuteCommand(runOnTargetPath, argsList.ToArray(), verbose);
        }

        public static bool ExecuteCommand(string commandPath, string[] args, bool verbose = false)
        {
            string argString = string.Empty;
            foreach (string arg in args)
            {
                argString += arg + " ";
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Execute {0}", commandPath);
            foreach (string arg in args)
            {
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "    {0}", arg);
            }

            int exitCode = -1;
            // ProcessOutputBuffer = null;
            ProcessStartInfo psInfo = new ProcessStartInfo();
            psInfo.FileName = commandPath;
            psInfo.Arguments = argString;
            psInfo.CreateNoWindow = true;
            psInfo.UseShellExecute = false;
            psInfo.RedirectStandardOutput = true;
            psInfo.RedirectStandardError = true;
            using (Process hProcess = Process.Start(psInfo))
            {
                hProcess.OutputDataReceived += CommandReceived;
                hProcess.ErrorDataReceived += CommandReceived;
                hProcess.BeginOutputReadLine();
                hProcess.BeginErrorReadLine();
                hProcess.WaitForExit();
                exitCode = hProcess.ExitCode;
                if (exitCode != 0)
                {
                    return false;
                }
            }

            return true;
        }

        private static void CommandReceived(object sender, DataReceivedEventArgs e)
        {
            LOG.LogLine(e.Data);
            // ProcessOutputBuffer += e.Data;
        }

        private static string GetRunOnTargetCommandPath()
        {
            string sdkRootPath = GetSdkRootPath();
            if (sdkRootPath == null) return null;
            const string RunOnTargetBasePath = @"Tools\CommandLineTools\RunOnTarget.exe";
            string runOnTargetPath = Path.Combine(sdkRootPath, RunOnTargetBasePath);
            if(!File.Exists(runOnTargetPath))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "{0} not found.", runOnTargetPath);
                return null;
            }
            return runOnTargetPath;
        }

        public static string GetSdkRootPath()
        {
            const string NintendoSdkRootMarkFileName = "NintendoSdkRootMark";
            string startPath = System.Reflection.Assembly.GetEntryAssembly().Location;
            string currentDirectory = startPath;
            string rootDirectory = Path.GetPathRoot(currentDirectory);

            while (rootDirectory != currentDirectory)
            {
                string sigloRootMarkFilePath = Path.Combine(currentDirectory, NintendoSdkRootMarkFileName);
                if (File.Exists(sigloRootMarkFilePath))
                {
                    return currentDirectory;
                }

                currentDirectory = Directory.GetParent(currentDirectory).ToString();
            }

            return null;
        }
    }
}
