﻿using System;
using System.IO;

namespace SdkEnvironmentCheckerLibrary
{
    internal static class PathUtility
    {
        internal static readonly char[] Slashes = new[] { Path.DirectorySeparatorChar, Path.AltDirectorySeparatorChar };

        /// <summary>
        /// 文字列中の環境変数を展開します。
        /// </summary>
        /// <param name="str">対象の文字列です。</param>
        /// <returns>環境変数を展開した文字列です。</returns>
        /// <exception cref="ArgumentNullException">str が null です。</exception>
        /// <exception cref="ArgumentException">環境変数の展開に失敗しました。</exception>
        public static string ExpandEnvironmentVariables(string str)
        {
            if (str == null) throw new ArgumentNullException(nameof(str));

            var expanded = Environment.ExpandEnvironmentVariables(str);

            // 文字列中の環境変数の展開に失敗
            if (expanded.Contains("%"))
            {
                throw new ArgumentException($"{str} に含まれる環境変数の展開に失敗しました。環境変数の設定を確認してください。", nameof(str));
            }

            return expanded;
        }

        /// <summary>
        /// パス文字列内の環境変数を展開し、絶対パスを返します。
        /// </summary>
        /// <param name="path">対象のパスです。</param>
        /// <returns>結果の絶対パスです。</returns>
        /// <exception cref="ArgumentNullException">path が null です。</exception>
        /// <exception cref="ArgumentException">
        /// <para>- path が長さが 0 の文字列であるか、空白しか含んでいません。または、パス文字列が長すぎます。</para>
        /// <para>- System.IO.Path.GetInvalidPathChars で定義されている無効な文字を 1 つ以上含んでいます。</para>
        /// <para>- 環境変数の展開に失敗しました。</para>
        /// <para>- 絶対パスを取得できませんでした。</para>
        /// </exception>
        public static string GetFullPath(string path)
        {
            if (path == null) throw new ArgumentNullException(nameof(path));

            try
            {
                return Path.GetFullPath(ExpandEnvironmentVariables(path)).TrimEnd(Slashes);
            }
            catch (Exception e)
            {
                throw new ArgumentException(e.Message, e);
            }
        }

        /// <summary>
        /// 比較用に正規化したパスを取得します。
        /// </summary>
        /// <param name="path">対象のパスです。</param>
        /// <returns>正規化後のパスを返します。</returns>
        /// <exception cref="ArgumentNullException">path が null です。</exception>
        /// <exception cref="ArgumentException">
        /// <para>- path が長さが 0 の文字列であるか、空白しか含んでいません。</para>
        /// <para>- System.IO.Path.GetInvalidPathChars で定義されている無効な文字を 1 つ以上含んでいます。</para>
        /// <para>- 環境変数の展開に失敗しました。</para>
        /// <para>- 絶対パスを取得できませんでした。</para>
        /// </exception>
        public static string NormalizePath(string path) => GetFullPath(path).ToLowerInvariant();

        /// <summary>
        /// パスとして等しいかどうかを比較します。
        /// パスのいずれかが無効である場合、false を返します。
        /// </summary>
        /// <param name="lhs">比較するパス1</param>
        /// <param name="rhs">比較するパス2</param>
        /// <returns>等しければ true。等しくない場合、またはパスとして無効な文字列である場合 false。</returns>
        public static bool EqualsPaths(string lhs, string rhs)
        {
            if (string.IsNullOrWhiteSpace(lhs) || string.IsNullOrWhiteSpace(rhs)) return false;

            try
            {
                return NormalizePath(lhs) == NormalizePath(rhs);
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// パス文字列から正規化した、'.' から始まる拡張子を取得します。
        /// 文字列が拡張子を含まない場合 null を返します。
        /// </summary>
        /// <param name="path">対象のパス文字列</param>
        /// <returns>正規化後の拡張子。または null。</returns>
        /// <exception cref="ArgumentException">System.IO.Path.GetInvalidPathChars で定義されている無効な文字を 1 つ以上含んでいます。</exception>
        public static string NormalizeExtension(string path)
        {
            var extension = Path.GetExtension(path);
            if (extension == string.Empty) return null;
            return extension?.ToLowerInvariant();
        }

        /// <summary>
        /// 二つのパスの拡張子が等しいかどうかを比較します。
        /// 拡張子は '.' から始まります。
        /// </summary>
        /// <param name="lhs">拡張子を比較する文字列1</param>
        /// <param name="rhs">拡張子を比較する文字列2</param>
        /// <returns>等しければ true。等しくない場合、またはパスとして無効な文字列である場合 false。</returns>
        public static bool EqualsExtensions(string lhs, string rhs)
        {
            if (string.IsNullOrWhiteSpace(lhs) || string.IsNullOrWhiteSpace(rhs)) return false;

            try
            {
                return NormalizeExtension(lhs) == NormalizeExtension(rhs);
            }
            catch
            {
                return false;
            }
        }
    }
}
