﻿using System;
using System.IO;

namespace SdkEnvironmentCheckerLibrary
{
    /// <summary>
    /// 比較可能な単一のファイルを示すファイルパスです。
    /// </summary>
    public class FilePath : IEquatable<FilePath>, IEquatable<string>
    {
        /// <summary>
        /// 空のファイルパスを示します。
        /// </summary>
        public static readonly FilePath Empty = new FilePath();

        /// <summary>
        /// ファイルパスから <see cref="FilePath"/> クラスのインスタンスの作成を試みます。
        /// ファイルパスが null, 空, 空白文字列である場合、FilePath.Empty を返します。
        /// </summary>
        /// <param name="filepath">ファイルパス</param>
        /// <param name="instance"><see cref="FilePath"/> インスタンス</param>
        /// <returns>インスタンス作成に成功すれば true</returns>
        public static bool TryCreate(string filepath, out FilePath instance)
        {
            try
            {
                instance = Create(filepath);
                return true;
            }
            catch
            {
                instance = null;
                return false;
            }
        }

        /// <summary>
        /// ファイルパスから <see cref="FilePath"/> クラスのインスタンスを作成します。
        /// ファイルパスが null, 空, 空白文字列である場合、FilePath.Empty を返します。
        /// </summary>
        /// <param name="filepath">ファイルパス</param>
        /// <returns><see cref="FilePath"/> インスタンス</returns>
        /// <exception cref="System.ArgumentException">
        /// <para>- filepath にパスとして不正な文字が含まれています。</para>
        /// <para>- 環境変数が指定されていますが、展開に失敗しました。</para>
        /// </exception>
        public static FilePath Create(string filepath) => string.IsNullOrWhiteSpace(filepath) ? Empty : new FilePath(filepath);

        // for Empty
        private FilePath() { }

        /// <summary>
        /// 指定されたファイルパスで FilePath の新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="filepath">単一のファイルを示すパスです。</param>
        /// <exception cref="System.ArgumentException">
        /// <para>- filepath が null, 空, または空白文字列です。</para>
        /// <para>- filepath にパスとして不正な文字が含まれています。</para>
        /// <para>- 環境変数が指定されていますが、展開に失敗しました。</para>
        /// </exception>
        /// <remarks>
        /// ファイルパスは絶対パスで管理されるため、相対パスを入力する際はカレントディレクトリに注意してください。
        /// </remarks>
        public FilePath(string filepath)
        {
            if (string.IsNullOrWhiteSpace(filepath)) throw new ArgumentException(nameof(filepath));
            FullPath = PathUtility.GetFullPath(filepath);
            NormalizedPath = PathUtility.NormalizePath(filepath);

            Directory = DirectoryPath.Create(Path.GetDirectoryName(FullPath));
        }

        /// <summary>
        /// 環境変数を展開したフルパスを取得します。
        /// </summary>
        public string FullPath { get; } = string.Empty;

        /// <summary>
        /// 比較用に正規化されたパスを取得します。
        /// </summary>
        public string NormalizedPath { get; } = string.Empty;

        /// <summary>
        /// ディレクトリパスを取得します。
        /// 空のファイルパスである場合、DirectoryPath.Empty を返します。
        /// </summary>
        public DirectoryPath Directory { get; } = DirectoryPath.Empty;

        /// <summary>
        /// ファイルが存在するかどうかを取得します。
        /// </summary>
        public bool Exists => File.Exists(NormalizedPath);

        /// <summary>
        /// 拡張子込みのファイル名を取得します。
        /// </summary>
        public string FileName => Path.GetFileName(FullPath);

        /// <summary>
        /// 拡張子を除くファイル名を取得します。
        /// </summary>
        public string FileNameWithoutExtension => Path.GetFileNameWithoutExtension(FullPath);

        /// <summary>
        /// 拡張子を「.」つきで取得します。拡張子を持たない場合、string.Empty を返します。
        /// </summary>
        public string Extension => Path.GetExtension(NormalizedPath);

        /// <summary>
        /// ファイルを読み取り可能どうかを返します。ファイルが存在しない場合、false を返します。
        /// </summary>
        public bool CanRead
        {
            get
            {
                try
                {
                    return Exists &&
                        (File.GetAttributes(NormalizedPath) &
                        (FileAttributes.Directory | FileAttributes.Offline)) == 0;
                }
                catch
                {
                    return false;
                }
            }
        }

        /// <summary>
        /// ファイルを作成、もしくは書き込み可能かどうかを返します。
        /// ファイルが存在しない場合、true を返します。
        /// </summary>
        public bool CanWrite
        {
            get
            {
                if (this == Empty) return false;

                try
                {
                    return !Exists || !File.GetAttributes(NormalizedPath).HasFlag(FileAttributes.ReadOnly | FileAttributes.Directory | FileAttributes.Offline);
                }
                catch
                {
                    return false;
                }
            }
        }


        /// <summary>
        /// ファイルが存在し、かつ読み取り専用属性を持つかどうかを返します。
        /// CanWrite との違いに注意してください。
        /// </summary>
        /// <returns>ファイルが存在し、読み取り専用であれば true。そうでない場合 false。</returns>
        public bool IsReadOnly
        {
            get
            {
                if (this == Empty) return false;

                try
                {
                    return Exists && File.GetAttributes(NormalizedPath).HasFlag(FileAttributes.ReadOnly);
                }
                catch
                {
                    return false;
                }
            }
        }

        /// <summary>
        /// 空のファイルパスかどうかを返します。
        /// </summary>
        public bool IsEmpty => Equals(Empty);

        /// <summary>
        /// ファイル属性を読み取り専用に設定します。
        /// </summary>
        /// <returns>読み取り専用の設定に成功すれば true。ファイルが存在しないなど、設定に失敗した場合 false。</returns>
        public bool SetReadOnly()
        {
            if (!Exists) return false;

            try
            {
                var attr = File.GetAttributes(NormalizedPath);
                attr = attr & FileAttributes.ReadOnly;
                File.SetAttributes(NormalizedPath, attr);
                return true;
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// ファイル属性から読み取り専用フラグを削除します。
        /// </summary>
        /// <returns>読み取り専用フラグの削除に成功すれば true。ファイルが存在しないなど、設定に失敗した場合 false。</returns>
        public bool ResetReadOnly()
        {
            if (!Exists) return false;

            try
            {
                var attr = File.GetAttributes(NormalizedPath);
                attr = attr & ~FileAttributes.ReadOnly;
                File.SetAttributes(NormalizedPath, attr);
                return true;
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// フルパスを取得します。
        /// </summary>
        /// <returns>フルパスを取得します。</returns>
        public override string ToString() => FullPath;

        /// <summary>
        /// 自身のパスから見た、対象のファイルパスへの相対パスを取得します。
        /// 自身が空のファイルパスである場合、対象のファイルパスを返します。
        /// </summary>
        /// <param name="target">対象のファイルパス</param>
        /// <returns>自身から見た対象ファイルパスへの相対パス</returns>
        /// <exception cref="ArgumentNullException">target が null です。</exception>
        /// <exception cref="ArgumentException">target が空のファイルパスです。</exception>
        public string CreateRelativePath(FilePath target)
        {
            if (target == null) throw new ArgumentNullException(nameof(target));
            if (target == Empty) throw new ArgumentException(nameof(target));

            if (IsEmpty) return target.FullPath;

            return new Uri(FullPath)
                .MakeRelativeUri(new Uri(target.FullPath))
                .ToString()
                .Replace(Path.AltDirectorySeparatorChar, Path.DirectorySeparatorChar);
        }

        /// <summary>
        /// 自身のパスから見た、対象のディレクトリへの相対パスを取得します。
        /// 自身が空のファイルパスである場合、対象のディレクトリパスを返します。
        /// </summary>
        /// <param name="target">対象のファイルパス</param>
        /// <returns>自身から見た対象ファイルパスへの相対パス</returns>
        /// <exception cref="ArgumentNullException">target が null です。</exception>
        /// <exception cref="ArgumentException">target が空のディレクトリパスです。</exception>
        public string CreateRelativePath(DirectoryPath target)
        {
            if (target == null) throw new ArgumentNullException(nameof(target));
            if (target == DirectoryPath.Empty) throw new ArgumentException(nameof(target));

            if (IsEmpty) return target.FullPath;

            return new Uri(FullPath)
                .MakeRelativeUri(new Uri(target.FullPath))
                .ToString()
                .Replace(Path.AltDirectorySeparatorChar, Path.DirectorySeparatorChar);
        }

        /// <summary>
        /// 自身のパスと相対パスから、対象のファイルパスを取得します。
        /// </summary>
        /// <param name="relativePath">相対パス</param>
        /// <returns>対象のファイルパス</returns>
        /// <exception cref="ArgumentException">
        /// <para>- relativePath が null か、空文字列か、空白文字列です。</para>
        /// <para>- relativePath に不正な文字列が含まれています。</para>
        /// <para>- relativePath 内の環境変数の展開に失敗しました。</para>
        /// </exception>
        public FilePath CreateFilePathWithRelative(string relativePath)
        {
            if (string.IsNullOrWhiteSpace(relativePath))
            {
                throw new ArgumentException($"{nameof(relativePath)} に null、空文字列、空白文字列を指定することはできません。", nameof(relativePath));
            }

            try
            {
                return new FilePath(Path.Combine(Directory.FullPath, relativePath.TrimStart(PathUtility.Slashes)));
            }
            catch (Exception e)
            {
                throw new ArgumentException(e.Message, e);
            }
        }

        #region equality

        /// <summary>
        /// FilePath を比較します。
        /// </summary>
        /// <param name="lhs">比較するパス</param>
        /// <param name="rhs">比較するパス</param>
        /// <returns>パスとして同一であれば true</returns>
        public static bool operator ==(FilePath lhs, FilePath rhs) => ReferenceEquals(null, lhs) ? ReferenceEquals(null, rhs) : lhs.Equals(rhs);

        /// <summary>
        /// FilePath を比較します。
        /// </summary>
        /// <param name="lhs">比較するパス</param>
        /// <param name="rhs">比較するパス</param>
        /// <returns>パスとして異なっていれば true</returns>
        public static bool operator !=(FilePath lhs, FilePath rhs) => !(lhs == rhs);

        /// <summary>
        /// FilePath を比較します。
        /// </summary>
        /// <param name="other">比較するパス</param>
        /// <returns>パスとして等しいかどうか</returns>
        public bool Equals(FilePath other) => !ReferenceEquals(null, other) && NormalizedPath == other.NormalizedPath;

        /// <summary>
        /// 入力文字列がパスとして等しいかどうかを比較します。
        /// </summary>
        /// <param name="path">比較するファイルパス</param>
        /// <returns>パスとして等しいかどうか</returns>
        public bool Equals(string path)
        {
            if (path == null) return false;
            if (string.IsNullOrWhiteSpace(path)) return IsEmpty;

            try
            {
                return NormalizedPath == PathUtility.NormalizePath(path);
            }
            catch
            {
                return false;
            }
        }

        public override bool Equals(object obj)
        {
            if (obj is FilePath filepath) return Equals(filepath);
            if (obj is string stringPath) return Equals(stringPath);
            return false;
        }

        public override int GetHashCode() => NormalizedPath.GetHashCode();

        #endregion equality

        #region string operators

        /// <summary>
        /// FilePath を文字列に変換します。
        /// </summary>
        /// <param name="path">パスのインスタンス</param>
        public static implicit operator string(FilePath path) => path?.FullPath;

        /// <summary>
        /// FilePath と文字列がパスとして等しいかどうかを返します。
        /// </summary>
        /// <param name="lhs">比較するパス</param>
        /// <param name="rhs">比較するパス文字列</param>
        /// <returns>パスとして等しいかどうか</returns>
        public static bool operator ==(FilePath lhs, string rhs) => ReferenceEquals(null, lhs) ? rhs == null : lhs.Equals(rhs);

        /// <summary>
        /// FilePath と文字列がパスとして異なるかどうかを返します。
        /// </summary>
        /// <param name="lhs">比較するパス</param>
        /// <param name="rhs">比較するパス文字列</param>
        /// <returns>パスとして異なるかどうか</returns>
        public static bool operator !=(FilePath lhs, string rhs) => !(lhs == rhs);

        /// <summary>
        /// FilePath と文字列がパスとして等しいかどうかを返します。
        /// </summary>
        /// <param name="lhs">比較するパス文字列</param>
        /// <param name="rhs">比較するパス</param>
        /// <returns>パスとして等しいかどうか</returns>
        public static bool operator ==(string lhs, FilePath rhs) => rhs == lhs;

        /// <summary>
        /// FilePath と文字列がパスとして異なるかどうかを返します。
        /// </summary>
        /// <param name="lhs">比較するパス文字列</param>
        /// <param name="rhs">比較するパス</param>
        /// <returns>パスとして異なるかどうか</returns>
        public static bool operator !=(string lhs, FilePath rhs) => rhs != lhs;

        #endregion string operators
    }
}
