﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Windows;
using Nintendo.ToolFoundation.CommandLine;
using SdkEnvironmentCheckerLibrary;
using SdkEnvironmentCheckerLibrary.Serializers;

namespace SdkEnvironmentChecker
{
    /// <summary>
    /// App.xaml の相互作用ロジック
    /// </summary>
    public partial class App : Application
    {
        private const string DefaultConfigPath = @"SdkEnvironmentChecker.config";

        protected override void OnStartup(StartupEventArgs e)
        {
            var assemblyDir = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);

            Debug.Assert(assemblyDir != null);

            IReadOnlyCollection<CheckerResult> results;

            // parse arguments
            var parseResult = ParseArgs(e.Args, out var parsedConfig, out var parsedExcludeNvn);

            if (parseResult == ParseResultType.Success)
            {
                var configPath = Path.Combine(assemblyDir, DefaultConfigPath);

                if (!string.IsNullOrEmpty(parsedConfig)) configPath = parsedConfig;
                var excludeNvn = parsedExcludeNvn;

                try
                {
                    var config = SdkEnvironmentCheckerConfig.Load(configPath);
                    if (excludeNvn) config.NvnGraphicsDriver = false;
                    results = SdkEnvironmentCheckerLibrary.SdkEnvironmentChecker.Check(config.ToCheckerInfo()).ToArray();
                }
                catch (Exception exception)
                {
                    results = new[] { CheckerResult.ConfirmFailed("SdkEnvironmentChecker", exception.Message) };
                }
            }
            else if (parseResult == ParseResultType.Failure)
            {
                results = new[] { CheckerResult.ConfirmFailed("SdkEnvironmentChecker", "failed to parse options, see --help.") };
            }
            else
            {
                results = new[] { CheckerResult.ConfirmFailed("SdkEnvironmentChecker", "--help option is specified.") };
            }

            new MainWindow { DataContext = results }.Show();

            base.OnStartup(e);
        }

        private static ParseResultType ParseArgs(IEnumerable<string> args, out string configPath, out bool excludeNvn)
        {
            var result = CommandLine.ParseArgs<CommandLineParam>(args, out var param);
            if (result == ParseResultType.Success)
            {
                configPath = param.ConfigPath;
                excludeNvn = param.Exclude == "nvn";
            }
            else
            {
                configPath = null;
                excludeNvn = false;
            }

            return result;
        }

        private class CommandLineParam
        {
            [CommandLineValueOption("config", IsHidden = true)]
            public string ConfigPath { get; set; }

            [CommandLineValueOption("exclude", Description = nameof(Exclude), DescriptionConverterName = nameof(DescriptionConverter))]
            public string Exclude { get; set; }

            public static string DescriptionConverter(string description)
            {
                if (description == nameof(Exclude)) return SdkEnvironmentChecker.Properties.Resources.CommandLineExcludeDescription;
                return string.Empty;
            }
        }
    }
}
