﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Diagnostics;
using System.Collections.Generic;

namespace QCIT_Maker
{
    internal class Executor
    {
        private const string AuthoringToolDirPath = @"Tools\CommandLineTools\AuthoringTool";
        private bool isVerbose = false;
        public void SetVerboseMode()
        {
            isVerbose = true;
        }

        /// <summary>
        /// 実行結果の情報を格納するクラス定義
        /// </summary>
        internal class ExecResult
        {
            public int ExitCode { get; set; }
            public string ErrorMessage { get; set; }

            public bool IsSuccess()
            {
                return (ExitCode == 0) ? true : false;
            }

            public bool IsFailure()
            {
                return !IsSuccess();
            }
        }

        /// <summary>
        /// AuthoringTool.exe を creatensp オプションで実行するメソッドです。(詳細形式版)
        /// </summary>
        /// <param name="outFilePath">出力nspファイルのパス</param>
        /// <param name="descFilePath">入力descファイルのパス</param>
        /// <param name="codeDirPath">入力codeフォルダのパス</param>
        /// <param name="dataDirPath">入力dataフォルダのパス</param>
        /// <param name="inSetParam">設定パラメータ情報</param>
        /// <returns>実行結果</returns>
        public ExecResult ExecuteAuthoringToolCreateNsp(
            string outFilePath, string descFilePath, string codeDirPath, string dataDirPath, SettingParams inSetParam)
        {
            // AuthoringToolに渡すオプション引数文字列を生成
            // 3PT に登録して量産化できる必要があるので keyindex に 0 を指定する
            var args = string.Format("creatensp -o \"{0}\" --meta \"{1}\" --desc \"{2}\" --type Application --program \"{3}\" \"{4}\" --keyindex 0",
                                outFilePath, inSetParam.NmetaFilePath, descFilePath, codeDirPath, dataDirPath);
            return InternalExecuteAuthoringTool(args);
        }

        /// <summary>
        /// AuthoringTool.exe を extractnsp オプションで実行するメソッドです。(詳細形式版)
        /// </summary>
        /// <param name="outFilePath">出力nspファイルのパス</param>
        /// <param name="descFilePath">入力descファイルのパス</param>
        /// <returns>実行結果</returns>
        public ExecResult ExecuteAuthoringToolExtractNsp(string outFilePath, string extractNspPath)
        {
            // AuthoringToolに渡すオプション引数文字列を生成
            var args = string.Format("extract -o \"{0}\" --targetregex .*\\.xml$ \"{1}\"", outFilePath, extractNspPath);
            return InternalExecuteAuthoringTool(args);
        }

        private ExecResult InternalExecuteAuthoringTool(string args)
        {
            var result = new ExecResult();
            using (var process = new Process())
            {
                var authoringToolDirPath = CommonParams.GetSigloRoot() + AuthoringToolDirPath;
                process.StartInfo.FileName = Path.Combine(authoringToolDirPath, "AuthoringTool.exe");
                process.StartInfo.Arguments = args;

                if (isVerbose == true)
                {
                    Console.WriteLine("Execute AuthoringTool ...");
                    Console.WriteLine("  arg : {0}", process.StartInfo.Arguments);
                }

                process.StartInfo.CreateNoWindow = true;
                process.StartInfo.UseShellExecute = false;
                process.StartInfo.RedirectStandardError = true;
                //process.StartInfo.StandardErrorEncoding = System.Text.Encoding.UTF8;
                process.Start();

                process.WaitForExit();
                result.ExitCode = process.ExitCode;
                result.ErrorMessage = process.StandardError.ReadToEnd();
            }

            return result;
        }

        /// <summary>
        /// MakeMeta.exe を実行するメソッドです。
        /// </summary>
        /// <param name="outNpdmFilePath">出力npdmファイルのパス</param>
        /// <param name="inDescFilePath">入力descファイルのパス</param>
        /// <param name="inMetaFilePath">入力metaファイルのパス</param>
        /// <returns>実行結果</returns>
        public ExecResult ExecuteMakeMeta(string outNpdmFilePath, string inDescFilePath, string inMetaFilePath)
        {
            var result = new ExecResult();
            using (var process = new Process())
            {
                process.StartInfo.FileName = CommonParams.GetSigloRoot() + @"Tools\CommandLineTools\MakeMeta\MakeMeta.exe";
                var args = string.Format("-o \"{0}\" --desc \"{1}\" --meta \"{2}\"", outNpdmFilePath, inDescFilePath, inMetaFilePath);
                process.StartInfo.Arguments = args;

                if (isVerbose == true)
                {
                    Console.WriteLine(" ExecuteMakeMeta arg : {0}", process.StartInfo.Arguments);
                }

                process.StartInfo.CreateNoWindow = true;
                process.StartInfo.UseShellExecute = false;
                process.StartInfo.RedirectStandardError = true;
                process.StartInfo.StandardErrorEncoding = System.Text.Encoding.UTF8;
                process.Start();

                process.WaitForExit();
                result.ExitCode = process.ExitCode;
                result.ErrorMessage = process.StandardError.ReadToEnd();
            }
            return result;
        }

        /// <summary>
        /// AuthoringTool.exe を makepatch オプションで実行するメソッドです。
        /// </summary>
        /// <param name="outFilePath">出力nspファイルのパス</param>
        /// <param name="descFilePath">入力descファイルのパス</param>
        /// <param name="currentNspFilePath">入力currentNspファイルのパス</param>
        /// <param name="inSetParam">設定パラメータ情報</param>
        /// <returns>実行結果</returns>
        public ExecResult ExecuteAuthoringToolMakePatch(string outFilePath, string descFilePath, string currentNspFilePath, SettingParams inSetParam)
        {
            // OriginalNspFilePath は設定されているはず
            // スレッドセーフ化のため作業ディレクトリにコピーする
            var dstOriginalNspFile = Path.Combine(inSetParam.ChildWorkDirPath, "__original__.nsp");
            File.Copy(inSetParam.OriginalNspFilePath, dstOriginalNspFile);

            // AuthoringToolに渡すオプション引数文字列を生成
            var args = string.Format("makepatch -o \"{0}\" --desc \"{1}\" --original \"{2}\" --current \"{3}\"",
                                        outFilePath, descFilePath, dstOriginalNspFile, currentNspFilePath);

            if (!string.IsNullOrEmpty(inSetParam.PreviousPatchFilePath))
            {
                // スレッドセーフ化のため作業ディレクトリにコピーする
                var dstPreviousNspFile = Path.Combine(inSetParam.ChildWorkDirPath, "__previous_patch__.nsp");
                File.Copy(inSetParam.PreviousPatchFilePath, dstPreviousNspFile);
                // previous のファイルパスを指定
                args += string.Format(" --previous \"{0}\"", dstPreviousNspFile);
            }

            return InternalExecuteAuthoringTool(args);
        }
    }
}
