﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
using System.IO;
using System.Globalization;
using System.Security.Cryptography;
using System.Text.RegularExpressions;

namespace MakeRepairInitialImage
{
    public class MakeRepairInitialImage
    {
        [CommandLineOption("image",
            Description = "input InitialImage file.",
            IsRequired = true)]
        public string Input { get; set; }

        [CommandLineOption("output",
            Description = "output RepairInitialImage file. If omit, Create in input image directory.",
            IsRequired = false)]
        public string Output { get; set; }

        [CommandLineOption("hashFile",
            Description = "output hash file. If omit, Use ${NINTENDO_SDK_ROOT}\\Programs\\Chris\\Sources\\TargetTools\\SystemInitializerRepair\\SystemInitializer_InitialImageHash.cpp",
            IsRequired = false)]
        public string HashFile { get; set; }

        [CommandLineOption("ignore-sizecheck",
            Description = "Ingnore input file size check.",
            DefaultValue = false,
            IsRequired = false)]
        public bool IgnoreSizeCheck { get; set; }

        public const Int64 RepairInitialImageFormatVersion = 1;
        public const Int64 RepairInitialImageHeaderSize = 8 + 8;
        public const Int64 SafeModePartitionSize = 0x04000000;
        public const Int64 InitialImageMaxSize = SafeModePartitionSize - RepairInitialImageHeaderSize;

        public int Run()
        {
            int returnValue = 0;

            // パラメータのチェック
            returnValue = ValidateParameter();
            if (returnValue != 0) return returnValue;

            // 初期化イメージがセーフモード領域に収まるかチェック
            if (!IgnoreSizeCheck)
            {
                returnValue = CheckInitImageSize(this.Input);
                if (returnValue != 0) return returnValue;
            }
            else
            {
                Console.WriteLine("[WARNING] Skip InitialImage file size check.");
            }

            // 修理用初期化イメージの作成
            CreateRepairInitImage(this.Output, this.Input);

            // SHA256 の計算
            var sha256DataBytes = ComputeSha256Hash(this.Output);

            // ソースコード内のハッシュ値を更新
            returnValue = UpdateHashFile(sha256DataBytes, this.HashFile);
            if (returnValue != 0) return returnValue;

            return 0;
        }

        // ファイルのハッシュ値を計算して返す
        private static Byte[] ComputeSha256Hash(string filePath)
        {
            var sha256managed = new SHA256Managed();
            var initImageBytes = File.ReadAllBytes(filePath);

            return sha256managed.ComputeHash(initImageBytes);
        }

        // 初期化イメージのサイズチェックを行う
        private static int CheckInitImageSize(string initImagePath)
        {
            var fileSize = new FileInfo(initImagePath).Length;
            if (fileSize > InitialImageMaxSize)
            {
                Console.Error.WriteLine($"[ERROR] InitialImage is too large. (current: {fileSize} byte, max: {InitialImageMaxSize} byte)");
                return 1;
            }
            return 0;
        }

        // 修理用初期化イメージを作成する
        private static void CreateRepairInitImage(string outFilePath, string inFilePath)
        {
            var fileSize = new FileInfo(inFilePath).Length + RepairInitialImageHeaderSize;
            var formatVersionBytes = BitConverter.GetBytes(RepairInitialImageFormatVersion);
            var fileSizeBytes = BitConverter.GetBytes(fileSize);

            using (var inFs = new FileStream(inFilePath, FileMode.Open, FileAccess.Read))
            using (var outFs = new FileStream(outFilePath, FileMode.Create, FileAccess.Write))
            {
                outFs.Write(formatVersionBytes, 0, formatVersionBytes.Length);
                outFs.Write(fileSizeBytes, 0, fileSizeBytes.Length);
                inFs.CopyTo(outFs);
            }
        }

        // ソースコードに書かれたハッシュ値を更新する
        private static int UpdateHashFile(Byte[] newHashData, string hashFile)
        {
            var hashFileText = File.ReadAllText(hashFile);
            var newHashDataText = "\n";

            for (int i=0; i< newHashData.Length; i++)
            {
                if (i % 16 == 0) newHashDataText += "    ";
                newHashDataText += String.Format("0x{0:X2},", newHashData[i]);
                if (i%16 == 15 || i+1 == newHashData.Length) newHashDataText += "\n";
            }

            var regex = new Regex(@"(InitialImageHash\[.*?\](?:\s|\n)*=(?:\s|\n)*{)(?:\s|\n)*(?:.|\n)+?(};)");
            if(!regex.IsMatch(hashFileText))
            {
                Console.Error.WriteLine("[ERROR] Can't replace hashfile");
                Console.Error.WriteLine("Please confirm HashFile format or replace manually to following hash value.");
                Console.Error.WriteLine($"{newHashDataText}");
                return 1;
            }

            hashFileText = regex.Replace(hashFileText, String.Format($"$1{newHashDataText}$2"));
            File.WriteAllText(hashFile, hashFileText);
            return 0;
        }

        private int ValidateParameter()
        {
            this.Input = Path.GetFullPath(this.Input);

            // output オプションが省略された場合は、input ファイルと同じディレクトリに生成する
            if (String.IsNullOrEmpty(this.Output))
            {
                this.Output = this.Input;
            }
            else
            {
                this.Output = Path.GetFullPath(this.Output);
            }
            this.Output = Path.ChangeExtension(this.Output, "repair.initimg");

            // hashFile オプションが省略された場合は以下のパスを使用する
            // ${NINTENDO_SDK_ROOT}\\Programs\\Chris\\Sources\\TargetTools\\SystemInitializerRepair\\SystemInitializer_InitialImageHash.cpp
            if (String.IsNullOrEmpty(this.HashFile))
            {
                var nintendoSdkRoot = Environment.GetEnvironmentVariable("NINTENDO_SDK_ROOT");

                if (String.IsNullOrEmpty(nintendoSdkRoot))
                {
                    Console.Error.WriteLine("[ERROR] NINTENDO_SDK_ROOT not defined.");
                    Console.Error.WriteLine("Please define NINTENDO_SDK_ROOT or specify --hashFile option");
                    return 1;
                }
                this.HashFile = Path.Combine(nintendoSdkRoot, "Programs", "Chris", "Sources", "TargetTools", "SystemInitializerRepair", "SystemInitializer_InitialImageHash.cpp");
            }
            else
            {
                this.HashFile = Path.GetFullPath(this.HashFile);
            }

            this.Input    = PathConverter.Windowsify(this.Input);
            this.Output   = PathConverter.Windowsify(this.Output);
            this.HashFile = PathConverter.Windowsify(this.HashFile);

            // パラメータの表示
            Console.WriteLine($"Input    : {this.Input}");
            Console.WriteLine($"Output   : {this.Output}");
            Console.WriteLine($"HashFile : {this.HashFile}");

            // ファイルの存在確認
            if (!File.Exists(this.Input))
            {
                Console.Error.WriteLine($"[ERROR] Input file not found. {this.Input}");
                return 1;
            }
            if (!File.Exists(this.HashFile))
            {
                Console.Error.WriteLine($"[ERROR] Hash file not found. {this.HashFile}");
                return 1;
            }

            return 0;
        }
    }

    public static class PathConverter
    {
        internal static string CYGWIN_PATH = Environment.GetEnvironmentVariable("CYGWIN_PATH");

        /// <summary>
        /// Cygwin形式のパスをWindowsのパスに変換します。既にWindowsのパスである場合は変換を行いません。
        /// </summary>
        /// <param name="input">Cygwin形式のパス</param>
        /// <returns>Windows形式に変換したパス</returns>
        internal static string Windowsify(string input)
        {
            if (string.IsNullOrEmpty(input))
            {
                return input;
            }

            int cygdriverIndex = -1;

            cygdriverIndex = input.IndexOf(@"/cygdrive///");

            if ((cygdriverIndex == 0 || cygdriverIndex == 1) &&
                 !string.IsNullOrEmpty(CYGWIN_PATH))
            {
                // Special treatment
                input = input.Replace(@"/cygdrive///", Path.Combine(CYGWIN_PATH, "cygdrive/"));
                // it should continue with normal conversion to fix the \\
            }

            // Normal conversion -> path starts with /cygdriver/<stuff>  or "/cygdriver/<stuff>"
            cygdriverIndex = input.IndexOf("/cygdrive/");
            if (cygdriverIndex == 0 || cygdriverIndex == 1)
            {
                char drive_letter = input.ToCharArray()[10 + cygdriverIndex];
                input = input.Remove(0, 11 + cygdriverIndex);
                input = input.Replace('/', '\\');
                input = drive_letter + ":" + input;
            }
            else
            {
                // This should be a path that doesn't start with cygdrive,
                // so just make sure it has the proper \\ instead of /
                input = input.Replace('/', '\\');
            }
            return input;
        }
    }

    class Program
    {
        static void Main(string[] args)
        {
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);

#if !DEBUG
            try
            {
#endif
            MakeRepairInitialImage parsed;
            var parser = new Nintendo.Foundation.IO.CommandLineParser();

            if (false == parser.ParseArgs<MakeRepairInitialImage>(args, out parsed))
            {
                System.Environment.Exit(1);
            }

            System.Environment.Exit(parsed.Run());
#if !DEBUG
            }
            catch (Exception exception)
            {
                PrintException(exception);
                System.Environment.Exit(1);
            }
#endif
        }

        public static void PrintException(Exception exception)
        {
            Console.Error.WriteLine("[ERROR] {0}", exception.Message);
            Console.Error.WriteLine(string.Format("StackTrace: {0}", exception.StackTrace));
        }
    }
}
