﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using System.Windows.Forms;
using System.Diagnostics;
using CommandUtility;
using System.IO;
using Nintendo.ManuHostTools.UsbLibrary;

namespace BackupRestoreGui
{
    public enum ToolMode
    {
        Backup,
        Restore
    }

    public partial class MainForm : Form
    {
        private ToolMode Mode { get; set; }
        private string LogFileName { get; set; }
        private Process BackupRestoreProcess { get; set; }
        private Process FrcmProcess { get; set; }
        private Thread WaitProcessThread { get; set; }
        private bool IsDevelop { get; set; }
        private bool IsForceInstall { get; set; }
        private bool IsSigned { get; set; }
        private UsbDeviceLocker FrcmLocker { get; set; }
        private UsbDeviceLocker ManuLocker { get; set; }
        private string TargetSerialNumber { get; set; }

        readonly Guid RecoveryModeDeviceGuid = Guid.Parse("EAD8C4F6-6102-45C7-AA66-36E6D7204600");
        readonly Guid ManuModeDeviceGuid = Guid.Parse("97FFFD48-2D1D-47A0-85A3-07FDE6FA0143");
        const string RecoveryModeDevicePathPattern = @"usb#vid_0955~pid_7321#(?<iSerialNumber>.*)#";
        const string ManuModeDevicePathPattern = @"usb#vid_057e~pid_3000#(?<iSerialNumber>.*)#";

        public MainForm(ToolMode mode, bool isDevelop, bool isForceInstall, bool isSigned)
        {
            IsDevelop = isDevelop;
            IsForceInstall = isForceInstall;
            IsSigned = isSigned;

            InitializeComponent();

            UpdateMode(mode);
        }

        private void UpdateMode(ToolMode mode)
        {
            this.Mode = mode;

            switch (this.Mode)
            {
                case ToolMode.Backup:
                    MenuItemModeBackup.Checked = true;
                    MenuItemModeRestore.Checked = false;
                    string modeText = "Backup" + (IsForceInstall ? " Forcibly" : String.Empty) + (IsSigned ? " Signed" : String.Empty);
                    StartButton.Text = "Start Backup";
                    this.Text = $"Backup Restore Tool (Mode: {modeText})";
                    break;
                case ToolMode.Restore:
                    MenuItemModeBackup.Checked = false;
                    MenuItemModeRestore.Checked = true;
                    StartButton.Text = "Start Restore";
                    this.Text = "Backup Restore Tool (Mode: Restore)";
                    break;
                default:
                    throw new Exception("unknown mode");
            }
        }

        private void openLogDirectoryToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Program.EnsureExistingLogDirectoryPath();
            Process.Start(Program.GetLogDirecotryPath());
        }

        private void saveCurrentLogToolStripMenuItem_Click(object sender, EventArgs e)
        {
            var dialog = new SaveFileDialog();

            dialog.Filter = "txt files (*.txt)|*.txt|All files (*.*)|*.*";
            dialog.FilterIndex = 0;
            dialog.RestoreDirectory = true;

            if (dialog.ShowDialog() == DialogResult.OK)
            {
                using (var writer = File.CreateText(dialog.FileName))
                {
                    writer.Write(LogTextBox.Text);
                }
            }
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            PrepareToExit();
            System.Environment.Exit(0);
        }

        private void MenuItemModeBackup_Click(object sender, EventArgs e)
        {
            UpdateMode(ToolMode.Backup);
        }

        private void MenuItemModeRestore_Click(object sender, EventArgs e)
        {
            UpdateMode(ToolMode.Restore);
        }

        string CreateFileName()
        {
            var time = DateTime.Now;
            return Path.Combine(Program.GetLogDirecotryPath(), string.Format("log-{0}-{1}.txt", time.ToString("yyyyMMdd-HHmmss"), Mode.ToString()));
        }

        private bool IsFrcmOrBackupRestoreInProgress()
        {
            return BackupRestoreProcess != null || FrcmProcess != null;
        }

        private void TerminateFrcmOrBackupRestoreProcess()
        {
            ProcessUtility.TerminateProcess(FrcmProcess);
            FrcmProcess = null;
            ProcessUtility.TerminateProcess(BackupRestoreProcess);
            BackupRestoreProcess = null;
        }

        private void StartButton_Click(object sender, EventArgs e)
        {
            if (IsFrcmOrBackupRestoreInProgress())
            {
                var result = MessageBox.Show("Stop the current process?", "Warning", MessageBoxButtons.OKCancel, MessageBoxIcon.Error);
                if (result == DialogResult.OK)
                {
                    TerminateCurrentBackupRestoreProcess();
                }
                else
                {
                    return;
                }
            }

            Program.EnsureExistingLogDirectoryPath();
            ClearLog();
            LogFileName = CreateFileName();

            if (IsDetectedRecoveryModeDevice())
            {
                AppendLog("Found the recovery mode device." + System.Environment.NewLine);
            }
            else
            {
                AppendLog("[ERROR] Can't find the recovery mode device." + System.Environment.NewLine);
                return;
            }

            var frcmScriptPath = Script.GetFrcmPath(IsDevelop, IsForceInstall, IsSigned);
            var recoverySerialNumber = GetFirstRecoveryModeSerialNumberWithLock();
            if (recoverySerialNumber == null)
            {
                AppendLog("[ERROR] Can't find the recovery mode device." + System.Environment.NewLine);
                return;
            }
            var devicePathOption = @"-DevicePath " + recoverySerialNumber;
            FrcmProcess = ProcessUtility.StartScript(frcmScriptPath, SdkPath.FindSdkRoot(), OutputDataReceived, ErrorDataReceived, devicePathOption);
            WaitProcessThread = new Thread(() =>
            {
                var exitCode = ProcessUtility.WaitProcess(FrcmProcess);
                FrcmProcess = null;
                FrcmLocker.Unlock();
                ShowProcessResult(exitCode, showResultOnlyIfFailed: true, hasTargetLog: false);
                if (exitCode == 0)
                {
                    var backupRestoreScriptPath = Script.GetPath(this.Mode, IsDevelop, IsSigned);
                    TargetSerialNumber = GetFirstManuModeSerialNumberWithLock();
                    SetSerialNumber(TargetSerialNumber.ToUpper());
                    var manuOption = @"-DevicePath " + TargetSerialNumber + " -DoProcessClean false";
                    BackupRestoreProcess = ProcessUtility.StartScript(backupRestoreScriptPath, SdkPath.FindSdkRoot(), OutputDataReceived, ErrorDataReceived, manuOption);
                    exitCode = ProcessUtility.WaitProcess(BackupRestoreProcess);
                    BackupRestoreProcess = null;
                    SetSerialNumber("");
                    ManuLocker.Unlock();
                    ShowProcessResult(exitCode, showResultOnlyIfFailed: false, hasTargetLog: true);
                }
            });

            WaitProcessThread.Start();
        }

        private void TerminateCurrentBackupRestoreProcess()
        {
            if (WaitProcessThread.IsAlive)
            {
                WaitProcessThread.Abort();
            }
            TerminateFrcmOrBackupRestoreProcess();
            FrcmLocker?.Unlock();
            ManuLocker?.Unlock();
            SetSerialNumber("");
        }

        private void ShowProcessResult(int exitCode, bool showResultOnlyIfFailed, bool hasTargetLog)
        {
            bool succeeded = hasTargetLog ? (exitCode == 0 && IsTargetCompletedSuccessfully()) : (exitCode == 0);
            if (succeeded)
            {
                if (showResultOnlyIfFailed)
                {
                    return;
                }
                AppendLog("============================================================" + System.Environment.NewLine);
                AppendLog("" + System.Environment.NewLine);
                AppendLog("  SUCCEEDED" + System.Environment.NewLine);
                AppendLog("" + System.Environment.NewLine);
                AppendLog("============================================================" + System.Environment.NewLine);
            }
            else
            {
                AppendLog("============================================================" + System.Environment.NewLine);
                AppendLog("" + System.Environment.NewLine);
                AppendLog("  FAILED" + System.Environment.NewLine);
                AppendLog("" + System.Environment.NewLine);
                AppendLog("    ExitStatus: " + exitCode.ToString() + System.Environment.NewLine);
                AppendLog("" + System.Environment.NewLine);
                AppendLog("============================================================" + System.Environment.NewLine);
            }
        }

        delegate void ClearLogCallback();

        private void ClearLog()
        {
            if (this.LogTextBox.InvokeRequired)
            {
                this.Invoke(new ClearLogCallback(ClearLog), new object[] { });
            }
            else
            {
                this.LogTextBox.Clear();
            }
        }

        private bool IsDetectedRecoveryModeDevice()
        {
            var devicePathList = UsbDevice.EnumerateDevicePathByGuid(RecoveryModeDeviceGuid);
            foreach (var i in devicePathList)
            {
                var serial = GetSerialNumberFromDevicePath(RecoveryModeDevicePathPattern, i);
                if (!UsbDeviceLocker.IsLocked(serial))
                {
                    return true;
                }
            }
            return false;
        }

        private string GetFirstRecoveryModeDevicePathWithLock()
        {
            var devicePathList = UsbDevice.EnumerateDevicePathByGuid(RecoveryModeDeviceGuid);
            foreach (var i in devicePathList)
            {
                var serial = GetSerialNumberFromDevicePath(RecoveryModeDevicePathPattern, i);
                if (!UsbDeviceLocker.IsLocked(serial))
                {
                    FrcmLocker = new UsbDeviceLocker();
                    if (FrcmLocker.Lock(serial))
                    {
                        return i;
                    }
                }
            }
            return null;
        }

        public delegate string GetDevicePathDelegate();
        private string GetConnectedUsbSerialNumber(string vidPidPattern, GetDevicePathDelegate GetDevicePath)
        {
            var devicePath = GetDevicePath();
            return GetSerialNumberFromDevicePath(vidPidPattern, devicePath);
        }

        private static string GetSerialNumberFromDevicePath(string vidPidPattern, string devicePath)
        {
            var r = new System.Text.RegularExpressions.Regex(
                vidPidPattern,
                System.Text.RegularExpressions.RegexOptions.IgnoreCase);

            var match = r.Match(devicePath);
            if (match.Success)
            {
                //一致した対象が見つかったときキャプチャした部分文字列を表示
                return match.Groups["iSerialNumber"].Value;
            }

            return null;
        }

        private string GetFirstRecoveryModeSerialNumberWithLock()
        {
            return GetConnectedUsbSerialNumber(RecoveryModeDevicePathPattern, GetFirstRecoveryModeDevicePathWithLock);
        }

        private string GetManuModeDevicePathWithLock()
        {
            string devicePath = null;
            RetryUtility.Do(() =>
            {
                var devicePathList = new List<string>();
                do
                {
                    devicePathList = UsbDevice.EnumerateDevicePathByGuid(ManuModeDeviceGuid);
                    Thread.Sleep(100);
                } while (devicePathList.Count == 0);

                foreach (var i in devicePathList)
                {
                    var serial = GetSerialNumberFromDevicePath(ManuModeDevicePathPattern, i);
                    if (!UsbDeviceLocker.IsLocked(serial))
                    {
                        ManuLocker = new UsbDeviceLocker();
                        if (ManuLocker.Lock(serial))
                        {
                            devicePath = i;
                            return;
                        }
                    }
                }
                throw new Exception("GetSerialNumberFromDevicePath failed.");
            },
            (e) => {
                AppendLog(e.Message);
            },
            100,
            TimeSpan.FromSeconds(3));

            return devicePath;
        }

        private string GetFirstManuModeSerialNumberWithLock()
        {
            return GetConnectedUsbSerialNumber(ManuModeDevicePathPattern, GetManuModeDevicePathWithLock);
        }

        delegate void AppendLogCallback(string text);

        private void AppendLog(string text)
        {
            if (this.LogTextBox.InvokeRequired)
            {
                this.Invoke(new AppendLogCallback(AppendLog), new object[] { text });
            }
            else
            {
                this.LogTextBox.AppendText(text);
                if (LogFileName != null)
                {
                    using (var writer = File.AppendText(LogFileName))
                    {
                        writer.Write(text);
                    }
                }
            }
        }

        private bool IsTargetCompletedSuccessfully()
        {
            return FindLog("[[ SUCCESS ]]") < 0 && FindLog("[[ LIMITED SUCCESS ]]") < 0 ? false : true;
        }

        private int FindLog(string text)
        {
            if (!File.Exists(LogFileName))
            {
                return -1;
            }

            using (var sr = new StreamReader(LogFileName))
            {
                return sr.ReadToEnd().IndexOf(text);
            }
        }

        private void ErrorDataReceived(object sender, DataReceivedEventArgs e)
        {
            AppendLog(e.Data + System.Environment.NewLine);
        }

        private void OutputDataReceived(object sender, DataReceivedEventArgs e)
        {
            AppendLog(e.Data + System.Environment.NewLine);
        }

        private void DeviceCheckTimer_Tick(object sender, EventArgs e)
        {
            StartButton.Enabled = (IsDetectedRecoveryModeDevice() && !IsFrcmOrBackupRestoreInProgress()) ? true : false;
            CancelProcessButton.Enabled = (IsFrcmOrBackupRestoreInProgress() || (WaitProcessThread?.IsAlive ?? false)) ? true : false;
        }

        private void PrepareToExit()
        {
            if (IsFrcmOrBackupRestoreInProgress())
            {
                TerminateCurrentBackupRestoreProcess();
            }
            if (WaitProcessThread?.IsAlive ?? false)
            {
                WaitProcessThread.Abort();
            }
            ProcessUtility.KillProcessBasedOnCommandLineArgument("RunOnTargetFromNand.exe", TargetSerialNumber);
        }

        private void Form1_FormClosed(Object sender, FormClosedEventArgs e)
        {
            PrepareToExit();
            System.Environment.Exit(0);
        }

        delegate void SetSerialNumberCallBack(string text);

        private void SetSerialNumber(string serial)
        {
            if (this.SerialNumberLabel.InvokeRequired)
            {
                this.Invoke(new SetSerialNumberCallBack(SetSerialNumber), new object[] { serial });
            }
            else
            {
                this.SerialNumberLabel.Text = serial;
            }
        }

        private void CancelButton_Click(object sender, EventArgs e)
        {
            var result = MessageBox.Show("Stop the current process?", "Warning", MessageBoxButtons.OKCancel, MessageBoxIcon.Error);
            if (result == DialogResult.OK)
            {
                PrepareToExit();
                ClearLog();
            }
            else
            {
                return;
            }
        }
    }
}
