﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.Contracts;

namespace VibrationConverterConsole.BnvibIO
{
    public static class BnvibFileWriter
    {
        public static void WriteBnvibFile(string outputPath, BnvibFile file)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(outputPath);
            Ensure.Argument.NotNull(file);

            using (var fileStream = new FileStream(outputPath, FileMode.Create, FileAccess.Write))
            {
                using (var writer = new BinaryWriter(fileStream))
                {
                    WriteBnvibFile(writer, file);
                }
            }
        }

        public static void WriteBnvibFile(BinaryWriter writer, BnvibFile file)
        {
            Ensure.Argument.NotNull(writer);
            Ensure.Argument.NotNull(file);
            Ensure.Argument.True(file.Info != null || file.Data != null);

            // Meta Data Region
            int[] ExpectedMetaDataSizeList = new int[] {
                BnvibFileInfo.MetaDataSizeBasic,
                BnvibFileInfo.MetaDataSizeWithLoop,
                BnvibFileInfo.MetaDataSizeWithLoopInterval
            };
            BnvibFileFormatId ExpectedFormatId = BnvibFileInfo.DefaultFormatId;
            Ensure.Argument.True(ExpectedMetaDataSizeList.Contains(file.Info.MetaDataSize));
            Ensure.Argument.Equals(file.Info.FormatId, ExpectedFormatId);
            Ensure.Argument.Range(file.Info.SamplingRate, 1, int.MaxValue);
            writer.Write((UInt32)file.Info.MetaDataSize);
            writer.Write((UInt16)file.Info.FormatId);
            writer.Write((UInt16)file.Info.SamplingRate);
            if (file.Info.HasLoop)
            {
                Ensure.Argument.Range(file.Info.LoopStart, 0, file.Info.SampleLength + 1);
                Ensure.Argument.Range(file.Info.LoopEnd, file.Info.LoopStart, file.Info.SampleLength + 1);
                writer.Write((UInt32)file.Info.LoopStart);
                writer.Write((UInt32)file.Info.LoopEnd);

                if (file.Info.HasLoopInterval)
                {
                    Ensure.Argument.Range(file.Info.LoopInterval, 1, int.MaxValue);
                    writer.Write((UInt32)file.Info.LoopInterval);
                }
            }

            // DataRegion
            byte[] rawData = file.Data.ToRawData();
            int expectedDataSize = rawData.Count();
            Ensure.Equals(file.Info.DataSize, expectedDataSize);
            writer.Write((UInt32)file.Info.DataSize);
            writer.Write(rawData);
        }

        public static void WriteNvibFile(string outputPath, BnvibFile file)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(outputPath);
            Ensure.Argument.NotNull(file);

            using (var fileStream = new FileStream(outputPath, FileMode.Create, FileAccess.Write))
            {
                using (var writer = new StreamWriter(fileStream))
                {
                    WriteNvibFile(writer, file);
                }
            }
        }

        public static void WriteNvibFile(TextWriter writer, BnvibFile file)
        {
            Ensure.Argument.NotNull(writer);
            Ensure.Argument.NotNull(file);
            Ensure.Argument.True(file.Info != null || file.Data != null);

            if(file.Info.HasLoop)
            {
                writer.WriteLine("$LoopStart={0}", file.Info.LoopStart);
                writer.WriteLine("$LoopEnd={0}", file.Info.LoopEnd);
            }

            if(file.Info.HasLoopInterval)
            {
                writer.WriteLine("$LoopInterval={0}", file.Info.LoopInterval);
            }

            for(int i = 0; i < file.Data.SampleCount; i++)
            {
                VibrationValue v = file.Data.GetVibrationValueAt(i);

                writer.WriteLine("{0,6},{1,7}, {2,6},{3,7}",
                    v.amplitudeLow.ToString("0.0000", System.Globalization.CultureInfo.InvariantCulture),
                    v.frequencyLow.ToString("0.00", System.Globalization.CultureInfo.InvariantCulture),
                    v.amplitudeHigh.ToString("0.0000", System.Globalization.CultureInfo.InvariantCulture),
                    v.frequencyHigh.ToString("0.00", System.Globalization.CultureInfo.InvariantCulture));
            }
        }
    }
}
