﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Json;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.Contracts;
using VibrationConverterConsole.BnvibIO;
using VibrationConverterConsole.Filter;

namespace VibrationConverterConsole.AmpFreqExtractor
{
    public class SubbandData
    {
        public float[] amplitudeArray;
        public float[] frequencyArray;
    }

    public class OutputKeeper
    {
        public float[] savedOutput = null;
        public bool canUseSavedOutput = false;
        public void SaveOutput(float[] output)
        {
            savedOutput = output;
            canUseSavedOutput = true;
        }
    }
    public class KeeperLinkedProperty<T> where T : IComparable
    {
        private T m_Value;
        private OutputKeeper m_SavedOutput;
        public event EventHandler ValueChanged;

        public KeeperLinkedProperty(T value, OutputKeeper savedOutput)
        {
            m_Value = value;
            m_SavedOutput = savedOutput;
        }
        public void SetValue(T value)
        {
            if (m_Value.CompareTo(value) != 0)
            {
                m_SavedOutput.canUseSavedOutput = false;
                m_Value = value;
                if (ValueChanged != null) ValueChanged(this, new EventArgs());
            }
        }
        public static implicit operator T(KeeperLinkedProperty<T> linkedProperty)
        {
            return linkedProperty.m_Value;
        }
    }

    public class SamplingRateChanger
    {
        public OutputKeeper keeper;
        public KeeperLinkedProperty<double> samplingRate;

        public SamplingRateChanger()
        {
            keeper = new OutputKeeper();
            samplingRate = new KeeperLinkedProperty<double>(8000.0, keeper);
        }
        public float[] ChangeSamplingRate(float[] source, double inputSamplingRate)
        {
            if (keeper.canUseSavedOutput)
            {
                return keeper.savedOutput;
            }


            // とりあえずダウンサンプリングだけに対応
            float[] dest;
            if (samplingRate < inputSamplingRate)
            {
                int destLength = (int)Math.Ceiling((double)source.Length * samplingRate / inputSamplingRate);
                dest = new float[destLength];

                for (int destIdx = 0; destIdx < destLength; destIdx++)
                {
                    int srcIdxStart = (int)((double)destIdx * inputSamplingRate / samplingRate);
                    int srcIdxEnd = (int)((double)(destIdx + 1) * inputSamplingRate / samplingRate);
                    if (srcIdxEnd > source.Length) srcIdxEnd = source.Length;

                    dest[destIdx] = 0.0f;
                    for (int srcIdx = srcIdxStart; srcIdx < srcIdxEnd; srcIdx++)
                    {
                        dest[destIdx] += source[srcIdx];
                    }
                    dest[destIdx] /= (srcIdxEnd - srcIdxStart);
                }
            }
            else
            {
                dest = new float[source.Length];
                Array.Copy(source, dest, source.Length);
            }

            keeper.SaveOutput(dest);
            return dest;
        }
    }

    [DataContract]
    public struct SingleBpfConfig
    {
        [DataMember]
        public FilterFamily filterFamily;
        [DataMember]
        public double lowerCutoff;
        [DataMember]
        public double higherCutoff;
        [DataMember]
        public int order;
        [DataMember]
        public double epsilon;
    }
    public class SingleBpfExecutor : TransferFunction
    {
        public OutputKeeper keeper;
        public KeeperLinkedProperty<FilterFamily> filterFamily;
        public KeeperLinkedProperty<double> lowerCutoff;
        public KeeperLinkedProperty<double> higherCutoff;
        public KeeperLinkedProperty<int> order;
        public KeeperLinkedProperty<double> epsilon;
        public KeeperLinkedProperty<double> samplingRate;

        private ShelfFilterFunction m_HpfFunc;
        private ShelfFilterFunction m_LpfFunc;

        public SingleBpfExecutor()
        {
            keeper = new OutputKeeper();
            filterFamily = new KeeperLinkedProperty<FilterFamily>(FilterFamily.Butterworth, keeper);
            filterFamily.ValueChanged += filterFamily_ValueChanged;
            lowerCutoff = new KeeperLinkedProperty<double>(80.0, keeper);
            lowerCutoff.ValueChanged += lowerCutoff_ValueChanged;
            higherCutoff = new KeeperLinkedProperty<double>(240.0, keeper);
            higherCutoff.ValueChanged += higherCutoff_ValueChanged;
            order = new KeeperLinkedProperty<int>(16, keeper);
            order.ValueChanged += order_ValueChanged;
            epsilon = new KeeperLinkedProperty<double>(1.0, keeper);
            epsilon.ValueChanged += epsilon_ValueChanged;
            samplingRate = new KeeperLinkedProperty<double>(44100.0, keeper);
            samplingRate.ValueChanged += samplingRate_ValueChanged;

            m_HpfFunc = new ShelfFilterFunction(filterFamily, FrequencyResponse.HighPass);
            m_LpfFunc = new ShelfFilterFunction(filterFamily, FrequencyResponse.LowPass);

            m_HpfFunc.CutoffLength = samplingRate / lowerCutoff;
            m_LpfFunc.CutoffLength = samplingRate / higherCutoff;
            m_HpfFunc.Order = order;
            m_LpfFunc.Order = order;
            m_HpfFunc.Epsilon = epsilon;
            m_LpfFunc.Epsilon = epsilon;
        }

        void filterFamily_ValueChanged(object sender, EventArgs e)
        {
            m_HpfFunc.FilterFamily = filterFamily;
            m_LpfFunc.FilterFamily = filterFamily;
        }
        void lowerCutoff_ValueChanged(object sender, EventArgs e)
        {
            m_HpfFunc.CutoffLength = samplingRate / lowerCutoff;
        }
        void higherCutoff_ValueChanged(object sender, EventArgs e)
        {
            m_LpfFunc.CutoffLength = samplingRate / higherCutoff;
        }
        void order_ValueChanged(object sender, EventArgs e)
        {
            m_HpfFunc.Order = order;
            m_LpfFunc.Order = order;
        }
        void epsilon_ValueChanged(object sender, EventArgs e)
        {
            m_HpfFunc.Epsilon = epsilon;
            m_LpfFunc.Epsilon = epsilon;
        }
        void samplingRate_ValueChanged(object sender, EventArgs e)
        {
            m_HpfFunc.CutoffLength = samplingRate / lowerCutoff;
            m_LpfFunc.CutoffLength = samplingRate / higherCutoff;
        }

        public void SetConfig(SingleBpfConfig c)
        {
            this.filterFamily.SetValue(c.filterFamily);
            this.lowerCutoff.SetValue(c.lowerCutoff);
            this.higherCutoff.SetValue(c.higherCutoff);
            this.order.SetValue(c.order);
            this.epsilon.SetValue(c.epsilon);
        }
        public SingleBpfConfig GetConfig()
        {
            SingleBpfConfig c = new SingleBpfConfig();
            c.filterFamily = this.filterFamily;
            c.lowerCutoff = this.lowerCutoff;
            c.higherCutoff = this.higherCutoff;
            c.order = this.order;
            c.epsilon = this.epsilon;
            return c;
        }
        public override Complex GetValue(Complex z)
        {
            Complex h = 1.0;
            h *= m_HpfFunc.GetValue(z);
            h *= m_LpfFunc.GetValue(z);
            return h;
        }
        public SerialFilter<BiQuadFilter> GetFilter()
        {
            SerialFilter<BiQuadFilter> s = new SerialFilter<BiQuadFilter>();
            for (int k = 0; k < order / 2; k++)
            {
                BiQuadFilter bqHPF = m_HpfFunc.GetBiquadFilterElement(k);
                BiQuadFilter bqLPF = m_LpfFunc.GetBiquadFilterElement(k);
                s.AddFilter(bqHPF, bqLPF);
            }
            return s;
        }
        public float[] ExecuteFilter(float[] source, MessageQueue messageQueue = null)
        {
            if (keeper.canUseSavedOutput) return keeper.savedOutput;
            IFilter filter = GetFilter();

            float[] dest = new float[source.Length];
            int largeStep = 10000;
            for (int i = 0; i < source.Length; )
            {
                if (messageQueue != null)
                {
                    string msg = "Executing BPF ("
                        + ((double)lowerCutoff).ToString("0.00") + ", "
                        + ((double)higherCutoff).ToString("0.00") + ")... ("
                        + i.ToString() + "/" + source.Length.ToString() + ")";
                    messageQueue.Write(msg);
                }
                int idxEnd = i + largeStep;
                if (idxEnd > source.Length) idxEnd = source.Length;
                for (; i < idxEnd; i++)
                {
                    dest[i] = filter.GetOutput(source[i]);
                }
            }

            keeper.SaveOutput(dest);
            return dest;
        }
    }

    [DataContract]
    public struct AmplitudeExtractorConfig
    {
        [DataMember]
        public float amplitudeCoeff;
    }
    public class AmplitudeExtractor
    {
        public OutputKeeper keeper;
        public KeeperLinkedProperty<double> updateFrequency;
        public KeeperLinkedProperty<float> amplitudeCoeff;

        public AmplitudeExtractor()
        {
            keeper = new OutputKeeper();
            updateFrequency = new KeeperLinkedProperty<double>(200.0, keeper);
            amplitudeCoeff = new KeeperLinkedProperty<float>(1.0f, keeper);
        }
        public float[] ExtractAmplitude(float[] source, double inputSamplingRate)
        {
            if (keeper.canUseSavedOutput) return keeper.savedOutput;

            int ampLength = (int)Math.Ceiling((double)source.Length * updateFrequency / inputSamplingRate);
            float[] amplitude = new float[ampLength];

            for (int ampIdx = 0; ampIdx < ampLength; ampIdx++)
            {
                int srcIdxStart = (int)((double)ampIdx * inputSamplingRate / updateFrequency);
                int srcIdxEnd = (int)((double)(ampIdx + 1) * inputSamplingRate / updateFrequency);
                if (srcIdxEnd > source.Length) srcIdxEnd = source.Length;

                amplitude[ampIdx] = 0.0f;
                for (int srcIdx = srcIdxStart; srcIdx < srcIdxEnd; srcIdx++)
                {
                    float abs = source[srcIdx];
                    if (abs < 0) abs = -abs;
                    if (abs > amplitude[ampIdx]) amplitude[ampIdx] = abs;
                }
                amplitude[ampIdx] *= amplitudeCoeff;
            }

            keeper.SaveOutput(amplitude);
            return amplitude;
        }
        public AmplitudeExtractorConfig GetConfig()
        {
            AmplitudeExtractorConfig c = new AmplitudeExtractorConfig();
            c.amplitudeCoeff = this.amplitudeCoeff;
            return c;
        }
        public void SetConfig(AmplitudeExtractorConfig c)
        {
            this.amplitudeCoeff.SetValue(c.amplitudeCoeff);
        }
    }

    [DataContract]
    public struct FrequencyExtractorConfig
    {
        [DataMember]
        public double baseFrequency;
        [DataMember]
        public double lowerCutoff;
        [DataMember]
        public double higherCutoff;
    }
    public class FrequencyExtractor
    {
        public OutputKeeper keeper;
        public KeeperLinkedProperty<double> updateFrequency;
        public KeeperLinkedProperty<double> lowerCutoff;
        public KeeperLinkedProperty<double> baseFrequency;
        public KeeperLinkedProperty<double> higherCutoff;

        public FrequencyExtractor()
        {
            keeper = new OutputKeeper();
            updateFrequency = new KeeperLinkedProperty<double>(200.0, keeper);
            lowerCutoff = new KeeperLinkedProperty<double>(128.0, keeper);
            baseFrequency = new KeeperLinkedProperty<double>(160.0, keeper);
            higherCutoff = new KeeperLinkedProperty<double>(200.0, keeper);
        }
        public float[] ExtractFrequency(float[] source, double inputSamplingRate)
        {
            if (keeper.canUseSavedOutput) return keeper.savedOutput;
            int freqLength = (int)Math.Ceiling((double)source.Length * updateFrequency / inputSamplingRate);

            // ゼロポイントを区間毎に探す
            List<int>[] zeroIdx = new List<int>[freqLength + 1];
            int minDistance = (int)(inputSamplingRate / (2.0 * higherCutoff));
            int lastZeroIdx = -minDistance;
            for (int freqIdx = 0; freqIdx < freqLength + 1; freqIdx++)
            {
                zeroIdx[freqIdx] = new List<int>();
                int srcIdxStart = (int)((double)freqIdx * inputSamplingRate / updateFrequency);
                int srcIdxEnd = (int)((double)(freqIdx + 1) * inputSamplingRate / updateFrequency);
                if (srcIdxEnd > source.Length - 1) srcIdxEnd = source.Length - 1;

                for (int srcIdx = srcIdxStart; srcIdx < srcIdxEnd; srcIdx++)
                {
                    if (source[srcIdx] * source[srcIdx + 1] <= 0)
                    {
                        if (srcIdx - lastZeroIdx >= minDistance)
                        {
                            zeroIdx[freqIdx].Add(srcIdx);
                            lastZeroIdx = srcIdx;
                        }
                    }
                }
            }

            // 各区間について、区間外で次に現れるゼロポイントを探す
            int[] nextZeroIdx = new int[freqLength];
            nextZeroIdx[freqLength - 1] = -1;   // 次のゼロポイントが存在しない場合は負の値で表現する
            for (int freqIdx = freqLength - 2; freqIdx >= 0; freqIdx--)
            {
                if(zeroIdx[freqIdx + 1].Count > 0)
                {
                    nextZeroIdx[freqIdx] = zeroIdx[freqIdx + 1].First();
                }
                else
                {
                    nextZeroIdx[freqIdx] = nextZeroIdx[freqIdx + 1];
                }
            }

            // 区間毎の周波数を計算
            float[] frequency = new float[freqLength];
            float lastFreq = (float)baseFrequency;
            for (int freqIdx = 0; freqIdx < freqLength; freqIdx++)
            {
                if (zeroIdx[freqIdx].Count == 0)
                {
                    frequency[freqIdx] = lastFreq;
                }
                else
                {
                    int numZeros = zeroIdx[freqIdx].Count;
                    int firstIdx = zeroIdx[freqIdx].First();
                    int lastIdx = nextZeroIdx[freqIdx];

                    // 次のゼロポイントが見つからない場合は区間内のゼロポイントだけで計算する
                    if (lastIdx < 0)
                    {
                        numZeros--;
                        lastIdx = zeroIdx[freqIdx].Last();
                    }

                    if (firstIdx == lastIdx)
                    {
                        frequency[freqIdx] = lastFreq;
                    }
                    else
                    {
                        frequency[freqIdx] = (float)(0.5 * inputSamplingRate * numZeros / (double)(lastIdx - firstIdx));
                        if (frequency[freqIdx] < lowerCutoff) frequency[freqIdx] = (float)lowerCutoff;
                        if (frequency[freqIdx] > higherCutoff) frequency[freqIdx] = (float)higherCutoff;
                    }
                }

                lastFreq = frequency[freqIdx];
            }

            keeper.SaveOutput(frequency);
            return frequency;
        }
        public FrequencyExtractorConfig GetConfig()
        {
            FrequencyExtractorConfig c = new FrequencyExtractorConfig();
            c.baseFrequency = this.baseFrequency;
            c.lowerCutoff = this.lowerCutoff;
            c.higherCutoff = this.higherCutoff;
            return c;
        }
        public void SetConfig(FrequencyExtractorConfig c)
        {
            this.baseFrequency.SetValue(c.baseFrequency);
            this.lowerCutoff.SetValue(c.lowerCutoff);
            this.higherCutoff.SetValue(c.higherCutoff);
        }
    }

    [DataContract]
    public struct SubbandAmpFreqExtractorConfig
    {
        [DataMember]
        public SingleBpfConfig bpfForAm;
        [DataMember]
        public SingleBpfConfig bpfForFm;
        [DataMember]
        public double amfmUpdateFrequency;
        [DataMember]
        public bool isSameBpfUsedForFm;
        [DataMember]
        public AmplitudeExtractorConfig ampExtractor;
        [DataMember]
        public FrequencyExtractorConfig freqExtractor;
    }
    public class SubbandAmpFreqExtractor
    {
        public SingleBpfExecutor bpfForAm;
        public SingleBpfExecutor bpfForFm;
        public double amfmUpdateFrequency
        {
            get { return ampExtractor.updateFrequency; }
            set
            {
                ampExtractor.updateFrequency.SetValue(value);
                freqExtractor.updateFrequency.SetValue(value);
            }
        }
        private bool m_IsSameBpfUsedForFm = false;
        public bool isSameBpfUsedForFm
        {
            get { return m_IsSameBpfUsedForFm; }
            set
            {
                if (m_IsSameBpfUsedForFm != value)
                {
                    m_IsSameBpfUsedForFm = value;
                    bpfForFm.keeper.canUseSavedOutput = false;
                    freqExtractor.keeper.canUseSavedOutput = false;
                }
            }
        }
        public AmplitudeExtractor ampExtractor;
        public FrequencyExtractor freqExtractor;
        public bool canUseSavedOutput
        {
            get
            {
                if (!bpfForAm.keeper.canUseSavedOutput) return false;
                if (!bpfForFm.keeper.canUseSavedOutput) return false;
                if (!ampExtractor.keeper.canUseSavedOutput) return false;
                if (!freqExtractor.keeper.canUseSavedOutput) return false;
                return true;
            }
        }

        public SubbandAmpFreqExtractor()
        {
            bpfForAm = new SingleBpfExecutor();
            bpfForFm = new SingleBpfExecutor();
            ampExtractor = new AmplitudeExtractor();
            freqExtractor = new FrequencyExtractor();
        }
        public SubbandAmpFreqExtractorConfig GetConfig()
        {
            SubbandAmpFreqExtractorConfig c = new SubbandAmpFreqExtractorConfig();
            c.bpfForAm = this.bpfForAm.GetConfig();
            c.bpfForFm = this.bpfForFm.GetConfig();
            c.amfmUpdateFrequency = this.amfmUpdateFrequency;
            c.isSameBpfUsedForFm = this.isSameBpfUsedForFm;
            c.ampExtractor = this.ampExtractor.GetConfig();
            c.freqExtractor = this.freqExtractor.GetConfig();
            return c;
        }
        public void SetConig(SubbandAmpFreqExtractorConfig c)
        {
            this.bpfForAm.SetConfig(c.bpfForAm);
            this.bpfForFm.SetConfig(c.bpfForFm);
            this.amfmUpdateFrequency = c.amfmUpdateFrequency;
            this.isSameBpfUsedForFm = c.isSameBpfUsedForFm;
            this.ampExtractor.SetConfig(c.ampExtractor);
            this.freqExtractor.SetConfig(c.freqExtractor);
        }
        private void SetDefaultCommon()
        {
            bpfForAm.epsilon.SetValue(1.0);
            bpfForAm.filterFamily.SetValue(FilterFamily.Butterworth);
            bpfForAm.order.SetValue(6);
            bpfForFm.epsilon.SetValue(1.0);
            bpfForFm.filterFamily.SetValue(FilterFamily.Butterworth);
            bpfForFm.order.SetValue(6);
            amfmUpdateFrequency = 200.0;
            isSameBpfUsedForFm = false;
        }
        public void SetDefaultForLow()
        {
            SetDefaultCommon();
            bpfForAm.lowerCutoff.SetValue(80.0);
            bpfForAm.higherCutoff.SetValue(180.0);
            bpfForFm.lowerCutoff.SetValue(128.0);
            bpfForFm.higherCutoff.SetValue(200.0);
            isSameBpfUsedForFm = false;
            freqExtractor.baseFrequency.SetValue(160.0);
            freqExtractor.lowerCutoff.SetValue(128.0);
            freqExtractor.higherCutoff.SetValue(200.0);
        }
        public void SetDefaultForHigh()
        {
            SetDefaultCommon();
            bpfForAm.lowerCutoff.SetValue(284.0);
            bpfForAm.higherCutoff.SetValue(400.0);
            bpfForFm.lowerCutoff.SetValue(256.0);
            bpfForFm.higherCutoff.SetValue(400.0);
            isSameBpfUsedForFm = false;
            freqExtractor.baseFrequency.SetValue(320.0);
            freqExtractor.lowerCutoff.SetValue(256.0);
            freqExtractor.higherCutoff.SetValue(400.0);
        }
        public void DisableUsingSavedOutput()
        {
            bpfForAm.keeper.canUseSavedOutput = false;
            bpfForFm.keeper.canUseSavedOutput = false;
            ampExtractor.keeper.canUseSavedOutput = false;
            freqExtractor.keeper.canUseSavedOutput = false;
        }
        private void UpdateKeeperStatus()
        {
            if (!bpfForAm.keeper.canUseSavedOutput)
            {
                ampExtractor.keeper.canUseSavedOutput = false;
                if (isSameBpfUsedForFm)
                {
                    freqExtractor.keeper.canUseSavedOutput = false;
                }
            }
            if (!isSameBpfUsedForFm)
            {
                if (!bpfForFm.keeper.canUseSavedOutput)
                {
                    freqExtractor.keeper.canUseSavedOutput = false;
                }
            }
        }
        public SubbandData Extract(float[] source, double samplingRate, MessageQueue messageQueue = null)
        {
            UpdateKeeperStatus();

            SubbandData subbandData = new SubbandData();

            bpfForAm.samplingRate.SetValue(samplingRate);
            float[] bpfBufferAm = bpfForAm.ExecuteFilter(source, messageQueue);

            if (messageQueue != null) messageQueue.Write("Extracting Amplitude...");
            ampExtractor.updateFrequency.SetValue(amfmUpdateFrequency);
            subbandData.amplitudeArray = ampExtractor.ExtractAmplitude(bpfBufferAm, samplingRate);

            float[] bpfBufferFm = bpfBufferAm;
            if (!isSameBpfUsedForFm)
            {
                bpfForFm.samplingRate.SetValue(samplingRate);
                bpfBufferFm = bpfForFm.ExecuteFilter(source, messageQueue);
            }

            if (messageQueue != null) messageQueue.Write("Extracting Frequency...");
            freqExtractor.updateFrequency.SetValue(amfmUpdateFrequency);
            subbandData.frequencyArray = freqExtractor.ExtractFrequency(bpfBufferFm, samplingRate);

            Ensure.Equals(subbandData.amplitudeArray.Count(), subbandData.frequencyArray.Count());
            return subbandData;
        }
    }

    [DataContract]
    public struct CombinedAmpFreqExtractorConfig
    {
        [DataMember]
        public double downSamplingRate;
        [DataMember]
        public SubbandAmpFreqExtractorConfig extractorLow;
        [DataMember]
        public SubbandAmpFreqExtractorConfig extractorHigh;
        [DataMember]
        public bool isAmpCoeffAutoAdjusted;
    }
    public class CombinedAmpFreqExtractor
    {
        public SamplingRateChanger samplingRateChanger;
        public SubbandAmpFreqExtractor extractorLow;
        public SubbandAmpFreqExtractor extractorHigh;
        public bool isAmpCoeffAutoAdjusted;
        public bool canUseSavedOutput
        {
            get
            {
                if (!samplingRateChanger.keeper.canUseSavedOutput) return false;
                if (!extractorLow.canUseSavedOutput) return false;
                if (!extractorHigh.canUseSavedOutput) return false;
                return true;
            }
        }

        public CombinedAmpFreqExtractor()
        {
            samplingRateChanger = new SamplingRateChanger();
            extractorLow = new SubbandAmpFreqExtractor();
            extractorLow.SetDefaultForLow();
            extractorHigh = new SubbandAmpFreqExtractor();
            extractorHigh.SetDefaultForHigh();
            isAmpCoeffAutoAdjusted = true;
        }
        public CombinedAmpFreqExtractorConfig GetConfig()
        {
            CombinedAmpFreqExtractorConfig c = new CombinedAmpFreqExtractorConfig();
            c.downSamplingRate = this.samplingRateChanger.samplingRate;
            c.extractorLow = this.extractorLow.GetConfig();
            c.extractorHigh = this.extractorHigh.GetConfig();
            c.isAmpCoeffAutoAdjusted = this.isAmpCoeffAutoAdjusted;
            return c;
        }
        public void SetConfig(CombinedAmpFreqExtractorConfig c)
        {
            this.samplingRateChanger.samplingRate.SetValue(c.downSamplingRate);
            this.extractorLow.SetConig(c.extractorLow);
            this.extractorHigh.SetConig(c.extractorHigh);
            this.isAmpCoeffAutoAdjusted = c.isAmpCoeffAutoAdjusted;
        }
        public void DisableUsingSavedOutput()
        {
            samplingRateChanger.keeper.canUseSavedOutput = false;
            extractorLow.DisableUsingSavedOutput();
            extractorHigh.DisableUsingSavedOutput();
        }
        public BnvibFile Extract(float[] source, double samplingRate, MessageQueue messageQueue = null)
        {
            if (!samplingRateChanger.keeper.canUseSavedOutput)
            {
                extractorLow.DisableUsingSavedOutput();
                extractorHigh.DisableUsingSavedOutput();
            }

            BnvibFile bnvibFile = new BnvibFile();
            SubbandData subbandLow, subbandHigh;

            if (messageQueue != null) messageQueue.Write("Down Sampling...");
            float[] dsBuffer = samplingRateChanger.ChangeSamplingRate(source, samplingRate);

            if (messageQueue != null) messageQueue.Write("Extracting Lower Subband...");
            subbandLow = extractorLow.Extract(dsBuffer, samplingRateChanger.samplingRate, messageQueue);

            if (messageQueue != null) messageQueue.Write("Extracting Higher Subband...");
            subbandHigh = extractorHigh.Extract(dsBuffer, samplingRateChanger.samplingRate, messageQueue);

            Ensure.Equals(subbandLow.amplitudeArray.Count(), subbandHigh.amplitudeArray.Count());

            bnvibFile.Info.MetaDataSize = 4;
            bnvibFile.Info.FormatId = BnvibFileFormatId.PcmVer1;
            bnvibFile.Info.SamplingRate = 200;
            bnvibFile.Info.SampleLength = subbandLow.amplitudeArray.Count();
            bnvibFile.Info.DataSize = bnvibFile.Info.SampleLength * 4;

            if (isAmpCoeffAutoAdjusted)
            {
                float maxAmp = 0.0f;
                for (int i = 0; i < bnvibFile.Info.SampleLength; i++)
                {
                    float ampLow = subbandLow.amplitudeArray[i];
                    float ampHigh = subbandHigh.amplitudeArray[i];
                    float amp = Math.Abs(ampLow) + Math.Abs(ampHigh);
                    if (amp > maxAmp) maxAmp = amp;
                }

                if (maxAmp > 0.0f)
                {
                    float ampCoeffLow = extractorLow.ampExtractor.amplitudeCoeff;
                    float ampCoeffHigh = extractorHigh.ampExtractor.amplitudeCoeff;
                    extractorLow.ampExtractor.amplitudeCoeff.SetValue(ampCoeffLow / maxAmp);
                    extractorHigh.ampExtractor.amplitudeCoeff.SetValue(ampCoeffHigh / maxAmp);
                }

                if (messageQueue != null) messageQueue.Write("Extracting Lower Subband Again...");
                subbandLow = extractorLow.Extract(dsBuffer, samplingRateChanger.samplingRate, messageQueue);

                if (messageQueue != null) messageQueue.Write("Extracting Higher Subband Again...");
                subbandHigh = extractorHigh.Extract(dsBuffer, samplingRateChanger.samplingRate, messageQueue);
            }

            // ふたつの SubbandData から BnvibSampleData を生成
            bnvibFile.Data = new BnvibSampleData();
            for (int i = 0; i < bnvibFile.Info.SampleLength; i++)
            {
                VibrationValue v = VibrationValue.Make(
                    subbandLow.amplitudeArray[i],
                    subbandLow.frequencyArray[i],
                    subbandHigh.amplitudeArray[i],
                    subbandHigh.frequencyArray[i]);
                bnvibFile.Data.AddVibrationValue(v);
            }

            return bnvibFile;
        }
    }

    public class ConfigWriter
    {
        public static void Write(System.IO.Stream stream, CombinedAmpFreqExtractor extractor)
        {
            CombinedAmpFreqExtractorConfig config = extractor.GetConfig();
            DataContractJsonSerializer serializer = new DataContractJsonSerializer(typeof(CombinedAmpFreqExtractorConfig));
            serializer.WriteObject(stream, config);
        }
    }
    public class ConfigReader
    {
        public static void Read(System.IO.Stream stream, ref CombinedAmpFreqExtractor extractor)
        {
            DataContractJsonSerializer serializer = new DataContractJsonSerializer(typeof(CombinedAmpFreqExtractorConfig));
            CombinedAmpFreqExtractorConfig config = (CombinedAmpFreqExtractorConfig)serializer.ReadObject(stream);
            extractor.SetConfig(config);
        }
    }
}
