﻿// ポート名の列挙に WMI を使う
//#define USE_WMI

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Management;
using System.Text;
using System.Text.RegularExpressions;

namespace Nintendo.NX.IrsensorNfcFirmwareUpdater
{
    class Util
    {
        /// <summary>
        /// COM ポート検出用の正規表現
        /// </summary>
#if USE_WMI
        public static readonly Regex RegexPortNum = new Regex(@"\((?<com>COM(?<num>\d+))\)");
#else
        public static readonly Regex RegexPortNum = new Regex(@"(?<com>COM(?<num>\d+))");
#endif

        /// <summary>
        /// ポート情報
        /// </summary>
        public class PortInfo
        {
            /// <summary>
            /// SerialPort クラス指定用のポート ID
            /// </summary>
            public string Id { get; set; }

            /// <summary>
            /// ポート番号
            /// </summary>
            public int Number { get; set; }

            /// <summary>
            /// ポート名
            /// </summary>
            public string Name { get; set; }

            public PortInfo()
            {
                Id     = "";
                Number = 0;
                Name   = "";
            }
        };

        /// <summary>
        /// 詳細ポート名一覧を取得
        /// </summary>
        /// <returns>PortInfo のリスト</returns>
        public static List<PortInfo> GetDetailedPortNames()
        {
            var portNames = new List<PortInfo>();

#if USE_WMI
            // (COMx) を含むプラグアンドプレイデバイス一覧を取得
            var pnpEntities = new ManagementClass("Win32_PnPEntity");
            foreach (var port in pnpEntities.GetInstances())
            {
                var info = new PortInfo();
                info.Name = port.GetPropertyValue("Caption").ToString();

                var match = RegexPortNum.Match(info.Name);
                if (match.Length > 0)
                {
                    info.Id = match.Groups["com"].Value;
                    info.Number = int.Parse(match.Groups["num"].Value);
                    portNames.Add(info);
                }
            }

#else
            var names = System.IO.Ports.SerialPort.GetPortNames();
            foreach (var portName in names)
            {
                var info = new PortInfo();
                info.Name = portName;

                var match = RegexPortNum.Match(info.Name);
                if (match.Length > 0)
                {
                    info.Id = match.Groups["com"].Value;
                    info.Number = int.Parse(match.Groups["num"].Value);
                    portNames.Add(info);
                }
            }
#endif
            // ID 順にソート
            portNames.Sort((a, b) => a.Number - b.Number);

            return portNames;
        }

    }
}
