﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector.Services
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Runtime.InteropServices;
    using System.Threading;
    using System.Threading.Tasks;

    /// <summary>
    /// DebugPad サービスを提供します。
    /// </summary>
    internal sealed class DebugPadService : ISessionService
    {
        private static readonly TimeSpan SamplerInterval =
            TimeSpan.FromMilliseconds(7);

        private readonly object syncObject = new object();

        private DebugPadState hostState = new DebugPadState();

        private DebugPadState autoState = new DebugPadState();

        private DebugPadState lastState = new DebugPadState();

        private uint port = HidShellPortAccessor.DefaultPort;

        private ISessionWriter writer = null;

        private CancellationTokenSource tokenSource = null;

        private Task task = null;

        /// <summary>
        /// DebugPadService クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal DebugPadService()
        {
        }

        [Flags]
        private enum DebugPadAttributes : uint
        {
            None = 0,
            IsConnected = 1u,
        }

        [Flags]
        private enum DebugPadButtons : uint
        {
            None = 0,
            A = 1u,
            B = 1u << 1,
            X = 1u << 2,
            Y = 1u << 3,
            L = 1u << 4,
            R = 1u << 5,
            ZL = 1u << 6,
            ZR = 1u << 7,
            Start = 1u << 8,
            Select = 1u << 9,
            Left = 1u << 10,
            Up = 1u << 11,
            Right = 1u << 12,
            Down = 1u << 13,
        }

        /// <summary>
        /// セッションの情報を設定します。
        /// </summary>
        public SessionInfo Info { private get; set; }

        /// <summary>
        /// サービスを開始します。
        /// </summary>
        /// <param name="writer">セッションへの書き込みを扱うライターです。</param>
        public void Start(ISessionWriter writer)
        {
            lock (this.syncObject)
            {
                this.hostState = new DebugPadState();
                this.autoState = this.hostState;
                this.lastState = this.hostState;
            }

            this.writer = writer;

            var ioPortManager = this.Info.IoPortManager;

            this.port = ioPortManager.GetPort(this.Info.Name);

            ioPortManager.Polling += this.OnPolling;

            this.tokenSource = new CancellationTokenSource();

            this.task = Task.Run(() => this.Sample(this.tokenSource.Token));
        }

        /// <summary>
        /// サービスを停止します。
        /// </summary>
        public void Stop()
        {
            this.Cancel();
        }

        /// <summary>
        /// サービスをキャンセルします。
        /// </summary>
        public void Cancel()
        {
            this.tokenSource.Cancel();

            this.task.Wait();
            this.task.Dispose();
            this.task = null;

            this.tokenSource.Dispose();
            this.tokenSource = null;

            var ioPortManager = this.Info.IoPortManager;
            ioPortManager.Polling -= this.OnPolling;

            this.writer = null;
        }

        /// <summary>
        /// メッセージを受け付けます。
        /// </summary>
        /// <param name="message">メッセージです。</param>
        public void AcceptMessage(byte[] message)
        {
            if (message.Length != Session.MessageSize)
            {
                return;
            }

            switch ((MessageType)message[0])
            {
                case MessageType.DebugPadButton:
                    {
                        var chunk = StructConverter.FromBytes<
                            DebugPadChunk>(message);

                        this.AcceptDebugPadButtonMessage(chunk.Buttons);

                        break;
                    }

                case MessageType.DebugPadStick:
                    {
                        var chunk = StructConverter.FromBytes<
                            DebugPadStickChunk>(message);

                        var stickType = (GamePadStickType)chunk.StickType;

                        this.AcceptDebugPadStickMessage(
                            stickType, chunk.X, chunk.Y);

                        break;
                    }

                case MessageType.DebugPadPower:
                    {
                        var chunk = StructConverter.FromBytes<
                            DebugPadChunk>(message);

                        var powerState = (GamePadPowerState)chunk.PowerState;

                        this.AcceptDebugPadPowerMessage(powerState);

                        break;
                    }

                default:
                    break;
            }
        }

        private static void GenerateChunks(
            List<byte[]> chunks,
            ref DebugPadState nextState,
            ref DebugPadState lastState)
        {
            var isConnected =
                nextState.Attributes.HasFlag(DebugPadAttributes.IsConnected);

            if (nextState.Attributes != lastState.Attributes)
            {
                var powerState = isConnected
                    ? GamePadPowerState.NoBattery
                    : GamePadPowerState.Disconnected;

                var chunk = new DebugPadChunk()
                {
                    Type = (byte)MessageType.DebugPadPower,
                    PowerState = (byte)powerState,
                };

                chunks.Add(StructConverter.ToBytes(chunk));
            }

            if (isConnected)
            {
                if (nextState.Buttons != lastState.Buttons)
                {
                    var chunk = new DebugPadChunk()
                    {
                        Type = (byte)MessageType.DebugPadButton,
                        Buttons = (uint)nextState.Buttons,
                    };

                    chunks.Add(StructConverter.ToBytes(chunk));
                }

                if (nextState.StickL.X != lastState.StickL.X ||
                    nextState.StickL.Y != lastState.StickL.Y)
                {
                    var chunk = new DebugPadStickChunk()
                    {
                        Type = (byte)MessageType.DebugPadStick,
                        StickType = (byte)GamePadStickType.Left,
                        X = (short)nextState.StickL.X,
                        Y = (short)nextState.StickL.Y,
                    };

                    chunks.Add(StructConverter.ToBytes(chunk));
                }

                if (nextState.StickR.X != lastState.StickR.X ||
                    nextState.StickR.Y != lastState.StickR.Y)
                {
                    var chunk = new DebugPadStickChunk()
                    {
                        Type = (byte)MessageType.DebugPadStick,
                        StickType = (byte)GamePadStickType.Right,
                        X = (short)nextState.StickR.X,
                        Y = (short)nextState.StickR.Y,
                    };

                    chunks.Add(StructConverter.ToBytes(chunk));
                }
            }
        }

        private static void SetDebugPadState(
            IntPtr handle, ref DebugPadState state, uint port)
        {
            var result = (HidShellResult)
                Native.SetHidShellDebugPadState(
                    handle, ref state, port, HidShellPortAccessor.Out);

            if (result != HidShellResult.Success)
            {
                throw new HidShellException(result);
            }
        }

        private static void GetDebugPadAutoPilotState(
            IntPtr handle, out DebugPadState outState, uint port)
        {
            var result = (HidShellResult)
                Native.GetHidShellDebugPadState(
                    handle, out outState, port, HidShellPortAccessor.In);

            switch (result)
            {
                case HidShellResult.Success:
                    return;

                case HidShellResult.StateNotSet:
                    outState = new DebugPadState();
                    return;

                default:
                    throw new HidShellException(result);
            }
        }

        private void AcceptDebugPadButtonMessage(uint buttons)
        {
            lock (this.syncObject)
            {
                this.hostState.Attributes = DebugPadAttributes.IsConnected;

                this.hostState.Buttons = (DebugPadButtons)buttons;
            }
        }

        private void AcceptDebugPadStickMessage(
            GamePadStickType stickType, int x, int y)
        {
            lock (this.syncObject)
            {
                this.hostState.Attributes = DebugPadAttributes.IsConnected;

                switch (stickType)
                {
                    case GamePadStickType.Left:
                        this.hostState.StickL.X = x;
                        this.hostState.StickL.Y = y;
                        break;

                    case GamePadStickType.Right:
                        this.hostState.StickR.X = x;
                        this.hostState.StickR.Y = y;
                        break;

                    default:
                        break;
                }
            }
        }

        private void AcceptDebugPadPowerMessage(
            GamePadPowerState powerState)
        {
            lock (this.syncObject)
            {
                this.hostState = new DebugPadState();

                switch (powerState)
                {
                    case GamePadPowerState.NoBattery:
                        this.hostState.Attributes =
                            DebugPadAttributes.IsConnected;
                        break;

                    default:
                        break;
                }
            }
        }

        private void OnPolling(object sender, HidShellPortAccessor accessor)
        {
            lock (this.syncObject)
            {
                GetDebugPadAutoPilotState(
                    accessor.Handle, out this.autoState, this.port);

                SetDebugPadState(
                    accessor.Handle, ref this.hostState, this.port);
            }
        }

        private void Sample(CancellationToken token)
        {
            var works = true;

            var stopwatch = new Stopwatch();

            var chunks = new List<byte[]>();

            var autoState = new DebugPadState();

            while (works && !token.IsCancellationRequested)
            {
                stopwatch.Start();

                lock (this.syncObject)
                {
                    autoState = this.autoState;
                }

                GenerateChunks(chunks, ref autoState, ref this.lastState);

                foreach (var chunk in chunks)
                {
                    try
                    {
                        this.writer.Write(chunk, 0, chunk.Length, token);
                    }
                    catch
                    {
                        works = false;

                        break;
                    }
                }

                chunks.Clear();

                this.lastState = autoState;

                stopwatch.Stop();

                if (stopwatch.Elapsed < SamplerInterval &&
                    works && !token.IsCancellationRequested)
                {
                    token.WaitHandle.WaitOne(
                        SamplerInterval - stopwatch.Elapsed);
                }

                stopwatch.Reset();
            }
        }

        [StructLayout(LayoutKind.Explicit)]
        private struct DebugPadChunk
        {
            [FieldOffset(0)]
            public byte Type;

            [FieldOffset(6)]
            public byte PowerState;

            [FieldOffset(7)]
            public byte BatteryLevel;

            [FieldOffset(8)]
            public uint Buttons;
        }

        [StructLayout(LayoutKind.Explicit)]
        private struct DebugPadStickChunk
        {
            [FieldOffset(0)]
            public byte Type;

            [FieldOffset(5)]
            public byte StickType;

            [FieldOffset(8)]
            public short X;

            [FieldOffset(10)]
            public short Y;
        }

        private struct DebugPadStickState
        {
            internal int X;
            internal int Y;
        }

        private struct DebugPadState
        {
            internal DebugPadAttributes Attributes;
            internal DebugPadButtons Buttons;
            internal DebugPadStickState StickL;
            internal DebugPadStickState StickR;
        }

        private static class Native
        {
            [DllImport("HidShellLibrary.dll")]
            internal static extern int GetHidShellDebugPadState(
                IntPtr handle,
                out DebugPadState outState, uint port, uint direction);

            [DllImport("HidShellLibrary.dll")]
            internal static extern int SetHidShellDebugPadState(
                IntPtr handle,
                ref DebugPadState state, uint port, uint direction);
        }
    }
}
