﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector.Services
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Runtime.InteropServices;
    using System.Threading;
    using System.Threading.Tasks;

    /// <summary>
    /// CaptureButton サービスを提供します。
    /// </summary>
    internal sealed class CaptureButtonService : ISessionService
    {
        private static readonly TimeSpan SamplerInterval =
            TimeSpan.FromMilliseconds(7);

        private readonly object syncObject = new object();

        private CaptureButtonState autoState = new CaptureButtonState();

        private CaptureButtonState lastState = new CaptureButtonState();

        private uint port = HidShellPortAccessor.DefaultPort;

        private ISessionWriter writer = null;

        private CancellationTokenSource tokenSource = null;

        private Task task = null;

        /// <summary>
        /// CaptureButtonService クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal CaptureButtonService()
        {
        }

        /// <summary>
        /// セッションの情報を設定します。
        /// </summary>
        public SessionInfo Info { private get; set; }

        /// <summary>
        /// サービスを開始します。
        /// </summary>
        /// <param name="writer">セッションへの書き込みを扱うライターです。</param>
        public void Start(ISessionWriter writer)
        {
            lock (this.syncObject)
            {
                this.autoState = new CaptureButtonState();
                this.lastState = this.autoState;
            }

            this.writer = writer;

            var ioPortManager = this.Info.IoPortManager;

            this.port = ioPortManager.GetPort(this.Info.Name);

            ioPortManager.Polling += this.OnPolling;

            this.tokenSource = new CancellationTokenSource();

            this.task = Task.Run(() => this.Sample(this.tokenSource.Token));
        }

        /// <summary>
        /// サービスを停止します。
        /// </summary>
        public void Stop()
        {
            this.Cancel();
        }

        /// <summary>
        /// サービスをキャンセルします。
        /// </summary>
        public void Cancel()
        {
            this.tokenSource.Cancel();

            this.task.Wait();
            this.task.Dispose();
            this.task = null;

            this.tokenSource.Dispose();
            this.tokenSource = null;

            var ioPortManager = this.Info.IoPortManager;
            ioPortManager.Polling -= this.OnPolling;

            this.writer = null;
        }

        /// <summary>
        /// メッセージを受け付けます。
        /// </summary>
        /// <param name="message">メッセージです。</param>
        public void AcceptMessage(byte[] message)
        {
        }

        private static void GenerateChunks(
            List<byte[]> chunks,
            ref CaptureButtonState nextState,
            ref CaptureButtonState lastState)
        {
            if (nextState.Buttons != lastState.Buttons)
            {
                var chunk = new CaptureButtonChunk
                {
                    Type = (byte)MessageType.CaptureButton,
                    Buttons = (uint)nextState.Buttons,
                };

                chunks.Add(StructConverter.ToBytes(chunk));
            }
        }

        private static void GetCaptureButtonAutoPilotState(
            IntPtr handle, out CaptureButtonState outState, uint port)
        {
            var result = (HidShellResult)
                Native.GetHidShellCaptureButtonState(
                    handle, out outState, port, HidShellPortAccessor.In);

            switch (result)
            {
                case HidShellResult.Success:
                    return;

                case HidShellResult.StateNotSet:
                    outState = new CaptureButtonState();
                    return;

                default:
                    throw new HidShellException(result);
            }
        }

        private void OnPolling(object sender, HidShellPortAccessor accessor)
        {
            lock (this.syncObject)
            {
                GetCaptureButtonAutoPilotState(
                    accessor.Handle, out this.autoState, this.port);
            }
        }

        private void Sample(CancellationToken token)
        {
            var works = true;

            var stopwatch = new Stopwatch();

            var chunks = new List<byte[]>();

            var autoState = new CaptureButtonState();

            while (works && !token.IsCancellationRequested)
            {
                stopwatch.Start();

                lock (this.syncObject)
                {
                    autoState = this.autoState;
                }

                GenerateChunks(chunks, ref autoState, ref this.lastState);

                foreach (var chunk in chunks)
                {
                    try
                    {
                        this.writer.Write(chunk, 0, chunk.Length, token);
                    }
                    catch
                    {
                        works = false;

                        break;
                    }
                }

                chunks.Clear();

                this.lastState = autoState;

                stopwatch.Stop();

                if (stopwatch.Elapsed < SamplerInterval &&
                    works && !token.IsCancellationRequested)
                {
                    token.WaitHandle.WaitOne(
                        SamplerInterval - stopwatch.Elapsed);
                }

                stopwatch.Reset();
            }
        }

        [StructLayout(LayoutKind.Explicit)]
        private struct CaptureButtonChunk
        {
            [FieldOffset(0)]
            public byte Type;

            [FieldOffset(4)]
            public uint Buttons;

            [FieldOffset(8)]
            private uint Padding;
        }

        private struct CaptureButtonState
        {
            internal ulong Buttons;
        }

        private static class Native
        {
            [DllImport("HidShellLibrary.dll")]
            internal static extern int GetHidShellCaptureButtonState(
                IntPtr handle,
                out CaptureButtonState outState, uint port, uint direction);
        }
    }
}
