﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"
#include "HidShellLibrary.h"
#include "HidShellLibrary_SharedMemory.h"
#include "HidShellLibrary_Utility.h"

namespace {

//!< ミューテックスオブジェクトの名前です。
const char* const MutexObjectName = "NINTENDO_HIDSHELL_MUTEX";

//!< ファイルマッピングオブジェクトの名前です。
const char* const FileMappingObjectName = "NINTENDO_HIDSHELL";

//!< ミューテックスオブジェクトを扱うクラスです。
class MutexObject final
{
    NN_DISALLOW_COPY(MutexObject);
    NN_DISALLOW_MOVE(MutexObject);

private:
    HANDLE m_Handle;

public:
    MutexObject() NN_NOEXCEPT;

    //!< ミューテックスをロックします。
    HidShellResult Lock() NN_NOEXCEPT;

    //!< ミューテックスをアンロックします。
    void Unlock() NN_NOEXCEPT;
};

//!< ファイルマッピングオブジェクトを扱うクラスです。
class FileMappingObject final
{
    NN_DISALLOW_COPY(FileMappingObject);
    NN_DISALLOW_MOVE(FileMappingObject);

private:
    HANDLE m_Handle;
    LPVOID m_Address;

public:
    FileMappingObject() NN_NOEXCEPT;

    //!< ファイルマッピングオブジェクトのアドレスを返します。
    LPVOID GetAddress() NN_NOEXCEPT
    {
        return m_Address;
    };

    //!< ファイルマッピングオブジェクトを作成します。
    HidShellResult Create(bool* pOutExists, DWORD size) NN_NOEXCEPT;
};

//!< ミューテックスオブジェクトを返します。
MutexObject& GetMutexObject() NN_NOEXCEPT;

//!< ファイルマッピングオブジェクトを返します。
FileMappingObject& GetFileMappingObject() NN_NOEXCEPT;

} // namespace

HidShellResult AquireHidShellSharedMemoryExclusiveRight(
    HidShellHandle* pOutHandle) NN_NOEXCEPT
{
    HID_SHELL_RESULT_THROW_UNLESS(
        pOutHandle != nullptr, HidShellResult::NullHandlePointer);

    MutexObject& mutex = GetMutexObject();

    HID_SHELL_RESULT_DO(mutex.Lock());

    *pOutHandle = reinterpret_cast<HidShellHandle>(&mutex);

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult ReleaseHidShellSharedMemoryExclusiveRight(
    HidShellHandle handle) NN_NOEXCEPT
{
    MutexObject& mutex = GetMutexObject();

    HID_SHELL_RESULT_THROW_UNLESS(
        &mutex == reinterpret_cast<MutexObject*>(handle),
        HidShellResult::InvalidHandle);

    mutex.Unlock();

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult CreateHidShellSharedMemory(
    HidShellHandle handle, bool* pOutExists, uint32_t size) NN_NOEXCEPT
{
    const MutexObject& mutex = GetMutexObject();

    HID_SHELL_RESULT_THROW_UNLESS(
        &mutex == reinterpret_cast<MutexObject*>(handle),
        HidShellResult::InvalidHandle);

    HID_SHELL_RESULT_DO(
        GetFileMappingObject().Create(pOutExists, static_cast<DWORD>(size)));

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult GetHidShellSharedMemoryAddress(
    HidShellHandle handle, void** pOutAddress) NN_NOEXCEPT
{
    HID_SHELL_RESULT_THROW_UNLESS(
        pOutAddress != nullptr, HidShellResult::NullAddressPointer);

    const MutexObject& mutex = GetMutexObject();

    HID_SHELL_RESULT_THROW_UNLESS(
        &mutex == reinterpret_cast<MutexObject*>(handle),
        HidShellResult::InvalidHandle);

    *pOutAddress = GetFileMappingObject().GetAddress();

    HID_SHELL_RESULT_SUCCESS;
}

namespace {

MutexObject::MutexObject() NN_NOEXCEPT
{
    m_Handle = ::CreateMutexA(NULL, FALSE, MutexObjectName);
}

HidShellResult MutexObject::Lock() NN_NOEXCEPT
{
    HID_SHELL_RESULT_THROW_UNLESS(
        m_Handle != nullptr, HidShellResult::FailedToCreateMutex);

    ::WaitForSingleObject(m_Handle, INFINITE);

    HID_SHELL_RESULT_SUCCESS;
}

void MutexObject::Unlock() NN_NOEXCEPT
{
    assert(m_Handle != nullptr);

    ::ReleaseMutex(m_Handle);
}

FileMappingObject::FileMappingObject() NN_NOEXCEPT
    : m_Handle(nullptr)
    , m_Address(nullptr)
{
}

HidShellResult FileMappingObject::Create(
    bool* pOutExists, DWORD size) NN_NOEXCEPT
{
    HID_SHELL_RESULT_THROW_UNLESS(
        pOutExists != nullptr, HidShellResult::NullExistenceFlagPointer);

    HID_SHELL_RESULT_THROW_UNLESS(
        m_Handle == nullptr, HidShellResult::FailedToCreateSharedMemory);

    const HANDLE handle = ::CreateFileMappingA(
        INVALID_HANDLE_VALUE, NULL, PAGE_READWRITE, 0, size,
        FileMappingObjectName);

    HID_SHELL_RESULT_THROW_UNLESS(
        handle != nullptr, HidShellResult::FailedToCreateSharedMemory);

    const bool exists = (::GetLastError() == ERROR_ALREADY_EXISTS);

    const LPVOID address = ::MapViewOfFile(handle, FILE_MAP_WRITE, 0, 0, size);

    if (address == 0)
    {
        ::CloseHandle(handle);

        HID_SHELL_RESULT_THROW(HidShellResult::FailedToCreateSharedMemory);
    }

    m_Address = address;

    m_Handle = handle;

    *pOutExists = exists;

    HID_SHELL_RESULT_SUCCESS;
}

MutexObject& GetMutexObject() NN_NOEXCEPT
{
    static MutexObject s_MutexObject;

    return s_MutexObject;
}

FileMappingObject& GetFileMappingObject() NN_NOEXCEPT
{
    static FileMappingObject s_FileMappingObject;

    return s_FileMappingObject;
}

} // namespace
