﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"
#include "HidShellLibrary.h"
#include "HidShellLibrary_IoPort.h"
#include "HidShellLibrary_SharedMemory.h"
#include "HidShellLibrary_Utility.h"

namespace {

//!< IO ポートの数です。
const size_t IoPortCount = 32;

//!< IO ポートの不正な ID です。
const int64_t InvalidIoPortId = ::std::numeric_limits<int64_t>::min();

//!< IO ポートのモードを表す列挙体です。
enum class IoPortMapMode : int32_t
{
    None,       //!< 無し
    AutoPilot,  //!< 自動操作中
};

//!< 撮影ボタンのアドレスマップを表す構造体です。
struct CaptureButtonMap final
{
    IoPortMapMode mode;                 //!< IO ポートのモード
    int32_t _padding;
    HidShellCaptureButtonState state;   //!< 撮影ボタンの入力状態
};

//!< デバッグパッドのアドレスマップを表す構造体です。
struct DebugPadMap final
{
    IoPortMapMode mode;             //!< IO ポートのモード
    int32_t _padding;
    HidShellDebugPadState state;    //!< デバッグパッドの入力状態
};

//!< ホームボタンのアドレスマップを表す構造体です。
struct HomeButtonMap final
{
    IoPortMapMode mode;             //!< IO ポートのモード
    int32_t _padding;
    HidShellHomeButtonState state;  //!< ホームボタンの入力状態
};

//!< タッチパネルのアドレスマップを表す構造体です。
struct TouchScreenMap final
{
    IoPortMapMode mode;             //!< IO ポートのモード
    int32_t _padding;
    HidShellTouchScreenState state; //!< タッチパネルの入力状態
};

//!< 基本的な構成を持つ Xpad のアドレスマップを表す構造体です。
struct BasicXpadMap final
{
    IoPortMapMode mode;             //!< 入出力ポートのモード
    HidShellBasicXpadState state;   //!< 基本的な構成を持つ Xpad の入力状態
};

/**
 * @brief       基本的な構成を持つ AbstractedPad の入力状態を表す構造体です。
 */
struct AbstractedPadMap final
{
    IoPortMapMode mode;                 //!< 入出力ポートのモード
    HidShellAbstractedPadState state;   //!< 基本的な構成を持つ Xpad の入力状態
};

//!< IO ポートのアドレスマップを表す構造体です。
struct IoPortMap final
{
    BasicXpadMap basicXpadMaps[
        HidShellBasicXpadCountMax];         //!< 基本的な構成を持つ Xpad
    TouchScreenMap touchScreenMap;          //!< タッチパネル
    HomeButtonMap homeButtonMap;            //!< ホームボタン
    CaptureButtonMap captureButtonMap;      //!< 撮影ボタン
    DebugPadMap debugPadMap;                //!< デバッグパッド
    AbstractedPadMap abstractedPadMap[
        HidShellAbstractedPadCountMax];     //!< abstractedPad
};

//!< IO ポート全体のアドレスマップを表す構造体です。
struct IoPortWholeMap final
{
    int64_t ioPortIds[IoPortCount];
    IoPortMap ioPortMaps[2 * (IoPortCount + 1)];
};

//!< ポート名のハッシュ値を返します。
int64_t GetDjbHash(const HidShellPortName& name) NN_NOEXCEPT;

//!< IO ポート全体を初期化します。
void InitializeIoPortWholeMap(IoPortWholeMap* pMap) NN_NOEXCEPT;

//!< IO ポートのアドレスマップを取得します。
HidShellResult GetIoPortMap(
    HidShellHandle handle,
    IoPortMap** ppOutMap,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT;

} // namespace

HidShellResult AquireHidShellIoPortExclusiveRight(
    HidShellHandle* pOutHandle) NN_NOEXCEPT
{
    HID_SHELL_RESULT_DO(AquireHidShellSharedMemoryExclusiveRight(pOutHandle));

    assert(pOutHandle != nullptr);

    static bool s_IsSharedMemoryCreated = false;

    if (!s_IsSharedMemoryCreated)
    {
        bool exists = true;

        HID_SHELL_RESULT_DO(
            CreateHidShellSharedMemory(
                *pOutHandle, &exists,
                static_cast<uint32_t>(sizeof(IoPortWholeMap))));

        if (!exists)
        {
            void* address = nullptr;

            HID_SHELL_RESULT_DO(
                GetHidShellSharedMemoryAddress(*pOutHandle, &address));

            InitializeIoPortWholeMap(
                reinterpret_cast<IoPortWholeMap*>(address));
        }

        s_IsSharedMemoryCreated = true;
    }

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult ReleaseHidShellIoPortExclusiveRight(
    HidShellHandle handle) NN_NOEXCEPT
{
    HID_SHELL_RESULT_DO(ReleaseHidShellSharedMemoryExclusiveRight(handle));

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult AttachHidShellIoPort(
    HidShellHandle handle, const HidShellPortName& name) NN_NOEXCEPT
{
    void* address = nullptr;

    HID_SHELL_RESULT_DO(GetHidShellSharedMemoryAddress(handle, &address));

    assert(address != nullptr);

    IoPortWholeMap& map = *reinterpret_cast<IoPortWholeMap*>(address);

    const int64_t hash = GetDjbHash(name);

    for (uint32_t i = 0; i < NN_ARRAY_SIZE(map.ioPortIds); ++i)
    {
        int64_t& ioPortId = map.ioPortIds[i];

        if (ioPortId == InvalidIoPortId)
        {
            ioPortId = hash;

            ::std::memset(
                &map.ioPortMaps[2 * (i + 1)], 0, 2 * sizeof(IoPortMap));

            HID_SHELL_RESULT_SUCCESS;
        }
    }

    HID_SHELL_RESULT_THROW(HidShellResult::NoAvailablePort);
}

HidShellResult DetachHidShellIoPort(
    HidShellHandle handle, const HidShellPortName& name) NN_NOEXCEPT
{
    void* address = nullptr;

    HID_SHELL_RESULT_DO(GetHidShellSharedMemoryAddress(handle, &address));

    assert(address != nullptr);

    IoPortWholeMap& map = *reinterpret_cast<IoPortWholeMap*>(address);

    const int64_t hash = GetDjbHash(name);

    for (uint32_t i = 0; i < NN_ARRAY_SIZE(map.ioPortIds); ++i)
    {
        int64_t& ioPortId = map.ioPortIds[i];

        if (ioPortId == hash)
        {
            ioPortId = InvalidIoPortId;

            ::std::memset(
                &map.ioPortMaps[2 * (i + 1)], 0, 2 * sizeof(IoPortMap));
        }
    }

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult GetHidShellIoPort(
    HidShellHandle handle,
    uint32_t* pOutPort, const HidShellPortName& name) NN_NOEXCEPT
{
    HID_SHELL_RESULT_THROW_UNLESS(
        pOutPort != nullptr, HidShellResult::NullPortPointer);

    void* address = nullptr;

    HID_SHELL_RESULT_DO(GetHidShellSharedMemoryAddress(handle, &address));

    assert(address != nullptr);

    IoPortWholeMap& map = *reinterpret_cast<IoPortWholeMap*>(address);

    const int64_t hash = GetDjbHash(name);

    for (uint32_t i = 0; i < NN_ARRAY_SIZE(map.ioPortIds); ++i)
    {
        if (map.ioPortIds[i] == hash)
        {
            *pOutPort = i + 1;

            HID_SHELL_RESULT_SUCCESS;
        }
    }

    HID_SHELL_RESULT_THROW(HidShellResult::PortNotFound);
}

HidShellResult GetHidShellIoPortList(
    HidShellHandle handle, HidShellPortList* pOutList) NN_NOEXCEPT
{
    HID_SHELL_RESULT_THROW_UNLESS(
        pOutList != nullptr, HidShellResult::NullPortListPointer);

    HidShellPortList& list = *pOutList;

    void* address = nullptr;

    HID_SHELL_RESULT_DO(GetHidShellSharedMemoryAddress(handle, &address));

    assert(address != nullptr);

    IoPortWholeMap& map = *reinterpret_cast<IoPortWholeMap*>(address);

    // デフォルト IO ポート
    list.count = 1;
    list.ports[0] = 0;

    for (const int64_t& ioPortId : map.ioPortIds)
    {
        if (ioPortId != InvalidIoPortId)
        {
            list.ports[list.count++] =
                static_cast<uint32_t>(&ioPortId - map.ioPortIds) + 1;
        }
    }

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult GetHidShellIoPortCaptureButtonState(
    HidShellHandle handle,
    HidShellCaptureButtonState* pOutValue,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    HID_SHELL_RESULT_THROW_UNLESS(
        pOutValue != nullptr, HidShellResult::NullStatePointer);

    IoPortMap* pMap = nullptr;

    HID_SHELL_RESULT_DO(GetIoPortMap(handle, &pMap, port, direction));

    assert(pMap != nullptr);

    const CaptureButtonMap& map = pMap->captureButtonMap;

    if (direction == HidShellPortDirection::In)
    {
        HID_SHELL_RESULT_THROW_UNLESS(
            map.mode == IoPortMapMode::AutoPilot, HidShellResult::StateNotSet);
    }

    *pOutValue = map.state;

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult SetHidShellIoPortCaptureButtonState(
    HidShellHandle handle,
    const HidShellCaptureButtonState& value,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    IoPortMap* pMap = nullptr;

    HID_SHELL_RESULT_DO(GetIoPortMap(handle, &pMap, port, direction));

    assert(pMap != nullptr);

    CaptureButtonMap& map = pMap->captureButtonMap;

    if (direction == HidShellPortDirection::In)
    {
        map.mode = IoPortMapMode::AutoPilot;
    }

    map.state = value;

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult GetHidShellIoPortDebugPadState(
    HidShellHandle handle,
    HidShellDebugPadState* pOutValue,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    HID_SHELL_RESULT_THROW_UNLESS(
        pOutValue != nullptr, HidShellResult::NullStatePointer);

    IoPortMap* pMap = nullptr;

    HID_SHELL_RESULT_DO(GetIoPortMap(handle, &pMap, port, direction));

    assert(pMap != nullptr);

    const DebugPadMap& map = pMap->debugPadMap;

    if (direction == HidShellPortDirection::In)
    {
        HID_SHELL_RESULT_THROW_UNLESS(
            map.mode == IoPortMapMode::AutoPilot, HidShellResult::StateNotSet);
    }

    *pOutValue = map.state;

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult SetHidShellIoPortDebugPadState(
    HidShellHandle handle,
    const HidShellDebugPadState& value,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    IoPortMap* pMap = nullptr;

    HID_SHELL_RESULT_DO(GetIoPortMap(handle, &pMap, port, direction));

    assert(pMap != nullptr);

    DebugPadMap& map = pMap->debugPadMap;

    if (direction == HidShellPortDirection::In)
    {
        map.mode = IoPortMapMode::AutoPilot;
    }

    map.state = value;

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult GetHidShellIoPortHomeButtonState(
    HidShellHandle handle,
    HidShellHomeButtonState* pOutValue,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    HID_SHELL_RESULT_THROW_UNLESS(
        pOutValue != nullptr, HidShellResult::NullStatePointer);

    IoPortMap* pMap = nullptr;

    HID_SHELL_RESULT_DO(GetIoPortMap(handle, &pMap, port, direction));

    assert(pMap != nullptr);

    const HomeButtonMap& map = pMap->homeButtonMap;

    if (direction == HidShellPortDirection::In)
    {
        HID_SHELL_RESULT_THROW_UNLESS(
            map.mode == IoPortMapMode::AutoPilot, HidShellResult::StateNotSet);
    }

    *pOutValue = map.state;

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult SetHidShellIoPortHomeButtonState(
    HidShellHandle handle,
    const HidShellHomeButtonState& value,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    IoPortMap* pMap = nullptr;

    HID_SHELL_RESULT_DO(GetIoPortMap(handle, &pMap, port, direction));

    assert(pMap != nullptr);

    HomeButtonMap& map = pMap->homeButtonMap;

    if (direction == HidShellPortDirection::In)
    {
        map.mode = IoPortMapMode::AutoPilot;
    }

    map.state = value;

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult GetHidShellIoPortTouchScreenState(
    HidShellHandle handle,
    HidShellTouchScreenState* pOutValue,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    HID_SHELL_RESULT_THROW_UNLESS(
        pOutValue != nullptr, HidShellResult::NullStatePointer);

    IoPortMap* pMap = nullptr;

    HID_SHELL_RESULT_DO(GetIoPortMap(handle, &pMap, port, direction));

    assert(pMap != nullptr);

    const TouchScreenMap& map = pMap->touchScreenMap;

    if (direction == HidShellPortDirection::In)
    {
        HID_SHELL_RESULT_THROW_UNLESS(
            map.mode == IoPortMapMode::AutoPilot, HidShellResult::StateNotSet);
    }

    *pOutValue = map.state;

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult SetHidShellIoPortTouchScreenState(
    HidShellHandle handle,
    const HidShellTouchScreenState& value,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    IoPortMap* pMap = nullptr;

    HID_SHELL_RESULT_DO(GetIoPortMap(handle, &pMap, port, direction));

    assert(pMap != nullptr);

    TouchScreenMap& map = pMap->touchScreenMap;

    if (direction == HidShellPortDirection::In)
    {
        map.mode = IoPortMapMode::AutoPilot;
    }

    map.state = value;

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult GetHidShellIoPortBasicXpadState(
    HidShellHandle handle,
    HidShellBasicXpadState* pOutValue, uint32_t id,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    HID_SHELL_RESULT_THROW_UNLESS(
        pOutValue != nullptr, HidShellResult::NullStatePointer);

    IoPortMap* pMap = nullptr;

    HID_SHELL_RESULT_DO(GetIoPortMap(handle, &pMap, port, direction));

    assert(pMap != nullptr);

    HID_SHELL_RESULT_THROW_UNLESS(
        id < NN_ARRAY_SIZE(pMap->basicXpadMaps),
        HidShellResult::InvalidBasicXpadId);

    const BasicXpadMap& map = pMap->basicXpadMaps[id];

    if (direction == HidShellPortDirection::In)
    {
        HID_SHELL_RESULT_THROW_UNLESS(
            map.mode == IoPortMapMode::AutoPilot, HidShellResult::StateNotSet);
    }

    *pOutValue = map.state;

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult SetHidShellIoPortBasicXpadState(
    HidShellHandle handle,
    const HidShellBasicXpadState& value, uint32_t id,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    IoPortMap* pMap = nullptr;

    HID_SHELL_RESULT_DO(GetIoPortMap(handle, &pMap, port, direction));

    assert(pMap != nullptr);

    HID_SHELL_RESULT_THROW_UNLESS(
        id < NN_ARRAY_SIZE(pMap->basicXpadMaps),
        HidShellResult::InvalidBasicXpadId);

    BasicXpadMap& map = pMap->basicXpadMaps[id];

    if (direction == HidShellPortDirection::In)
    {
        map.mode = IoPortMapMode::AutoPilot;
    }

    map.state = value;

    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult GetHidShellIoPortAbstractedPadState(
    HidShellHandle handle,
    HidShellAbstractedPadState* pOutValue, uint32_t id,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    HID_SHELL_RESULT_THROW_UNLESS(
        pOutValue != nullptr, HidShellResult::NullStatePointer);

    IoPortMap* pMap = nullptr;

    HID_SHELL_RESULT_DO(GetIoPortMap(handle, &pMap, port, direction));

    assert(pMap != nullptr);

    HID_SHELL_RESULT_THROW_UNLESS(
        id < NN_ARRAY_SIZE(pMap->abstractedPadMap),
        HidShellResult::InvalidAbstractedPadId);

    AbstractedPadMap& map = pMap->abstractedPadMap[id];
    if (direction == HidShellPortDirection::In)
    {
        HID_SHELL_RESULT_THROW_UNLESS(
            map.mode == IoPortMapMode::AutoPilot, HidShellResult::StateNotSet);
    }

    *pOutValue = map.state;
    HID_SHELL_RESULT_SUCCESS;
}

HidShellResult SetHidShellIoPortAbstractedPadState(
    HidShellHandle handle,
    const HidShellAbstractedPadState& value, uint32_t id,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    IoPortMap* pMap = nullptr;

    HID_SHELL_RESULT_DO(GetIoPortMap(handle, &pMap, port, direction));

    assert(pMap != nullptr);

    HID_SHELL_RESULT_THROW_UNLESS(
        id < NN_ARRAY_SIZE(pMap->abstractedPadMap),
        HidShellResult::InvalidAbstractedPadId);

    AbstractedPadMap& map = pMap->abstractedPadMap[id];

    if (direction == HidShellPortDirection::In)
    {
        map.mode = IoPortMapMode::AutoPilot;
    }

    map.state   = value;

    HID_SHELL_RESULT_SUCCESS;
}

namespace {

int64_t GetDjbHash(const HidShellPortName& name) NN_NOEXCEPT
{
    int64_t hashes[] = { 5381, 5381 };

    for (size_t i = 0; i < NN_ARRAY_SIZE(name.string); ++i)
    {
        const char c = name.string[i];

        if (c == '\0')
        {
            break;
        }
        else
        {
            int64_t& hash = hashes[i % NN_ARRAY_SIZE(hashes)];

            hash = ((hash << 5) + hash) ^ c;
        }
    }

    return hashes[0] + (hashes[1] * 1566083941);
}

void InitializeIoPortWholeMap(IoPortWholeMap* pMap) NN_NOEXCEPT
{
    assert(pMap != nullptr);

    IoPortWholeMap& map = *pMap;

    ::std::memset(&map, 0, sizeof(map));

    for (int64_t& ioPortId : map.ioPortIds)
    {
        ioPortId = InvalidIoPortId;
    }
}

HidShellResult GetIoPortMap(
    HidShellHandle handle,
    IoPortMap** ppOutMap,
    uint32_t port, HidShellPortDirection direction) NN_NOEXCEPT
{
    assert(ppOutMap != nullptr);

    void* address = nullptr;

    HID_SHELL_RESULT_DO(GetHidShellSharedMemoryAddress(handle, &address));

    assert(address != nullptr);

    IoPortWholeMap& map = *reinterpret_cast<IoPortWholeMap*>(address);

    HID_SHELL_RESULT_THROW_UNLESS(
        port <= NN_ARRAY_SIZE(map.ioPortIds), HidShellResult::InvalidPort);

    HID_SHELL_RESULT_THROW_UNLESS(
        port == 0 || map.ioPortIds[port - 1] != InvalidIoPortId,
        HidShellResult::PortNotFound);

    uint32_t offset = 2 * port;

    switch (direction)
    {
    case HidShellPortDirection::In:
    case HidShellPortDirection::Out:
        offset += static_cast<uint32_t>(direction);
        break;

    default:
        HID_SHELL_RESULT_THROW(HidShellResult::InvalidPortDirection);
    }

    HID_SHELL_RESULT_THROW_UNLESS(
        offset < NN_ARRAY_SIZE(map.ioPortMaps), HidShellResult::InvalidPort);

    *ppOutMap = &map.ioPortMaps[offset];

    HID_SHELL_RESULT_SUCCESS;
}

} // namespace
