﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace HidShell.Devices
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.Diagnostics;
    using System.Linq;
    using System.Runtime.InteropServices;
    using System.Threading;
    using System.Threading.Tasks;
    using Properties;

    /// <summary>
    /// DebugPad を扱うクラスです。
    /// </summary>
    [PartCreationPolicy(CreationPolicy.NonShared)]
    internal sealed class DebugPad : IDevice
    {
        private const string DeviceType = "DebugPad";

        private const int PlayerCountMax = 1;

        private static readonly TimeSpan Interval =
            TimeSpan.FromMilliseconds(7);

        private readonly List<string> deviceIds = new List<string>();

        private readonly List<EventDumper> eventDumpers;

        private readonly List<EventSender> eventSenders;

        private uint port = PortAccessor.DefaultPort;

        private Logger logger = null;

        /// <summary>
        /// DebugPad クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal DebugPad()
        {
            this.eventDumpers = new List<EventDumper>();

            this.eventSenders = new List<EventSender>();

            foreach (int i in Enumerable.Range(0, PlayerCountMax))
            {
                string deviceId = i.ToString();

                this.deviceIds.Add(deviceId);

                this.eventDumpers.Add(
                    new EventDumper(deviceId, GetDebugPadEventDumper(i)));

                this.eventSenders.AddRange(new[]
                {
                    new EventSender(
                        deviceId, EventType.Connected,
                        GetDebugPadConnectedEventSender(i)),
                    new EventSender(
                        deviceId, EventType.Disconnected,
                        GetDebugPadDisconnectedEventSender(i)),
                    new EventSender(
                        deviceId, EventType.ButtonDown,
                        GetDebugPadKeyDownEventSender(i)),
                    new EventSender(
                        deviceId, EventType.ButtonUp,
                        GetDebugPadKeyUpEventSender(i)),
                    new EventSender(
                        deviceId, EventType.StickMove,
                        GetDebugPadStickMoveEventSender(i)),
                });
            }
        }

        [Flags]
        private enum DebugPadAttributes : uint
        {
            None = 0,
            IsConnected = 1u,
        }

        [Flags]
        private enum DebugPadButtons : uint
        {
            None = 0,
            A = 1u,
            B = 1u << 1,
            X = 1u << 2,
            Y = 1u << 3,
            L = 1u << 4,
            R = 1u << 5,
            ZL = 1u << 6,
            ZR = 1u << 7,
            Start = 1u << 8,
            Select = 1u << 9,
            Left = 1u << 10,
            Up = 1u << 11,
            Right = 1u << 12,
            Down = 1u << 13,
        }

        /// <summary>
        /// ポートを設定します。
        /// </summary>
        /// <param name="port">ポートです。</param>
        public void SetPort(uint port)
        {
            this.port = port;
        }

        /// <summary>
        /// ロガーを設定します。
        /// </summary>
        /// <param name="logger">ロガーです。</param>
        public void SetLogger(Logger logger)
        {
            this.logger = logger;
        }

        /// <summary>
        /// 指定されたデバイス種別をサポートするか否かを表す値を返します。
        /// </summary>
        /// <param name="deviceType">デバイス種別を表す文字列です。</param>
        /// <returns>指定されたデバイス種別をサポートするか否かを表す値です。</returns>
        public bool Supports(string deviceType)
        {
            return deviceType == DeviceType;
        }

        /// <summary>
        /// デバイス識別子で指定されたデバイスについてイベントをダンプします。
        /// </summary>
        /// <param name="deviceId">デバイス識別子です。</param>
        public void DumpEvent(string deviceId)
        {
            var dumpers = deviceId == null
                ? this.eventDumpers
                : this.eventDumpers.Where(x => x.SupportsDeviceId(deviceId));

            if (deviceId != null && dumpers.Count() == 0)
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorDeviceIdUnexpected, deviceId));
            }

            Parallel.ForEach(dumpers, dumper => dumper.Invoke(this));
        }

        /// <summary>
        /// デバイス識別子で指定されたデバイスに対してイベントを発行します。
        /// </summary>
        /// <param name="deviceId">デバイス識別子です。</param>
        /// <param name="eventType">イベント種別です。</param>
        /// <param name="eventArgs">イベント引数です。</param>
        public void SendEvent(
            string deviceId, string eventType, string[] eventArgs)
        {
            var senders =
                this.eventSenders.Where(x => x.SupportsDeviceId(deviceId));

            if (senders.Count() == 0)
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorDeviceIdUnexpected, deviceId));
            }

            var sender =
                senders.FirstOrDefault(x => x.SupportsEventType(eventType));

            if (sender == null)
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorEventTypeUnexpected, eventType));
            }

            sender.Invoke(this, eventArgs);
        }

        private static void GetDebugPadState(
            IntPtr handle, out DebugPadState outState, uint port)
        {
            var result = (HidShellResult)
                Native.GetHidShellDebugPadState(
                    handle, out outState, port, PortAccessor.Out);

            if (result != HidShellResult.Success)
            {
                throw new HidShellException(result);
            }
        }

        private static void GetDebugPadAutoPilotState(
            IntPtr handle, out DebugPadState outState, uint port)
        {
            var result = (HidShellResult)
                Native.GetHidShellDebugPadState(
                    handle, out outState, port, PortAccessor.In);

            switch (result)
            {
                case HidShellResult.Success:
                    return;

                case HidShellResult.StateNotSet:
                    outState = new DebugPadState();
                    return;

                default:
                    throw new HidShellException(result);
            }
        }

        private static void SetDebugPadAutoPilotState(
            IntPtr handle, ref DebugPadState state, uint port)
        {
            var result = (HidShellResult)
                Native.SetHidShellDebugPadState(
                    handle, ref state, port, PortAccessor.In);

            if (result != HidShellResult.Success)
            {
                throw new HidShellException(result);
            }
        }

        private static string EncodeCoordinate(int value)
        {
            if (value >= 0)
            {
                return string.Format("P{0:D5}", value);
            }
            else
            {
                return string.Format("M{0:D5}", -value);
            }
        }

        private static string DecodeCoordinate(string value)
        {
            if (value.StartsWith("P"))
            {
                return value.Substring(1);
            }

            if (value.StartsWith("M"))
            {
                return "-" + value.Substring(1);
            }

            return value;
        }

        private static Action<DebugPad> GetDebugPadEventDumper(int index)
        {
            return (DebugPad that) => {
                DumpDebugPadEvent(
                    that.port, that.logger, that.deviceIds[index]);
            };
        }

        private static void DumpDebugPadEvent(
            uint port, Logger logger, string deviceId)
        {
            var buttons =
                Enum.GetValues(typeof(DebugPadButtons))
                    .Cast<DebugPadButtons>();

            var stopwatch = new Stopwatch();

            var nextState = new DebugPadState();
            var lastState = new DebugPadState();

            using (var accessor = new PortAccessor())
            {
                GetDebugPadState(accessor.Handle, out lastState, port);
            }

            while (true)
            {
                stopwatch.Start();

                using (var accessor = new PortAccessor())
                {
                    GetDebugPadState(accessor.Handle, out nextState, port);
                }

                if (nextState.Attributes != lastState.Attributes)
                {
                    var attribute = DebugPadAttributes.IsConnected;

                    if ((attribute & lastState.Attributes) == 0 &&
                        (attribute & nextState.Attributes) != 0)
                    {
                        logger.WriteLine(
                            "{0,-16}{1,-16}{2}",
                            DeviceType, deviceId, EventType.Connected);
                    }

                    if ((attribute & lastState.Attributes) != 0 &&
                        (attribute & nextState.Attributes) == 0)
                    {
                        logger.WriteLine(
                            "{0,-16}{1,-16}{2}",
                            DeviceType, deviceId, EventType.Disconnected);
                    }
                }

                foreach (var button in buttons)
                {
                    if ((button & lastState.Buttons) == 0 &&
                        (button & nextState.Buttons) != 0)
                    {
                        logger.WriteLine(
                            "{0,-16}{1,-16}{2} {3}",
                            DeviceType, deviceId,
                            EventType.ButtonDown, button);
                    }

                    if ((button & lastState.Buttons) != 0 &&
                        (button & nextState.Buttons) == 0)
                    {
                        logger.WriteLine(
                            "{0,-16}{1,-16}{2} {3}",
                            DeviceType, deviceId,
                            EventType.ButtonUp, button);
                    }
                }

                if (nextState.StickL.X != lastState.StickL.X ||
                    nextState.StickL.Y != lastState.StickL.Y)
                {
                    logger.WriteLine(
                        "{0,-16}{1,-16}{2} {3} {4} {5}",
                        DeviceType, deviceId,
                        EventType.StickMove, StickType.Left,
                        EncodeCoordinate(nextState.StickL.X),
                        EncodeCoordinate(nextState.StickL.Y));
                }

                if (nextState.StickR.X != lastState.StickR.X ||
                    nextState.StickR.Y != lastState.StickR.Y)
                {
                    logger.WriteLine(
                        "{0,-16}{1,-16}{2} {3} {4} {5}",
                        DeviceType, deviceId,
                        EventType.StickMove, StickType.Right,
                        EncodeCoordinate(nextState.StickR.X),
                        EncodeCoordinate(nextState.StickR.Y));
                }

                lastState = nextState;

                stopwatch.Stop();

                if (stopwatch.Elapsed < Interval)
                {
                    Thread.Sleep(Interval - stopwatch.Elapsed);
                }

                stopwatch.Reset();
            }
        }

        private static Action<DebugPad, string[]> GetDebugPadConnectedEventSender(
            int index)
        {
            var state = new DebugPadState();

            state.Attributes = DebugPadAttributes.IsConnected;

            return (DebugPad that, string[] args) =>
            {
                if (args.Count() != 0)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.Connected));
                }

                using (var accessor = new PortAccessor())
                {
                    SetDebugPadAutoPilotState(
                        accessor.Handle, ref state, that.port);
                }
            };
        }

        private static Action<DebugPad, string[]> GetDebugPadDisconnectedEventSender(
            int index)
        {
            var state = new DebugPadState();

            return (DebugPad that, string[] args) =>
            {
                if (args.Count() != 0)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.Disconnected));
                }

                using (var accessor = new PortAccessor())
                {
                    SetDebugPadAutoPilotState(
                        accessor.Handle, ref state, that.port);
                }
            };
        }

        private static Action<DebugPad, string[]> GetDebugPadKeyDownEventSender(
            int index)
        {
            var buttons = Enum.GetValues(
                typeof(DebugPadButtons)).Cast<DebugPadButtons>().ToArray();

            var state = new DebugPadState();

            return (DebugPad that, string[] args) =>
            {
                if (args.Count() != 1)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.ButtonDown));
                }

                DebugPadButtons button =
                    buttons.FirstOrDefault(x => x.ToString() == args[0]);

                if (button == DebugPadButtons.None)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorButtonUnexpectedName, args[0]));
                }

                using (var accessor = new PortAccessor())
                {
                    GetDebugPadAutoPilotState(
                        accessor.Handle, out state, that.port);

                    state.Attributes = DebugPadAttributes.IsConnected;

                    state.Buttons |= button;

                    SetDebugPadAutoPilotState(
                        accessor.Handle, ref state, that.port);
                }
            };
        }

        private static Action<DebugPad, string[]> GetDebugPadKeyUpEventSender(
            int index)
        {
            var buttons = Enum.GetValues(
                typeof(DebugPadButtons)).Cast<DebugPadButtons>().ToArray();

            var state = new DebugPadState();

            return (DebugPad that, string[] args) =>
            {
                if (args.Count() != 1)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.ButtonUp));
                }

                DebugPadButtons button =
                    buttons.FirstOrDefault(x => x.ToString() == args[0]);

                if (button == DebugPadButtons.None)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorButtonUnexpectedName, args[0]));
                }

                using (var accessor = new PortAccessor())
                {
                    GetDebugPadAutoPilotState(
                        accessor.Handle, out state, that.port);

                    state.Attributes = DebugPadAttributes.IsConnected;

                    state.Buttons &= ~button;

                    SetDebugPadAutoPilotState(
                        accessor.Handle, ref state, that.port);
                }
            };
        }

        private static Action<DebugPad, string[]> GetDebugPadStickMoveEventSender(
            int index)
        {
            var state = new DebugPadState();

            return (DebugPad that, string[] args) =>
            {
                if (args.Count() != 3)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.StickMove));
                }

                var stickType = args[0];

                if (stickType != StickType.Left &&
                    stickType != StickType.Right)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorStickUnexpectedType, stickType));
                }

                var x = 0;

                if (!int.TryParse(DecodeCoordinate(args[1]), out x))
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorCoordinateInvalidX, args[1]));
                }

                var y = 0;

                if (!int.TryParse(DecodeCoordinate(args[2]), out y))
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorCoordinateInvalidY, args[2]));
                }

                using (var accessor = new PortAccessor())
                {
                    GetDebugPadAutoPilotState(
                        accessor.Handle, out state, that.port);

                    state.Attributes = DebugPadAttributes.IsConnected;

                    if (stickType == StickType.Left)
                    {
                        state.StickL.X = x;
                        state.StickL.Y = y;
                    }
                    else if (stickType == StickType.Right)
                    {
                        state.StickR.X = x;
                        state.StickR.Y = y;
                    }

                    SetDebugPadAutoPilotState(
                        accessor.Handle, ref state, that.port);
                }
            };
        }

        private struct DebugPadStickState
        {
            internal int X;
            internal int Y;
        }

        private struct DebugPadState
        {
            internal DebugPadAttributes Attributes;
            internal DebugPadButtons Buttons;
            internal DebugPadStickState StickL;
            internal DebugPadStickState StickR;
        }

        private static class Native
        {
            [DllImport("HidShellLibrary.dll")]
            internal static extern int GetHidShellDebugPadState(
                IntPtr handle,
                out DebugPadState outState, uint port, uint direction);

            [DllImport("HidShellLibrary.dll")]
            internal static extern int SetHidShellDebugPadState(
                IntPtr handle,
                ref DebugPadState state, uint port, uint direction);
        }

        private static class StickType
        {
            internal const string Left = "L";

            internal const string Right = "R";
        }

        private static class EventType
        {
            internal const string ButtonDown = "ButtonDown";

            internal const string ButtonUp = "ButtonUp";

            internal const string StickMove = "StickMove";

            internal const string Connected = "Connected";

            internal const string Disconnected = "Disconnected";
        }

        private sealed class EventDumper
        {
            private string deviceId;

            private Action<DebugPad> dumper;

            internal EventDumper(string deviceId, Action<DebugPad> dumper)
            {
                this.deviceId = deviceId;

                this.dumper = dumper;
            }

            internal bool SupportsDeviceId(string deviceId)
            {
                return deviceId == this.deviceId;
            }

            internal void Invoke(DebugPad that)
            {
                this.dumper(that);
            }
        }

        private sealed class EventSender
        {
            private string deviceId;

            private string eventType;

            private Action<DebugPad, string[]> sender;

            internal EventSender(
                string deviceId,
                string eventType,
                Action<DebugPad, string[]> sender)
            {
                this.deviceId = deviceId;

                this.eventType = eventType;

                this.sender = sender;
            }

            internal bool SupportsDeviceId(string deviceId)
            {
                return deviceId == this.deviceId;
            }

            internal bool SupportsEventType(string eventType)
            {
                return eventType == this.eventType;
            }

            internal void Invoke(DebugPad that, string[] eventArgs)
            {
                this.sender(that, eventArgs);
            }
        }
    }
}
