﻿//#define ENABLE_LOG //USBの受信ログを表示する場合はコメントアウトを外してください
#define KOTETSU　//KOTETSUで試す場合はコメントアウトを外してください
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Diagnostics;
using HidSharp;

namespace Nintendo.NX.KuinaFirmwareUpdater
{
    class UsbHid
    {

        const int vendorID = 0x057e;
        const int checkID = 0x01;　//左

        static readonly int[] productID = { 0x2009, 0x200E };
        static readonly string[] deviceName = { "ProController", "ChagringGrip" };

        public enum DeviceType
        {
            ProController = 0,
            ChagringGrip,
        }
        /// <summary>
        /// デバイス情報
        /// </summary>
        public class KuinaDevice
        {
            /// <summary>
            /// ID
            /// </summary>
            public int Id { get; private set; } = 0;

            /// <summary>
            /// FWバージョン
            /// </summary>
            public byte[] Version { get; private set; } = { 0x00, 0x00, 0x00 };

            /// <summary>
            /// デバイス種類
            /// </summary>
            public DeviceType Type { get; private set; } = 0;
            public string DeviceTypeStr
            {
                get
                {
                    return (Type == DeviceType.ProController) ? deviceName[0] : deviceName[1];
                }
            }

            private HidDevice hidDevice;
            private HidStream hidStream;
            private static int count = 0;

            public KuinaDevice(HidDevice device, HidStream stream)
            {
                hidDevice = device;
                hidStream = stream;
                hidStream.ReadTimeout = 1000;

                Id = count;
                count++;

                GetVersion();

                Debug.WriteLine("Device{0} Type : {1} Fw Version : {2}.{3}.{4}",
                    Id, Type, Version[0], Version[1], Version[2]);

            }

            public void GotoDfu()
            {
                //Version取得
                byte[] output = new byte[hidDevice.MaxInputReportLength];
                output[0] = 0x82;
                output[1] = 0x01;

                USBSend(hidDevice, hidStream, output);
                Debug.WriteLine("Go To DFU Mode");
            }

            private void GetVersion()
            {
                //Version取得
                byte[] output = new byte[hidDevice.MaxInputReportLength];
                output[0] = 0x80;
                output[1] = 0x07;
                var input = USBSendAndReceive(hidDevice, hidStream, output);
                Version[0] = input[4];
                Version[1] = input[5];
                Version[2] = input[6];

                Type = (input[3] == 0x01) ? DeviceType.ChagringGrip : DeviceType.ProController;
            }
        };


        public static List<KuinaDevice> GetDeviceList()
        {
            HidDeviceLoader loader = new HidDeviceLoader();
            //Thread.Sleep(2000);

            var list = new List<KuinaDevice>();

            for (int pid = 0; pid < productID.Length; pid++)
            {
                //デバイス列挙
                var deviceList = loader.GetDevices(vendorID, productID[pid]).ToArray();
                {
                    Debug.WriteLine("Complete device list ({0} devices):", deviceList.Length);

                    if (deviceList.Length == 0)
                    {
                        Debug.WriteLine("Fail to found {0}.", deviceName[pid]);
                        continue;
                    }

                    int count = 0;
                    foreach (var dev in deviceList)
                    {
                        Debug.WriteLine("Decice {0}:", count++);
                        Debug.WriteLine(dev);
                    }
                    Debug.WriteLine("");
                }

                //接続確認
                Debug.WriteLine("Try to open {0}...", deviceName[pid]);
                HidStream[] hidStream = new HidStream[deviceList.Length];
                for (int i = 0; i < deviceList.Length; i++)
                {
                    var device = deviceList[i];

                    if (device == null)
                    {
                        Debug.WriteLine("Failed to open {0}...", deviceName[pid]);
                        return list;
                    }

                    if (!device.TryOpen(out hidStream[i]))
                    {
                        Debug.WriteLine("Failed to open device.");
                        return list;
                    }
                }

                //左コントローラだけを登録
                for (int i = 0; i < deviceList.Length; i++)
                {
                    var device = deviceList[i];
                    var stream = hidStream[i];
                    stream.ReadTimeout = 1000;

                    //右コントローラか左コントローラを判断するコマンド
                    byte[] output = new byte[device.MaxInputReportLength];
                    output[0] = 0x80;
                    output[1] = 0x08;

                    var input = USBSendAndReceive(device, stream, output);

                    if (input != null && input[0] == 0x81 && input[1] == 0x08 && input[2] == 0x00 && input[3] == checkID)
                    {
                        list.Add(new KuinaDevice(device, stream));
                    }
                }
            }

            return list;
        }


        private static byte[] USBSendAndReceive(HidSharp.HidDevice device, HidStream stream, byte[] output)
        {
            try
            {
                stream.Write(output);
            }
            catch (Exception)
            {
                return null;
            }

            int count = 0;

            var input = USBReceive(device, stream);

            //送ったコマンド以外のレスポンスはスキップする
            while (input[0] != 0x81 || input[1] != output[1])
            {
                input = USBReceive(device, stream);
                count++;

                if (count == 3)
                    return null;
            }

            return input;
        }

        private static void USBSend(HidDevice device, HidStream stream, byte[] output)
        {
            try
            {
                stream.Write(output);
            }
            catch (Exception)
            {
                return;
            }
        }


        private static byte[] USBReceive(HidSharp.HidDevice device, HidStream stream)
        {
            var input = new byte[device.MaxInputReportLength];
            int count = 0;

            try
            {
                count = stream.Read(input, 0, input.Length);
            }
            catch (TimeoutException)
            {
                Debug.WriteLine("Read timed out.");
            }

#if ENABLE_LOG
            if (count > 0)
            {
                Console.Write("* {0} : ", count);
                for (int j = 0; j < count && j < input.Length; j++)
                {
                    Console.Write("{0:X} ", input[j]);
                }
                Console.WriteLine();
            }
#endif
            return input;
        }
    }
}
