﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cassert>
#include <Windows.h>

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>
#include <GL/gl.h>

namespace
{
    struct NvnGraphicsDriverCheckResult
    {
        bool isNvnGraphicsDriverInstalled;
        bool isNvnGpuSupported;
        int nvnDriverApiMajorVersion;
        int nvnDriverApiMinorVersion;
        int nvnSdkApiMajorVersion;
        int nvnSdkApiMinorVersion;
        const uint8_t* GlVendor;
        const uint8_t* GlRenderer;
        const uint8_t* GlVersion;
    };

    LRESULT CALLBACK WindowProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
    {
        switch (uMsg)
        {
        case WM_DESTROY:
            PostQuitMessage(0);
            return 0;
        }

        return DefWindowProcA(hWnd, uMsg, wParam, lParam);
    }
}

extern "C" void main()
{
    HWND hWnd;
    {
        BOOL result;

        HINSTANCE hInstance = GetModuleHandleA(nullptr);
        assert(hInstance);

        WNDCLASSEXA winc;
        result = GetClassInfoExA(hInstance, "MainWindow", &winc);
        assert(!result);

        WNDCLASSEXA newWinc = {
            sizeof(WNDCLASSEXA),
            CS_OWNDC,
            WindowProc,
            0,
            0,
            hInstance,
            nullptr,
            LoadCursor(nullptr, IDC_ARROW),
            static_cast<HBRUSH>(GetStockObject(BLACK_BRUSH)),
            nullptr,
            "MainWindow",
            nullptr
        };

        result = RegisterClassExA(&newWinc);
        assert(result);

        DWORD style = WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX;
        DWORD exStyle = WS_EX_APPWINDOW;
        RECT rect = { 0, 0, 1280, 720 };
        AdjustWindowRectEx(&rect, style, FALSE, exStyle);

        hWnd = CreateWindowExA(
            exStyle,
            "MainWindow",
            "NvnGraphicsDriverChecker",
            style,
            0,
            0,
            rect.right - rect.left,
            rect.bottom - rect.top,
            nullptr,
            nullptr,
            hInstance,
            nullptr);
        assert(hWnd);
    }

    {
        BOOL result;

        HDC hDc = GetDC(hWnd);
        assert(hDc);

        PIXELFORMATDESCRIPTOR pfd =
        {
            sizeof(PIXELFORMATDESCRIPTOR),
            1,
            PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER,
            PFD_TYPE_RGBA,
            24,
            0, 0, 0, 0, 0, 0,
            0,
            0,
            0,
            0, 0, 0, 0,
            0,
            0,
            0,
            PFD_MAIN_PLANE,
            0,
            0, 0, 0
        };
        int pixelFormat = ChoosePixelFormat(hDc, &pfd);
        result = SetPixelFormat(hDc, pixelFormat, &pfd);
        assert(result);

        HGLRC hGlrc = wglCreateContext(hDc);
        assert(hGlrc);

        result = wglMakeCurrent(hDc, hGlrc);
        assert(result);
    }

    NvnGraphicsDriverCheckResult checkResult = {};

    // NVN ヘッダのバージョンを取得
    checkResult.nvnSdkApiMajorVersion = NVN_API_MAJOR_VERSION;
    checkResult.nvnSdkApiMinorVersion = NVN_API_MINOR_VERSION;

    PFNNVNBOOTSTRAPLOADERPROC pBootstrapLoader;
    {
        pBootstrapLoader = reinterpret_cast<PFNNVNBOOTSTRAPLOADERPROC>(wglGetProcAddress("rq34nd2ffz"));

        // NVN ドライバが入っているなら、pBootstrapLoader は 非 nullptr である
        checkResult.isNvnGraphicsDriverInstalled = pBootstrapLoader != nullptr;
    }

    if (pBootstrapLoader != nullptr)
    {
        pfnc_nvnDeviceInitialize = reinterpret_cast<PFNNVNDEVICEINITIALIZEPROC>((*pBootstrapLoader)("nvnDeviceInitialize"));
        pfnc_nvnDeviceGetProcAddress = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>((*pBootstrapLoader)("nvnDeviceGetProcAddress"));

        nvnLoadCProcs(NULL, pfnc_nvnDeviceGetProcAddress);

        {
            int nvnApiMajorVersion, nvnApiMinorVersion;

            nvnDeviceGetInteger(NULL, NVN_DEVICE_INFO_API_MAJOR_VERSION, &nvnApiMajorVersion);
            nvnDeviceGetInteger(NULL, NVN_DEVICE_INFO_API_MINOR_VERSION, &nvnApiMinorVersion);

            // ドライバの NVN API バージョンを取得
            checkResult.nvnDriverApiMajorVersion = nvnApiMajorVersion;
            checkResult.nvnDriverApiMinorVersion = nvnApiMinorVersion;
        }

        {
            NVNdeviceBuilder deviceBuilder;
            nvnDeviceBuilderSetDefaults(&deviceBuilder);
            nvnDeviceBuilderSetFlags(&deviceBuilder, 0);

            NVNdevice device;
            auto deviceInitialized = nvnDeviceInitialize(&device, &deviceBuilder);

            // NVN ドライバのサポート対象の GPU であるならば、nvnDeviceInitialize は成功する。
            checkResult.isNvnGpuSupported = deviceInitialized != 0;
        }
    }

    {
        // e.g. NVIDIA Corporation
        checkResult.GlVendor = glGetString(GL_VENDOR);
        // e.g. GeForce GTX 960/PCIe/SSE2
        checkResult.GlRenderer = glGetString(GL_RENDERER);
        // e.g. 4.5.0 NVIDIA 373.21
        checkResult.GlVersion = glGetString(GL_VERSION);
    }

    std::printf("NvnGraphicsDriverInstalled: %s\n", checkResult.isNvnGraphicsDriverInstalled ? "true" : "false");
    std::printf("NvnGpuSupported: %s\n", checkResult.isNvnGpuSupported ? "true" : "false");
    std::printf("NvnSdkApiVersion: %d.%d\n", checkResult.nvnSdkApiMajorVersion, checkResult.nvnSdkApiMinorVersion);
    std::printf("NvnDriverApiVersion: %d.%d\n", checkResult.nvnDriverApiMajorVersion, checkResult.nvnDriverApiMinorVersion);
    std::printf("GL_VENDOR: %s\n", checkResult.GlVendor);
    std::printf("GL_RENDERER: %s\n", checkResult.GlRenderer);
    std::printf("GL_VERSION: %s\n", checkResult.GlVersion);
}
