﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace InputCapture
{
    using System;
    using System.Collections.Generic;
    using System.Drawing;
    using System.Globalization;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using InputCapture.Communication;
    using InputCapture.Interop;

    /// <summary>
    /// ユーティリティクラスです。
    /// </summary>
    public class Utility
    {
        /// <summary>
        /// 仮想キーの列挙型です。
        /// </summary>
        public enum VirtKey : int
        {
            /// <summary>
            /// Shift
            /// </summary>
            VK_SHIFT = 0x10,

            /// <summary>
            /// Ctrl
            /// </summary>
            VK_CONTROL = 0x11,

            /// <summary>
            /// Alt
            /// </summary>
            VK_MENU = 0x12
        }

        /// <summary>
        /// 画面サイズ（マルチスクリーン全体を包含した全スクリーン矩形）を計算します。
        /// </summary>
        /// <returns>画面サイズを返します。</returns>
        public static Rectangle CalcScreenSize()
        {
            int top = 0;
            int bottom = 0;
            int left = 0;
            int right = 0;

            foreach (Screen screen in Screen.AllScreens)
            {
                if (screen == Screen.AllScreens[0])
                {
                    top = screen.Bounds.Top;
                    bottom = screen.Bounds.Bottom;
                    left = screen.Bounds.Left;
                    right = screen.Bounds.Right;

                    continue;
                }

                top = Math.Min(top, screen.Bounds.Top);
                bottom = Math.Max(bottom, screen.Bounds.Bottom);
                left = Math.Min(left, screen.Bounds.Left);
                right = Math.Max(right, screen.Bounds.Right);
            }

            return new Rectangle(left, top, right - left, bottom - top);
        }

        /// <summary>
        /// マウスホイールを回転したときにスクロールする行数を取得します。
        /// </summary>
        /// <param name="delta">回転量です。</param>
        /// <returns>スクロール行数を返します。</returns>
        public static int GetWheelScrollLines(int delta)
        {
            return delta * SystemInformation.MouseWheelScrollLines / 120;
        }

        /// <summary>
        /// キーが押されているかどうかを取得します。
        /// </summary>
        /// <param name="key">仮想キーです。</param>
        /// <returns>キーが押されている場合に true を返します。</returns>
        public static bool IsKeyPressed(VirtKey key)
        {
            return (User32.GetAsyncKeyState((int)key) >> (8 * sizeof(short) - 1)) != 0;
        }

        /// <summary>
        /// KeyEventArg を生成します。
        /// </summary>
        /// <param name="keyCode">キーコードです。</param>
        /// <param name="keyModifier">修飾キーです。</param>
        /// <returns>生成した KeyEventArg を返します。</returns>
        public static KeyEventArg CreateKeyEventArg(int keyCode, KeyModifier keyModifier)
        {
            KeyEventArg packet = new KeyEventArg();
            packet.modifier = keyModifier;

            KeyCategory category = (KeyCategory)0;
            int indexValue = 0;

            bool isNumPad = false;

            Keys key = (Keys)keyCode;
            indexValue = (int)key;

            if (Keys.A <= key && key <= Keys.Z)
            {
                category = KeyCategory.Alphabet;
            }
            else if (Keys.D0 <= key && key <= Keys.D9)
            {
                category = KeyCategory.Numeric;
            }
            else if (Keys.NumPad0 <= key && key <= Keys.NumPad9)
            {
                category = KeyCategory.Numeric;
                isNumPad = true;
            }
            else if (Keys.F1 <= key && key <= Keys.F24)
            {
                category = KeyCategory.Function;
            }
            else
            {
                switch (key)
                {
                    case Keys.Oemtilde:         // '`'/'~' (106 '@'/'`' )
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.OemMinus:         // '-'/'_' (106 '-'/'=' )
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.Oemplus:          // '='/'+' (106 ';'/'+' )
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.OemPipe:          // '\'/'|' (106 '\'/'|' )
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.OemOpenBrackets:  // '['/'{' (106 '['/'{' )
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.OemCloseBrackets: // ']'/'}' (106 ']'/'}' )
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.OemSemicolon:     // ';'/':' (106 ':'/'*' )
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.OemQuotes:        // [']/["] (106 [^]/[~] )
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.Oemcomma:         // ','/'<' (106 ','/'<' )
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.OemPeriod:        // '.'/'>' (106 '.'/'>' )
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.OemQuestion:      // '/' '?' (106 '/' '?' )
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.OemBackslash:     // 106 '\' '_'
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.Space:
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.Divide:           // 数値パッドの'/'
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.Multiply:         // 数値パッドの'*'
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.Subtract:         // 数値パッドの'-'
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.Add:              // 数値パッドの'+'
                        category = KeyCategory.Symbol;
                        break;
                    case Keys.Decimal:          // 数値パッドの'.'
                        category = KeyCategory.Symbol;
                        break;

                    case Keys.Escape:
                        category = KeyCategory.System;
                        break;
                    case Keys.Back:
                        category = KeyCategory.System;
                        break;
                    case Keys.Tab:  // Tabキーによってフォーカスを受け取れるキーが他にない場合はこのキーイベントが発生するようです。
                        category = KeyCategory.System;
                        break;
                    case Keys.Enter:
                        category = KeyCategory.System;
                        break;
                    case Keys.CapsLock:
                        category = KeyCategory.System;
                        break;
                    case Keys.ShiftKey:
                    case Keys.LShiftKey:
                    case Keys.RShiftKey:
                        category = KeyCategory.System;
                        break;
                    case Keys.ControlKey:
                    case Keys.LControlKey:
                    case Keys.RControlKey:
                        category = KeyCategory.System;
                        break;
                    case Keys.Menu:
                    case Keys.LMenu:
                    case Keys.RMenu:
                        category = KeyCategory.System;
                        break;

                    case Keys.PrintScreen:
                        category = KeyCategory.System;
                        break;
                    case Keys.Scroll:
                        category = KeyCategory.System;
                        break;
                    case Keys.Pause:
                        category = KeyCategory.System;
                        break;
#if false
                case Keys.Cancel:           // Ctrl + Pause
                    category = KeyCategory.System;
                    indexValue = (byte)SystemKeyType.Cancel;
                    break;
#endif
                    case Keys.Insert:
                        category = KeyCategory.System;
                        break;
                    case Keys.Delete:
                        category = KeyCategory.System;
                        break;
                    case Keys.Home:
                        category = KeyCategory.System;
                        break;
                    case Keys.End:
                        category = KeyCategory.System;
                        break;
                    case Keys.PageUp:
                        category = KeyCategory.System;
                        break;
                    case Keys.PageDown:
                        category = KeyCategory.System;
                        break;
                    case Keys.Up:
                        category = KeyCategory.System;
                        break;
                    case Keys.Down:
                        category = KeyCategory.System;
                        break;
                    case Keys.Left:
                        category = KeyCategory.System;
                        break;
                    case Keys.Right:
                        category = KeyCategory.System;
                        break;
                    case Keys.NumLock:
                        category = KeyCategory.System;
                        break;
                    case Keys.Clear:
                        category = KeyCategory.System;
                        break;

                    case Keys.KanjiMode:        // 106 漢字
                        category = KeyCategory.System;
                        break;
                    case Keys.IMENonconvert:    // 106 無変換
                        category = KeyCategory.System;
                        break;
                    case Keys.IMEConvert:       // 106 変換
                        category = KeyCategory.System;
                        break;

                    default:
                        return null;
                }
            }

            packet.category = category;
            if (isNumPad)
            {
                packet.attribute |= KeyAttribute.NumPad;
            }

            packet.indexValue = (byte)indexValue;

            StringBuilder sbKLID = new StringBuilder();
            if (User32.GetKeyboardLayoutName(sbKLID))
            {
                uint klid = uint.Parse(sbKLID.ToString(), NumberStyles.AllowHexSpecifier);
                packet.keyboardLanguageId = klid & 0xffff;
            }

            return packet;
        }
    }
}
