﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace InputCapture
{
    using System;
    using System.Diagnostics;
    using System.IO;
    using System.Reflection;
    using System.Windows.Forms;
    using System.Xml.Serialization;

    /// <summary>
    /// マウスカーソル移動によるキャプチャ開始/停止を行う際の境界位置を表します。
    /// </summary>
    public enum BorderType : byte
    {
        None,
        Right,
        Left,
        Top,
        Bottom
    }

    /// <summary>
    /// マウスカーソル移動によるキャプチャ開始を行う際の操作を表します。
    /// </summary>
    public enum BorderSwitchType : byte
    {
        None,
        LClick,
        LDoubleClick
    }

    /// <summary>
    /// ユーザ設定を格納するクラスです。
    /// </summary>
    public class UserSettings
    {
        public const uint DefaultCheckAliveTimeout = 1;
        public const uint DefaultPollingInterval = 100;

        const string SettingsFileName = "user.config";

        bool _appSettingsChanged = false;
        uint _pollingInterval = DefaultPollingInterval;

        bool _bCaptureOnHotKey = true;   // ホットキーよるキャプチャ起動有効
        Keys _hotKey = Keys.F12;
        Keys _hotKeyModifier = Keys.Alt;

        bool _bCaptureOnMousePosition = true; // マウスポインタの画面端移動によるキャプチャ切り替え有効
        BorderType _captureBorderType = BorderType.Right;

        bool _bAutoConnect = true;
        bool _bAutoDisconnect = true;

        bool _bCheckAlive = true;
        uint _checkAliveTimeout = DefaultCheckAliveTimeout;

        BorderSwitchType _borderSwichType = BorderSwitchType.None;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UserSettings()
        {
        }

        /// <summary>
        /// ホットキーよるキャプチャ起動有効設定
        /// </summary>
        public bool IsCaptureOnHotKey
        {
            get
            {
                return _bCaptureOnHotKey;
            }
            set
            {
                if (_bCaptureOnHotKey != value)
                {
                    _bCaptureOnHotKey = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// マウスポインタの画面端移動によるキャプチャ切り替え有効設定
        /// </summary>
        public bool IsCaptureOnMousePosition
        {
            get
            {
                return _bCaptureOnMousePosition;
            }
            set
            {
                if (_bCaptureOnMousePosition != value)
                {
                    _bCaptureOnMousePosition = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// キャプチャ切り替えホットキー設定
        /// </summary>
        public Keys HotKey
        {
            get
            {
                return _hotKey;
            }
            set
            {
                if (_hotKey != value)
                {
                    _hotKey = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// キャプチャ切り替えホットキー修飾子設定
        /// </summary>
        public Keys HotKeyModifier
        {
            get
            {
                return _hotKeyModifier;
            }
            set
            {
                if (_hotKeyModifier != value)
                {
                    _hotKeyModifier = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// マウス移動によるキャプチャ開始の画面境界タイプの設定
        /// </summary>
        public BorderType CaptureBorderType
        {
            get
            {
                return _captureBorderType;
            }
            set
            {
                if (_captureBorderType != value)
                {
                    _captureBorderType = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// マウス移動によるキャプチャ開始の操作タイプの設定
        /// </summary>
        public BorderSwitchType CaptureBorderSwitchType
        {
            get
            {
                return _borderSwichType;
            }
            set
            {
                if (_borderSwichType != value)
                {
                    _borderSwichType = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// ポーリング間隔です。
        /// </summary>
        public uint PollingInterval
        {
            get
            {
                return _pollingInterval;
            }
            set
            {
                if (_pollingInterval != value)
                {
                    _pollingInterval = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// 自動接続有効設定です。
        /// </summary>
        public bool IsAutoConnect
        {
            get
            {
                return _bAutoConnect;
            }
            set
            {
                if (_bAutoConnect != value)
                {
                    _bAutoConnect = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// 自動切断有効設定です。
        /// </summary>
        public bool IsAutoDisconnect
        {
            get
            {
                return _bAutoDisconnect;
            }
            set
            {
                if (_bAutoDisconnect != value)
                {
                    _bAutoDisconnect = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// ターゲットプログラムの生存確認を行うか。
        /// </summary>
        public bool IsCheckAlive
        {
            get
            {
                return _bCheckAlive;
            }
            set
            {
                if (_bCheckAlive != value)
                {
                    _bCheckAlive = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// 生存確認判定間隔[秒]。
        /// </summary>
        public uint CheckAliveInterval
        {
            get
            {
                return _checkAliveTimeout;
            }
            set
            {
                if (_checkAliveTimeout != value)
                {
                    _checkAliveTimeout = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// 構成ファイルからクラスをシリアル化解除します。
        /// </summary>
        /// <returns></returns>
        public static UserSettings LoadAppSettings()
        {
            try
            {
                // 構成ファイルが存在する場合は、それを開きます。
                FileInfo fi = new FileInfo(GetAppDataPath() + @"\" + Application.ProductName + "." + SettingsFileName);
                if (fi.Exists)
                {
                    using (FileStream fs = fi.OpenRead())
                    {
                        XmlSerializer xmlSerializer = new XmlSerializer(typeof(UserSettings));
                        return (UserSettings)xmlSerializer.Deserialize(fs);
                    }
                }
            }
            catch (IOException ex)
            {
                Debug.WriteLine(ex.Message);
            }
            catch (InvalidOperationException ex)    // 構成ファイルの内容が正しくない場合
            {
                Debug.WriteLine(ex.Message);
            }
            return new UserSettings();
        }

        /// <summary>
        /// 変更されている設定がある場合、クラスを構成ファイルにシリアル化します。
        /// </summary>
        /// <returns></returns>
        public bool SaveAppSettings()
        {
            if (_appSettingsChanged)
            {
                XmlSerializer xmlSerializer = new XmlSerializer(typeof(UserSettings));

                try
                {
                    DirectoryInfo saveFileDir = new DirectoryInfo(GetAppDataPath());

                    if (! saveFileDir.Exists)
                    {
                        saveFileDir.Create();
                    }

                    using (FileStream writer = new FileStream(saveFileDir.FullName + @"\" + Application.ProductName + "." + SettingsFileName, FileMode.Create, FileAccess.Write))
                    {
                        xmlSerializer.Serialize(writer, this);
                    }
                }
                catch (IOException ex)  // サンプルコードがtry/catchで囲んでいたため残しているが、MSDNを見る限り特に例外が記述されていない。
                {
                    Debug.WriteLine(ex.Message);
                }
            }

            return _appSettingsChanged;
        }

        static string GetAppDataPath()
        {
            Assembly assembly = Assembly.GetEntryAssembly();
            return Path.GetDirectoryName(assembly.Location);
        }
    }
}
