﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//=============================================================================
// include
//=============================================================================
#include "Common.h"
#include <wincodec.h>

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//=============================================================================
//! @brief WIC ユーティリティのクラスです。
//=============================================================================
class WicUtility
{
protected:
    bool m_InitializesCom; //!< ユーティリティ内で COM を初期化するなら true です。
    bool m_IsComInitialized; //!< ユーティリティ内で COM を初期化済みなら true です。
    int m_CoInit; //!< COM の初期化オプションです。
    int m_FactoryVersion; //!< ファクトリのバージョンです。
    IWICImagingFactory* m_pFactory; //!< ファクトリへのポインタです。

public:
    //! @brief コンストラクタです。
    WicUtility()
    : m_InitializesCom(false),
      m_IsComInitialized(false),
      m_CoInit(COINIT_APARTMENTTHREADED),
      m_FactoryVersion(1),
      m_pFactory(nullptr)
    {
    }

    //! @brief デストラクタです。
    virtual ~WicUtility()
    {
        Finalize();
    }

    //! @brief COM のモードを設定します。
    //!        ユーティリティ内で COM を初期化する場合、
    //!        Initialize または GetFactory を呼ぶ前に呼ぶ必要があります。
    //!
    //! @param[in] initializesCom ユーティリティ内で COM を初期化するなら true、外部で初期化するなら false を指定します。
    //!                           デフォルトでは false の動作です。
    //! @param[in] coInit CoInitializeEx で指定する COM の初期化オプションです。
    //!
    void SetComMode(const bool initializesCom, const int coInit)
    {
        m_InitializesCom = initializesCom;
        m_CoInit = coInit;
    }

    //! @brief 必要なら COM を初期化します。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus InitializeCom();

    //! @brief 終了します。
    //!        ユーティリティ内で COM を初期化した場合は初期化終了します。
    void Finalize();

    //! @brief ファクトリを取得します。
    //!        初めて取得する際は内部で InitializeCom を呼びます。
    //!
    //! @param[out] pStatus 処理結果へのポインタです。nullptr なら何も格納しません。
    //!
    //! @return ファクトリへのポインタ（取得できなければ nullptr）を返します。
    //!
    IWICImagingFactory* GetFactory(RStatus* pStatus);

    //! @brief GetFactory で取得したファクトリのバージョンを取得します。
    //!
    //! @return ファクトリのバージョンを返します。
    //!
    int GetFactoryVersion() const
    {
        return m_FactoryVersion;
    }
};

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

