﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//=============================================================================
// include
//=============================================================================
#include "Common.h"

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//=============================================================================
//! @brief OpenEXR ユーティリティのクラスです。
//=============================================================================
class ROpenExr
{
protected:
    std::string m_DllPath; //!< DLL ファイルのパスです。
    HMODULE m_hDLL; //!< DLL のインスタンスハンドルです。

public:
    //! @brief 各種フラグを表す列挙型です。
    enum Flags
    {
        FLAGS_UINT    = (1 <<  0), //!< ピクセルタイプが UINT なら ON です。
        FLAGS_HALF    = (1 <<  1), //!< ピクセルタイプが HALF なら ON です。
        FLAGS_YC      = (1 <<  2), //!< YC カラーなら ON です。
        FLAGS_MIPMAP  = (1 <<  3), //!< ミップマップなら ON です。
        FLAGS_CUBEMAP = (1 <<  4), //!< キューブマップなら ON です。
        FLAGS_TILED   = (1 <<  5), //!< タイル形式なら ON です。
    };

    //! @brief EXR ファイルのヘッダをリードして情報を取得します。
    //!
    //! @param[out] pWidth 幅を格納する変数へのポインタです。
    //! @param[out] pHeight 高さを格納する変数へのポインタです。
    //! @param[out] pDepth 奥行きを格納する変数へのポインタです。
    //! @param[out] pChannelCount チャンネル数を格納する変数へのポインタです。
    //! @param[out] pFlags 各種フラグを格納する変数へのポインタです。
    //! @param[in] filePath EXR ファイルのパスです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool (*GetExrInfo)(
        int* pWidth,
        int* pHeight,
        int* pDepth,
        int* pChannelCount,
        uint32_t* pFlags,
        const wchar_t* filePath
    );

    //! @brief メモリ上の EXR ファイルデータから情報を取得します。
    //!
    //! @param[out] pWidth 幅を格納する変数へのポインタです。
    //! @param[out] pHeight 高さを格納する変数へのポインタです。
    //! @param[out] pDepth 奥行きを格納する変数へのポインタです。
    //! @param[out] pChannelCount チャンネル数を格納する変数へのポインタです。
    //! @param[out] pFlags 各種フラグを格納する変数へのポインタです。
    //! @param[in] pData メモリ上の EXR ファイルデータのポインタです。
    //! @param[in] dataSize メモリ上の EXR ファイルデータのサイズです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool (*ParseExrInfo)(
        int* pWidth,
        int* pHeight,
        int* pDepth,
        int* pChannelCount,
        uint32_t* pFlags,
        const void* pData,
        const size_t dataSize
    );

    //! @brief EXR ファイルをリードして 16-bit float のビットマップを取得します。
    //!
    //! @param[out] pBitmapF16 16-bit float のビットマップへのポインタです。
    //!                        幅 x 高さ x 4 x 2 バイトのメモリを確保しておきます。
    //! @param[in] filePath EXR ファイルのパスです。
    //! @param[in] channelCount チャンネル数です。
    //! @param[in] flags 各種フラグです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool (*ReadExrAsFloat16)(
        void* pBitmapF16,
        const wchar_t* filePath,
        const int channelCount,
        const uint32_t flags
    );

    //! @brief メモリ上の EXR ファイルデータから 16-bit float のビットマップを取得します。
    //!
    //! @param[out] pBitmapF16 16-bit float のビットマップへのポインタです。
    //!                        幅 x 高さ x 4 x 2 バイトのメモリを確保しておきます。
    //! @param[in] pData メモリ上の EXR ファイルデータのポインタです。
    //! @param[in] dataSize メモリ上の EXR ファイルデータのサイズです。
    //! @param[in] channelCount チャンネル数です。
    //! @param[in] flags 各種フラグです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool (*ParseExrAsFloat16)(
        void* pBitmapF16,
        const void* pData,
        const size_t dataSize,
        const int channelCount,
        const uint32_t flags
    );

    //! @brief EXR ファイルをリードして 32-bit float のビットマップを取得します。
    //!
    //! @param[out] pBitmapF32 32-bit float のビットマップへのポインタです。
    //!                        幅 x 高さ x 4 x 4 バイトのメモリを確保しておきます。
    //! @param[in] filePath EXR ファイルのパスです。
    //! @param[in] channelCount チャンネル数です。
    //! @param[in] flags 各種フラグです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool (*ReadExrAsFloat32)(
        void* pBitmapF32,
        const wchar_t* filePath,
        const int channelCount,
        const uint32_t flags
    );

    //! @brief メモリ上の EXR ファイルデータから 32-bit float のビットマップを取得します。
    //!
    //! @param[out] pBitmapF32 32-bit float のビットマップへのポインタです。
    //!                        幅 x 高さ x 4 x 4 バイトのメモリを確保しておきます。
    //! @param[in] pData メモリ上の EXR ファイルデータのポインタです。
    //! @param[in] dataSize メモリ上の EXR ファイルデータのサイズです。
    //! @param[in] channelCount チャンネル数です。
    //! @param[in] flags 各種フラグです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool (*ParseExrAsFloat32)(
        void* pBitmapF32,
        const void* pData,
        const size_t dataSize,
        const int channelCount,
        const uint32_t flags
    );

    //! @brief 16-bit float のビットマップを EXR ファイルにライトします。
    //!
    //! @param[in] filePath EXR ファイルのパスです。
    //! @param[in] pBitmapF16 16-bit float のビットマップへのポインタです。
    //! @param[in] imageW ビットマップの幅です。
    //! @param[in] imageH ビットマップの高さです。
    //! @param[in] imageD ビットマップの奥行きです。
    //! @param[in] flags 各種フラグです。
    //! @param[in] hasAlpha アルファ成分を持つなら true です。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool (*WriteExrFloat16)(
        const wchar_t* filePath,
        const void* pBitmapF16,
        const int imageW,
        const int imageH,
        const int imageD,
        const uint32_t flags,
        const bool hasAlpha
    );

    //! @brief 32-bit float のビットマップを EXR ファイルにライトします。
    //!
    //! @param[in] filePath EXR ファイルのパスです。
    //! @param[in] pBitmapF32 32-bit float のビットマップへのポインタです。
    //! @param[in] imageW ビットマップの幅です。
    //! @param[in] imageH ビットマップの高さです。
    //! @param[in] imageD ビットマップの奥行きです。
    //! @param[in] flags 各種フラグです。
    //! @param[in] hasAlpha アルファ成分を持つなら true です。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool (*WriteExrFloat32)(
        const wchar_t* filePath,
        const void* pBitmapF32,
        const int imageW,
        const int imageH,
        const int imageD,
        const uint32_t flags,
        const bool hasAlpha
    );

    //! @brief 32-bit float のビットマップを 16-bit float 形式でEXR ファイルにライトします。
    //!
    //! @param[in] filePath EXR ファイルのパスです。
    //! @param[in] pBitmapF32 32-bit float のビットマップへのポインタです。
    //! @param[in] imageW ビットマップの幅です。
    //! @param[in] imageH ビットマップの高さです。
    //! @param[in] imageD ビットマップの奥行きです。
    //! @param[in] flags 各種フラグです。
    //! @param[in] hasAlpha アルファ成分を持つなら true です。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool (*WriteExrFloat32AsFloat16)(
        const wchar_t* filePath,
        const void* pBitmapF32,
        const int imageW,
        const int imageH,
        const int imageD,
        const uint32_t flags,
        const bool hasAlpha
    );

    //! @brief エラーのユニコード文字列を返します。
    //!
    //! @return エラーのユニコード文字列を返します。
    //!
    const wchar_t* (*GetErrorString)();

    //! @brief エラーの Shift JIS 文字列を返します。
    //!
    //! @return エラーの Shift JIS 文字列を返します。
    //!
    const char* (*GetErrorShiftJisString)();

public:
    //! @biref コンストラクタです。
    ROpenExr()
    : m_hDLL(NULL),
      GetExrInfo(NULL),
      ParseExrInfo(NULL),
      ReadExrAsFloat16(NULL),
      ParseExrAsFloat16(NULL),
      ReadExrAsFloat32(NULL),
      ParseExrAsFloat32(NULL),
      WriteExrFloat16(NULL),
      WriteExrFloat32(NULL),
      WriteExrFloat32AsFloat16(NULL),
      GetErrorString(NULL),
      GetErrorShiftJisString(NULL)
    {
    }

    //! @biref デストラクタです。
    ~ROpenExr()
    {
        Finalize();
    }

    //! @biref DLL ファイルのパスを設定します。
    //!
    //! @param[in] dllPath OpenEXR ユーティリティの DLL ファイルのパスです。
    //!
    void SetDllPath(const char* dllPath)
    {
        m_DllPath = dllPath;
    }

    //! @biref DLL ファイルのパスを返します。
    std::string GetDllPath() const
    {
        return m_DllPath;
    }

    //! @biref 初期化します。
    //!
    //! @param[in] dllPath OpenEXR ユーティリティの DLL ファイルのパスです。
    //!                    NULL なら SetDllPath で設定したパスを使用します。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool Initialize(const char* dllPath);

    //! @biref 終了します。
    void Finalize();

    //! @biref 初期化済みなら true を返します。
    bool IsInitialized() const
    {
        return (m_hDLL != NULL);
    }
};

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

