﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// RXMLElement LoadDocument
// RDataStream RBinDataStream
// RUserData
// RFileBuf

//=============================================================================
// include
//=============================================================================
#include "Common.h"
#pragma comment(lib, "shlwapi.lib")

using namespace std;

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//=============================================================================
// constants
//=============================================================================
const std::string TgaExtension = "tga"; //!< TGA ファイルの拡張子です。
const std::string PngExtension = "png"; //!< PNG ファイルの拡張子です。
const std::string DdsExtension = "dds"; //!< DDS ファイルの拡張子です。
const std::string ExrExtension = "exr"; //!< EXR ファイルの拡張子です。
const std::string FtxaExtension = "ftxa"; //!< テクスチャー中間ファイル（アスキー形式）の拡張子です。
const std::string FtxbExtension = "ftxb"; //!< テクスチャー中間ファイル（バイナリー形式）の拡張子です。
const std::string BntxExtension = "bntx"; //!< テクスチャーバイナリーファイルの拡張子です。

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//=============================================================================
//! @brief 整数値と文字列のポインタの構造体です。
//=============================================================================
struct RIntAndCharPtr
{
public:
    int m_Value;
    const char* m_String;
};

//-----------------------------------------------------------------------------
//! @brief Base64 形式の文字列を設定します。
//!
//! @param[out] pDstStr Base64 形式データへのポインターです。
//! @param[in] packedData 入力データです。1 ～ 3 バイトを uint32_t 型に格納します。
//!               1 バイトの場合は byte0
//!               2 バイトの場合は (byte0 <<  8) | byte1
//!               3 バイトの場合は (byte0 << 16) | (byte1 << 8) | byte2
//! @param[in] packedBytes 入力データのバイト数（1 ～ 3）です。
//-----------------------------------------------------------------------------
void SetBase64String(char* pDstStr, const uint32_t packedData, const int packedBytes)
{
    static const char* s_Base64Chars =
        "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

    uint32_t packed = packedData << (3 - packedBytes) * 8;
    char* pDst = pDstStr;
    int base = 18;
    int dstIdx;
    for (dstIdx = 0; dstIdx < packedBytes + 1; ++dstIdx)
    {
        *pDst++ = s_Base64Chars[(packed >> base) & 0x3f];
        base -= 6;
    }
    while (dstIdx < 4)
    {
        *pDst++ = '='; // 端数
        ++dstIdx;
    }
}

//-----------------------------------------------------------------------------
//! @brief Base64 の入力データの 1 文字を 0 ～ 63 の整数値に変換して返します。
//!        入力データの 1 文字が不正なら -1 を返します。
//!
//! @param[in] c 入力データの 1 文字です。
//!
//! @return 整数値を返します。
//-----------------------------------------------------------------------------
int GetBase64CharValue(const int c)
{
    if (c >= static_cast<int>('A') && c <= static_cast<int>('Z'))
    {
        return c - static_cast<int>('A');
    }
    else if (c >= static_cast<int>('a') && c <= static_cast<int>('z'))
    {
        return c - static_cast<int>('a') + 26;
    }
    else if (c >= static_cast<int>('0') && c <= static_cast<int>('9'))
    {
        return c - static_cast<int>('0') + 52;
    }
    else if (static_cast<int>('+') == c)
    {
        return 62;
    }
    else if (static_cast<int>('/') == c)
    {
        return 63;
    }
    else if (static_cast<int>('=') == c)
    {
        return 0;
    }
    else
    {
        return -1;
    }
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 開いているメモ帳のウィンドウにメッセージを出力します（デバッグ用）。
//-----------------------------------------------------------------------------
void NoteTrace(const char* format, ...)
{
    HWND hWnd = FindWindow("Notepad", nullptr);
    if (hWnd != nullptr)
    {
        static const POINT pt = { 0, 0 };
        hWnd = ChildWindowFromPoint(hWnd, pt);
        if (hWnd != nullptr)
        {
            char strBuf[1024];
            va_list args;
            va_start(args, format);
            const int size = vsprintf_s(strBuf, format, args);
            va_end(args);
            SendMessage(hWnd, EM_REPLACESEL, 0, reinterpret_cast<LPARAM>(strBuf));
            SendMessage(hWnd, EM_REPLACESEL, 0, reinterpret_cast<LPARAM>("\r\n"));
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 2 のべき乗の数なら true を返します。
//-----------------------------------------------------------------------------
bool RIsPower2Number(const int number, const bool includeOne)
{
    if (includeOne && number == 1)
    {
        return true;
    }

    int cur = 1;
    do
    {
        cur *= 2;
        if (number == cur)
        {
            return true;
        }
    } while (cur < number);

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 指定した数以上で最小の 2 のべき乗の数を返します。
//-----------------------------------------------------------------------------
int RGetPower2Number(const int number, const bool includeOne)
{
    int ret = (includeOne) ? 1 : 2;
    while (ret < number)
    {
        ret <<= 1;
    }
    return ret;
}

//-----------------------------------------------------------------------------
//! @brief 指定した文字が Shift JIS の 2 バイト文字の 1 バイト目なら true を返します。
//-----------------------------------------------------------------------------
bool RIsShiftJisFirstByte(const unsigned char val)
{
    return ((0x81 <= val && val <= 0x9f) ||
            (0xe0 <= val && val <= 0xef));
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS の文字列を UTF-8 の文字列に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetUtf8FromShiftJis(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // Shift JIS -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    int wcharSize = MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, nullptr, 0);
    if (wcharSize == 0)
    {
        return std::string();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, wcharBuf, wcharSize);

    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> UTF-8
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    int utf8Size = WideCharToMultiByte(CP_UTF8, 0, wcharBuf, -1, nullptr, 0, nullptr, nullptr);
    char* utf8Buf = new char[utf8Size];
    WideCharToMultiByte(CP_UTF8, 0, wcharBuf, -1, utf8Buf, utf8Size, nullptr, nullptr);
    std::string dst(utf8Buf, strlen(utf8Buf));
    // 通常 strlen(utf8Buf) = utf8Size - 1

    //cerr << "[" << src.c_str() << "] [" << dst.c_str() << "] " << src.size() << ", " << wcharSize << ", " << utf8Size << ", " << dst.size() << endl;
    //for (int i = 0; i < utf8Size; ++i)
    //{
    //  cerr << hex << static_cast<int>(static_cast<uint8_t>(utf8Buf[i])) << dec << endl;
    //}

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;
    delete[] utf8Buf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief UTF-8 の文字列を Shift-JIS の文字列に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetShiftJisFromUtf8(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // UTF-8 -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    int wcharSize = MultiByteToWideChar(CP_UTF8, 0, src.c_str(), -1, nullptr, 0);
    if (wcharSize == 0)
    {
        return std::string();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_UTF8, 0, src.c_str(), -1, wcharBuf, wcharSize);

    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> Shift JIS
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    int sjisSize = WideCharToMultiByte(CP_ACP, 0, wcharBuf, -1, nullptr, 0, nullptr, nullptr);
    char* sjisBuf = new char[sjisSize];
    WideCharToMultiByte(CP_ACP, 0, wcharBuf, -1, sjisBuf, sjisSize, nullptr, nullptr);
    std::string dst(sjisBuf, strlen(sjisBuf));
    // 通常 strlen(sjisBuf) = sjisSize - 1

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;
    delete[] sjisBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief get Unicode (UCS-2) from Shift JIS
//-----------------------------------------------------------------------------
std::wstring RGetUnicodeFromShiftJis(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // Shift JIS -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    const int wcharSize = MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, nullptr, 0);
    if (wcharSize == 0)
    {
        return std::wstring();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, wcharBuf, wcharSize);
    std::wstring dst(wcharBuf, wcslen(wcharBuf));

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief get Shift JIS from Unicode (UCS-2)
//-----------------------------------------------------------------------------
std::string RGetShiftJisFromUnicode(const std::wstring& src)
{
    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> Shift JIS
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    const int sjisSize = WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, nullptr, 0, nullptr, nullptr);
    if (sjisSize == 0)
    {
        return std::string();
    }

    char* sjisBuf = new char[sjisSize];
    WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, sjisBuf, sjisSize, nullptr, nullptr);
    std::string dst(sjisBuf, strlen(sjisBuf));

    //-----------------------------------------------------------------------------
    // free memory
    delete[] sjisBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief ユニコード C 配列を RStringArray に変換します。
//-----------------------------------------------------------------------------
int RGetStringArray(RStringArray& dsts, const wchar_t* srcs[])
{
    dsts.clear();
    int iStr = 0;
    while (srcs[iStr] != nullptr)
    {
        dsts.push_back(RGetShiftJisFromUnicode(srcs[iStr]));
        ++iStr;
    }
    return static_cast<int>(dsts.size());
}

//-----------------------------------------------------------------------------
//! @brief 文字列を小文字に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetLowerCaseString(const std::string& src)
{
    std::string dst = src;
    int size = static_cast<int>(dst.size());
    for (int ic = 0; ic < size; ++ic)
    {
        char c = dst[ic];
        if ('A' <= c && c <= 'Z')
        {
            dst[ic] = c - 'A' + 'a';
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 文字列を大文字に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetUpperCaseString(const std::string& src)
{
    std::string dst = src;
    int size = static_cast<int>(dst.size());
    for (int ic = 0; ic < size; ++ic)
    {
        char c = dst[ic];
        if ('a' <= c && c <= 'z')
        {
            dst[ic] = c - 'a' + 'A';
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 大文字／小文字を無視して 2 つの文字列を比較し、同じなら true を返します。
//-----------------------------------------------------------------------------
bool RIsSameStringNoCase(const std::string& str1, const std::string& str2)
{
    return (RGetLowerCaseString(str1) == RGetLowerCaseString(str2));
}

//-----------------------------------------------------------------------------
//! @brief 文字列の前の不要な文字を削除します。
//-----------------------------------------------------------------------------
std::string RTrimLeftString(const std::string& src, const char* targets)
{
    const size_t iLeft = src.find_first_not_of(targets);
    return (iLeft != std::string::npos) ? src.substr(iLeft) : std::string();
}

//-----------------------------------------------------------------------------
//! @brief 文字列の後の不要な文字を削除します。
//-----------------------------------------------------------------------------
std::string RTrimRightString(const std::string& src, const char* targets)
{
    const size_t iRight = src.find_last_not_of(targets);
    return (iRight != std::string::npos) ? src.substr(0, iRight + 1) : std::string();
}

//-----------------------------------------------------------------------------
//! @brief 文字列の前後の不要な文字を削除します。
//-----------------------------------------------------------------------------
std::string RTrimString(const std::string& src, const char* targets)
{
    const size_t iLeft = src.find_first_not_of(targets);
    if (iLeft != std::string::npos)
    {
        const size_t iRight = src.find_last_not_of(targets);
        if (iRight != std::string::npos)
        {
            return src.substr(iLeft, iRight - iLeft + 1);
        }
    }
    return std::string();
}

//-----------------------------------------------------------------------------
//! @brief 文字列の前後の引用符を削除します。
//-----------------------------------------------------------------------------
std::string RDequoteString(const std::string& src, const char quote)
{
    std::string dst(src);
    const size_t iquotL = dst.find(quote);
    if (iquotL != std::string::npos)
    {
        if (iquotL == dst.size() - 1) // 末尾にのみ引用符が存在する場合、その引用符を削除します。
        {
            dst = dst.substr(0, iquotL);
        }
        else
        {
            dst = dst.substr(iquotL + 1);
            const size_t iquotR = dst.rfind(quote);
            if (iquotR != std::string::npos)
            {
                dst = dst.substr(0, iquotR);
            }
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS 文字列の改行コード CR + LF を LF に置換します。
//-----------------------------------------------------------------------------
std::string RReplaceCrLfToLf(const std::string& src)
{
    std::string dst;
    for (size_t ic = 0; ic < src.size(); ++ic)
    {
        const char c = src[ic];
        if (c == 0x0d && ic + 1 < src.size() && src[ic + 1] == 0x0a)
        {
            // CR + LF なら CR をスキップします。
        }
        else
        {
            dst += c;
            if (RIsShiftJisFirstByte(static_cast<unsigned char>(c)))
            {
                if (++ic < src.size())
                {
                    dst += src[ic];
                }
            }
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 文字列をトークンに分解します。
//-----------------------------------------------------------------------------
int RTokenizeString(RStringArray& tokens, const std::string& input, const char* delims)
{
    tokens.clear();
    size_t iEnd = 0;
    for (;;)
    {
        const size_t iStart = input.find_first_not_of(delims, iEnd);
        if (iStart == std::string::npos)
        {
            break;
        }
        iEnd = input.find_first_of(delims, iStart);
        if (iEnd == std::string::npos)
        {
            tokens.push_back(input.substr(iStart));
            break;
        }
        tokens.push_back(input.substr(iStart, iEnd - iStart));
    }
    return static_cast<int>(tokens.size());
}

//-----------------------------------------------------------------------------
//! @brief int 型の数値を文字列に変換します。
//-----------------------------------------------------------------------------
std::string RGetNumberString(const int num, const char* format)
{
    char buf[64];
    const char* fmt = (format != nullptr) ? format : "%d";
    _snprintf_s(buf, sizeof(buf), _TRUNCATE, fmt, num);
    return std::string(buf);
}

//-----------------------------------------------------------------------------
//! @brief uint32_t 型の数値を文字列に変換します。
//-----------------------------------------------------------------------------
std::string RGetNumberString(const uint32_t num, const char* format)
{
    char buf[64];
    const char* fmt = (format != nullptr) ? format : "%u";
    _snprintf_s(buf, sizeof(buf), _TRUNCATE, fmt, num);
    return std::string(buf);
}

//-----------------------------------------------------------------------------
//! @brief ポインタを文字列に変換します。
//-----------------------------------------------------------------------------
std::string RGetPointerString(const void* ptr, const char* format)
{
    char buf[64];
    const char* fmt = (format != nullptr) ? format : "%p";
    _snprintf_s(buf, sizeof(buf), _TRUNCATE, fmt, ptr);
    return std::string(buf);
}

//-----------------------------------------------------------------------------
//! @brief 文字列の末尾の整数値を取得します。
//!
//! @param[out] pNumber 整数値を格納します。文字列の末尾が数値でない場合は -1 を格納します。
//! @param[in] str 文字列です。
//!
//! @return 文字列の末尾の数値を削除した文字列を返します。
//-----------------------------------------------------------------------------
std::string RGetSuffixNumberFromString(int* pNumber, const std::string& str)
{
    int digitCount = 0;
    for (size_t ic = str.size() - 1; ic >= 0; --ic)
    {
        const char c = str[ic];
        if ('0' <= c && c <= '9')
        {
            ++digitCount;
        }
        else
        {
            break;
        }
    }

    if (digitCount > 0)
    {
        const size_t prefixCount = str.size() - digitCount;
        *pNumber = atoi(str.c_str() + prefixCount);
        return (prefixCount > 0) ? str.substr(0, prefixCount) : std::string();
    }
    else
    {
        *pNumber = -1;
        return str;
    }
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS 文字列を XML 用にエンコードします。
//-----------------------------------------------------------------------------
namespace
{
const std::string XML_AMP_STR  = "&amp;";
const std::string XML_LT_STR   = "&lt;";
const std::string XML_GT_STR   = "&gt;";
const std::string XML_QUOT_STR = "&quot;";
} // unnamed namespace

std::string REncodeXmlString(const std::string& src)
{
    std::string dst;
    for (size_t ic = 0; ic < src.size(); ++ic)
    {
        const char c = src[ic];
        if (RIsShiftJisFirstByte(static_cast<unsigned char>(c)))
        {
            dst += c;
            if (++ic < src.size())
            {
                dst += src[ic];
            }
        }
        else if (c == '&')
        {
            dst += XML_AMP_STR;
        }
        else if (c == '<')
        {
            dst += XML_LT_STR;
        }
        else if (c == '>')
        {
            dst += XML_GT_STR;
        }
        else if (c == '\"')
        {
            dst += XML_QUOT_STR;
        }
        else
        {
            dst += c;
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief XML 用にエンコードされた Shift-JIS 文字列をデコードします。
//-----------------------------------------------------------------------------
std::string RDecodeXmlString(const std::string& src)
{
    std::string dst;
    for (size_t ic = 0; ic < src.size(); ++ic)
    {
        const char c = src[ic];
        if (c == '&')
        {
            if (src.find(XML_AMP_STR, ic) == ic)
            {
                dst += '&';
                ic += XML_AMP_STR.size() - 1;
            }
            else if (src.find(XML_LT_STR, ic) == ic)
            {
                dst += '<';
                ic += XML_LT_STR.size() - 1;
            }
            else if (src.find(XML_GT_STR, ic) == ic)
            {
                dst += '>';
                ic += XML_GT_STR.size() - 1;
            }
            else if (src.find(XML_QUOT_STR, ic) == ic)
            {
                dst += '\"';
                ic += XML_QUOT_STR.size() - 1;
            }
            else
            {
                dst += c;
            }
        }
        else
        {
            dst += c;
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief uint8_t 型のデータ文字列をデコードします。
//-----------------------------------------------------------------------------
void RDecodeDataStringU8(void* pDst, const std::string& str, const size_t count)
{
    std::istringstream iss(str.c_str());
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
    size_t iVal = 0;
    while (iss && iVal < count)
    {
        std::string value;
        iss >> value;
        if (!value.empty())
        {
            pDstU8[iVal++] = static_cast<uint8_t>(strtol(value.c_str(), nullptr, 16));
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief float 型のデータ文字列をデコードします。
//-----------------------------------------------------------------------------
void RDecodeDataStringF32(void* pDst, const std::string& str, const size_t count)
{
    std::istringstream iss(str.c_str());
    float* pDstF32 = reinterpret_cast<float*>(pDst);
    size_t iVal = 0;
    while (iss && iVal < count)
    {
        std::string value;
        iss >> value;
        if (!value.empty())
        {
            pDstF32[iVal++] = static_cast<float>(atof(value.c_str()));
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief Windows 形式のファイルパスを返します。"/" を "\" に変換します。
//!        Shift-JIS の std::string に対応しています。
//!
//! @param[in] path UNIX 形式のファイルパスです。
//!
//! @return Windows 形式のファイルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetWindowsFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    char* buf = new char[size + 1];

    const char* src = path.c_str();
    char* dst = buf;

    while (*src)
    {
        char c = *src++;
        if (c == '/')
        {
            c = '\\';
        }
        *dst++ = c;
    }
    *dst = '\0';

    std::string ret = buf;
    delete[] buf;

    return ret;
}

//-----------------------------------------------------------------------------
//! @brief UNIX 形式のファイルパスを返します。"\" を "/" に変換します。
//!        Shift-JIS の std::string に対応しています。
//!
//! @param[in] path Windows 形式のファイルパスです。
//!
//! @return UNIX 形式のファイルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetUnixFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    char* buf = new char[size + 1];

    const char* src = path.c_str();
    char* dst = buf;

    while (*src)
    {
        char c = *src++;
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            if (*src)
            {
                *dst++ = c;
                c = *src++;
            }
        }
        else if (c == '\\')
        {
            c = '/';
        }
        *dst++ = c;
    }
    *dst = '\0';

    std::string ret = buf;
    delete[] buf;

    return ret;
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS のファイルパス文字列からファイル名を抽出して返します。
//-----------------------------------------------------------------------------
std::string RGetFileNameFromFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
    }
    if (iSlash >= 0)
    {
        return std::string(str + iSlash + 1);
    }
    else
    {
        return path;
    }
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS のファイルパス文字列からフォルダーのパス部分を抽出して返します。
//!        返り値の最後に / は付きません。
//-----------------------------------------------------------------------------
std::string RGetFolderFromFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
    }
    if (iSlash >= 0)
    {
        return std::string(str, iSlash); // without last slash
        //return std::string(str, iSlash + 1); // with last slash
    }
    else
    {
        return std::string(".");
    }
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS のファイルパス文字列から拡張子を抽出して返します。
//-----------------------------------------------------------------------------
std::string RGetExtensionFromFilePath(const std::string& path, const bool lower)
{
    const int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    int iDot = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
        else if (c == '.')
        {
            iDot = ic;
        }
    }

    if (iDot >= 0 && (iSlash == -1 || iDot > iSlash))
    {
        const std::string ext = std::string(str + iDot + 1);
        return (lower) ? RGetLowerCaseString(ext) : ext;
    }
    else
    {
        return std::string(); // empty
    }
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS のファイルパス文字列から拡張子を除いた部分を抽出して返します。
//-----------------------------------------------------------------------------
std::string RGetNoExtensionFilePath(const std::string& path)
{
    int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    int iDot = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
        else if (c == '.')
        {
            iDot = ic;
        }
    }
    if (iDot >= 0 && (iSlash == -1 || iDot > iSlash))
    {
        return std::string(str, iDot);
    }
    else
    {
        return path;
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイルパスがフルパスなら true を返します。
//-----------------------------------------------------------------------------
bool RIsFullFilePath(const std::string& path)
{
    if (path.size() >= 2)
    {
        if (path[1] == ':'                       ||
            (path[0] == '\\' && path[1] == '\\') ||
            (path[0] == '/'  && path[1] == '/' ))
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief ファイルパスをフルパスに変換します。
//!
//! @param[in] path 元のファイルパスです。
//! @param[in] isUnix フォルダーの区切りを / にするなら true を指定します。
//!
//! @return フルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetFullFilePath(const std::string& path, const bool isUnix)
{
    // 環境変数を展開します。
    std::string dst(path);
    if (dst.find('%') != std::string::npos)
    {
        char expandPath[MAX_PATH];
        if (::ExpandEnvironmentStringsA(dst.c_str(), expandPath, sizeof(expandPath)) != 0)
        {
            dst = expandPath;
        }
    }

    // フルパスに変換します。
    char fullPath[MAX_PATH];
    if (::PathSearchAndQualifyA(dst.c_str(), fullPath, sizeof(fullPath)))
    {
        dst = fullPath;
    }

    // ドライブ文字を大文字にします。
    if (dst.size() >= 2 && dst[1] == ':')
    {
        const char drive = dst[0];
        if ('a' <= drive && drive <= 'z')
        {
            dst[0] = drive - 'a' + 'A';
        }
    }

    return (isUnix) ? RGetUnixFilePath(dst) : dst;
}

//-----------------------------------------------------------------------------
//! @brief 末尾にスラッシュを追加したファイルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetFilePathWithEndingSlash(const std::string& path)
{
    return (!path.empty() && path[path.size() - 1] == '/') ? path : path + '/';
}

//-----------------------------------------------------------------------------
//! @brief 末尾にバックスラッシュを追加したファイルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetFilePathWithEndingBackslash(const std::string& path)
{
    return (!path.empty() && path[path.size() - 1] == '\\') ? path : path + '\\';
}

//-----------------------------------------------------------------------------
//! @brief ファイルが存在するなら true を返します。
//-----------------------------------------------------------------------------
bool RFileExists(const std::string& path)
{
    WIN32_FIND_DATA ffd;
    #ifdef UNICODE
    HANDLE  hFindFile = FindFirstFile(RGetUnicodeFromShiftJis(path).c_str(), &ffd);
    #else
    HANDLE  hFindFile = FindFirstFile(path.c_str(), &ffd);
    #endif
    if (hFindFile == INVALID_HANDLE_VALUE)
    {
        return false;
    }
    FindClose(hFindFile);

    if (!(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) // check is file
    {
        return true;
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief フォルダーが存在するなら true を返します。
//!        path の最後に '\' または '/' が付いていても付いていなくても構いません。
//-----------------------------------------------------------------------------
bool RFolderExists(const std::string& path)
{
    std::string unixPath = RGetUnixFilePath(path);
    int len = static_cast<int>(unixPath.length());
    if (len == 0)
    {
        return false;
    }

    while (len > 0)
    {
        if (unixPath[len - 1] != '/')
        {
            break;
        }
        --len;
    }
    string noSlashPath = string(unixPath.c_str(), len);
    //if (len != strlen(path))
    //  cerr << "no slash: " << path.c_str() << " -> " << noSlashPath.c_str() << endl;

    WIN32_FIND_DATA ffd;
    #ifdef UNICODE
    HANDLE  hFindFile = FindFirstFile(RGetUnicodeFromShiftJis(noSlashPath).c_str(), &ffd);
    #else
    HANDLE  hFindFile = FindFirstFile(noSlashPath.c_str(), &ffd);
    #endif
    if (hFindFile == INVALID_HANDLE_VALUE)
    {
        return false;
    }
    FindClose(hFindFile);

    if (ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) // check is folder
    {
        return true;
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイル用出力ストリームのフォーマットを初期化します。
//-----------------------------------------------------------------------------
void RInitOutStreamFormat(std::ostream& os)
{
    //-----------------------------------------------------------------------------
    // set format
    os.setf(ios_base::dec, ios_base::basefield);
    os.fill(' ');
    os.setf(ios_base::left, ios_base::adjustfield);
    os.setf(ios_base::uppercase); // 浮動小数の e を大文字に
    #if (_MSC_VER < 1900)
    _set_output_format(_TWO_DIGIT_EXPONENT); // 浮動小数の指数部を 2 桁に
        // VS2015 ではデフォルトで 2 桁です。
    #endif
    os.precision(R_FLOAT_OUT_PRECISION);
    //os.setf(ios_base::showpoint);
    //os.setf(ios_base::fixed, ios_base::floatfield);
    //os.setf(ios_base::scientific, ios_base::floatfield);
    //os.setf(ios_base::fmtflags(0), ios_base::floatfield); // default
}

//-----------------------------------------------------------------------------
//! @brief UTF-8 の BOM を出力します。
//-----------------------------------------------------------------------------
void ROutUtf8Bom(std::ostream& os)
{
    static const unsigned char utf8BOM[] = { 0xef, 0xbb, 0xbf };
    os.write(reinterpret_cast<const char*>(utf8BOM), 3);
}

//-----------------------------------------------------------------------------
//! @brief UTF-16 の BOM を出力します。
//-----------------------------------------------------------------------------
void ROutUtf16Bom(std::ostream& os)
{
    static const unsigned char utf16BOM[] = { 0xff, 0xfe };
    os.write(reinterpret_cast<const char*>(utf16BOM), 2);
}

//-----------------------------------------------------------------------------
//! @brief Shift JIS 文字列を UTF-8 文字列に変換して出力します。
//-----------------------------------------------------------------------------
void ROutUtf8FromShiftJis(std::ostream& os, const std::string& sjisStr)
{
    // UTF-8 の複数バイトコードは 0x80 から 0xfd までの範囲で表現されるので、
    // os はテキストモードでも改行の問題はない
    std::string utf8Str = RGetUtf8FromShiftJis(sjisStr);
    if (utf8Str.size() > 0)
    {
        os.write(utf8Str.c_str(), static_cast<int>(utf8Str.size()));
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定した数のタブ文字を出力します。
//-----------------------------------------------------------------------------
const char* RTab(const int count)
{
    const int TABS_MAX = 16;
    static const char* tabs = "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t"; // TABS_MAX 個

    if (count == 0)
    {
        return "";
    }
    else
    {
        return tabs + (TABS_MAX - count);
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの bool 型用文字列を出力します。
//-----------------------------------------------------------------------------
const char* RBoolStr(const bool value)
{
    return (value) ? "true" : "false";
}

//-----------------------------------------------------------------------------
//! @brief パディングを出力します。
//-----------------------------------------------------------------------------
void ROutPadding(std::ostream& os, const size_t size, const char value)
{
    if (size > 0)
    {
        std::string padding;
        for (size_t iByte = 0; iByte < size; ++iByte)
        {
            padding += value;
        }
        os << padding;
    }
}

//-----------------------------------------------------------------------------
//! @brief Base64 形式にエンコードした際のサイズを取得します。
//-----------------------------------------------------------------------------
size_t GetBase64EncodedSize(const size_t dataSize)
{
    const size_t dataBits = dataSize * 8;
    return (dataBits / 24) * 4 + (((dataBits % 24) != 0) ? 4 : 0);
}

//-----------------------------------------------------------------------------
//! @brief データを Base64 形式にエンコードします。
//!        Base64 形式データの末尾に \0 は含まれません。
//-----------------------------------------------------------------------------
size_t EncodeToBase64(void* base64Buf, const void* dataBuf, const size_t dataSize)
{
    char* pDst = reinterpret_cast<char*>(base64Buf);
    const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(dataBuf);
    const uint8_t* pSrcEnd = pSrc + dataSize;
    uint32_t packedData = 0;
    int packedBytes = 0;
    while (pSrc < pSrcEnd)
    {
        packedData = (packedData << 8) | *pSrc++;
        ++packedBytes;
        if (packedBytes == 3)
        {
            SetBase64String(pDst, packedData, packedBytes);
            packedData = 0;
            packedBytes = 0;
            pDst += 4;
        }
    }
    if (packedBytes != 0)
    {
        SetBase64String(pDst, packedData, packedBytes);
        pDst += 4;
    }
    return pDst - reinterpret_cast<char*>(base64Buf);
}

//-----------------------------------------------------------------------------
//! @brief Base64 形式のデータをデコードした際のサイズを取得します。
//-----------------------------------------------------------------------------
size_t GetBase64DecodedSize(const void* base64Buf, const size_t base64Size)
{
    const char* pSrc = reinterpret_cast<const char*>(base64Buf);
    const char* pBottom = pSrc + base64Size - 1;
    while (pBottom >= pSrc && *pBottom == '=')
    {
        --pBottom;
    }
    const size_t validSize = pBottom - pSrc + 1;
    const int remainder4 = validSize % 4;
    //cerr << "get decoded size: " << validSize << " / " << base64Size << ", " << remainder4 << endl;
    return (validSize / 4) * 3 + (
        (remainder4 >= 3) ? 2 :
        (remainder4 >= 2) ? 1 :
        0);
}

//-----------------------------------------------------------------------------
//! @brief Base64 形式のデータをデコードします。
//-----------------------------------------------------------------------------
size_t DecodeFromBase64(void* dataBuf, const void* base64Buf, const size_t base64Size)
{
    const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(base64Buf);
    const uint8_t* pSrcEnd = pSrc + base64Size;
    uint8_t* pDst = reinterpret_cast<uint8_t*>(dataBuf);
    while (pSrc < pSrcEnd)
    {
        int charCount = 0;
        uint32_t packedData = 0;
        while (pSrc < pSrcEnd)
        {
            const int c = *pSrc++;
            if (c == '=')
            {
                break;
            }
            const int value = GetBase64CharValue(c);
            if (value != -1)
            {
                packedData = (packedData << 6) | value;
                ++charCount;
                if (charCount >= 4)
                {
                    break;
                }
            }
        }
        if (charCount == 0)
        {
            break;
        }

        int packedBytes = 3;
        if (charCount < 4)
        {
            packedData <<= (4 - charCount) * 6;
            packedBytes = (charCount == 2) ? 1 : 2;
        }
        for (int dstIdx = 0; dstIdx < packedBytes; ++dstIdx)
        {
            *pDst++ = static_cast<uint8_t>((packedData >> ((2 - dstIdx) * 8)) & 0xff);
        }
    }
    return pDst - reinterpret_cast<const uint8_t*>(dataBuf);
}

//-----------------------------------------------------------------------------
//! @brief 環境変数の値を取得します。
//-----------------------------------------------------------------------------
std::string RGetEnvVar(const char* name)
{
    std::string ret;
    char* pValue;
    size_t length;
    if (_dupenv_s(&pValue, &length, name) == 0)
    {
        // 環境変数が見つからない場合は、pValue が nullptr に
        // length が 0、戻り値が 0（成功）となります。
        if (pValue != nullptr)
        {
            ret = pValue;
            free(pValue);
        }
    }
    return ret;
}

//-----------------------------------------------------------------------------
//! @brief PC のユーザー名を返します。
//-----------------------------------------------------------------------------
std::string RGetUserName()
{
    const int userNameSize = 256;
    TCHAR userName[userNameSize];
    DWORD userNameSizeRet = userNameSize;
    GetUserName(userName, &userNameSizeRet);
    #ifdef UNICODE
    return RGetShiftJisFromUnicode(userName);
    #else
    return std::string(userName);
    #endif
}

//-----------------------------------------------------------------------------
//! @brief PC のコンピュータ名を返します。
//-----------------------------------------------------------------------------
std::string RGetComputerName()
{
    const int computerNameSize = MAX_COMPUTERNAME_LENGTH + 1;
    TCHAR computerName[computerNameSize];
    DWORD computerNameSizeRet = computerNameSize;
    GetComputerName(computerName, &computerNameSizeRet);
    #ifdef UNICODE
    return RGetShiftJisFromUnicode(computerName);
    #else
    return std::string(computerName);
    #endif
}

//-----------------------------------------------------------------------------
//! @brief Windows のシステムメッセージ文字列を取得します。
//-----------------------------------------------------------------------------
std::string GetWindowsSystemMessage(const int messageId)
{
    void* pMessage = nullptr;
    FormatMessageA(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM |
        FORMAT_MESSAGE_IGNORE_INSERTS, nullptr, messageId,
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), reinterpret_cast<LPSTR>(&pMessage), 0, nullptr);
    std::string message;
    if (pMessage != nullptr)
    {
        message = reinterpret_cast<const char*>(pMessage);
        LocalFree(pMessage);
        const size_t spaceIdx = message.find_last_not_of(" \t\r\n"); // 末尾の空白と改行を削除します。
        message = (spaceIdx != std::string::npos) ? message.substr(0, spaceIdx + 1) : "";
    }
    else
    {
        char strBuf[32];
        sprintf_s(strBuf, "0x%08X", messageId);
        message = strBuf;
    }
    return message;
}

//-----------------------------------------------------------------------------
//! @brief Windows の最後のエラーのメッセージ文字列を取得します。
//-----------------------------------------------------------------------------
std::string GetWindowsLastErrorMessage()
{
    return GetWindowsSystemMessage(GetLastError());
}

//-----------------------------------------------------------------------------
//! @brief SYSTEMTIME から time_t を取得して返します。
//-----------------------------------------------------------------------------
time_t RGetTimeT(const SYSTEMTIME& st)
{
    FILETIME ft;
    ::SystemTimeToFileTime(&st, &ft);

    LONGLONG ll;
    ll = (static_cast<LONGLONG>(ft.dwHighDateTime) << 32) + ft.dwLowDateTime;
    return static_cast<time_t>((ll - 116444736000000000) / 10000000);
}

//-----------------------------------------------------------------------------
//! @brief ISO 8601 に準拠した書式で現在の日時を返します。
//-----------------------------------------------------------------------------
std::string RGetDateTimeString()
{
    SYSTEMTIME time;
    GetLocalTime(&time);

    TIME_ZONE_INFORMATION tzi;
    GetTimeZoneInformation(&tzi);
    const int diffMin = -tzi.Bias; // UTC からの差（分単位）にするため符号反転します。

    char dtStr[64];
    sprintf_s(dtStr, "%d-%02d-%02dT%02d:%02d:%02d%c%02d:%02d",
        time.wYear, time.wMonth, time.wDay,
        time.wHour, time.wMinute, time.wSecond,
        (diffMin >= 0) ? '+' : '-', RAbs(diffMin) / 60, diffMin % 60);
    return std::string(dtStr);
}

//-----------------------------------------------------------------------------
//! @brief 指定されたモジュールを含む実行ファイルのフルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetModuleFileName(HMODULE hModule)
{
    TCHAR processPathBuf[MAX_PATH];
    if (GetModuleFileName(hModule, processPathBuf, MAX_PATH))
    {
        #ifdef UNICODE
        return RGetShiftJisFromUnicode(processPathBuf);
        #else
        return std::string(processPathBuf);
        #endif
    }
    else
    {
        return std::string();
    }
}

//-----------------------------------------------------------------------------
//! @brief ユーザーが使用できるアドレス空間の空き容量を返します。
//-----------------------------------------------------------------------------
size_t RGetAvailableVirtualMemorySize()
{
    MEMORYSTATUS ms;
    ms.dwLength = sizeof(MEMORYSTATUS);
    GlobalMemoryStatus(&ms);
    return ms.dwAvailVirtual;
    //return ms.dwAvailPhys;
    //return ms.dwAvailPageFile;
}

//-----------------------------------------------------------------------------
//! @brief テキストファイルの 1 行から引数のトークンを取得します。
//!
//! @param[in,out] tokens 取得した引数のトークンを追加する配列です。
//! @param[in] lineStr 1 行の文字列です。
//-----------------------------------------------------------------------------
void GetArgTokensFromLine(RStringArray& tokens, const std::string& lineStr)
{
    bool inQuat = false;
    std::string token;
    const char* str = lineStr.c_str();
    const size_t lineBytes = lineStr.size();
    for (size_t iByte = 0; iByte < lineBytes; ++iByte)
    {
        const char c = str[iByte];
        if (RIsSpace(c) && !inQuat)
        {
            if (!token.empty())
            {
                tokens.push_back(token);
                token = "";
            }
        }
        else
        {
            if (c == '\"')
            {
                inQuat = !inQuat;
            }
            token += c;
        }
    }

    if (!token.empty())
    {
        tokens.push_back(token);
    }
}

//-----------------------------------------------------------------------------
//! @brief テキストファイルから引数のトークンを取得します。
//!
//! @param[in,out] tokens 取得した引数のトークンを追加する配列です。
//! @param[in] filePath テキストファイルのパスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus GetArgTokensFromFile(RStringArray& tokens, const std::string& filePath)
{
    //-----------------------------------------------------------------------------
    // open file
    ifstream ifs(filePath.c_str());
    if (!ifs)
    {
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + filePath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // read
    std::string lineStr;
    while (!ifs.eof())
    {
        getline(ifs, lineStr);
        if (!lineStr.empty())
        {
            lineStr = RTrimString(lineStr);
            if (lineStr.find('#') != 0)
            {
                GetArgTokensFromLine(tokens, lineStr);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // close file
    ifs.close();

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 引数からオプション名を取得します。
//!
//! @param[out] pIsLongName ロングネームなら true、ショートネームなら false を格納します。
//! @param[in] arg 引数です。
//!
//! @return オプション名を返します。オプション引数でなければ空文字を返します。
//-----------------------------------------------------------------------------
namespace
{

std::string GetOptionNameFromArg(bool* pIsLongName, const std::string& arg)
{
    *pIsLongName = false;
    if (arg.size() >= 2 && arg[0] == '-')
    {
        *pIsLongName = (arg.size() >= 3 && arg[1] == '-');
        if (*pIsLongName)
        {
            const size_t eqIdx = arg.find('=');
            return arg.substr(2, (eqIdx != std::string::npos) ? eqIdx - 2 : std::string::npos);
        }
        else
        {
            return arg.substr(1, 1);
        }
    }
    return "";
}

} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 引数配列からオプション値を取得します。
//!
//! @param[out] pValue オプション値を格納します。
//! @param[in,out] pArgIdx 現在の引数インデックスです。次の引数がオプション値の場合は +1 した値を格納します。
//! @param[in] args 引数配列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus GetOptionArgValue(
    std::string* pValue,
    int* pArgIdx,
    const RStringArray& args
)
{
    const std::string& arg = args[*pArgIdx];
    const bool isLongName = (arg.find("--") == 0);
    const size_t eqIdx = arg.find('=');

    *pValue = "";
    if (isLongName && eqIdx != std::string::npos)
    {
        *pValue = arg.substr(eqIdx + 1);
    }
    else if (!isLongName && arg.size() >= 3)
    {
        *pValue = (arg[2] == '=') ? arg.substr(3) : arg.substr(2);
    }
    else if (*pArgIdx < static_cast<int>(args.size()) - 1)
    {
        *pArgIdx += 1;
        *pValue = args[*pArgIdx];
    }
    else
    {
        return RStatus(RStatus::FAILURE, "Option has no value: " + arg); // RShowError
    }
    *pValue = RDequoteString(*pValue);
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 引数ファイルが指定されていればファイル内の引数を展開します。
//!
//! @param[out] pDstArgs 引数配列を格納します。
//! @param[in] srcArgs 展開前の引数配列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ExpandArguments(RStringArray* pDstArgs, const RStringArray& srcArgs)
{
    RStatus status;
    RStringArray& dstArgs = *pDstArgs;
    dstArgs.clear();
    for (int argIdx = 0; argIdx < static_cast<int>(srcArgs.size()); ++argIdx)
    {
        const std::string& arg = srcArgs[argIdx];
        if (arg == "--args-file" || arg.find("--args-file=") == 0)
        {
            std::string argsFilePath;
            status = GetOptionArgValue(&argsFilePath, &argIdx, srcArgs);
            if (status)
            {
                status = GetArgTokensFromFile(dstArgs, argsFilePath);
            }
        }
        else
        {
            dstArgs.push_back(arg);
        }
        RCheckStatus(status);
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief ジョブリストファイルの 1 行を解析します。
//!
//! @param[in,out] pConvArg コンバーター引数へのポインタです。
//! @param[in] commonJobArg 共通ジョブ引数です。
//! @param[in] lineStr 1 行の文字列です。
//! @param[in] isGlobalOptionArgFunc 引数がグローバルオプションなら true を返す関数です。
//! @param[in] isNextArgJobOptionValueFunc 指定した引数の次の引数がジョブオプションの値なら true を返す関数です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
namespace
{

RStatus ParseJobListLine(
    ConverterArgument* pConvArg,
    const JobArgument& commonJobArg,
    const std::string& lineStr,
    IsGlobalOptionArgFunc isGlobalOptionArgFunc,
    IsNextArgJobOptionValueFunc isNextArgJobOptionValueFunc
)
{
    RStringArray srcArgs;
    GetArgTokensFromLine(srcArgs, lineStr);
    RStringArray args;
    RStatus status = ExpandArguments(&args, srcArgs);
    RCheckStatus(status);

    JobArgument jobArg = commonJobArg;
    for (int argIdx = 0; argIdx < static_cast<int>(args.size()); ++argIdx)
    {
        const std::string& arg = args[argIdx];
        const std::wstring argW = RGetUnicodeFromShiftJis(arg);
        if (arg.find('-') == 0)
        {
            if (isGlobalOptionArgFunc(argW.c_str()))
            {
                return RStatus(RStatus::FAILURE, "Wrong option is specified in the job list file: " + arg); // RShowError
            }
            else
            {
                jobArg.options.push_back(arg);
                if (isNextArgJobOptionValueFunc(argW.c_str()))
                {
                    ++argIdx;
                    if (argIdx < static_cast<int>(args.size()))
                    {
                        jobArg.options.push_back(args[argIdx]);
                    }
                }
            }
        }
        else
        {
            const std::string inputPath = RDequoteString(arg);
            jobArg.filePaths.push_back(RGetFullFilePath(inputPath, true));
        }
    }

    if (jobArg.filePaths.empty())
    {
        return RStatus(RStatus::FAILURE, "No input file: " + lineStr); // RShowError
    }
    pConvArg->jobArgs.push_back(jobArg);
    return RStatus::SUCCESS;
}

} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief ジョブリストファイルを解析します。
//!
//! @param[in,out] pConvArg コンバーター引数へのポインタです。
//! @param[in] commonJobArg 共通ジョブ引数です。
//! @param[in] jobListFilePath ジョブリストファイルのパスです。
//! @param[in] isGlobalOptionArgFunc 引数がグローバルオプションなら true を返す関数です。
//! @param[in] isNextArgJobOptionValueFunc 指定した引数の次の引数がジョブオプションの値なら true を返す関数です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseJobListFile(
    ConverterArgument* pConvArg,
    const JobArgument& commonJobArg,
    const std::string& jobListFilePath,
    IsGlobalOptionArgFunc isGlobalOptionArgFunc,
    IsNextArgJobOptionValueFunc isNextArgJobOptionValueFunc
)
{
    RStatus status;

    ifstream ifs(jobListFilePath.c_str());
    if (!ifs)
    {
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + jobListFilePath); // RShowError
    }

    std::string lineStr;
    while (!ifs.eof())
    {
        getline(ifs, lineStr);
        lineStr = RTrimString(lineStr);
        if (!lineStr.empty())
        {
            if (lineStr.find('#') != 0)
            {
                status = ParseJobListLine(pConvArg, commonJobArg, lineStr,
                    isGlobalOptionArgFunc, isNextArgJobOptionValueFunc);
                RCheckStatus(status);
            }
        }
    }
    ifs.close();
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 引数からオプション引数情報へのポインタを取得します。
//!
//! @param[out] pIsLongName ロングネームなら true、ショートネームなら false を格納します。
//! @param[out] pSuffixNum オプションの末尾の整数値を格納します。末尾が数値でない場合は -1 を格納します。
//! @param[in] optionArgInfos オプション引数情報配列です。
//! @param[in] arg 引数です。
//!
//! @return オプション引数情報へのポインタを返します。
//!         該当するオプション引数情報がなければ nullptr を返します。
//-----------------------------------------------------------------------------
const OptionArgInfo* GetOptionArgInfo(
    bool* pIsLongName,
    int* pSuffixNum,
    const OptionArgInfo* optionArgInfos,
    const std::string& arg
)
{
    const std::string optionName = GetOptionNameFromArg(pIsLongName, arg);
    if (!optionName.empty())
    {
        const std::string noSuffixName = RGetSuffixNumberFromString(pSuffixNum, optionName);

        const OptionArgInfo* pInfo = optionArgInfos;
        while (pInfo->longName != nullptr)
        {
            if (*pIsLongName)
            {
                const size_t longNameSize = strlen(pInfo->longName);
                if (*pSuffixNum != -1 && pInfo->longName[longNameSize - 1] == '#')
                {
                    if (noSuffixName == std::string(pInfo->longName, longNameSize - 1))
                    {
                        return pInfo;
                    }
                }
                else
                {
                    if (optionName == pInfo->longName)
                    {
                        return pInfo;
                    }
                }
            }
            else
            {
                if (pInfo->shortName != ' ' && optionName[0] == pInfo->shortName)
                {
                    return pInfo;
                }
            }
            ++pInfo;
        }
    }
    return nullptr;
}

//-----------------------------------------------------------------------------
//! @brief 指定した引数の次の引数がオプションの値なら true を返します。
//!
//! @param[in] optionArgInfos オプション引数情報配列です。
//! @param[in] arg 引数です。
//!
//! @brief 指定した引数の次の引数がオプションの値なら true を返します。
//-----------------------------------------------------------------------------
bool IsNextArgOptionValue(const OptionArgInfo* optionArgInfos, const std::string& arg)
{
    bool isLongName;
    int suffixNum;
    const OptionArgInfo* pInfo = GetOptionArgInfo(&isLongName, &suffixNum, optionArgInfos, arg);
    if (pInfo != nullptr && pInfo->requiresValue)
    {
        if (isLongName)
        {
            return (arg.find('=') == std::string::npos);
        }
        else
        {
            return (arg.size() <= 2);
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief オプション引数を解析します。
//!
//! @param[out] ppInfo オプション引数情報へのポインタを格納します。
//! @param[out] pSuffixNum オプションの末尾の整数値を格納します。末尾が数値でない場合は -1 を格納します。
//! @param[out] pValue オプション値を格納します。
//! @param[in,out] pArgIdx 現在の引数インデックスです。次の引数がオプション値の場合は +1 した値を格納します。
//! @param[in] args 引数配列です。
//! @param[in] optionArgInfos オプション引数情報配列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseOptionArg(
    const OptionArgInfo** ppInfo,
    int* pSuffixNum,
    std::string* pValue,
    int* pArgIdx,
    const RStringArray& args,
    const OptionArgInfo* optionArgInfos
)
{
    //-----------------------------------------------------------------------------
    // オプション引数情報へのポインタを取得します。
    const std::string& arg = args[*pArgIdx];
    bool isLongName;
    const OptionArgInfo* pInfo = GetOptionArgInfo(&isLongName, pSuffixNum, optionArgInfos, arg);
    *ppInfo = pInfo;
    if (pInfo == nullptr)
    {
        return RStatus(RStatus::FAILURE, "Option is wrong: " + arg); // RShowError
    }

    //-----------------------------------------------------------------------------
    // オプション値を取得します。
    const size_t eqIdx = arg.find('=');
    const bool isWithValue =
        ( isLongName && eqIdx != std::string::npos) ||
        (!isLongName && arg.size() >= 3           );
    if (pInfo->requiresValue)
    {
        if (isWithValue)
        {
            if (isLongName)
            {
                *pValue = arg.substr(eqIdx + 1);
            }
            else
            {
                *pValue = (arg[2] == '=') ? arg.substr(3) : arg.substr(2);
            }
        }
        else
        {
            if (*pArgIdx >= static_cast<int>(args.size()) - 1)
            {
                return RStatus(RStatus::FAILURE, "Option has no value: " + arg); // RShowError
            }
            *pArgIdx += 1;
            *pValue = args[*pArgIdx];
        }
        *pValue = RDequoteString(*pValue);
    }
    else // 値を必要としないオプション
    {
        if (isWithValue)
        {
            return RStatus(RStatus::FAILURE, "Option is wrong: " + arg); // RShowError
        }
        *pValue = "";
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief XML 要素のエラーを表示します。
//-----------------------------------------------------------------------------
void RXMLElement::DisplayError(const std::string& message) const
{
    if (m_ErrorFunc != nullptr)
    {
        m_ErrorFunc(message.c_str(), m_pErrorUserData);
    }
    else
    {
        cerr << message << endl;
    }
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の属性を解析してノードに記録します。
//-----------------------------------------------------------------------------
int RXMLElement::LoadAttributes(const char* src, const char end)
{
    int pos = 0;
    for (;;)
    {
        pos += static_cast<int>(strspn(&src[pos], " \t\r\n"));
        if (src[pos] == '>' || src[pos] == end)
        {
            break;
        }
        int namelen = static_cast<int>(strcspn(&src[pos], "="));
        std::string attrName(&src[pos], namelen);
        pos += namelen;
        std::string value;
        if (src[pos] == '=')
        {
            ++pos;
            if (src[pos] == '\"' || src[pos] == '\'')
            {
                char end2[2] = { src[pos], };
                ++pos;
                int valuelen = static_cast<int>(strcspn(&src[pos], end2));
                value = std::string(&src[pos], valuelen);
                pos += valuelen;
                pos += (src[pos] != '\0');
            }
            else
            {
                char end2[] = " \t\r\n";
                int valuelen = static_cast<int>(strcspn(&src[pos], end2));
                value = std::string(&src[pos], valuelen);
                pos += valuelen;
            }
        }
        attributes[attrName] = value;
    }
    return pos;
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の代入演算子です。
//-----------------------------------------------------------------------------
RXMLElement& RXMLElement::operator=(const RXMLElement& r)
{
    name = r.name;
    text = r.text;
    attributes.clear();
    for (AttrMap::const_iterator it = r.attributes.begin(); it != r.attributes.end(); ++it)
    {
        attributes[it->first] = it->second;
    }
    nodes.clear();
    nodes.resize(r.nodes.size());
    for (size_t i = 0; i < r.nodes.size(); ++i)
    {
        nodes[i] = r.nodes[i];
    }

    m_ErrorFunc = r.m_ErrorFunc;
    m_pErrorUserData = r.m_pErrorUserData;

    return *this;
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の指定のパスに該当する要素を指定個数分検索します。
//-----------------------------------------------------------------------------
void RXMLElement::FindElements(
    std::vector<const RXMLElement*>& list,
    const std::string& path,
    const bool showError,
    const int maxCount
) const
{
    // パス名を階層に区切ります。
    RStringArray entries;
    std::string subPath(path);
    for (size_t ic = 0; ic < subPath.size(); ++ic)
    {
        if (subPath[ic] == '\\')
        {
            subPath[ic] = '/';
        }
    }
    while (!subPath.empty())
    {
        int span = static_cast<int>(strcspn(subPath.c_str(), "/"));
        entries.push_back(subPath.substr(0, span));
        span += (subPath[span] == '/');
        subPath = subPath.substr(span);
    }
    const int entryCount = static_cast<int>(entries.size());

    // 階層の上から順に探索します（一意性が無いので総当たり）。
    int count = maxCount;
    std::vector<const RXMLElement*> stack(entryCount);
    std::vector<int> index(entryCount);
    int level = 0;
    index[0] = 0;
    stack[0] = this;
    const RXMLElement* node = this;
    int pos = 0;
    for (;;)
    {
        bool found = false;
        if (level >= entryCount)
        {
            // 全階層マッチしたら現在位置を記録します。
            list.push_back(node);
            if (--count <= 0)
            {
                // 指定の個数が見つかればここで終了です。
                break;
            }
        }
        else
        {
            // そうでなければ現在の階層を探索します。
            for (; pos < static_cast<int>(node->nodes.size()); ++pos)
            {
                if (node->nodes[pos].name == entries[level])
                {
                    found = true;
                    break;
                }
            }
        }
        if (found)
        {
            // マッチするノードがあればさらに下層へ。
            // 失敗時のために現在位置を退避します。
            index[level] = pos + 1;
            stack[level] = node;
            ++level;
            node = &node->nodes[pos];
            pos = 0;
        }
        else if (--level >= 0)
        {
            // 該当ノードが無ければスタックを戻ります。
            node = stack[level];
            pos = index[level];
        }
        else // 全要素の探索が終了した。
        {
            break;
        }
    }

    if (showError && list.size() == 0)
    {
        DisplayError("Element is not found: <" + path + ">");
    }
}

//-----------------------------------------------------------------------------
//! @brief XML 文書を解析してツリーを構築します（ASCII テキスト限定）。
//-----------------------------------------------------------------------------
void RXMLElement::LoadDocument(RXMLElement* decl, const std::string& xmlDoc)
{
    RXMLElement dummy;
    decl = (decl != nullptr) ? decl : &dummy;
    const char* src = xmlDoc.c_str();
    int len = static_cast<int>(xmlDoc.size());
    if (static_cast<uint8_t>(src[0]) == 0xef)
    {
        // BOM を読み捨てます。
        src += 3;
        len -= 3;
    }
    std::vector<RXMLElement*> stack;
    RXMLElement* node = this;
    stack.push_back(node);
    for (int pos = 0; pos < len; )
    {
        if (src[pos] == '\0')
        {
            break;
        }
        const int span = static_cast<int>(strcspn(&src[pos], "<"));
        node->text += std::string(&src[pos], span);
        pos += span;
        if (src[pos] == '<')
        {
            ++pos;
            if (src[pos] == '?') // 宣言要素。
            {
                ++pos;
                int namelen = static_cast<int>(strcspn(&src[pos], " \t\r\n?"));
                std::string nodeName(&src[pos], namelen);
                pos += namelen;
                pos += decl->LoadAttributes(&src[pos], '?');
                pos += (src[pos] == '?');
            }
            else // 開始・終了タグ。
            {
                bool endtag = (src[pos] == '/');
                pos += endtag;
                int namelen = static_cast<int>(strcspn(&src[pos], " \t\r\n/>"));
                std::string nodeName(&src[pos], namelen);
                pos += namelen;
                if (!endtag) // 開始タグ。
                {
                    node = node->CreateElement();
                    node->name = nodeName;
                    stack.push_back(node);
                    pos += node->LoadAttributes(&src[pos], '/');
                    endtag = (src[pos] == '/');
                    pos += endtag;
                }
                if (endtag) // 終了タグ。
                {
                    for (int i = static_cast<int>(stack.size()) - 1; i >= 0; --i)
                    {
                        if (stack[i]->name == nodeName)
                        {
                            if (i > 0 && stack[i] == node)
                            {
                                node = stack[i - 1];
                            }
                            stack.erase(stack.end() - 1);
                            break;
                        }
                    }
                }
                pos += static_cast<int>(strcspn(&src[pos], ">"));
            }
            pos += (src[pos] == '>');
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief XML 文書を取得します。
//-----------------------------------------------------------------------------
std::string RXMLElement::GetDocument(const int tabCount, const char* lf) const
{
    const int& tc = tabCount;
    std::string doc;
    const std::string txt = RTrimString(text);
    if (name.empty())
    {
        doc += std::string("<?xml version=\"1.0\" encoding=\"utf-8\"?>") + lf;
        for (size_t iChild = 0; iChild < nodes.size(); ++iChild)
        {
            doc += nodes[iChild].GetDocument(tc, lf);
        }
    }
    else
    {
        doc += std::string(RTab(tc)) + "<" + name;
        for (AttrMap::const_iterator it = attributes.begin(); it != attributes.end(); ++it)
        {
            doc += " " + it->first + "=\"" + it->second + "\"";
        }
        if (txt.empty() && nodes.size() == 0)
        {
            doc += std::string(" />") + lf;
        }
        else
        {
            doc += std::string(">") + lf;
            if (!txt.empty())
            {
                doc += RTab(tc + 1) + txt + lf;
            }
            for (size_t iChild = 0; iChild < nodes.size(); ++iChild)
            {
                doc += nodes[iChild].GetDocument(tc + 1, lf);
            }
            doc += std::string(RTab(tc)) + "</" + name + ">" + lf;
        }
    }
    return doc;
}

//-----------------------------------------------------------------------------
//! @brief グラフィックスツールの exe および dll ファイルのフルパスを取得します。
//-----------------------------------------------------------------------------
std::string RGetGraphicsToolsExeDllFullPath(const std::string& fileName, HMODULE hModule)
{
    const std::string moduleFolderPath = RGetFolderFromFilePath(RGetModuleFileName(hModule)) + "\\";
    #ifdef _M_IX86
    const std::string fullPath32 = moduleFolderPath + "x86\\" + fileName;
    if (RFileExists(fullPath32))
    {
        return fullPath32;
    }
    #endif
    return moduleFolderPath + fileName;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルバージョンを整数（major * 100 + minor）で返します。
//-----------------------------------------------------------------------------
int RGetNw4f3difVerAsInt(const std::string& value)
{
    int ver = 0;
    RStringArray tokens;
    if (RTokenizeString(tokens, value, ".") >= 2)
    {
        ver = atoi(tokens[0].c_str()) * 100 + atoi(tokens[1].c_str());
    }
    return ver;
}

//-----------------------------------------------------------------------------
// @brief データ列のバイト値用のコンストラクタです。
//-----------------------------------------------------------------------------
RDataStream::RDataStream(const uint8_t* array, const size_t bytes, const int column)
:   m_Type(BYTE),
    m_Column(column)
{
    m_ByteValues.resize(bytes);
    for (size_t iByte = 0; iByte < bytes; ++iByte)
    {
        m_ByteValues[iByte] = array[iByte];
    }
}

//-----------------------------------------------------------------------------
// @brief データ列にバイト値を追加します。
//-----------------------------------------------------------------------------
void RDataStream::Append(const uint8_t* array, const size_t bytes)
{
    const size_t prevSize = m_ByteValues.size();
    m_ByteValues.resize(prevSize + bytes);
    for (size_t iByte = 0; iByte < bytes; ++iByte)
    {
        m_ByteValues[prevSize + iByte] = array[iByte];
    }
}

//-----------------------------------------------------------------------------
// @brief データ列のデータへのポインターを取得します。
//-----------------------------------------------------------------------------
const void* RDataStream::GetDataPtr() const
{
    switch (m_Type)
    {
    case FLOAT:
        return m_FloatValues.data();
    case INT:
        return m_IntValues.data();
    case BYTE:
        return m_ByteValues.data();
    case STRING:
    case WSTRING:
        return m_StringValue.c_str();
    default:
        return nullptr;
    }
}

//-----------------------------------------------------------------------------
//! @brief 整数型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount インデントに必要なタブの数です。
//! @param[in] array データ値配列です。
//! @param[in] column 列数です。
//-----------------------------------------------------------------------------
void ROutDataValues(
    std::ostream& os,
    const int tabCount,
    const RIntArray& array,
    const int column
)
{
    const int valSize = static_cast<int>(array.size());
    if (valSize == 0)
    {
        return;
    }

    const int& tc = tabCount;
    int ic = 0;
    for (int iVal = 0; iVal < valSize; ++iVal)
    {
        if (ic == 0)
        {
            os << RTab(tc);
        }
        os << array[iVal];
        if (iVal < valSize - 1)
        {
            if (ic == column - 1)
            {
                os << R_ENDL;
            }
            else
            {
                os << " ";
            }
        }
        if (++ic >= column)
        {
            ic = 0;
        }
    }
    os << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 浮動小数点数型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount インデントに必要なタブの数です。
//! @param[in] array データ値配列です。
//! @param[in] column 列数です。
//-----------------------------------------------------------------------------
void ROutDataValues(
    std::ostream& os,
    const int tabCount,
    const RFloatArray& array,
    const int column
)
{
    const int valSize = static_cast<int>(array.size());
    if (valSize == 0)
    {
        return;
    }

    const int& tc = tabCount;
    int ic = 0;
    for (int iVal = 0; iVal < valSize; ++iVal)
    {
        if (ic == 0)
        {
            os << RTab(tc);
        }
        os << array[iVal];
        if (iVal < valSize - 1)
        {
            if (ic == column - 1)
            {
                os << R_ENDL;
            }
            else
            {
                os << " ";
            }
        }
        if (++ic >= column)
        {
            ic = 0;
        }
    }
    os << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief バイト型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount インデントに必要なタブの数です。
//! @param[in] array データ値配列です。
//! @param[in] column 列数です。
//-----------------------------------------------------------------------------
void ROutDataValues(
    std::ostream& os,
    const int tabCount,
    const RUCharArray& array,
    const int column
)
{
    //-----------------------------------------------------------------------------
    // check size
    const size_t valSize = array.size();
    if (valSize == 0)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // out
    const int& tc = tabCount;
    int ic = 0;
    for (size_t iVal = 0; iVal < valSize; ++iVal)
    {
        if (ic == 0)
        {
            os << RTab(tc);
        }
        const uint32_t val = static_cast<uint32_t>(array[iVal]);
        const uint32_t vh = val >> 4;
        const uint32_t vl = val & 0x0f;
        os << static_cast<char>((vh >= 10) ? 'A' + vh - 10 : '0' + vh)
           << static_cast<char>((vl >= 10) ? 'A' + vl - 10 : '0' + vl);
        if (ic == column - 1 || iVal == valSize - 1)
        {
            os << R_ENDL;
        }
        else if (column == 12)
        {
            os << (((ic % 3) == 2) ? "  " : " ");
        }
        else
        {
            os << (((ic & 3) == 3) ? "  " : " ");
        }
        if (++ic >= column)
        {
            ic = 0;
        }
    }
}

//-----------------------------------------------------------------------------
// @brief データ列を出力します。
//-----------------------------------------------------------------------------
void RDataStream::Out(std::ostream& os, const int tabCount, const int index) const
{
    static const char* const typeStrs[] = { "float", "int", "byte", "string", "wstring" };

    const int& tc = tabCount;
    const size_t count = GetCount();
    if (count > 0)
    {
        const bool isString = (m_Type == STRING || m_Type == WSTRING);
        os << RTab(tc) << "<stream stream_index=\"" << index
           << "\" type=\"" << typeStrs[m_Type]
           << "\" count=\"" << count
           << "\" column=\"" << m_Column
           << "\">";
        if (!isString)
        {
            os << R_ENDL;
        }

        if (m_Type == FLOAT)
        {
            ROutDataValues(os, tc + 1, m_FloatValues, m_Column);
        }
        else if (m_Type == INT)
        {
            ROutDataValues(os, tc + 1, m_IntValues, m_Column);
        }
        else if (m_Type == BYTE)
        {
            ROutDataValues(os, tc + 1, m_ByteValues, m_Column);
        }
        else // STRING, WSTRING
        {
            os << RGetUtf8FromShiftJis(REncodeXmlString(m_StringValue));
        }

        if (!isString)
        {
            os << RTab(tc);
        }
        os << "</stream>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
// @brief データ列配列を出力します。
//-----------------------------------------------------------------------------
void ROutDataStreams(std::ostream& os, const RDataStreamArray& array)
{
    ROutArrayElement(os, 0, array, "stream_array");
}

//-----------------------------------------------------------------------------
//! @brief データ列配をバイナリー形式で出力します。
//-----------------------------------------------------------------------------
void RDataStream::OutBinary(std::ostream& os) const
{
    static const char* const STREAM_IDENTIFIER = "stream  ";

    //-----------------------------------------------------------------------------
    // ヘッダを出力します。
    os.write(STREAM_IDENTIFIER, strlen(STREAM_IDENTIFIER));

    int binType = 0;
    switch (m_Type)
    {
    case FLOAT   : binType = 0; break;
    case INT     : binType = 1; break;
    case BYTE    : binType = 2; break;
    case STRING  : binType = 3; break;
    case WSTRING : binType = 4; break;
    default:
        break;
    }
    os.write(reinterpret_cast<const char*>(&binType), sizeof(binType));

    const uint32_t count = static_cast<uint32_t>(GetCount());
    os.write(reinterpret_cast<const char*>(&count), sizeof(count));

    os.write(reinterpret_cast<const char*>(&m_Column), sizeof(m_Column));

    const uint32_t dataSize = static_cast<uint32_t>(GetSize());
    os.write(reinterpret_cast<const char*>(&dataSize), sizeof(dataSize));

    uint32_t padding = 0;
    os.write(reinterpret_cast<const char*>(&padding), sizeof(padding));
    os.write(reinterpret_cast<const char*>(&padding), sizeof(padding));

    //-----------------------------------------------------------------------------
    // データを出力します。
    uint8_t* dataBuf = new uint8_t[dataSize];
    uint8_t* dst = dataBuf;
    if (m_Type == FLOAT)
    {
        for (uint32_t iData = 0; iData < count; ++iData, dst += sizeof(float))
        {
            memcpy(dst, &m_FloatValues[iData], sizeof(float));
        }
    }
    else if (m_Type == INT)
    {
        for (uint32_t iData = 0; iData < count; ++iData, dst += sizeof(int))
        {
            memcpy(dst, &m_IntValues[iData], sizeof(int));
        }
    }
    else if (m_Type == BYTE)
    {
        for (uint32_t iData = 0; iData < count; ++iData)
        {
            dst[iData] = m_ByteValues[iData];
        }
    }
    else if (m_Type == STRING)
    {
        memcpy(dst, m_StringValue.c_str(), m_StringValue.size());
        dst[m_StringValue.size()] = '\0'; // 終端文字を出力します。
    }
    else if (m_Type == WSTRING)
    {
        const std::wstring wstr = RGetUnicodeFromShiftJis(m_StringValue);
        memcpy(dst, wstr.c_str(), wstr.size() * sizeof(wchar_t));
        *(reinterpret_cast<wchar_t*>(dst) + wstr.size()) = 0; // 終端文字を出力します。
    }
    os.write(reinterpret_cast<const char*>(dataBuf), GetSize());
    delete[] dataBuf;

    //-----------------------------------------------------------------------------
    // パディングを出力します。
    ROutPadding(os, GetBinarySize() - (HEADER_SIZE + GetSize()));
}

//-----------------------------------------------------------------------------
//! @brief データ列配列をバイナリー形式で出力します。
//-----------------------------------------------------------------------------
void ROutBinaryDataStreams(
    std::ostream& os,
    const RDataStreamArray& dataStreams,
    const bool outZeroPadding
)
{
    // 参考
    // Tool\G3dTool\g3diflib\build\src\Io\Format\IfBinaryFormatter.cs
    // Tool\G3dTool\g3diflib\build\src\Data\streamBinaryType.cs の Write

    static const char* const STREAM_ARRAY_IDENTIFIER = "g3dstrma";

    //-----------------------------------------------------------------------------
    // データ列がなければ何も出力しません。
    const int streamCount = static_cast<int>(dataStreams.size());
    if (streamCount == 0)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // バイナリー形式の後半部の前のゼロパディングを出力します。
    if (outZeroPadding)
    {
        const size_t xmlSize = static_cast<int>(os.tellp());
        const size_t binOfs = RAlignValue(xmlSize + 1, R_BINARY_ALIGNMENT);
        ROutPadding(os, binOfs - xmlSize);
    }

    //-----------------------------------------------------------------------------
    // ストリーム配列チャンクを出力します。
    os.write(STREAM_ARRAY_IDENTIFIER, strlen(STREAM_ARRAY_IDENTIFIER));
    os.write(reinterpret_cast<const char*>(&streamCount), sizeof(streamCount));
    const uint32_t headerSize = 8 + 4 + (4 + 4) * streamCount;
    const uint32_t streamAddrTop = RAlignValue(headerSize, R_BINARY_ALIGNMENT);

    uint32_t streamAddr = streamAddrTop;
    for (int istream = 0; istream < streamCount; ++istream)
    {
        const uint32_t streamSize = static_cast<uint32_t>(dataStreams[istream].GetBinarySize());
        os.write(reinterpret_cast<const char*>(&streamAddr), sizeof(streamAddr));
        os.write(reinterpret_cast<const char*>(&streamSize), sizeof(streamSize));
        streamAddr += streamSize;
    }
    ROutPadding(os, streamAddrTop - headerSize);

    //-----------------------------------------------------------------------------
    // 各ストリームチャンクを出力します。
    for (int istream = 0; istream < streamCount; ++istream)
    {
        dataStreams[istream].OutBinary(os);
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの先頭からバイナリーデータまでのオフセットを取得します。
//-----------------------------------------------------------------------------
size_t RGetBinaryOffset(const void* fileBuf, const size_t fileSize)
{
    size_t binOfs = fileSize;
    const uint8_t* buf = reinterpret_cast<const uint8_t*>(fileBuf);
    for (size_t iBuf = 0; iBuf < fileSize; ++iBuf)
    {
        if (buf[iBuf] == 0x00)
        {
            binOfs = RAlignValue(iBuf + 1, R_BINARY_ALIGNMENT);
            break;
        }
    }
    return binOfs;
}

//-----------------------------------------------------------------------------
//! @brief バイナリーデータ列へのポインタを取得します（static 関数）。
//-----------------------------------------------------------------------------
const RBinDataStream* RBinDataStream::Get(const void* binTop, const int streamIdx)
{
    const RBinDataStream* streamTop = nullptr;
    const uint8_t* src = reinterpret_cast<const uint8_t*>(binTop);
    src += 8; // "g3dstrma"
    const int streamCount = *reinterpret_cast<const int*>(src);
    src += sizeof(int);
    if (streamIdx < streamCount)
    {
        src += (4 + 4) * streamIdx; // address, size
        const int streamAddr = *reinterpret_cast<const int*>(src);
        src = reinterpret_cast<const uint8_t*>(binTop) + streamAddr;
        streamTop = reinterpret_cast<const RBinDataStream*>(src);
    }
    return streamTop;
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータのコンストラクタです。
//-----------------------------------------------------------------------------
RUserData::RUserData(
    const RXMLElement* userDataElem,
    const RXMLElement* streamsElem,
    const void* binTop
)
{
    m_Name = RGetShiftJisFromUtf8(userDataElem->GetAttribute("name"));

    const RXMLElement* valueElem = &(userDataElem->nodes[0]);
    if (valueElem->name == "user_float" || valueElem->name == "user_int")
    {
        //-----------------------------------------------------------------------------
        // 数値型ユーザーデータを解析します。
        m_Type = (valueElem->name == "user_float") ? FLOAT : INT;
        m_NumberValueCount = atoi(valueElem->GetAttribute("count").c_str());
        if (m_NumberValueCount > 0)
        {
            m_NumberValuesText = valueElem->text;
        }
    }
    else if (valueElem->name == "user_string" || valueElem->name == "user_wstring")
    {
        //-----------------------------------------------------------------------------
        // 文字列型ユーザーデータを解析します。
        m_Type = (valueElem->name == "user_string") ? STRING : WSTRING;
        for (size_t iString = 0; iString < valueElem->nodes.size(); ++iString)
        {
            const RXMLElement* stringElem = &(valueElem->nodes[iString]);
            m_StringValues.push_back(
                RDecodeXmlString(RGetShiftJisFromUtf8(stringElem->text)));
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // ストリーム型ユーザーデータを解析します。
        m_Type = STREAM;
        const int iStream = atoi(valueElem->GetAttribute("stream_index").c_str());
        if (iStream == -1) // データなし
        {

        }
        else if (streamsElem != nullptr)
        {
            const RXMLElement* streamElem = &(streamsElem->nodes[iStream]);
            const int dataCount = atoi(streamElem->GetAttribute("count").c_str());
            m_StreamValues.resize(dataCount);
            int iData = 0;
            std::istringstream iss(streamElem->text.c_str());
            while (iss && iData < dataCount)
            {
                std::string value;
                iss >> value;
                if (!value.empty())
                {
                    m_StreamValues[iData++] = static_cast<uint8_t>(strtol(value.c_str(), nullptr, 16));
                }
            }
        }
        else
        {
            const RBinDataStream* binStream = RBinDataStream::Get(binTop, iStream);
            m_StreamValues.resize(binStream->m_Count);
            const unsigned char* pSrc = reinterpret_cast<const unsigned char*>(binStream->m_Data);
            for (int iData = 0; iData < binStream->m_Count; ++iData)
            {
                m_StreamValues[iData] = *pSrc++;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータを出力します。
//-----------------------------------------------------------------------------
void RUserData::Out(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const int tabCount,
    const int index
) const
{
    static const char* const valueElemNames[] =
    {
        "user_float",
        "user_int",
        "user_string",
        "user_wstring",
        "user_stream",
    };

    const int& tc = tabCount;
    os << RTab(tc) << "<user_data index=\"" << index
       << "\" name=\"" << RGetUtf8FromShiftJis(m_Name)
       << "\">" << R_ENDL;

    const char* valueElemName = valueElemNames[m_Type];
    os << RTab(tc + 1) << "<" << valueElemName;
    if (m_Type != STREAM)
    {
        int dataCount = 0;
        switch (m_Type)
        {
        case FLOAT: dataCount = m_NumberValueCount; break;
        case INT  : dataCount = m_NumberValueCount; break;
        default   : dataCount = static_cast<int>(m_StringValues.size()); break;
        }
        os << " count=\"" << dataCount << "\"";
        if (dataCount == 0)
        {
            os << " />" << R_ENDL;
        }
        else
        {
            os << ">";
            if (m_Type == FLOAT || m_Type == INT)
            {
                // 現在は中間ファイルの文字列をそのまま取得して出力します。
                // 値の配列から出力する場合は次のコードと同様に処理します。
                //   3dTools\3dIntermediateFileSerializer\Format\G3dDataFormatter.cs
                //   FormatUserIntData、FormatUserFloatData
                // (4 タブの間隔を空けつつ一行あたり 4 つの値)
                // (符号が＋の場合は数値の前にスペースを挿入)
                os << m_NumberValuesText;
            }
            else // STRING, WSTRING
            {
                os << R_ENDL;
                const char* strElemName = (m_Type == STRING) ? "string" : "wstring";
                for (int iData = 0; iData < dataCount; ++iData)
                {
                    os << RTab(tc + 2) << "<" << strElemName << ">"
                       << RGetUtf8FromShiftJis(REncodeXmlString(m_StringValues[iData]))
                       << "</" << strElemName << ">" << R_ENDL;
                }
                os << RTab(tc + 1);
            }
            os << "</" << valueElemName << ">" << R_ENDL;
        }
    }
    else // STREAM
    {
        const int dataCount = static_cast<int>(m_StreamValues.size());
        const int iStream = (dataCount > 0) ? static_cast<int>(dataStreams.size()) : -1;
        os << " size=\"" << dataCount
           << "\" stream_index=\"" << iStream
           << "\" />" << R_ENDL;
        if (dataCount > 0)
        {
            dataStreams.push_back(RDataStream(&m_StreamValues[0], dataCount, 16));
        }
    }

    os << RTab(tc) << "</user_data>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief C 言語のソース形式でバイト型のデータ値配列を出力します。
//-----------------------------------------------------------------------------
void ROutCSourceData(
    std::ostream& os,
    const int tabCount,
    const RUCharArray& array,
    const int column
)
{
    //-----------------------------------------------------------------------------
    // check size
    const int valSize = static_cast<int>(array.size());
    if (valSize == 0)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // set os style
    const char fillBak = os.fill('0');
    const long flagBak = os.setf(ios_base::hex, ios_base::basefield);
    os.setf(ios_base::uppercase);
    os.setf(ios_base::right, ios_base::adjustfield);

    //-----------------------------------------------------------------------------
    // out
    const int& tc = tabCount;
    int ic = 0;
    for (int iVal = 0; iVal < valSize; ++iVal)
    {
        if (ic == 0)
        {
            os << RTab(tc);
        }
        os << "0x" << setw(2) << static_cast<unsigned int>(array[iVal]) << ",";
        if (iVal < valSize - 1)
        {
            if (ic == column - 1)
            {
                os << R_ENDL;
            }
            else
            {
                //os << (((ic & 3) == 3) ? "  " : " ");
            }
        }
        if (++ic >= column)
        {
            ic = 0;
        }
    }
    os << R_ENDL;

    //-----------------------------------------------------------------------------
    // restore os style
    os.fill(fillBak);
    os.flags(flagBak);
}

//-----------------------------------------------------------------------------
//! @brief ファイルバッファにファイルをリードします。
//-----------------------------------------------------------------------------
bool RFileBuf::Read(const std::string& path)
{
    RFreeAndClearArray(m_FileBuf);

    m_FilePath = path;
    m_FileSize = 0;

    std::ifstream ifs(path.c_str(), ios_base::in | ios_base::binary);
    if (ifs)
    {
        //-----------------------------------------------------------------------------
        // read file
        ifs.seekg(0L, ios_base::end);
        m_FileSize = static_cast<size_t>(ifs.tellg());
        ifs.seekg(0L, ios_base::beg);
        if (m_FileSize > 0)
        {
            m_FileBuf = new uint8_t[m_FileSize];
            ifs.read(reinterpret_cast<char*>(m_FileBuf), m_FileSize);
        }
        else
        {
            m_FileBuf = new uint8_t[1];
        }

        //-----------------------------------------------------------------------------
        // close file
        ifs.close();
        return true;
    }
    else
    {
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルのファイルパスなら true を返します。
//-----------------------------------------------------------------------------
bool IsFtxFilePath(const std::string& filePath)
{
    const std::string ext = RGetExtensionFromFilePath(filePath);
    return (ext == FtxaExtension || ext == FtxbExtension);
}

//-----------------------------------------------------------------------------
//! @brief バイナリー形式の ftx ファイルのファイルパスなら true を返します。
//-----------------------------------------------------------------------------
bool IsFtxBinaryFilePath(const std::string& filePath)
{
    return (RGetExtensionFromFilePath(filePath) == FtxbExtension);
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列からテクスチャーの次元を取得します。
//-----------------------------------------------------------------------------
FtxDimension RGetTextureDimensionFromString(const std::string& str)
{
    return
        (str == "1d"           ) ? FtxDimension_1d                 :
        (str == "2d"           ) ? FtxDimension_2d                 :
        (str == "3d"           ) ? FtxDimension_3d                 :
        (str == "cube"         ) ? FtxDimension_CubeMap            :
        (str == "cube_array"   ) ? FtxDimension_CubeMapArray       :
        (str == "1d_array"     ) ? FtxDimension_1dArray            :
        (str == "2d_array"     ) ? FtxDimension_2dArray            :
        (str == "2d_msaa"      ) ? FtxDimension_2dMultisample      :
        (str == "2d_msaa_array") ? FtxDimension_2dMultisampleArray :
        FtxDimension_2d;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーの次元から中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetTextureDimensionString(const FtxDimension dimension)
{
    switch (dimension)
    {
    case FtxDimension_1d:                 return "1d";
    case FtxDimension_2d:                 return "2d";
    case FtxDimension_3d:                 return "3d";
    case FtxDimension_CubeMap:            return "cube";
    case FtxDimension_CubeMapArray:       return "cube_array";
    case FtxDimension_1dArray:            return "1d_array";
    case FtxDimension_2dArray:            return "2d_array";
    case FtxDimension_2dMultisample:      return "2d_msaa";
    case FtxDimension_2dMultisampleArray: return "2d_msaa_array";
    default:                              return "invalid";
    }
}

//-----------------------------------------------------------------------------
//! @brief フォーマットと文字列のペアのデータです。
//-----------------------------------------------------------------------------
namespace {

const RIntAndCharPtr s_FormatStrings[] =
{
    { FtxFormat_Unorm_8      , "unorm_8"       },
    { FtxFormat_Snorm_8      , "snorm_8"       },
    { FtxFormat_Uint_8       , "uint_8"        },
    { FtxFormat_Sint_8       , "sint_8"        },
    { FtxFormat_Unorm_16     , "unorm_16"      },
    { FtxFormat_Snorm_16     , "snorm_16"      },
    { FtxFormat_Uint_16      , "uint_16"       },
    { FtxFormat_Sint_16      , "sint_16"       },
    { FtxFormat_Uint_32      , "uint_32"       },
    { FtxFormat_Sint_32      , "sint_32"       },

    { FtxFormat_Unorm_4_4    , "unorm_4_4"     },
    { FtxFormat_Unorm_8_8    , "unorm_8_8"     },
    { FtxFormat_Snorm_8_8    , "snorm_8_8"     },
    { FtxFormat_Uint_8_8     , "uint_8_8"      },
    { FtxFormat_Sint_8_8     , "sint_8_8"      },
    { FtxFormat_Unorm_16_16  , "unorm_16_16"   },
    { FtxFormat_Snorm_16_16  , "snorm_16_16"   },
    { FtxFormat_Uint_16_16   , "uint_16_16"    },
    { FtxFormat_Sint_16_16   , "sint_16_16"    },
    { FtxFormat_Uint_32_32   , "uint_32_32"    },
    { FtxFormat_Sint_32_32   , "sint_32_32"    },

    { FtxFormat_Unorm_5_6_5  , "unorm_5_6_5"   },

    { FtxFormat_Unorm_5_5_5_1, "unorm_5_5_5_1" },
    { FtxFormat_Unorm_4_4_4_4, "unorm_4_4_4_4" },

    { FtxFormat_Unorm_8_8_8_8    , "unorm_8_8_8_8"     },
    { FtxFormat_Snorm_8_8_8_8    , "snorm_8_8_8_8"     },
    { FtxFormat_Srgb_8_8_8_8     , "srgb_8_8_8_8"      },
    { FtxFormat_Uint_8_8_8_8     , "uint_8_8_8_8"      },
    { FtxFormat_Sint_8_8_8_8     , "sint_8_8_8_8"      },
    { FtxFormat_Unorm_10_10_10_2 , "unorm_10_10_10_2"  },
    { FtxFormat_Uint_10_10_10_2  , "uint_10_10_10_2"   },
    { FtxFormat_Unorm_16_16_16_16, "unorm_16_16_16_16" },
    { FtxFormat_Snorm_16_16_16_16, "snorm_16_16_16_16" },
    { FtxFormat_Uint_16_16_16_16 , "uint_16_16_16_16"  },
    { FtxFormat_Sint_16_16_16_16 , "sint_16_16_16_16"  },
    { FtxFormat_Uint_32_32_32_32 , "uint_32_32_32_32"  },
    { FtxFormat_Sint_32_32_32_32 , "sint_32_32_32_32"  },

    { FtxFormat_Float_16         , "float_16"          },
    { FtxFormat_Float_32         , "float_32"          },
    { FtxFormat_Float_16_16      , "float_16_16"       },
    { FtxFormat_Float_32_32      , "float_32_32"       },
    { FtxFormat_Float_11_11_10   , "float_11_11_10"    },
    { FtxFormat_Float_16_16_16_16, "float_16_16_16_16" },
    { FtxFormat_Float_32_32_32_32, "float_32_32_32_32" },

    { FtxFormat_Unorm_Bc1, "unorm_bc1" },
    { FtxFormat_Srgb_Bc1 , "srgb_bc1"  },
    { FtxFormat_Unorm_Bc2, "unorm_bc2" },
    { FtxFormat_Srgb_Bc2 , "srgb_bc2"  },
    { FtxFormat_Unorm_Bc3, "unorm_bc3" },
    { FtxFormat_Srgb_Bc3 , "srgb_bc3"  },
    { FtxFormat_Unorm_Bc4, "unorm_bc4" },
    { FtxFormat_Snorm_Bc4, "snorm_bc4" },
    { FtxFormat_Unorm_Bc5, "unorm_bc5" },
    { FtxFormat_Snorm_Bc5, "snorm_bc5" },

    { FtxFormat_Ufloat_Bc6, "ufloat_bc6" },
    { FtxFormat_Float_Bc6 , "float_bc6"  },
    { FtxFormat_Unorm_Bc7 , "unorm_bc7"  },
    { FtxFormat_Srgb_Bc7  , "srgb_bc7"   },

    { FtxFormat_Unorm_Etc1      , "unorm_etc1"       },
    { FtxFormat_Unorm_Etc2      , "unorm_etc2"       },
    { FtxFormat_Srgb_Etc2       , "srgb_etc2"        },
    { FtxFormat_Unorm_Etc2_Mask , "unorm_etc2_mask"  },
    { FtxFormat_Srgb_Etc2_Mask  , "srgb_etc2_mask"   },
    { FtxFormat_Unorm_Etc2_Alpha, "unorm_etc2_alpha" },
    { FtxFormat_Srgb_Etc2_Alpha , "srgb_etc2_alpha"  },

    { FtxFormat_Unorm_Eac_11   , "unorm_eac_11"    },
    { FtxFormat_Snorm_Eac_11   , "snorm_eac_11"    },
    { FtxFormat_Unorm_Eac_11_11, "unorm_eac_11_11" },
    { FtxFormat_Snorm_Eac_11_11, "snorm_eac_11_11" },

    { FtxFormat_Unorm_Pvrtc1_2Bpp      , "unorm_pvrtc1_2bpp"       },
    { FtxFormat_Srgb_Pvrtc1_2Bpp       , "srgb_pvrtc1_2bpp"        },
    { FtxFormat_Unorm_Pvrtc1_4Bpp      , "unorm_pvrtc1_4bpp"       },
    { FtxFormat_Srgb_Pvrtc1_4Bpp       , "srgb_pvrtc1_4bpp"        },
    { FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp, "unorm_pvrtc1_alpha_2bpp" },
    { FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp , "srgb_pvrtc1_alpha_2bpp"  },
    { FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp, "unorm_pvrtc1_alpha_4bpp" },
    { FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp , "srgb_pvrtc1_alpha_4bpp"  },
    { FtxFormat_Unorm_Pvrtc2_Alpha_2Bpp, "unorm_pvrtc2_alpha_2bpp" },
    { FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp , "srgb_pvrtc2_alpha_2bpp"  },
    { FtxFormat_Unorm_Pvrtc2_Alpha_4Bpp, "unorm_pvrtc2_alpha_4bpp" },
    { FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp , "srgb_pvrtc2_alpha_4bpp"  },

    { FtxFormat_Unorm_Astc_4x4  , "unorm_astc_4x4"   },
    { FtxFormat_Srgb_Astc_4x4   , "srgb_astc_4x4"    },
    { FtxFormat_Unorm_Astc_5x4  , "unorm_astc_5x4"   },
    { FtxFormat_Srgb_Astc_5x4   , "srgb_astc_5x4"    },
    { FtxFormat_Unorm_Astc_5x5  , "unorm_astc_5x5"   },
    { FtxFormat_Srgb_Astc_5x5   , "srgb_astc_5x5"    },
    { FtxFormat_Unorm_Astc_6x5  , "unorm_astc_6x5"   },
    { FtxFormat_Srgb_Astc_6x5   , "srgb_astc_6x5"    },
    { FtxFormat_Unorm_Astc_6x6  , "unorm_astc_6x6"   },
    { FtxFormat_Srgb_Astc_6x6   , "srgb_astc_6x6"    },
    { FtxFormat_Unorm_Astc_8x5  , "unorm_astc_8x5"   },
    { FtxFormat_Srgb_Astc_8x5   , "srgb_astc_8x5"    },
    { FtxFormat_Unorm_Astc_8x6  , "unorm_astc_8x6"   },
    { FtxFormat_Srgb_Astc_8x6   , "srgb_astc_8x6"    },
    { FtxFormat_Unorm_Astc_8x8  , "unorm_astc_8x8"   },
    { FtxFormat_Srgb_Astc_8x8   , "srgb_astc_8x8"    },
    { FtxFormat_Unorm_Astc_10x5 , "unorm_astc_10x5"  },
    { FtxFormat_Srgb_Astc_10x5  , "srgb_astc_10x5"   },
    { FtxFormat_Unorm_Astc_10x6 , "unorm_astc_10x6"  },
    { FtxFormat_Srgb_Astc_10x6  , "srgb_astc_10x6"   },
    { FtxFormat_Unorm_Astc_10x8 , "unorm_astc_10x8"  },
    { FtxFormat_Srgb_Astc_10x8  , "srgb_astc_10x8"   },
    { FtxFormat_Unorm_Astc_10x10, "unorm_astc_10x10" },
    { FtxFormat_Srgb_Astc_10x10 , "srgb_astc_10x10"  },
    { FtxFormat_Unorm_Astc_12x10, "unorm_astc_12x10" },
    { FtxFormat_Srgb_Astc_12x10 , "srgb_astc_12x10"  },
    { FtxFormat_Unorm_Astc_12x12, "unorm_astc_12x12" },
    { FtxFormat_Srgb_Astc_12x12 , "srgb_astc_12x12"  },

    { FtxFormat_Uint_32_32_32 , "uint_32_32_32"  },
    { FtxFormat_Sint_32_32_32 , "sint_32_32_32"  },
    { FtxFormat_Float_32_32_32, "float_32_32_32" },
};

} // namespace

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列からテクスチャーフォーマットを取得します。
//-----------------------------------------------------------------------------
FtxFormat RGetTextureFormatFromString(const std::string& str)
{
    const int fmtCount = sizeof(s_FormatStrings) / sizeof(RIntAndCharPtr);
    for (int fmtIdx = 0; fmtIdx < fmtCount; ++fmtIdx)
    {
        const RIntAndCharPtr& fmtStr = s_FormatStrings[fmtIdx];
        if (str == fmtStr.m_String)
        {
            return static_cast<FtxFormat>(fmtStr.m_Value);
        }
    }
    return FtxFormat_Invalid;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットから中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetTextureFormatString(const FtxFormat format)
{
    const int fmtCount = sizeof(s_FormatStrings) / sizeof(RIntAndCharPtr);
    for (int fmtIdx = 0; fmtIdx < fmtCount; ++fmtIdx)
    {
        const RIntAndCharPtr& fmtStr = s_FormatStrings[fmtIdx];
        if (format == fmtStr.m_Value)
        {
            return fmtStr.m_String;
        }
    }
    return "invalid";
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが BC1/BC2/BC3 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RIsBc123TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Bc1:
    case FtxFormat_Srgb_Bc1:
    case FtxFormat_Unorm_Bc2:
    case FtxFormat_Srgb_Bc2:
    case FtxFormat_Unorm_Bc3:
    case FtxFormat_Srgb_Bc3:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが BC6/BC7 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RIsBc67TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Ufloat_Bc6:
    case FtxFormat_Float_Bc6:
    case FtxFormat_Unorm_Bc7:
    case FtxFormat_Srgb_Bc7:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが BC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RIsBcTextureFormat(const FtxFormat format)
{
    if (RIsBc123TextureFormat(format) ||
        RIsBc67TextureFormat(format))
    {
        return true;
    }
    else
    {
        switch (format)
        {
        case FtxFormat_Unorm_Bc4:
        case FtxFormat_Snorm_Bc4:
        case FtxFormat_Unorm_Bc5:
        case FtxFormat_Snorm_Bc5:
            return true;
        default:
            return false;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが 1 成分あたり 16bit の unorm / snorm なら true を返します。
//-----------------------------------------------------------------------------
bool RIsUnormSnorm16TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_16:
    case FtxFormat_Snorm_16:
    case FtxFormat_Unorm_16_16:
    case FtxFormat_Snorm_16_16:
    case FtxFormat_Unorm_16_16_16_16:
    case FtxFormat_Snorm_16_16_16_16:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが 1 成分あたり 16bit の uint / sint なら true を返します。
//-----------------------------------------------------------------------------
bool RIsUintSint16TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Uint_16:
    case FtxFormat_Sint_16:
    case FtxFormat_Uint_16_16:
    case FtxFormat_Sint_16_16:
    case FtxFormat_Uint_16_16_16_16:
    case FtxFormat_Sint_16_16_16_16:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが 1 成分あたり 32bit の uint / sint なら true を返します。
//-----------------------------------------------------------------------------
bool RIsUintSint32TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Uint_32:
    case FtxFormat_Sint_32:
    case FtxFormat_Uint_32_32:
    case FtxFormat_Sint_32_32:
    case FtxFormat_Uint_32_32_32:
    case FtxFormat_Sint_32_32_32:
    case FtxFormat_Uint_32_32_32_32:
    case FtxFormat_Sint_32_32_32_32:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが浮動小数点数型なら true を返します。
//-----------------------------------------------------------------------------
bool RIsFloatTextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Float_16:
    case FtxFormat_Float_32:
    case FtxFormat_Float_16_16:
    case FtxFormat_Float_32_32:
    case FtxFormat_Float_11_11_10:
    case FtxFormat_Float_32_32_32:
    case FtxFormat_Float_16_16_16_16:
    case FtxFormat_Float_32_32_32_32:
    case FtxFormat_Ufloat_Bc6:
    case FtxFormat_Float_Bc6:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief 実数型として扱うテクスチャーフォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RIsRealNumberTextureFormat(const FtxFormat format)
{
    return (format == FtxFormat_Unorm_10_10_10_2 ||
            format == FtxFormat_Uint_10_10_10_2  ||
            RIsUnormSnorm16TextureFormat(format) ||
            RIsUintSint16TextureFormat(format)   ||
            RIsUintSint32TextureFormat(format)   ||
            RIsFloatTextureFormat(format));
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが整数型なら true を返します。
//-----------------------------------------------------------------------------
bool RIsIntTextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Uint_8:
    case FtxFormat_Sint_8:
    case FtxFormat_Uint_8_8:
    case FtxFormat_Sint_8_8:
    case FtxFormat_Uint_8_8_8_8:
    case FtxFormat_Sint_8_8_8_8:
    case FtxFormat_Uint_10_10_10_2:
        return true;
    default:
        return (RIsUintSint16TextureFormat(format) ||
                RIsUintSint32TextureFormat(format));
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが符号ありなら true を返します。
//-----------------------------------------------------------------------------
bool RIsSignedTextureFormat(const FtxFormat format)
{
    if (RIsFloatTextureFormat(format) && format != FtxFormat_Ufloat_Bc6)
    {
        return true;
    }

    switch (format)
    {
    case FtxFormat_Snorm_8:
    case FtxFormat_Sint_8:
    case FtxFormat_Snorm_16:
    case FtxFormat_Sint_16:
    case FtxFormat_Sint_32:
    case FtxFormat_Snorm_8_8:
    case FtxFormat_Sint_8_8:
    case FtxFormat_Snorm_16_16:
    case FtxFormat_Sint_16_16:
    case FtxFormat_Sint_32_32:
    case FtxFormat_Sint_32_32_32:
    case FtxFormat_Snorm_8_8_8_8:
    case FtxFormat_Sint_8_8_8_8:
    case FtxFormat_Snorm_16_16_16_16:
    case FtxFormat_Sint_16_16_16_16:
    case FtxFormat_Sint_32_32_32_32:
    case FtxFormat_Snorm_Bc4:
    case FtxFormat_Snorm_Bc5:
    case FtxFormat_Snorm_Eac_11:
    case FtxFormat_Snorm_Eac_11_11:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが sRGB フェッチなら true を返します。
//-----------------------------------------------------------------------------
bool RIsSrgbFetchTextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Srgb_8_8_8_8:
    case FtxFormat_Srgb_Bc1:
    case FtxFormat_Srgb_Bc2:
    case FtxFormat_Srgb_Bc3:
    case FtxFormat_Srgb_Bc7:
    case FtxFormat_Srgb_Etc2:
    case FtxFormat_Srgb_Etc2_Mask:
    case FtxFormat_Srgb_Etc2_Alpha:
    case FtxFormat_Srgb_Pvrtc1_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp:
    case FtxFormat_Srgb_Astc_4x4:
    case FtxFormat_Srgb_Astc_5x4:
    case FtxFormat_Srgb_Astc_5x5:
    case FtxFormat_Srgb_Astc_6x5:
    case FtxFormat_Srgb_Astc_6x6:
    case FtxFormat_Srgb_Astc_8x5:
    case FtxFormat_Srgb_Astc_8x6:
    case FtxFormat_Srgb_Astc_8x8:
    case FtxFormat_Srgb_Astc_10x5:
    case FtxFormat_Srgb_Astc_10x6:
    case FtxFormat_Srgb_Astc_10x8:
    case FtxFormat_Srgb_Astc_10x10:
    case FtxFormat_Srgb_Astc_12x10:
    case FtxFormat_Srgb_Astc_12x12:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが ETC1 なら true を返します。
//-----------------------------------------------------------------------------
bool RIsEtc1TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Etc1:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが ETC2 なら true を返します。
//-----------------------------------------------------------------------------
bool RIsEtc2TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Etc2:
    case FtxFormat_Srgb_Etc2:
    case FtxFormat_Unorm_Etc2_Mask:
    case FtxFormat_Srgb_Etc2_Mask:
    case FtxFormat_Unorm_Etc2_Alpha:
    case FtxFormat_Srgb_Etc2_Alpha:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが ETC1 または ETC2 なら true を返します。
//-----------------------------------------------------------------------------
bool RIsEtcTextureFormat(const FtxFormat format)
{
    return RIsEtc1TextureFormat(format) || RIsEtc2TextureFormat(format);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが EAC なら true を返します。
//-----------------------------------------------------------------------------
bool RIsEacTextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Eac_11:
    case FtxFormat_Snorm_Eac_11:
    case FtxFormat_Unorm_Eac_11_11:
    case FtxFormat_Snorm_Eac_11_11:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが PVRTC1 なら true を返します。
//-----------------------------------------------------------------------------
bool RIsPvrtc1TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Pvrtc1_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_2Bpp:
    case FtxFormat_Unorm_Pvrtc1_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_4Bpp:
    case FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが PVRTC2 なら true を返します。
//-----------------------------------------------------------------------------
bool RIsPvrtc2TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Pvrtc2_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp:
    case FtxFormat_Unorm_Pvrtc2_Alpha_4Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが PVRTC1 または PVRTC2 なら true を返します。
//-----------------------------------------------------------------------------
bool RIsPvrtcTextureFormat(const FtxFormat format)
{
    return RIsPvrtc1TextureFormat(format) || RIsPvrtc2TextureFormat(format);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが ASTC なら true を返します。
//-----------------------------------------------------------------------------
bool RIsAstcTextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Astc_4x4:
    case FtxFormat_Srgb_Astc_4x4:
    case FtxFormat_Unorm_Astc_5x4:
    case FtxFormat_Srgb_Astc_5x4:
    case FtxFormat_Unorm_Astc_5x5:
    case FtxFormat_Srgb_Astc_5x5:
    case FtxFormat_Unorm_Astc_6x5:
    case FtxFormat_Srgb_Astc_6x5:
    case FtxFormat_Unorm_Astc_6x6:
    case FtxFormat_Srgb_Astc_6x6:
    case FtxFormat_Unorm_Astc_8x5:
    case FtxFormat_Srgb_Astc_8x5:
    case FtxFormat_Unorm_Astc_8x6:
    case FtxFormat_Srgb_Astc_8x6:
    case FtxFormat_Unorm_Astc_8x8:
    case FtxFormat_Srgb_Astc_8x8:
    case FtxFormat_Unorm_Astc_10x5:
    case FtxFormat_Srgb_Astc_10x5:
    case FtxFormat_Unorm_Astc_10x6:
    case FtxFormat_Srgb_Astc_10x6:
    case FtxFormat_Unorm_Astc_10x8:
    case FtxFormat_Srgb_Astc_10x8:
    case FtxFormat_Unorm_Astc_10x10:
    case FtxFormat_Srgb_Astc_10x10:
    case FtxFormat_Unorm_Astc_12x10:
    case FtxFormat_Srgb_Astc_12x10:
    case FtxFormat_Unorm_Astc_12x12:
    case FtxFormat_Srgb_Astc_12x12:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが圧縮形式なら true を返します。
//-----------------------------------------------------------------------------
bool RIsCompressedTextureFormat(const FtxFormat format)
{
    return (RIsBcTextureFormat(format)    ||
            RIsEtcTextureFormat(format)   ||
            RIsEacTextureFormat(format)   ||
            RIsPvrtcTextureFormat(format) ||
            RIsAstcTextureFormat(format));
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットから成分数を取得します。
//-----------------------------------------------------------------------------
int RGetComponentCount(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_8:
    case FtxFormat_Uint_8:
    case FtxFormat_Snorm_8:
    case FtxFormat_Sint_8:
    case FtxFormat_Unorm_16:
    case FtxFormat_Snorm_16:
    case FtxFormat_Uint_16:
    case FtxFormat_Sint_16:
    case FtxFormat_Uint_32:
    case FtxFormat_Sint_32:
    case FtxFormat_Float_16:
    case FtxFormat_Float_32:
    case FtxFormat_Unorm_Bc4:
    case FtxFormat_Snorm_Bc4:
    case FtxFormat_Unorm_Eac_11:
    case FtxFormat_Snorm_Eac_11:
        return 1;

    case FtxFormat_Unorm_4_4:
    case FtxFormat_Unorm_8_8:
    case FtxFormat_Uint_8_8:
    case FtxFormat_Snorm_8_8:
    case FtxFormat_Sint_8_8:
    case FtxFormat_Unorm_16_16:
    case FtxFormat_Snorm_16_16:
    case FtxFormat_Uint_16_16:
    case FtxFormat_Sint_16_16:
    case FtxFormat_Uint_32_32:
    case FtxFormat_Sint_32_32:
    case FtxFormat_Float_16_16:
    case FtxFormat_Float_32_32:
    case FtxFormat_Unorm_Bc5:
    case FtxFormat_Snorm_Bc5:
    case FtxFormat_Unorm_Eac_11_11:
    case FtxFormat_Snorm_Eac_11_11:
        return 2;

    case FtxFormat_Unorm_5_6_5:
    case FtxFormat_Uint_32_32_32:
    case FtxFormat_Sint_32_32_32:
    case FtxFormat_Float_11_11_10:
    case FtxFormat_Float_32_32_32:
    case FtxFormat_Ufloat_Bc6:
    case FtxFormat_Float_Bc6:
    case FtxFormat_Unorm_Etc1:
    case FtxFormat_Unorm_Etc2:
    case FtxFormat_Srgb_Etc2:
        return 3;

    default:
        return 4;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットの 1 ピクセルのビット数を返します。
//-----------------------------------------------------------------------------
int RGetBitsPerPixel(const FtxFormat format)
{
    // ASTC フォーマットの場合は 0 を返します。
    if (RIsAstcTextureFormat(format))
    {
        return 0;
    }

    switch (format)
    {
    case FtxFormat_Unorm_Pvrtc1_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_2Bpp:
    case FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Unorm_Pvrtc2_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp:
        return 2;

    case FtxFormat_Unorm_Bc1:
    case FtxFormat_Srgb_Bc1:
    case FtxFormat_Unorm_Bc4:
    case FtxFormat_Snorm_Bc4:
    case FtxFormat_Unorm_Etc1:
    case FtxFormat_Unorm_Etc2:
    case FtxFormat_Srgb_Etc2:
    case FtxFormat_Unorm_Etc2_Mask:
    case FtxFormat_Srgb_Etc2_Mask:
    case FtxFormat_Unorm_Eac_11:
    case FtxFormat_Snorm_Eac_11:
    case FtxFormat_Unorm_Pvrtc1_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_4Bpp:
    case FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp:
    case FtxFormat_Unorm_Pvrtc2_Alpha_4Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp:
        return 4;

    case FtxFormat_Unorm_8:
    case FtxFormat_Uint_8:
    case FtxFormat_Snorm_8:
    case FtxFormat_Sint_8:
    case FtxFormat_Unorm_4_4:
    case FtxFormat_Unorm_Bc2:
    case FtxFormat_Srgb_Bc2:
    case FtxFormat_Unorm_Bc3:
    case FtxFormat_Srgb_Bc3:
    case FtxFormat_Unorm_Bc5:
    case FtxFormat_Snorm_Bc5:
    case FtxFormat_Ufloat_Bc6:
    case FtxFormat_Float_Bc6:
    case FtxFormat_Unorm_Bc7:
    case FtxFormat_Srgb_Bc7:
    case FtxFormat_Unorm_Etc2_Alpha:
    case FtxFormat_Srgb_Etc2_Alpha:
    case FtxFormat_Unorm_Eac_11_11:
    case FtxFormat_Snorm_Eac_11_11:
        return 8;

    case FtxFormat_Unorm_16:
    case FtxFormat_Snorm_16:
    case FtxFormat_Uint_16:
    case FtxFormat_Sint_16:
    case FtxFormat_Float_16:
    case FtxFormat_Unorm_8_8:
    case FtxFormat_Uint_8_8:
    case FtxFormat_Snorm_8_8:
    case FtxFormat_Sint_8_8:
    case FtxFormat_Unorm_5_6_5:
    case FtxFormat_Unorm_5_5_5_1:
    case FtxFormat_Unorm_4_4_4_4:
        return 16;

    case FtxFormat_Unorm_16_16:
    case FtxFormat_Snorm_16_16:
    case FtxFormat_Uint_16_16:
    case FtxFormat_Sint_16_16:
    case FtxFormat_Uint_32:
    case FtxFormat_Sint_32:
    case FtxFormat_Float_32:
    case FtxFormat_Float_16_16:
    case FtxFormat_Float_11_11_10:
    case FtxFormat_Unorm_8_8_8_8:
    case FtxFormat_Uint_8_8_8_8:
    case FtxFormat_Snorm_8_8_8_8:
    case FtxFormat_Sint_8_8_8_8:
    case FtxFormat_Srgb_8_8_8_8:
    case FtxFormat_Unorm_10_10_10_2:
    case FtxFormat_Uint_10_10_10_2:
        return 32;

    case FtxFormat_Unorm_16_16_16_16:
    case FtxFormat_Snorm_16_16_16_16:
    case FtxFormat_Uint_16_16_16_16:
    case FtxFormat_Sint_16_16_16_16:
    case FtxFormat_Uint_32_32:
    case FtxFormat_Sint_32_32:
    case FtxFormat_Float_32_32:
    case FtxFormat_Float_16_16_16_16:
        return 64;

    case FtxFormat_Uint_32_32_32:
    case FtxFormat_Sint_32_32_32:
    case FtxFormat_Float_32_32_32:
        return 96;

    case FtxFormat_Uint_32_32_32_32:
    case FtxFormat_Sint_32_32_32_32:
    case FtxFormat_Float_32_32_32_32:
        return 128;

    default:
        return 32;
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief テクスチャーの圧縮ブロックの幅を取得します。
//-----------------------------------------------------------------------------
int RGetTextureBlockWidth(const FtxFormat format)
{
    if (!RIsCompressedTextureFormat(format))
    {
        return 1;
    }

    switch( format )
    {
    case FtxFormat_Unorm_Pvrtc1_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_2Bpp:
    case FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp:
        return 16;

    case FtxFormat_Unorm_Pvrtc1_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_4Bpp:
    case FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp:
    case FtxFormat_Unorm_Pvrtc2_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp:
        return 8;

    case FtxFormat_Unorm_Astc_5x4:
    case FtxFormat_Srgb_Astc_5x4:
    case FtxFormat_Unorm_Astc_5x5:
    case FtxFormat_Srgb_Astc_5x5:
        return 5;

    case FtxFormat_Unorm_Astc_6x5:
    case FtxFormat_Srgb_Astc_6x5:
    case FtxFormat_Unorm_Astc_6x6:
    case FtxFormat_Srgb_Astc_6x6:
        return 6;

    case FtxFormat_Unorm_Astc_8x5:
    case FtxFormat_Srgb_Astc_8x5:
    case FtxFormat_Unorm_Astc_8x6:
    case FtxFormat_Srgb_Astc_8x6:
    case FtxFormat_Unorm_Astc_8x8:
    case FtxFormat_Srgb_Astc_8x8:
        return 8;

    case FtxFormat_Unorm_Astc_10x5:
    case FtxFormat_Srgb_Astc_10x5:
    case FtxFormat_Unorm_Astc_10x6:
    case FtxFormat_Srgb_Astc_10x6:
    case FtxFormat_Unorm_Astc_10x8:
    case FtxFormat_Srgb_Astc_10x8:
    case FtxFormat_Unorm_Astc_10x10:
    case FtxFormat_Srgb_Astc_10x10:
        return 10;

    case FtxFormat_Unorm_Astc_12x10:
    case FtxFormat_Srgb_Astc_12x10:
    case FtxFormat_Unorm_Astc_12x12:
    case FtxFormat_Srgb_Astc_12x12:
        return 12;

    default:
        return 4;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーの圧縮ブロックの高さを取得します。
//-----------------------------------------------------------------------------
int RGetTextureBlockHeight(const FtxFormat format)
{
    if (!RIsCompressedTextureFormat(format))
    {
        return 1;
    }

    switch( format )
    {
    case FtxFormat_Unorm_Astc_5x5:
    case FtxFormat_Srgb_Astc_5x5:
    case FtxFormat_Unorm_Astc_6x5:
    case FtxFormat_Srgb_Astc_6x5:
    case FtxFormat_Unorm_Astc_8x5:
    case FtxFormat_Srgb_Astc_8x5:
    case FtxFormat_Unorm_Astc_10x5:
    case FtxFormat_Srgb_Astc_10x5:
        return 5;

    case FtxFormat_Unorm_Astc_6x6:
    case FtxFormat_Srgb_Astc_6x6:
    case FtxFormat_Unorm_Astc_8x6:
    case FtxFormat_Srgb_Astc_8x6:
    case FtxFormat_Unorm_Astc_10x6:
    case FtxFormat_Srgb_Astc_10x6:
        return 6;

    case FtxFormat_Unorm_Pvrtc1_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_2Bpp:
    case FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Unorm_Pvrtc1_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_4Bpp:
    case FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp:
    case FtxFormat_Unorm_Astc_8x8:
    case FtxFormat_Srgb_Astc_8x8:
    case FtxFormat_Unorm_Astc_10x8:
    case FtxFormat_Srgb_Astc_10x8:
        return 8;

    case FtxFormat_Unorm_Astc_10x10:
    case FtxFormat_Srgb_Astc_10x10:
    case FtxFormat_Unorm_Astc_12x10:
    case FtxFormat_Srgb_Astc_12x10:
        return 10;

    case FtxFormat_Unorm_Astc_12x12:
    case FtxFormat_Srgb_Astc_12x12:
        return 12;

    default:
        return 4;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーのデータサイズ（バイト数）を取得します。
//-----------------------------------------------------------------------------
size_t RGetTextureDataSize(
    std::vector<size_t>* pLevelOffsets,
    const FtxFormat format,
    const FtxDimension dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipCount
)
{
    if (pLevelOffsets != nullptr)
    {
        pLevelOffsets->clear();
    }
    const int pixelBits = RGetBitsPerPixel(format);
    const bool isCompressed = RIsCompressedTextureFormat(format);
    const int blockW = RGetTextureBlockWidth(format);
    const int blockH = RGetTextureBlockHeight(format);
    const size_t unitBytes =
        (!isCompressed               ) ? pixelBits / 8 :
        (RIsAstcTextureFormat(format)) ? 16            :
        pixelBits * blockW * blockH / 8;

    size_t dataSize = 0;
    for (int level = 0; level < mipCount; ++level)
    {
        if (pLevelOffsets != nullptr)
        {
            pLevelOffsets->push_back(dataSize);
        }
        const int levelW = RMax(imageW >> level, 1);
        const int levelH = RMax(imageH >> level, 1);
        const int levelD = (dimension == FtxDimension_3d) ?
            RMax(imageD >> level, 1) : imageD;
        int unitW = levelW;
        int unitH = levelH;
        int unitD = levelD;
        if (isCompressed)
        {
            unitW = (levelW + blockW - 1) / blockW;
            unitH = (levelH + blockH - 1) / blockH;
        }
        const size_t levelSize = unitBytes * unitW * unitH * unitD;
        dataSize += levelSize;
    }
    return dataSize;
}

//-----------------------------------------------------------------------------
//! @brief 成分選択の成分をあらわす文字列のデータです。
//-----------------------------------------------------------------------------
namespace {

const char* const CompSelStrs[FtxComponent_Count] =
{
    "r", "g", "b", "a", "0", "1"
};

} // namespace

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列から成分選択を取得します。
//-----------------------------------------------------------------------------
FtxCompSel RGetCompSelFromString(const std::string& str)
{
    int comps[R_RGBA_BYTES] =
    {
        FtxComponent_Red,
        FtxComponent_Green,
        FtxComponent_Blue,
        FtxComponent_Alpha,
    };
    int validCount = 0;
    std::istringstream iss(str);
    for (int iRgba = 0; iRgba < R_RGBA_BYTES; ++iRgba)
    {
        if (iss)
        {
            std::string c;
            iss >> c;
            for (int iComp = 0; iComp < FtxComponent_Count; ++iComp)
            {
                if (c == CompSelStrs[iComp])
                {
                    comps[iRgba] = iComp;
                    ++validCount;
                    break;
                }
            }
        }
    }
    return FTX_GET_COMP_SEL(comps[0], comps[1], comps[2], comps[3]);
}

//-----------------------------------------------------------------------------
//! @brief 成分選択から中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetCompSelString(const FtxCompSel compSel)
{
    std::string value;
    for (int iRgba = 0; iRgba < R_RGBA_BYTES; ++iRgba)
    {
        const int iComp = (compSel >> (3 - iRgba) * 8) & 0xff;
        if (iRgba > 0)
        {
            value += " ";
        }
        value += ((iComp < FtxComponent_Count) ? CompSelStrs[iComp] : "0");
    }
    return value;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットに対応したデフォルトの成分選択を取得します。
//-----------------------------------------------------------------------------
FtxCompSel RGetDefaultCompSel(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_8:
    case FtxFormat_Uint_8:
    case FtxFormat_Snorm_8:
    case FtxFormat_Sint_8:
    case FtxFormat_Unorm_16:
    case FtxFormat_Snorm_16:
    case FtxFormat_Uint_16:
    case FtxFormat_Sint_16:
    case FtxFormat_Uint_32:
    case FtxFormat_Sint_32:
    case FtxFormat_Float_16:
    case FtxFormat_Float_32:
    case FtxFormat_Unorm_Bc4:
    case FtxFormat_Snorm_Bc4:
    case FtxFormat_Unorm_Eac_11:
    case FtxFormat_Snorm_Eac_11:
        // rrr1
        return FTX_GET_COMP_SEL(FtxComponent_Red, FtxComponent_Red, FtxComponent_Red, FtxComponent_One);

    case FtxFormat_Unorm_4_4:
    case FtxFormat_Unorm_8_8:
    case FtxFormat_Unorm_Bc5:
    case FtxFormat_Unorm_Eac_11_11:
        // rrrg
        return FTX_GET_COMP_SEL(FtxComponent_Red, FtxComponent_Red, FtxComponent_Red, FtxComponent_Green);

    case FtxFormat_Snorm_8_8:
    case FtxFormat_Uint_8_8:
    case FtxFormat_Sint_8_8:
    case FtxFormat_Unorm_16_16:
    case FtxFormat_Snorm_16_16:
    case FtxFormat_Uint_16_16:
    case FtxFormat_Sint_16_16:
    case FtxFormat_Uint_32_32:
    case FtxFormat_Sint_32_32:
    case FtxFormat_Float_16_16:
    case FtxFormat_Float_32_32:
    case FtxFormat_Snorm_Bc5:
    case FtxFormat_Snorm_Eac_11_11:
        // rg01
        return FTX_GET_COMP_SEL(FtxComponent_Red, FtxComponent_Green, FtxComponent_Zero, FtxComponent_One);

    case FtxFormat_Unorm_5_6_5:
    case FtxFormat_Uint_32_32_32:
    case FtxFormat_Sint_32_32_32:
    case FtxFormat_Float_11_11_10:
    case FtxFormat_Float_32_32_32:
    case FtxFormat_Ufloat_Bc6:
    case FtxFormat_Float_Bc6:
    case FtxFormat_Unorm_Etc1:
    case FtxFormat_Unorm_Etc2:
    case FtxFormat_Srgb_Etc2:
        // rgb1
        return FTX_GET_COMP_SEL(FtxComponent_Red, FtxComponent_Green, FtxComponent_Blue, FtxComponent_One);

    default:
        // rgba
        return FtxCompSelRgba;
    }
}

//-----------------------------------------------------------------------------
//! @brief 法線用のヒント情報を取得します。
//-----------------------------------------------------------------------------
std::string RGetHintForNormal()
{
    return "normal";
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットに対応したデフォルトのヒント情報を取得します。
//-----------------------------------------------------------------------------
std::string RGetDefaultHint(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Snorm_Bc5:
    case FtxFormat_Snorm_Eac_11_11:
        return RGetHintForNormal();

    default:
        return "";
    }
}

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

