﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// DllMain LoadNvnDll FreeNvnDll
// ConvertTiling ReleaseTexture GetDataSizeNX
// ConvertToNX ConvertToLinear
// GetNvnFormat

//=============================================================================
// include
//=============================================================================
#include <algorithm>
#include <cstdint>
#include <ctime>
#include <iostream>
#include <string>

#include <nvnTool/nvnTool_TexpkgProducerCore.h>

#include <windows.h>

#include "../Sources/BinaryTypes.h"
#include "../Sources/NXTypes.h"

using namespace std;
using namespace nvnTool::texpkg;
using namespace nn;
using namespace nn::gfx::tool::texcvtr;

#define DLLEXPORT extern "C" __declspec(dllexport)

//=============================================================================
//! @brief 未使用変数の警告を抑止するためのマクロです。
//=============================================================================
#define NXDLL_UNUSED_VARIABLE(Variable) (void)(&Variable);

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//=============================================================================
// variables
//=============================================================================
std::string g_DllFolderPath; //!< この dll ファイルが存在するフォルダのパスです。

bool g_IsNvnLoaded = false; //!< NVN の dll ファイルをロード済みなら true です。
RawImageLib g_ImageLib; //!< NVN のイメージライブラリです。
TexpkgLib g_TexpkgLib; //!< NVN の TEXPKG ライブラリです。

//-----------------------------------------------------------------------------
//! @brief 指定した文字が Shift JIS の 2 バイト文字の 1 バイト目なら true を返します。
//-----------------------------------------------------------------------------
bool IsShiftJisFirstByte(const uint8_t val)
{
    return ((0x81 <= val && val <= 0x9f) ||
            (0xe0 <= val && val <= 0xef));
}

//-----------------------------------------------------------------------------
//! @brief ANSI 文字列からワイド文字列（Unicode）を取得します。
//-----------------------------------------------------------------------------
std::wstring GetUnicodeFromAnsi(const std::string& src)
{
    std::wstring dst;
    const int wcharSize = MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, nullptr, 0);
    if (wcharSize != 0)
    {
        wchar_t* wcharBuf = new wchar_t[wcharSize];
        MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, wcharBuf, wcharSize);
        dst = std::wstring(wcharBuf, wcslen(wcharBuf));
        delete[] wcharBuf;
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief ワイド文字列（Unicode）から ANSI 文字列を取得します。
//-----------------------------------------------------------------------------
std::string GetAnsiFromUnicode(const std::wstring& src)
{
    std::string dst;
    const int ansiSize = WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, nullptr, 0, nullptr, nullptr);
    if (ansiSize != 0)
    {
        char* ansiBuf = new char[ansiSize];
        WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, ansiBuf, ansiSize, nullptr, nullptr);
        dst = std::string(ansiBuf, strlen(ansiBuf));
        delete[] ansiBuf;
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief ファイルパス文字列からフォルダのパス部分を抽出して返します。
//!        返り値の最後に / は付きません。
//-----------------------------------------------------------------------------
std::string GetFolderFromFilePath(const std::string& path)
{
    size_t delimIdx = std::string::npos;
    for (size_t charIdx = 0; charIdx < path.size(); ++charIdx)
    {
        const char c = path[charIdx];
        if (IsShiftJisFirstByte(static_cast<uint8_t>(c)))
        {
            ++charIdx; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            delimIdx = charIdx;
        }
    }
    return (delimIdx != std::string::npos) ? path.substr(0, delimIdx) : ".";
}

//-----------------------------------------------------------------------------
//! @brief ファイルが存在するなら true を返します。
//-----------------------------------------------------------------------------
bool FileExists(const std::string& path)
{
    WIN32_FIND_DATA ffd;
    HANDLE  hFindFile = FindFirstFileA(path.c_str(), &ffd);
    if (hFindFile == INVALID_HANDLE_VALUE)
    {
        return false;
    }
    FindClose(hFindFile);
    if (!(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
    {
        return true;
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief Windows のシステムメッセージ文字列を取得します。
//!
//! @param[in] messageId メッセージ ID です。
//!
//! @return Windows のシステムメッセージ文字列を返します。
//-----------------------------------------------------------------------------
std::string GetWindowsSystemMessage(const int messageId)
{
    void* pMessage = nullptr;
    FormatMessageA(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM |
        FORMAT_MESSAGE_IGNORE_INSERTS, nullptr, messageId,
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), reinterpret_cast<LPSTR>(&pMessage), 0, nullptr);
    std::string message;
    if (pMessage != nullptr)
    {
        message = reinterpret_cast<const char*>(pMessage);
        LocalFree(pMessage);
        const size_t spaceIdx = message.find_last_not_of(" \t\r\n"); // 末尾の空白と改行を削除します。
        message = (spaceIdx != std::string::npos) ? message.substr(0, spaceIdx + 1) : "";
    }
    else
    {
        char strBuf[32];
        sprintf_s(strBuf, "0x%08X", messageId);
        message = strBuf;
    }
    return message;
}

//-----------------------------------------------------------------------------
//! @brief Windows の最後のエラーのメッセージ文字列を取得します。
//!
//! @return Windows の最後のエラーのメッセージ文字列を返します。
//-----------------------------------------------------------------------------
std::string GetWindowsLastErrorMessage()
{
    return GetWindowsSystemMessage(GetLastError());
}

//=============================================================================
//! @brief 文字列と値の構造体です。
//=============================================================================
template <typename T>
struct StringAndValue
{
    const char* pString; //!< 文字列へのポインタです。
    T value; //!< 値です。
};

//! @brief 値に対応する文字列を取得します。
//!
//! @param[in] svs 文字列と値配列です。配列の最後の文字列は nullptr である必要があります。
//! @param[in] value 値です。
//! @param[in] alt 値に対応する文字列が存在しなかった場合に返す文字列です。
//!
//! @return 文字列を返します。
//!
template <typename T>
std::string GetStringFromValue(const StringAndValue<T>* svs, const T value, const char* alt = "")
{
    for (const StringAndValue<T>* pSv = svs; pSv->pString != nullptr; ++pSv)
    {
        if (value == pSv->value)
        {
            return pSv->pString;
        }
    }
    return alt;
}

//! @brief 文字列に対応する値を取得します。
//!
//! @param[in] svs 文字列と値配列です。配列の最後の文字列は nullptr である必要があります。
//! @param[in] str 文字列です。
//! @param[in] alt 文字列に対応する値が存在しなかった場合に返す値です。
//!
//! @return 値を返します。
//!
template <typename T>
T GetValueFromString(const StringAndValue<T>* svs, const std::string& str, const T alt)
{
    for (const StringAndValue<T>* pSv = svs; pSv->pString != nullptr; ++pSv)
    {
        if (str == pSv->pString)
        {
            return pSv->value;
        }
    }
    return alt;
}

//-----------------------------------------------------------------------------
//! @brief NVN の dll ファイルをロードします。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool LoadNvnDll()
{
    g_ImageLib.module = nullptr;
    g_TexpkgLib.module = nullptr;

    const std::string imageLibFileName = GetAnsiFromUnicode(NVN_IMAGE_DLL_FILENAME);
    #ifdef _M_IX86
    std::string nvnToolsPath = g_DllFolderPath + "\\..\\..\\NvnTools";
    #else
    std::string nvnToolsPath = g_DllFolderPath + "\\..\\NvnTools";
    #endif
    if (!FileExists(nvnToolsPath + "\\" + imageLibFileName))
    {
        if (FileExists(g_DllFolderPath + "\\" + imageLibFileName))
        {
            // NVN の dll ファイルが NvnTools フォルダに存在せず、
            // TextureConverterNX.dll と同じフォルダに存在するならそれをロードします。
            nvnToolsPath = g_DllFolderPath;
        }
        #ifdef _DEBUG
        else
        {
            #ifdef _M_IX86
            const std::string outputRelativePath = g_DllFolderPath +
                "\\..\\..\\..\\..\\..\\..\\..\\..\\..\\..\\Tools\\Graphics\\NvnTools";
            #else
            const std::string outputRelativePath = g_DllFolderPath +
                "\\..\\..\\..\\..\\..\\..\\..\\..\\..\\Tools\\Graphics\\NvnTools";
            #endif
            if (FileExists(outputRelativePath + "\\" + imageLibFileName))
            {
                nvnToolsPath = outputRelativePath;
            }
        }
        #endif
    }

    const std::string imageLibPath  = nvnToolsPath + "\\" + imageLibFileName;
    //cerr << "nvn image dll: " << imageLibPath << endl;
    if (!LoadNVNImageLibrary(&g_ImageLib, GetUnicodeFromAnsi(imageLibPath).c_str()))
    {
        cerr << "Error: Cannot load dll: " << imageLibPath
             << " (" << GetWindowsLastErrorMessage() << ")" << endl;
        return false;
    }

    const std::string texpkgLibPath = nvnToolsPath + "\\" +
        GetAnsiFromUnicode(NVN_TEXPKG_DLL_FILENAME);
    //cerr << "nvn texpkg dll: " << texpkgLibPath << endl;
    if (!LoadTexpkgLibrary(&g_TexpkgLib, GetUnicodeFromAnsi(texpkgLibPath).c_str()))
    {
        cerr << "Error: Cannot load dll: " << texpkgLibPath
             << " (" << GetWindowsLastErrorMessage() << ")" << endl;
        return false;
    }

    return true;
}

//-----------------------------------------------------------------------------
//! @brief NVN の dll ファイルを解放します。
//-----------------------------------------------------------------------------
void FreeNvnDll()
{
    if (g_ImageLib.module != nullptr)
    {
        FreeLibrary(g_ImageLib.module);
        g_ImageLib.module = nullptr;
    }
    if (g_TexpkgLib.module != nullptr)
    {
        FreeLibrary(g_TexpkgLib.module);
        g_TexpkgLib.module = nullptr;
    }
}

//-----------------------------------------------------------------------------
//! @brief 配列テクスチャなら true を返します。
//-----------------------------------------------------------------------------
bool IsArray(const NVNtextureTarget target)
{
    return (target == NVN_TEXTURE_TARGET_1D_ARRAY             ||
            target == NVN_TEXTURE_TARGET_2D_ARRAY             ||
            target == NVN_TEXTURE_TARGET_CUBEMAP_ARRAY        ||
            target == NVN_TEXTURE_TARGET_2D_MULTISAMPLE_ARRAY);
}

//-----------------------------------------------------------------------------
//! @brief キューブマップなら true を返します。
//-----------------------------------------------------------------------------
bool IsCubeMap(const NVNtextureTarget target)
{
    return (target == NVN_TEXTURE_TARGET_CUBEMAP       ||
            target == NVN_TEXTURE_TARGET_CUBEMAP_ARRAY);
}

//-----------------------------------------------------------------------------
//! @brief NVN テクスチャターゲットを取得します。
//!
//! @param[in] dimensionStr 中間ファイルの次元文字列です。
//!
//! @return NVN テクスチャターゲットを返します。
//-----------------------------------------------------------------------------
NVNtextureTarget GetNvnTextureTarget(const std::string& dimensionStr)
{
    return
        (dimensionStr == "1d"           ) ? NVN_TEXTURE_TARGET_1D                   :
        (dimensionStr == "2d"           ) ? NVN_TEXTURE_TARGET_2D                   :
        (dimensionStr == "3d"           ) ? NVN_TEXTURE_TARGET_3D                   :
        (dimensionStr == "cube"         ) ? NVN_TEXTURE_TARGET_CUBEMAP              :
        (dimensionStr == "cube_array"   ) ? NVN_TEXTURE_TARGET_CUBEMAP_ARRAY        :
        (dimensionStr == "1d_array"     ) ? NVN_TEXTURE_TARGET_1D_ARRAY             :
        (dimensionStr == "2d_array"     ) ? NVN_TEXTURE_TARGET_2D_ARRAY             :
        (dimensionStr == "2d_msaa"      ) ? NVN_TEXTURE_TARGET_2D_MULTISAMPLE       :
        (dimensionStr == "2d_msaa_array") ? NVN_TEXTURE_TARGET_2D_MULTISAMPLE_ARRAY :
        NVN_TEXTURE_TARGET_2D;
}

//-----------------------------------------------------------------------------
//! @brief NVN フォーマットを取得します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return NVN フォーマットを返します。
//-----------------------------------------------------------------------------
const StringAndValue<NVNformat> g_NvnFormatStringValues[] =
{
    { "unorm_8", NVN_FORMAT_R8   },
    { "snorm_8", NVN_FORMAT_R8SN },
    { "uint_8" , NVN_FORMAT_R8UI },
    { "sint_8" , NVN_FORMAT_R8I  },

    { "unorm_16", NVN_FORMAT_R16   },
    { "snorm_16", NVN_FORMAT_R16SN },
    { "uint_16" , NVN_FORMAT_R16UI },
    { "sint_16" , NVN_FORMAT_R16I  },

    { "uint_32", NVN_FORMAT_R32UI },
    { "sint_32", NVN_FORMAT_R32I  },

    { "unorm_8_8", NVN_FORMAT_RG8   },
    { "snorm_8_8", NVN_FORMAT_RG8SN },
    { "uint_8_8" , NVN_FORMAT_RG8UI },
    { "sint_8_8" , NVN_FORMAT_RG8I  },

    { "unorm_16_16", NVN_FORMAT_RG16   },
    { "snorm_16_16", NVN_FORMAT_RG16SN },
    { "uint_16_16" , NVN_FORMAT_RG16UI },
    { "sint_16_16" , NVN_FORMAT_RG16I  },

    { "uint_32_32", NVN_FORMAT_RG32UI },
    { "sint_32_32", NVN_FORMAT_RG32I },

    { "unorm_5_6_5"  , NVN_FORMAT_RGB565  },
    { "uint_32_32_32", NVN_FORMAT_RGB32UI },
    { "sint_32_32_32", NVN_FORMAT_RGB32I  },

    { "unorm_5_5_5_1", NVN_FORMAT_RGB5A1 },
    { "unorm_4_4_4_4", NVN_FORMAT_RGBA4  },

    { "unorm_8_8_8_8", NVN_FORMAT_RGBA8      },
    { "snorm_8_8_8_8", NVN_FORMAT_RGBA8SN    },
    { "srgb_8_8_8_8" , NVN_FORMAT_RGBA8_SRGB },
    { "uint_8_8_8_8" , NVN_FORMAT_RGBA8UI    },
    { "sint_8_8_8_8" , NVN_FORMAT_RGBA8I     },

    { "unorm_10_10_10_2", NVN_FORMAT_RGB10A2   },
    { "uint_10_10_10_2" , NVN_FORMAT_RGB10A2UI },

    { "unorm_16_16_16_16", NVN_FORMAT_RGBA16   },
    { "snorm_16_16_16_16", NVN_FORMAT_RGBA16SN },
    { "uint_16_16_16_16" , NVN_FORMAT_RGBA16UI },
    { "sint_16_16_16_16" , NVN_FORMAT_RGBA16I  },

    { "uint_32_32_32_32", NVN_FORMAT_RGBA32UI },
    { "sint_32_32_32_32", NVN_FORMAT_RGBA32I  },

    { "float_16"         , NVN_FORMAT_R16F       },
    { "float_32"         , NVN_FORMAT_R32F       },
    { "float_16_16"      , NVN_FORMAT_RG16F      },
    { "float_32_32"      , NVN_FORMAT_RG32F      },
    { "float_11_11_10"   , NVN_FORMAT_R11G11B10F },
    { "float_32_32_32"   , NVN_FORMAT_RGB32F     },
    { "float_16_16_16_16", NVN_FORMAT_RGBA16F    },
    { "float_32_32_32_32", NVN_FORMAT_RGBA32F    },

    { "unorm_bc1", NVN_FORMAT_RGBA_DXT1      },
    { "srgb_bc1" , NVN_FORMAT_RGBA_DXT1_SRGB },
    { "unorm_bc2", NVN_FORMAT_RGBA_DXT3      },
    { "srgb_bc2" , NVN_FORMAT_RGBA_DXT3_SRGB },
    { "unorm_bc3", NVN_FORMAT_RGBA_DXT5      },
    { "srgb_bc3" , NVN_FORMAT_RGBA_DXT5_SRGB },
    { "unorm_bc4", NVN_FORMAT_RGTC1_UNORM    },
    { "snorm_bc4", NVN_FORMAT_RGTC1_SNORM    },
    { "unorm_bc5", NVN_FORMAT_RGTC2_UNORM    },
    { "snorm_bc5", NVN_FORMAT_RGTC2_SNORM    },

    { "ufloat_bc6", NVN_FORMAT_BPTC_UFLOAT     },
    { "float_bc6" , NVN_FORMAT_BPTC_SFLOAT     },
    { "unorm_bc7" , NVN_FORMAT_BPTC_UNORM      },
    { "srgb_bc7"  , NVN_FORMAT_BPTC_UNORM_SRGB },

    { "unorm_astc_4x4"  , NVN_FORMAT_RGBA_ASTC_4x4        },
    { "srgb_astc_4x4"   , NVN_FORMAT_RGBA_ASTC_4x4_SRGB   },
    { "unorm_astc_5x4"  , NVN_FORMAT_RGBA_ASTC_5x4        },
    { "srgb_astc_5x4"   , NVN_FORMAT_RGBA_ASTC_5x4_SRGB   },
    { "unorm_astc_5x5"  , NVN_FORMAT_RGBA_ASTC_5x5        },
    { "srgb_astc_5x5"   , NVN_FORMAT_RGBA_ASTC_5x5_SRGB   },
    { "unorm_astc_6x5"  , NVN_FORMAT_RGBA_ASTC_6x5        },
    { "srgb_astc_6x5"   , NVN_FORMAT_RGBA_ASTC_6x5_SRGB   },
    { "unorm_astc_6x6"  , NVN_FORMAT_RGBA_ASTC_6x6        },
    { "srgb_astc_6x6"   , NVN_FORMAT_RGBA_ASTC_6x6_SRGB   },
    { "unorm_astc_8x5"  , NVN_FORMAT_RGBA_ASTC_8x5        },
    { "srgb_astc_8x5"   , NVN_FORMAT_RGBA_ASTC_8x5_SRGB   },
    { "unorm_astc_8x6"  , NVN_FORMAT_RGBA_ASTC_8x6        },
    { "srgb_astc_8x6"   , NVN_FORMAT_RGBA_ASTC_8x6_SRGB   },
    { "unorm_astc_8x8"  , NVN_FORMAT_RGBA_ASTC_8x8        },
    { "srgb_astc_8x8"   , NVN_FORMAT_RGBA_ASTC_8x8_SRGB   },
    { "unorm_astc_10x5" , NVN_FORMAT_RGBA_ASTC_10x5       },
    { "srgb_astc_10x5"  , NVN_FORMAT_RGBA_ASTC_10x5_SRGB  },
    { "unorm_astc_10x6" , NVN_FORMAT_RGBA_ASTC_10x6       },
    { "srgb_astc_10x6"  , NVN_FORMAT_RGBA_ASTC_10x6_SRGB  },
    { "unorm_astc_10x8" , NVN_FORMAT_RGBA_ASTC_10x8       },
    { "srgb_astc_10x8"  , NVN_FORMAT_RGBA_ASTC_10x8_SRGB  },
    { "unorm_astc_10x10", NVN_FORMAT_RGBA_ASTC_10x10      },
    { "srgb_astc_10x10" , NVN_FORMAT_RGBA_ASTC_10x10_SRGB },
    { "unorm_astc_12x10", NVN_FORMAT_RGBA_ASTC_12x10      },
    { "srgb_astc_12x10" , NVN_FORMAT_RGBA_ASTC_12x10_SRGB },
    { "unorm_astc_12x12", NVN_FORMAT_RGBA_ASTC_12x12      },
    { "srgb_astc_12x12" , NVN_FORMAT_RGBA_ASTC_12x12_SRGB },

    { nullptr, NVN_FORMAT_NONE },
};

NVNformat GetNvnFormat(const std::string& formatStr)
{
    return GetValueFromString(g_NvnFormatStringValues, formatStr, NVN_FORMAT_NONE);
}

//-----------------------------------------------------------------------------
//! @brief サブ画像数を取得します。
//!
//! @param[in] target テクスチャターゲットです。
//! @param[in] arrayLength 配列長です（キューブマップと配列以外は 1）。
//!
//! @return サブ画像数を返します。
//-----------------------------------------------------------------------------
int GetSubImageCount(const NVNtextureTarget target, const int arrayLength)
{
    return (IsCubeMap(target) || IsArray(target)) ? arrayLength : 1;
}

//-----------------------------------------------------------------------------
//! @brief NX 向けのタイリング変換のストレージフラグを取得します。
//!
//! @param[in] pExporter ハードウェアテクスチャエクスポータへのポインタです。
//! @param[in] target テクスチャターゲットです。
//! @param[in] format フォーマットです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです（3D 以外は 1）。
//! @param[in] mipCount ミップマップのレベル数です。
//! @param[in] subCount サブ画像数です。
//! @param[in] tilingFlags タイリングフラグです。
//! @param[in] sizeThreshold OptimizeSizeAuto の場合のデータ削減率のしきい値 [%] です。
//!
//! @return タイリング変換のストレージフラグを返します。
//-----------------------------------------------------------------------------
uint32_t GetStorageFlags(
    RawImageHardwareTextureExporter* pExporter,
    const NVNtextureTarget target,
    const NVNformat format,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipCount,
    const int subCount,
    const uint32_t tilingFlags,
    const int sizeThreshold
)
{
    const bool isSparse          = ((tilingFlags & NXTilingFlag_Sparse          ) != 0);
    const bool optimizesSizeAuto = ((tilingFlags & NXTilingFlag_OptimizeSizeAuto) != 0) &&
        !isSparse;
    const bool optimizesSize     = ((tilingFlags & NXTilingFlag_OptimizeSize    ) != 0) &&
        !isSparse && !optimizesSizeAuto;

    bool isMinimal = false;
    if (optimizesSize)
    {
        isMinimal = true;
    }
    else if (optimizesSizeAuto)
    {
        if (0 < sizeThreshold && sizeThreshold < 100)
        {
            const uint64_t minimalSize = pExporter->GetDataSize(
                target, imageW, imageH, imageD, format, mipCount, subCount,
                NVN_TEXPKG_STORAGE_FLAGS_MINIMAL_LAYOUT);
            const uint64_t defaultSize = pExporter->GetDataSize(
                target, imageW, imageH, imageD, format, mipCount, subCount,
                0);
            const double sizePercentage =
                static_cast<double>(minimalSize) / defaultSize * 100.0;
            isMinimal = (sizePercentage <= static_cast<double>(sizeThreshold));
        }
        else if (sizeThreshold >= 100)
        {
            isMinimal = true;
        }
    }

    return
        ((isSparse ) ? NVN_TEXPKG_STORAGE_FLAGS_SPARSE         : 0) |
        ((isMinimal) ? NVN_TEXPKG_STORAGE_FLAGS_MINIMAL_LAYOUT : 0);
}

//-----------------------------------------------------------------------------
//! @brief NX 向けのタイリング変換をします。
//!
//! @param[out] pDstTexture 変換後の NX テクスチャへのポインタです。
//! @param[in] pSrcData 変換前のデータです。
//! @param[in] srcDataSize 変換前のデータサイズです。
//! @param[in] target テクスチャターゲットです。
//! @param[in] format フォーマットです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです（3D 以外は 1）。
//! @param[in] arrayLength 配列長です（キューブマップと配列以外は 1）。
//! @param[in] mipCount ミップマップのレベル数です。
//! @param[in] tilingFlags タイリングフラグです。
//! @param[in] sizeThreshold OptimizeSizeAuto の場合のデータ削減率のしきい値 [%] です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool ConvertToNX(
    NXTexture* pDstTexture,
    const void* pSrcData,
    const size_t srcDataSize,
    const NVNtextureTarget target,
    const NVNformat format,
    const int imageW,
    const int imageH,
    const int imageD,
    const int arrayLength,
    const int mipCount,
    const uint32_t tilingFlags,
    const int sizeThreshold
)
{
    //-----------------------------------------------------------------------------
    // 変換元の画像を作成します。
    const int subCount = GetSubImageCount(target, arrayLength);
    RawImage* pImg = g_ImageLib.createRawImage();
    TPError error = pImg->Create(target, imageW, imageH, imageD, format, mipCount, subCount);
    if (error != TP_OK)
    {
        cerr << "Error: Cannot create raw image: " << TPErrorToStr(error) << ", format = 0x" << hex << format << dec << endl;
        g_ImageLib.releaseRawImage(pImg);
        return false;
    }

    if (IsCubeMap(target))
    {
        pImg->SetCubeFaceMask(CUBE_FACE_All);
    }

    //-----------------------------------------------------------------------------
    // 変換元の画像にテクスチャデータを設定します。
    GetImageSliceDataSizeFunction sizeFunc = pImg->CurrentImageSliceDataSizeFunction();
    const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(pSrcData);
    const uint8_t* pSrcEnd = pSrc + srcDataSize;
    for (int level = 0; level < mipCount; ++level)
    {
        const int curW = std::max(imageW >> level, 1);
        const int curH = std::max(imageH >> level, 1);
        const int curD = (target == NVN_TEXTURE_TARGET_3D) ?
            std::max(imageD >> level, 1) : 1;
        for (int subIdx = 0; subIdx < subCount; ++subIdx)
        {
            RawMipMapLevel* pLevel = pImg->MipMapLevel(level, subIdx);
            uint32_t pitch = 0;
            uint32_t sliceSize = sizeFunc(&g_ImageLib, curW, curH, 1, format, &pitch);
                // ↑引数 depth は影響しません。
            if (pSrc + sliceSize * curD > pSrcEnd)
            {
                cerr << "Error: Source data size is wrong: level = " << level << ", sub = " << subIdx << endl;
                g_ImageLib.releaseRawImage(pImg);
                return false;
            }
            for (int sliceIdx = 0; sliceIdx < curD; ++sliceIdx)
            {
                memcpy(g_ImageLib.rawMipMapLevelDataSlice(pLevel, sliceIdx), pSrc, sliceSize);
                pSrc += sliceSize;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // ハードウェア依存のテクスチャデータに変換します。
    RawImageHardwareTextureExporter* pExporter =
        g_TexpkgLib.createRawImageHardwareTextureExporter(&g_ImageLib);
    NVNHWTexture hwTex;
    const uint32_t storageFlags = GetStorageFlags(pExporter,
        target, format, imageW, imageH, imageD, mipCount, subCount,
        tilingFlags, sizeThreshold);
    error = pExporter->Convert(pImg, &hwTex, storageFlags);
    g_ImageLib.releaseRawImage(pImg); // 変換元の画像を解放します。
    if (error != TP_OK)
    {
        cerr << "Error: Cannot convert to HW texture: " << TPErrorToStr(error) << endl;
        g_TexpkgLib.releaseRawImageHardwareTextureExporter(pExporter);
        return false;
    }

    //-----------------------------------------------------------------------------
    // NX テクスチャを作成します。
    const NVNHWTextureHeader& header = hwTex.header;
    NXTexture& nxTex = *pDstTexture;
    nxTex.dataSize = static_cast<size_t>(header.dataSize);
    nxTex.pData = new uint8_t[nxTex.dataSize];
    memcpy(nxTex.pData, hwTex.data, nxTex.dataSize);
    nxTex.alignment = header.align;
    memset(nxTex.levelOffsets, 0, sizeof(nxTex.levelOffsets));
    for (int level = 0; level < mipCount; ++level)
    {
        nxTex.levelOffsets[level] = static_cast<size_t>(header.mipmapLevelOffset[level]);
    }
    memcpy(nxTex.textureLayout, &header.layout, sizeof(nxTex.textureLayout));
    pExporter->ReleaseTextureData(&hwTex);
    g_TexpkgLib.releaseRawImageHardwareTextureExporter(pExporter);

    return true;
}

//-----------------------------------------------------------------------------
//! @brief NX 向けのタイリング変換を解除します。
//!
//! @param[out] pDstTexture 変換後の NX テクスチャへのポインタです。
//! @param[in] pSrcData 変換前のデータです。
//! @param[in] srcDataSize 変換前のデータサイズです。
//! @param[in] target テクスチャターゲットです。
//! @param[in] format フォーマットです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです（3D 以外は 1）。
//! @param[in] arrayLength 配列長です（キューブマップと配列以外は 1）。
//! @param[in] mipCount ミップマップのレベル数です。
//! @param[in] pTextureLayout テクスチャレイアウトへのポインタです。
//!                           タイリング変換した際の値を指定します。
//! @param[in] tilingFlags タイリングフラグです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool ConvertToLinear(
    NXTexture* pDstTexture,
    const void* pSrcData,
    const size_t srcDataSize,
    const NVNtextureTarget target,
    const NVNformat format,
    const int imageW,
    const int imageH,
    const int imageD,
    const int arrayLength,
    const int mipCount,
    const void* pTextureLayout,
    const uint32_t tilingFlags
)
{
    //-----------------------------------------------------------------------------
    // HW テクスチャを作成します。
    const int subCount = GetSubImageCount(target, arrayLength);
    NVNHWTexture hwTex;
    NVNHWTextureHeader& header = hwTex.header;
    header.dataSize = srcDataSize;
    header.align = 1;
    header.width  = imageW;
    header.height = imageH;
    header.depth =
        (target == NVN_TEXTURE_TARGET_CUBEMAP) ? 1           :
        (IsArray(target)                     ) ? arrayLength :
        imageD;
    header.target = target;
    header.format = format;
    header.mipmapLevels = mipCount;
    header.subImagePitch = static_cast<uint32_t>(srcDataSize / subCount);
    memset(header.mipmapLevelOffset, 0, sizeof(header.mipmapLevelOffset));
    memset(&header.layout, 0x00, sizeof(header.layout));
    if (pTextureLayout != nullptr)
    {
        memcpy(&header.layout, pTextureLayout, sizeof(header.layout));
    }
    header.sparse = ((tilingFlags & NXTilingFlag_Sparse) != 0) ? NVN_TRUE : NVN_FALSE;
    hwTex.data = reinterpret_cast<uint8_t*>(const_cast<void*>(pSrcData));

    //-----------------------------------------------------------------------------
    // タイリングがリニアな画像に変換します。
    RawImageHardwareTextureExporter* pExporter =
        g_TexpkgLib.createRawImageHardwareTextureExporter(&g_ImageLib);
    RawImage* pImg = g_ImageLib.createRawImage();
    TPError error = pExporter->Unconvert(pImg, &hwTex);
    g_TexpkgLib.releaseRawImageHardwareTextureExporter(pExporter);
    if (error != TP_OK)
    {
        cerr << "Error: Cannot convert to linear texture: " << TPErrorToStr(error) << endl;
        g_ImageLib.releaseRawImage(pImg);
        return false;
    }

    //-----------------------------------------------------------------------------
    // タイリングがリニアなテクスチャデータのサイズを求めます。
    NXTexture& nxTex = *pDstTexture;
    GetImageSliceDataSizeFunction sizeFunc = pImg->CurrentImageSliceDataSizeFunction();
    nxTex.dataSize = 0;
    memset(nxTex.levelOffsets, 0, sizeof(nxTex.levelOffsets));
    for (int level = 0; level < mipCount; ++level)
    {
        nxTex.levelOffsets[level] = nxTex.dataSize;
        const int curW = std::max(imageW >> level, 1);
        const int curH = std::max(imageH >> level, 1);
        const int curD = (target == NVN_TEXTURE_TARGET_3D) ?
            std::max(imageD >> level, 1) : 1;
        uint32_t pitch = 0;
        uint32_t sliceSize = sizeFunc(&g_ImageLib, curW, curH, 1, format, &pitch);
        nxTex.dataSize += sliceSize * curD * subCount;
    }

    //-----------------------------------------------------------------------------
    // NX テクスチャを作成します。
    nxTex.pData = new uint8_t[nxTex.dataSize];
    uint8_t* pDst = nxTex.pData;
    for (int level = 0; level < mipCount; ++level)
    {
        const int curW = std::max(imageW >> level, 1);
        const int curH = std::max(imageH >> level, 1);
        const int curD = (target == NVN_TEXTURE_TARGET_3D) ?
            std::max(imageD >> level, 1) : 1;
        for (int subIdx = 0; subIdx < subCount; ++subIdx)
        {
            RawMipMapLevel* pLevel = pImg->MipMapLevel(level, subIdx);
            uint32_t pitch = 0;
            uint32_t sliceSize = sizeFunc(&g_ImageLib, curW, curH, 1, format, &pitch);
            for (int sliceIdx = 0; sliceIdx < curD; ++sliceIdx)
            {
                memcpy(pDst, g_ImageLib.rawMipMapLevelDataSlice(pLevel, sliceIdx), sliceSize);
                pDst += sliceSize;
            }
        }
    }
    nxTex.alignment = 1;
    memset(nxTex.textureLayout, 0x00, sizeof(nxTex.textureLayout));
    g_ImageLib.releaseRawImage(pImg);

    return true;
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 構造化例外の変換関数です。
//-----------------------------------------------------------------------------
void TranslateSe(unsigned int code, struct _EXCEPTION_POINTERS* ep)
{
    throw ep; // 標準 C++ の例外を発生させます。
    NXDLL_UNUSED_VARIABLE(code);
}

//-----------------------------------------------------------------------------
//! @brief DLL のメイン関数です。
//-----------------------------------------------------------------------------
BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
    switch (fdwReason)
    {
        case DLL_PROCESS_ATTACH:
        {
            _set_se_translator(TranslateSe); // このスレッドでのみ有効です。

            char dllPath[MAX_PATH];
            GetModuleFileNameA(hinstDLL, dllPath, MAX_PATH);
            g_DllFolderPath = GetFolderFromFilePath(dllPath);
            break;
        }

        case DLL_THREAD_ATTACH:
            break;

        case DLL_THREAD_DETACH:
            break;

        case DLL_PROCESS_DETACH:
            if (g_IsNvnLoaded)
            {
                FreeNvnDll();
                g_IsNvnLoaded = false;
            }
            break;

        default:
            break;
    }
    NXDLL_UNUSED_VARIABLE(lpvReserved);
    return TRUE;
}

//-----------------------------------------------------------------------------
//! @brief タイリングを変換します。
//!
//! @param[out] pDstTexture 変換後の NX テクスチャへのポインタです。
//! @param[in] isTiling NX 向けのタイリング変換をするなら true、タイリング変換を解除するなら false です。
//! @param[in] pSrcData 変換前のデータです。
//! @param[in] srcDataSize 変換前のデータサイズです。
//! @param[in] dimension 中間ファイルの次元文字列です。
//! @param[in] format 中間ファイルのフォーマット文字列です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです（3D 以外は 1）。
//! @param[in] arrayLength 配列長です（キューブマップと配列以外は 1）。
//! @param[in] mipCount ミップマップのレベル数です。
//! @param[in] pTextureLayout テクスチャレイアウトへのポインタです。
//!                           NX 向けのタイリング変換をする場合は使用されません（nullptr でも構いません）。
//!                           タイリング変換を解除する場合はタイリング変換した際の値を指定します。
//! @param[in] tilingFlags タイリングフラグです。
//!                        タイリング変換を解除する場合、Sparse 以外は処理に影響しません。
//! @param[in] sizeThreshold OptimizeSizeAuto の場合のデータ削減率のしきい値 [%] です。
//!                          タイリング変換を解除する場合は処理に影響しません。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool ConvertTiling(
    NXTexture* pDstTexture,
    const bool isTiling,
    const void* pSrcData,
    const size_t srcDataSize,
    const wchar_t* dimension,
    const wchar_t* format,
    const int imageW,
    const int imageH,
    const int imageD,
    const int arrayLength,
    const int mipCount,
    const void* pTextureLayout,
    const uint32_t tilingFlags,
    const int sizeThreshold
)
{
    //-----------------------------------------------------------------------------
    // 引数をチェックします。
    if (pDstTexture == nullptr ||
        pSrcData    == nullptr ||
        dimension   == nullptr ||
        format      == nullptr)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // NVN の dll ファイルをロードします。
    if (!g_IsNvnLoaded)
    {
        if (!LoadNvnDll())
        {
            return false;
        }
        g_IsNvnLoaded = true;
    }

    //-----------------------------------------------------------------------------
    // テクスチャターゲットとフォーマットを取得します。
    const std::string dimensionStr = GetAnsiFromUnicode(dimension);
    const std::string formatStr    = GetAnsiFromUnicode(format);
    const NVNtextureTarget target = GetNvnTextureTarget(dimensionStr);
    const NVNformat nvnFormat = GetNvnFormat(formatStr);
    //cerr << "nvn format: " << formatStr << " = 0x" << hex << nvnFormat << dec << endl;
    if (nvnFormat == NVN_FORMAT_NONE)
    {
        cerr << "Error: Unsupported format: " << formatStr << endl;
        return false;
    }

    //-----------------------------------------------------------------------------
    // タイリングを変換します。
    if (isTiling)
    {
        return ConvertToNX(pDstTexture, pSrcData, srcDataSize,
            target, nvnFormat, imageW, imageH, imageD, arrayLength, mipCount,
            tilingFlags, sizeThreshold);
    }
    else
    {
        return ConvertToLinear(pDstTexture, pSrcData, srcDataSize,
            target, nvnFormat, imageW, imageH, imageD, arrayLength, mipCount,
            pTextureLayout, tilingFlags);
    }
}

//-----------------------------------------------------------------------------
//! @brief NX テクスチャを解放します。
//!
//! @param[in,out] pTexture NX テクスチャへのポインタです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool ReleaseTexture(NXTexture* pTexture)
{
    if (pTexture != nullptr)
    {
        if (pTexture->pData != nullptr)
        {
            delete[] pTexture->pData;
        }
        memset(pTexture, 0, sizeof(*pTexture));
        return true;
    }
    else
    {
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief NX 向けのタイリング変換をした場合のデータサイズを取得します。
//!
//! @param[in] dimension 中間ファイルの次元文字列です。
//! @param[in] format 中間ファイルのフォーマット文字列です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです（3D 以外は 1）。
//! @param[in] arrayLength 配列長です（キューブマップと配列以外は 1）。
//! @param[in] mipCount ミップマップのレベル数です。
//! @param[in] tilingFlags タイリングフラグです。
//! @param[in] sizeThreshold OptimizeSizeAuto の場合のデータ削減率のしきい値 [%] です。
//!
//! @return NX 向けのタイリング変換をした場合のデータサイズを返します。
//!         処理失敗なら 0 を返します。
//-----------------------------------------------------------------------------
DLLEXPORT size_t GetDataSizeNX(
    const wchar_t* dimension,
    const wchar_t* format,
    const int imageW,
    const int imageH,
    const int imageD,
    const int arrayLength,
    const int mipCount,
    const uint32_t tilingFlags,
    const int sizeThreshold
)
{
    //-----------------------------------------------------------------------------
    // 引数をチェックします。
    if (dimension == nullptr ||
        format    == nullptr)
    {
        return 0;
    }

    //-----------------------------------------------------------------------------
    // NVN の dll ファイルをロードします。
    if (!g_IsNvnLoaded)
    {
        if (!LoadNvnDll())
        {
            return 0;
        }
        g_IsNvnLoaded = true;
    }

    //-----------------------------------------------------------------------------
    // テクスチャターゲットとフォーマットを取得します。
    const std::string dimensionStr = GetAnsiFromUnicode(dimension);
    const std::string formatStr    = GetAnsiFromUnicode(format);
    const NVNtextureTarget target = GetNvnTextureTarget(dimensionStr);
    const NVNformat nvnFormat = GetNvnFormat(formatStr);
    if (nvnFormat == NVN_FORMAT_NONE)
    {
        cerr << "Error: Unsupported format: " << formatStr << endl;
        return 0;
    }

    //-----------------------------------------------------------------------------
    // NX 向けのタイリング変換をした場合のデータサイズを取得します。
    RawImageHardwareTextureExporter* pExporter =
        g_TexpkgLib.createRawImageHardwareTextureExporter(&g_ImageLib);
    const int subCount = GetSubImageCount(target, arrayLength);
    const uint32_t storageFlags = GetStorageFlags(pExporter,
        target, nvnFormat, imageW, imageH, imageD, mipCount, subCount,
        tilingFlags, sizeThreshold);
    const size_t dataSize = static_cast<size_t>(pExporter->GetDataSize(
        target, imageW, imageH, imageD, nvnFormat, mipCount, subCount,
        storageFlags));
    g_TexpkgLib.releaseRawImageHardwareTextureExporter(pExporter);
    return dataSize;
}

//-----------------------------------------------------------------------------
//! @brief メモリプールのアライメント値を取得します。
//!
//! @param[out] pAddressAlignment アドレスのアライメント値へのポインタです。
//! @param[out] pSizeAlignment サイズのアライメント値へのポインタです。
//-----------------------------------------------------------------------------
DLLEXPORT void GetMemoryPoolAlignment(int* pAddressAlignment, int* pSizeAlignment)
{
    *pAddressAlignment = NVN_MEMORY_POOL_STORAGE_ALIGNMENT;
    *pSizeAlignment    = NVN_MEMORY_POOL_STORAGE_GRANULARITY;
}

