﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <locale>

#include <gfxTool_VariationDefinitionArg.h>
#include <gfxTool_VariationXml.h>

namespace nn {
namespace gfxTool {

void VariationConstantDefinition::Initialize( nngfxToolShaderCompilerVariationConstantDefinition* pArg,
        const VariationConstantDefinitionNode* pVariationConstantDefinitionNode, int index )
{
    m_pArg = pArg;
    auto name = pVariationConstantDefinitionNode->GetNameAttribute();
    if( name.empty() )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlNodeNotFound, "%s::name is not found.",
            pVariationConstantDefinitionNode->GetId() );
    }
    m_Name.assign( name.data(), name.length() );
    m_pArg->name.pValue = m_Name.data();
    m_pArg->name.length = NumericCastAuto( m_Name.length() );
    m_pArg->indexInVariationConstantValueArray = index;

    auto type = pVariationConstantDefinitionNode->GetTypeAttribute();
    if( type.empty() )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlNodeNotFound, "%s::name is not found.",
            pVariationConstantDefinitionNode->GetId() );
    }
    auto pChar = type.data();
    if( strncmp( type.data(), "bool", 4 ) == 0 )
    {
        m_pArg->type = nngfxToolShaderCompilerVariableType_Bool32;
        pChar += 4;
    }
    else if( strncmp( type.data(), "int", 3 ) == 0 )
    {
        m_pArg->type = nngfxToolShaderCompilerVariableType_Int32;
        pChar += 3;
    }
    else if( strncmp( type.data(), "uint", 4 ) == 0 )
    {
        m_pArg->type = nngfxToolShaderCompilerVariableType_Uint32;
        pChar += 4;
    }
    else if( strncmp( type.data(), "float", 5 ) == 0 )
    {
        m_pArg->type = nngfxToolShaderCompilerVariableType_Float32;
        pChar += 5;
    }
    else if( strncmp( type.data(), "double", 6 ) == 0 )
    {
        m_pArg->type = nngfxToolShaderCompilerVariableType_Float64;
        pChar += 6;
    }
    else
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "%s is invalid.", type.data() );
    }

    std::locale locale( "C" );
    m_pArg->vectorComponents = 0;
    m_pArg->matrix.column = 0;
    m_pArg->matrix.row = 0;
    m_pArg->arrayLength = 0;
    int vectorComponents = *pChar - '0';
    if( vectorComponents >= 2 && vectorComponents <= 4 )
    {
        if( *( ++pChar ) == 'x' )
        {
            m_pArg->matrix.column = NumericCastAuto( vectorComponents );
            m_pArg->matrix.row = NumericCastAuto( *( ++pChar ) - '0' );
            if( m_pArg->matrix.row < 2 || m_pArg->matrix.row > 4 )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "%s is invalid.", type.data() );
            }
            ++pChar;
        }
        else
        {
            m_pArg->vectorComponents = NumericCastAuto( vectorComponents );
        }
    }
    while( *pChar != '\0' && std::isblank( *pChar, locale ) )
    {
        ++pChar;
    }
    if( *pChar == '[' )
    {
        Custom< std::stringstream >::Type ss;
        int arrayLength;
        ss << ++pChar;
        ss >> arrayLength;
        if( ss.fail() )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "%s is invalid.", type.data() );
        }
        m_pArg->arrayLength = NumericCastAuto( arrayLength );
        pChar = strchr( pChar, ']' );
        pChar = pChar == nullptr ? nullptr : pChar + 1;
    }
    if( pChar == nullptr || *pChar != '\0' )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "%s is invalid.", type.data() );
    }
}

void PreprocessorDefinitionDefinition::Initialize(
    nngfxToolShaderCompilerPreprocessorDefinitionDefinition* pArg,
    const PreprocessorDefinitionDefinitionNode* pPreprocessorDefinitionDefinitionNode, int index )
{
    m_pArg = pArg;
    auto name = pPreprocessorDefinitionDefinitionNode->GetNameAttribute();
    if( name.empty() )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlAttributeNotFound, "%s::name is not found.",
            pPreprocessorDefinitionDefinitionNode->GetId() );
    }
    m_Name.assign( name.data(), name.length() );
    m_pArg->name.pValue = m_Name.data();
    m_pArg->name.length = NumericCastAuto( m_Name.length() );
    m_pArg->indexInPreprocessorDefinitionValueArray = index;
}

void VariationDefinition::Initialize(
    const StageShaderVariationDefinitionNode* pStageShaderVariationDefinitionNode )
{
    if( auto pVariationConstantBufferNode =
        pStageShaderVariationDefinitionNode->GetVariationConstantBufferNode() )
    {
        auto bufferName = pVariationConstantBufferNode->GetNameAttribute();
        if( bufferName.empty() )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlAttributeNotFound, "%s::name is not found.",
                pVariationConstantBufferNode->GetId() );
        }
        m_BufferName.assign( bufferName.data(), bufferName.length() );
        this->variationConstantBufferName.length = NumericCastAuto( m_BufferName.length() );
        this->variationConstantBufferName.pValue = m_BufferName.data();
    }

    if( auto pVariationConstantDefinitionArrayNode =
        pStageShaderVariationDefinitionNode->GetVariationConstantDefinitionArrayNode() )
    {
        auto length = pVariationConstantDefinitionArrayNode->GetLengthAttribute();
        if( length.empty() )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlAttributeNotFound, "%s::length is not found.",
                pVariationConstantDefinitionArrayNode->GetId() );
        }
        this->variationConstantDefinitionCount = NumericCastAuto( LexicalCast< int >( length.data() ) );
        m_variationConstantDefinitionArrayRaw.resize( this->variationConstantDefinitionCount );
        m_variationConstantDefinitionArray.resize( this->variationConstantDefinitionCount );
        decltype( pVariationConstantDefinitionArrayNode->GetNextChild( nullptr ) )
            pVariationConstantDefinitionNode = nullptr;
        for( int idxVariationConstantDefinition = 0, variationConstantDefinitionCountMax = NumericCastAuto(
            this->variationConstantDefinitionCount ); idxVariationConstantDefinition <
            variationConstantDefinitionCountMax; ++idxVariationConstantDefinition )
        {
            pVariationConstantDefinitionNode =
                pVariationConstantDefinitionArrayNode->GetNextChild( pVariationConstantDefinitionNode );
            if( pVariationConstantDefinitionNode == nullptr )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlNodeNotFound, "%s is not found.",
                    pVariationConstantDefinitionNode->GetId() );
            }
            m_variationConstantDefinitionArray[ idxVariationConstantDefinition ].Initialize(
                &m_variationConstantDefinitionArrayRaw[ idxVariationConstantDefinition ],
                pVariationConstantDefinitionNode, idxVariationConstantDefinition );
        }
        this->pVariationConstantDefinitionArray = &m_variationConstantDefinitionArrayRaw[ 0 ];
    }

    if( auto pPreprocessorDefinitionDefinitionArrayNode =
        pStageShaderVariationDefinitionNode->GetPreprocessorDefinitionDefinitionArrayNode() )
    {
        auto length = pPreprocessorDefinitionDefinitionArrayNode->GetLengthAttribute();
        if( length.empty() )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlAttributeNotFound, "%s::length is not found.",
                pPreprocessorDefinitionDefinitionArrayNode->GetId() );
        }
        this->preprocessorDefinitionDefinitionCount = NumericCastAuto( LexicalCast< int >( length.data() ) );
        m_preprocessorDefinitionDefinitionArrayRaw.resize( this->preprocessorDefinitionDefinitionCount );
        m_preprocessordefinitionDefinitionArray.resize( this->preprocessorDefinitionDefinitionCount );
        decltype( pPreprocessorDefinitionDefinitionArrayNode->GetNextChild( nullptr ) )
            pPreprocessorDefinitionDefinitionNode = nullptr;
        for( int idxPreprocessorDefinitionDefinition = 0, preprocessorDefinitionDefinitionCountMax =
            NumericCastAuto( this->preprocessorDefinitionDefinitionCount ); idxPreprocessorDefinitionDefinition <
            preprocessorDefinitionDefinitionCountMax; ++idxPreprocessorDefinitionDefinition )
        {
            pPreprocessorDefinitionDefinitionNode =
                pPreprocessorDefinitionDefinitionArrayNode->GetNextChild( pPreprocessorDefinitionDefinitionNode );
            if( pPreprocessorDefinitionDefinitionNode == nullptr )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlNodeNotFound, "%s is not found.",
                    pPreprocessorDefinitionDefinitionNode->GetId() );
            }
            m_preprocessordefinitionDefinitionArray[ idxPreprocessorDefinitionDefinition ].Initialize(
                &m_preprocessorDefinitionDefinitionArrayRaw[ idxPreprocessorDefinitionDefinition ],
                pPreprocessorDefinitionDefinitionNode, idxPreprocessorDefinitionDefinition );
        }
        this->pPreprocessorDefinitionDefinitionArray = &m_preprocessorDefinitionDefinitionArrayRaw[ 0 ];
    }
}

void VariationDefinitionArg::Initialize( const ShaderVariationDefinitionNode* pShaderVariationDefinitionNode )
{
    static const nngfxToolShaderCompilerVariationDefinition*
        nngfxToolShaderCompilerVariationDefinitionArg::*pVariationDefinition[] =
    {
        &nngfxToolShaderCompilerVariationDefinitionArg::pVertexShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pHullShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pDomainShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pGeometryShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pPixelShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pComputeShaderVariationDefinition
    };

    for( int stage = 0; stage < static_cast< int >( ShaderStage::End ); ++stage )
    {
        if( auto pStageShaderVariationDefinitionNode =
            pShaderVariationDefinitionNode->GetStageShaderVariationDefinitionNode(
            StaticCastAuto( stage ) ) )
        {
            m_pVariationDefinition[ stage ].reset( new VariationDefinition() );
            m_pVariationDefinition[ stage ]->Initialize( pStageShaderVariationDefinitionNode );
            this->*pVariationDefinition[ stage ] = m_pVariationDefinition[ stage ].get();
        }
    }
}

}
}
