﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <filesystem>

#include <nn/gfx/gfx_ResShader.h>
#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_CompileOutput.h>

#include <gfxTool_CompilerManager.h>

namespace {

const char* GetShaderCompilerDllName()
{
    return "ShaderCompiler.dll";
}

}

namespace nn {
namespace gfxTool {

CompilerManager::CompilerManager()
    : m_ShaderCompilerDll()
    , m_hShaderCompiler()
    , m_pCompileOutput()
{
}

CompilerManager::~CompilerManager()
{
    DeleteCompileOutput();

    if( m_hShaderCompiler )
    {
        if( m_ShaderCompilerDll.IsInitialized() )
        {
            m_ShaderCompilerDll.DeleteHandle( m_hShaderCompiler );
        }
    }

    if( m_ShaderCompilerDll.IsInitialized() )
    {
        m_ShaderCompilerDll.Finalize();
    }
}

void CompilerManager::Reset()
{
    DeleteCompileOutput();
}

void CompilerManager::Compile( const nngfxToolShaderCompilerCompileArg* pArg )
{
    const char* const pShaderCompilerErrorMessage =
        "An error occurred in ShaderCompiler.dll";

    if( !m_ShaderCompilerDll.IsInitialized() )
    {
        auto path = std::tr2::sys::path( GetModulePath(
            GetShaderConverterModuleHandle() ) ).parent_path().string();
        path.append( "/" ).append( GetShaderCompilerDllName() );
        if( !m_ShaderCompilerDll.Initialize( path.c_str() ) )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_DllNotFound, "Failed to load %s.", path.c_str() );
        }
    }

    nngfxToolResultCode result;
    if( m_hShaderCompiler )
    {
        result = m_ShaderCompilerDll.ResetHandle( m_hShaderCompiler );
    }
    else
    {
        nngfxToolShaderCompilerCreateHandleArg shaderCompilerCreateHandleArg = {};
        result = m_ShaderCompilerDll.CreateHandle( &m_hShaderCompiler, &shaderCompilerCreateHandleArg );
    }
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_THROW_MSG( result, pShaderCompilerErrorMessage );
    }

    result = m_ShaderCompilerDll.Compile( &m_pCompileOutput, m_hShaderCompiler, pArg );
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_THROW_MSG( result, pShaderCompilerErrorMessage );
    }
}

void CompilerManager::DeleteCompileOutput()
{
    if( m_pCompileOutput )
    {
        if( !m_ShaderCompilerDll.IsInitialized() )
        {
            NN_GFXTOOL_THROW( nngfxToolResultCode_DllNotFound );
        }

        m_ShaderCompilerDll.DeleteCompileOutput( m_pCompileOutput );
        m_pCompileOutput = nullptr;
    }
}

}
}
