﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <sstream>

#include <nn/util/util_BytePtr.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_GroupSource.h>

namespace {

inline const char* GetScalarName( nngfxToolShaderCompilerVariableType type )
{
    if( type >= nngfxToolShaderCompilerVariableType_End )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "VariableType" );
    }

    static const char* scalarNameTable[] =
    {
        "bool",
        "int",
        "uint",
        "float",
        "double"
    };

    return scalarNameTable[ type ];
}

inline const char* GetVectorName( nngfxToolShaderCompilerVariableType type, int vectorComponents )
{
    if( type >= nngfxToolShaderCompilerVariableType_End )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "VariableType" );
    }

    if( vectorComponents < 2 || vectorComponents > 4 )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "VectorComponents" );
    }

    static const char* vectorNameTable[][ 3 ] =
    {
        { "bvec2", "bvec3", "bvec4" },
        { "ivec2", "ivec3", "ivec4" },
        { "uvec2", "uvec3", "uvec4" },
        { "vec2", "vec3", "vec4" },
        { "dvec2", "dvec3", "dvec4" }
    };

    return vectorNameTable[ type ][ vectorComponents - 2 ];
}

inline const char* GetMatrixName( nngfxToolShaderCompilerVariableType type, int column, int row )
{
    if( type >= nngfxToolShaderCompilerVariableType_End )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "VariableType" );;
    }
    if( column < 2 || column > 4 )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "MatrixColumn(%d)", column );;
    }
    if( row < 2 || row > 4 )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "MatrixRow(%d)", row );
    }

    static const char* matrixNameTable[][ 3 ][ 3 ] =
    {
        { { "mat2", "mat2x3", "mat2x4" }, { "mat3x2", "mat3", "mat3x4" }, { "mat4x2", "mat4x3", "mat4" } },
        { { "dmat2", "dmat2x3", "dmat2x4" }, { "dmat3x2", "dmat3", "dmat3x4" }, { "dmat4x2", "dmat4x3", "dmat4" } }
    };

    int idxType;
    switch( type )
    {
    case nngfxToolShaderCompilerVariableType_Float32:
        {
            idxType = 0;
        }
        break;
    case nngfxToolShaderCompilerVariableType_Float64:
        {
            idxType = 1;
        }
        break;
    default:
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument,
            "%s matrix is not supported.", GetScalarName( type ) );
    }
    int idxColumn = column - 2;
    int idxRow = row - 2;

    return matrixNameTable[ idxType ][ idxColumn ][ idxRow ];
}

inline const char* GetTypeName( const nngfxToolShaderCompilerVariationConstantDefinition& definition )
{
    auto type = static_cast< nngfxToolShaderCompilerVariableType >( definition.type );
    if( definition.matrix.row > 1 )
    {
        return GetMatrixName( type, definition.matrix.column, definition.matrix.row );
    }
    else if( definition.vectorComponents > 1 )
    {
        return GetVectorName( type, definition.vectorComponents );
    }
    return GetScalarName( type );
}

inline const void* WriteScalar( nn::gfxTool::Custom< std::stringstream >::Type& ss,
    nngfxToolShaderCompilerVariableType type, const void* pValue )
{
    ptrdiff_t offset = 0;
    switch( type )
    {
    case nngfxToolShaderCompilerVariableType_Bool32:
        {
            ss << *static_cast< const int32_t* >( pValue ) ? "true" : "false";
            offset = sizeof( int32_t );
        }
        break;
    case nngfxToolShaderCompilerVariableType_Int32:
        {
            ss << *static_cast< const int32_t* >( pValue );
            offset = sizeof( int32_t );
        }
        break;
    case nngfxToolShaderCompilerVariableType_Uint32:
        {
            ss << *static_cast< const uint32_t* >( pValue ) << "u";
            offset = sizeof( uint32_t );
        }
        break;
    case nngfxToolShaderCompilerVariableType_Float32:
        {
            ss << *static_cast< const float* >( pValue );
            offset = sizeof( float );
        }
        break;
    case nngfxToolShaderCompilerVariableType_Float64:
        {
            ss << *static_cast< const double* >( pValue );
            offset = sizeof( double );
        }
        break;
    default:
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument,
            "Invalid variable type (%d).", nn::gfxTool::NumericCast< int >( type ) );
    }
    return nn::util::ConstBytePtr( pValue, offset ).Get();
}

inline const void* WriteVector( nn::gfxTool::Custom< std::stringstream >::Type& ss,
    nngfxToolShaderCompilerVariableType type, int vectorComponent, const void* pValue )
{
    ss << GetVectorName( type, vectorComponent );
    ss.write( "(", 1 );
    for( int idxComponent = 0; idxComponent < vectorComponent; ++idxComponent )
    {
        if( idxComponent != 0 )
        {
            ss.write( ",", 1 );
        }
        pValue = WriteScalar( ss, type, pValue );
    }
    ss.write( ")", 1 );
    return pValue;
}

inline const void* WriteMatrix( nn::gfxTool::Custom< std::stringstream >::Type& ss,
    nngfxToolShaderCompilerVariableType type, int column, int row, const void* pValue )
{
    ss << GetMatrixName( type, column, row );
    ss.write( "(", 1 );
    for( int idxRow = 0; idxRow < row; ++idxRow )
    {
        if( idxRow != 0 )
        {
            ss.write( ",", 1 );
        }
        pValue = WriteVector( ss, type, column, pValue );
    }
    ss.write( ")", 1 );
    return pValue;
}

inline const void* WriteValue( nn::gfxTool::Custom< std::stringstream >::Type& ss,
    const nngfxToolShaderCompilerVariationConstantDefinition& definition, const void* pValue )
{
    auto type = static_cast< nngfxToolShaderCompilerVariableType >( definition.type );

    ss << GetTypeName( definition );
    if( definition.arrayLength > 1 )
    {
        ss.write( "[]", 2 );
    }
    ss.write( "(", 1 );

    for( int idxArray = 0, arrayCount = definition.arrayLength == 0 ? 1 :
        nn::gfxTool::NumericCast< int >( definition.arrayLength ); idxArray < arrayCount; ++idxArray )
    {
        if( idxArray != 0 )
        {
            ss.write( ",", 1 );
        }
        if( definition.matrix.row > 1 )
        {
            pValue = WriteMatrix( ss, type, definition.matrix.column, definition.matrix.row, pValue );
        }
        else if( definition.vectorComponents > 1 )
        {
            pValue = WriteVector( ss, type, definition.vectorComponents, pValue );
        }
        else
        {
            pValue = WriteScalar( ss, type, pValue );
        }
    }

    ss.write( ")", 1 );

    return pValue;
}

}

namespace nn {
namespace gfxTool {

template<>
void VariationConstantSource::Initialize< nngfxToolShaderCompilerShaderSourceFormat_Glsl >(
    const nngfxToolShaderCompilerCompileArg* pArg,
    const Custom< std::vector< int > >::Type* pGroupToHeadVariationTable, ShaderStage stage )
{
    if( pGroupToHeadVariationTable == nullptr )
    {
        NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
    }

    m_Sources.resize( pGroupToHeadVariationTable->size() );

    if( pArg->pVariationDefinitionArg == nullptr )
    {
        return;
    }

    auto pVariationDefinition = GetStageVariationDefinition( pArg->pVariationDefinitionArg, stage );

    for( int idxGroup = 0, groupCount = NumericCastAuto(
        pGroupToHeadVariationTable->size() ); idxGroup < groupCount; ++idxGroup )
    {
        auto& dst = m_Sources[ idxGroup ];
        auto pStageVariationValue = GetStageVariationValue(
            pArg->pVariationValueArray + ( *pGroupToHeadVariationTable )[ idxGroup ], stage );
        if( pStageVariationValue == nullptr )
        {
            continue;
        }
        auto pValues = pStageVariationValue->pVariationConstantValueArray;
        if( pValues == nullptr )
        {
            continue;
        }
        nn::gfxTool::Custom< std::stringstream >::Type ss;
        for( int idxDefinition = 0, definitionCount = NumericCastAuto(
            pVariationDefinition->variationConstantDefinitionCount );
            idxDefinition < definitionCount; ++idxDefinition )
        {
            auto& constantDefinition = pVariationDefinition->pVariationConstantDefinitionArray[ idxDefinition ];
            auto& value = pValues[ constantDefinition.indexInVariationConstantValueArray ];
            auto typeName = GetTypeName( constantDefinition );
            ss.write( "const ", 6 );
            ss << typeName;
            ss.write( " ", 1 );
            ss.write( constantDefinition.name.pValue, constantDefinition.name.length );
            if( constantDefinition.arrayLength > 1 )
            {
                ss.write( "[]", 2 );
            }
            ss.write( "=", 1 );
            auto pValueEnd = WriteValue( ss, constantDefinition, value.pValue );
            auto read = NumericCast< int >( nn::util::ConstBytePtr( value.pValue ).Distance( pValueEnd ) );
            if( read > NumericCast< decltype( read ) >( value.valueSizeIn32Bit ) * 4 )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_NotEnoughSize,
                    "Variation Constant Definition(%d):%s, ValueSize:%d\n",
                    idxDefinition, typeName, value.valueSizeIn32Bit );
            }
            ss.write( ";\n", 2 );
        }
        dst = ss.str();
    }
}

}
}
