﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <mutex>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_BoostPreprocessor.h>

// 増えてきたら glsl ヘルパーのようなものを作って移動
namespace {

static const char* const FloatOpaqueTypeNames[] =
{
    "sampler1D", "sampler2D", "sampler3D", "samplerCube", "sampler2DRect", "sampler1DArray",
    "sampler2DArray", "samplerBuffer", "sampler2DMS", "sampler2DMSArray", "samplerCubeArray",
    "sampler1DShadow", "sampler2DShadow", "sampler2DRectShadow", "sampler1DArrayShadow",
    "sampler2DArrayShadow", "samplerCubeShadow", "samplerCubeArrayShadow",
    "image1D", "image2D", "image3D", "imageCube", "image2DRect", "image1DArray",
    "image2DArray", "imageBuffer", "image2DMS", "image2DMSArray", "imageCubeArray"
};

static const char* const SintOpaqueTypeNames[] =
{
    "isampler1D", "isampler2D", "isampler3D", "isamplerCube", "isampler2DRect", "isampler1DArray",
    "isampler2DArray", "isamplerBuffer", "isampler2DMS", "isampler2DMSArray", "isamplerCubeArray",
    "iimage1D", "iimage2D", "iimage3D", "iimageCube", "iimage2DRect", "iimage1DArray",
    "iimage2DArray", "iimageBuffer", "iimage2DMS", "iimage2DMSArray", "iimageCubeArray"
};

static const char* const UintOpaqueTypeNames[] =
{
    "atomic_uint",
    "usampler1D", "usampler2D", "usampler3D", "usamplerCube", "usampler2DRect", "usampler1DArray",
    "usampler2DArray", "usamplerBuffer", "usampler2DMS", "usampler2DMSArray", "usamplerCubeArray",
    "uimage1D", "uimage2D", "uimage3D", "uimageCube", "uimage2DRect", "uimage1DArray",
    "uimage2DArray", "uimageBuffer", "uimage2DMS", "uimage2DMSArray", "uimageCubeArray"
};

bool IsOpaqueType( const char* pTypeName )
{
    if( pTypeName == nullptr )
    {
        return false;
    }

    auto Contains = []( const char* pValue, const char* const * ppStrings, int stringCount )
    {
        for( int idx = 0; idx < stringCount; ++idx )
        {
            if( strcmp( pValue, ppStrings[ idx ] ) == 0 )
            {
                return true;
            }
        }
        return false;
    };

    switch( pTypeName[ 0 ] )
    {
    case 's':
        return Contains( pTypeName, FloatOpaqueTypeNames, NN_ARRAY_SIZE( FloatOpaqueTypeNames ) );
    case 'i':
        switch( pTypeName[ 1 ] )
        {
        case 'm':
            return Contains( pTypeName, FloatOpaqueTypeNames, NN_ARRAY_SIZE( FloatOpaqueTypeNames ) );
        case 's':
        case 'i':
            return Contains( pTypeName, SintOpaqueTypeNames, NN_ARRAY_SIZE( SintOpaqueTypeNames ) );
        default:
            return false;
        }
    case 'a':
    case 'u':
        return Contains( pTypeName, UintOpaqueTypeNames, NN_ARRAY_SIZE( UintOpaqueTypeNames ) );
    default:
        return false;
    }
}

}

namespace nn {
namespace gfxTool {

struct BoostPreprocessor::Impl
{
    enum class ResultPiece
    {
        Default,
        AfterFirstUniformRegister,

        End
    };

    struct UniformRegisterInfo
    {
        Custom< std::string >::Type name;
        Custom< std::string >::Type declaration;
    };

    BoostPreprocessor::Impl()
        : pReadFileCallback()
        , pReadFileCallbackParam()
        , pDefinition()
        , isVariationConstantEmulationEnabled()
        , isUniformRegisterToBlockEnabled()
        , isInvertYEnabled()
        , isRemapZEnabled()
        , pResult( Results )
        , pContext()
    {
    }

    nngfxToolReadFileCallback pReadFileCallback;
    void* pReadFileCallbackParam;
    Custom< std::stringstream >::Type Results[ static_cast< int >( ResultPiece::End ) ];
    context_type::iterator_type iter;
    const nngfxToolShaderCompilerVariationDefinition* pDefinition;
    bool isVariationConstantEmulationEnabled;
    bool isUniformRegisterToBlockEnabled;
    bool isInvertYEnabled;
    bool isRemapZEnabled;
    Custom< std::stringstream >::Type* pResult;
    Custom< std::vector< UniformRegisterInfo > >::Type uniformRegisters;
    std::unique_ptr< context_type > pContext;
};

BoostPreprocessor::BoostPreprocessor()
    : m_pImpl( new Impl() )
{
}

BoostPreprocessor::~BoostPreprocessor()
{
}

void BoostPreprocessor::SetReadFileCallback( nngfxToolReadFileCallback callback, void* pCallbackParam )
{
    m_pImpl->pReadFileCallback = callback;
    m_pImpl->pReadFileCallbackParam = pCallbackParam;
}

void BoostPreprocessor::SetVariationDefinition( const nngfxToolShaderCompilerVariationDefinition* pDefinition )
{
    m_pImpl->pDefinition = pDefinition;
}

void BoostPreprocessor::SetVariationConstantEmulationEnabled( bool value )
{
    m_pImpl->isVariationConstantEmulationEnabled = value;
}

void BoostPreprocessor::SetUniformRegisterToBlockEnabled( bool value )
{
    m_pImpl->isUniformRegisterToBlockEnabled = value;
}

void BoostPreprocessor::SetInvertYEnabled( bool value )
{
    m_pImpl->isInvertYEnabled = value;
}

void BoostPreprocessor::SetRemapZEnabled( bool value )
{
    m_pImpl->isRemapZEnabled = value;
}

const Custom< std::stringstream >::Type& BoostPreprocessor::GetResult() const
{
    return m_pImpl->Results[ static_cast< int >( Impl::ResultPiece::Default ) ];
}

void BoostPreprocessor::Preprocess( const char* pSource, size_t sourceLength )
{
    {
        static std::mutex s_Mutex;
        std::lock_guard< decltype( s_Mutex ) > lock( s_Mutex );
        m_pImpl->pContext.reset( new context_type( pSource, pSource + sourceLength, "<Unknown>", *this ) );
    }

    Custom< std::string >::Type errorLog = "";

    try
    {
        m_pImpl->pContext->set_language( StaticCastAuto(
            boost::wave::support_cpp
            //| boost::wave::support_option_long_long
            | boost::wave::support_option_variadics
            //| boost::wave::support_option_insert_whitespace
            | boost::wave::support_option_preserve_comments
            | boost::wave::support_option_no_character_validation
            //|boost::wave::support_option_convert_trigraphs
            //|boost::wave::support_option_single_line
            | boost::wave::support_option_prefer_pp_numbers
            //| boost::wave::support_option_emit_line_directives
            //| boost::wave::support_option_include_guard_detection
            | boost::wave::support_option_emit_pragma_directives
            ));

        m_pImpl->pContext->set_sysinclude_delimiter();

        bool needPreprocessUniform = m_pImpl->isVariationConstantEmulationEnabled
            || m_pImpl->isUniformRegisterToBlockEnabled;
        bool needGlPositionModify = m_pImpl->isInvertYEnabled || m_pImpl->isRemapZEnabled;
        bool insideMain = false;
        int braceDepth = 0;
        m_pImpl->pResult->clear();
        auto end = m_pImpl->pContext->end();
        for( m_pImpl->iter = m_pImpl->pContext->begin(); m_pImpl->iter != end; )
        {
            if( *m_pImpl->iter == boost::wave::T_IDENTIFIER &&
                needPreprocessUniform && m_pImpl->iter->get_value() == "uniform" )
            {
                PreprocessUniform();
            }
            else if ( needGlPositionModify && !insideMain && *m_pImpl->iter == boost::wave::T_IDENTIFIER && m_pImpl->iter->get_value() == "main" )
            {
                *m_pImpl->pResult << m_pImpl->iter->get_value();
                insideMain = true;
            }
            else if ( needGlPositionModify && insideMain )
            {
                switch ( *m_pImpl->iter )
                {
                    case boost::wave::T_LEFTBRACE:
                    {
                        braceDepth++;
                        *m_pImpl->pResult << m_pImpl->iter->get_value();
                        break;
                    }

                    case boost::wave::T_RIGHTBRACE:
                    case boost::wave::T_RETURN:
                    {
                        bool isReturn = ( *m_pImpl->iter == boost::wave::T_RETURN );

                        if ( !isReturn )
                        {
                            braceDepth--;
                            insideMain = braceDepth != 0;
                        }
                        else
                        {
                            // Preemptively place inside of a block
                            *m_pImpl->pResult << "{ ";
                        }

                        if ( braceDepth == 0 || isReturn )
                        {
                            // Insert y flipping code
                            if ( m_pImpl->isInvertYEnabled )
                            {
                                *m_pImpl->pResult << "gl_Position.y = -gl_Position.y;\n";
                            }

                            if ( m_pImpl->isRemapZEnabled )
                            {
                                *m_pImpl->pResult << "gl_Position.z = (gl_Position.z + gl_Position.w) / 2.0;\n";
                            }
                        }

                        if ( isReturn )
                        {
                            *m_pImpl->pResult << m_pImpl->iter->get_value();
                            // Preemptively place inside of a block
                            *m_pImpl->pResult << "; }";
                        }
                        else
                        {
                            *m_pImpl->pResult << m_pImpl->iter->get_value();
                        }
                        break;
                    }

                    default:
                    {
                        *m_pImpl->pResult << m_pImpl->iter->get_value();
                        break;
                    }
                }
            }
            else
            {
                *m_pImpl->pResult << m_pImpl->iter->get_value();
            }
            Advance();
        }
    }
    catch( nn::gfxTool::Exception const& exception )
    {
        errorLog = exception.what();
    }
    catch( boost::wave::cpp_exception const& exception )
    {
        Custom< std::stringstream >::Type ss;
        ss << exception.file_name() << "(" << exception.line_no() << "): " << exception.description();
        errorLog = ss.str();
    }
    catch( std::exception const& exception )
    {
        Custom< std::stringstream >::Type ss;
        ss << m_pImpl->iter->get_position().get_file() << "(" <<
            m_pImpl->iter->get_position().get_line() << "): "  << "exception: " << exception.what();
        errorLog = ss.str();
    }
    catch( ... )
    {
        Custom< std::stringstream >::Type ss;
        ss << m_pImpl->iter->get_position().get_file() << "(" <<
            m_pImpl->iter->get_position().get_line() << "): " << "unexpected exception";
        errorLog = ss.str();
    }

    if( !errorLog.empty() )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_FailedToPreprocess, "%s", errorLog.c_str() );
    }

    m_pImpl->pContext.reset( nullptr );
}

void BoostPreprocessor::ResolveUniformRegisterBlock( int preprocessorCount,
    BoostPreprocessor* pPreprocessors, const char* pUniformRegisterBlockName )
{
    if( pUniformRegisterBlockName == nullptr )
    {
        return;
    }

    Custom< std::stringstream >::Type uniformRegisterBlock;
    for( int idx = 0; idx < preprocessorCount; ++idx )
    {
        auto& preprocessor = pPreprocessors[ idx ];
        for( auto&& uniform : preprocessor.m_pImpl->uniformRegisters )
        {
            bool found = false;
            for( int idxFind = 0; !found && idxFind < idx; ++idxFind )
            {
                auto& preprocessorFind = pPreprocessors[ idxFind ];
                for( auto&& uniformFind : preprocessorFind.m_pImpl->uniformRegisters )
                {
                    if( uniform.name == uniformFind.name )
                    {
                        found = true;
                        break;
                    }
                }
            }
            if( !found )
            {
                uniformRegisterBlock << uniform.declaration << "\n";
            }
        }
    }
    auto uniformRegisterBlockStr = uniformRegisterBlock.str();

    if( !uniformRegisterBlockStr.empty() )
    {
        for( int idx = 0; idx < preprocessorCount; ++idx )
        {
            auto& preprocessor = pPreprocessors[ idx ];
            auto afterFirstUniformRegister = preprocessor.m_pImpl->Results[
                static_cast<int>( Impl::ResultPiece::AfterFirstUniformRegister ) ].str();
            if( preprocessor.m_pImpl->isUniformRegisterToBlockEnabled &&
                afterFirstUniformRegister.length() > 0 )
            {
                preprocessor.m_pImpl->Results[ static_cast<int>( Impl::ResultPiece::Default ) ]
                    << "layout( std140 ) uniform " << pUniformRegisterBlockName << "\n{\n"
                    << uniformRegisterBlockStr << "};\n" << afterFirstUniformRegister;
            }
        }
    }
}

void BoostPreprocessor::PreprocessUniform()
{
    const char* pVariationConstantBufferName = m_pImpl->pDefinition ?
        m_pImpl->pDefinition->variationConstantBufferName.pValue : nullptr;

    auto end = m_pImpl->pContext->end();
    bool isTargetUniformBlock = false;
    bool isOpaqueType = false;
    Custom< std::stringstream >::Type ssUniform;
    Custom< std::string >::Type identifier;
    while( m_pImpl->iter != end )
    {
        Advance();
        ssUniform << m_pImpl->iter->get_value();

        if( *m_pImpl->iter == boost::wave::T_IDENTIFIER )
        {
            if( pVariationConstantBufferName && strncmp( m_pImpl->iter->get_value().c_str(),
                pVariationConstantBufferName, m_pImpl->iter->get_value().length() ) == 0 )
            {
                isTargetUniformBlock = true;
            }
            if( m_pImpl->isUniformRegisterToBlockEnabled )
            {
                if( IsOpaqueType( m_pImpl->iter->get_value().c_str() ) )
                {
                    isOpaqueType = true;
                }
                identifier = m_pImpl->iter->get_value().c_str();
            }
        }
        else if( *m_pImpl->iter == boost::wave::T_SEMICOLON )
        {
            // ユニフォームレジスタ
            if( m_pImpl->isUniformRegisterToBlockEnabled && !isOpaqueType )
            {
                m_pImpl->uniformRegisters.emplace_back();
                auto& uniformRegister = m_pImpl->uniformRegisters.back();
                uniformRegister.name = identifier;
                uniformRegister.declaration = ssUniform.str();
                m_pImpl->pResult = m_pImpl->Results + static_cast<int>(
                    Impl::ResultPiece::AfterFirstUniformRegister );
            }
            else
            {
                *m_pImpl->pResult << "uniform " << ssUniform.str();
            }
            return;
        }
        else if( *m_pImpl->iter == boost::wave::T_LEFTBRACE )
        {
            // ユニフォームブロック
            break;
        }
    }
    *m_pImpl->pResult << "uniform" << ssUniform.str();
    if( !m_pImpl->isVariationConstantEmulationEnabled || !isTargetUniformBlock )
    {
        return;
    }

    int braceDepth = 1;
    while( m_pImpl->iter != end )
    {
        Advance();
        *m_pImpl->pResult << m_pImpl->iter->get_value();
        if( *m_pImpl->iter == boost::wave::T_LEFTBRACE )
        {
            ++braceDepth;
        }
        else if( *m_pImpl->iter == boost::wave::T_RIGHTBRACE )
        {
            if( --braceDepth <= 0 )
            {
                break;
            }
        }
        else if( *m_pImpl->iter == boost::wave::T_IDENTIFIER )
        {
            auto last = m_pImpl->pDefinition->pVariationConstantDefinitionArray
                + m_pImpl->pDefinition->variationConstantDefinitionCount;
            if( std::find_if( m_pImpl->pDefinition->pVariationConstantDefinitionArray, last,
                [ & ]( decltype( *m_pImpl->pDefinition->pVariationConstantDefinitionArray )& variationConstantDefinition )
            {
                return strncmp( m_pImpl->iter->get_value().c_str(), variationConstantDefinition.name.pValue,
                    variationConstantDefinition.name.length ) == 0; } ) != last )
            {
                *m_pImpl->pResult << "_nngfx_unused";
            }
        }
    }
}

void BoostPreprocessor::Advance()
{
    try
    {
        ++m_pImpl->iter;
    }
    catch( boost::wave::preprocess_exception& exception )
    {
        if( !exception.is_recoverable() )
        {
            throw exception;
        }
    }
    catch( boost::wave::cpplexer::lexing_exception& exception )
    {
        if( !exception.is_recoverable() )
        {
            throw exception;
        }
    }
}

bool BoostPreprocessor::ReadFile( void** ppOutFileData, size_t* pOutFileDataSize, const char* pFileName ) const
{
    if( m_pImpl->pReadFileCallback )
    {
        return m_pImpl->pReadFileCallback( ppOutFileData,
            pOutFileDataSize, pFileName, m_pImpl->pReadFileCallbackParam );
    }
    return false;
}

}
}
