﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_System.h>
#include "PVRTextureDefines.h"
#include "PVRTDecompress.h"


//------------------------------------------------------------------------------
//  PVRテクスチャを初期化します
//------------------------------------------------------------------------------
static bool PvrTextureResourceInitializer( nw::eft2::TextureResource* texture, u64 id, nw::eft2::ResTexture* resTexture, void* binary, u32 binarySize, nw::eft2::ResourceHeap* resHeap )
{
    EFT_UNUSED_VARIABLE( texture );
    EFT_UNUSED_VARIABLE( resTexture );
    EFT_UNUSED_VARIABLE( binary );
    EFT_UNUSED_VARIABLE( binarySize );

    PVRTuint8* pTextureData=NULL;

    pTextureData = (PVRTuint8*)binary;

    // フォーマットの変換と、データサイズ計算用の除算値を決定
    int bpp_inv = 2;
    PVRTuint64 PixelFormat = ePVRTPF_PVRTCI_4bpp_RGBA;
    switch(resTexture->format)
    {
        case nw::eft2::EFT_TEXTURE_FORMAT_UNORM_PVRTC1_2BPP:               //!< RGB_PVRTC_2BPPV1
        case nw::eft2::EFT_TEXTURE_FORMAT_SRGB_PVRTC1_2BPP:                //!< SRGB_PVRTC_2BPPV
            PixelFormat = ePVRTPF_PVRTCI_2bpp_RGB;
            bpp_inv = 4;
            break;
        case nw::eft2::EFT_TEXTURE_FORMAT_UNORM_PVRTC1_4BPP:               //!< RGB_PVRTC_4BPPV1
        case nw::eft2::EFT_TEXTURE_FORMAT_SRGB_PVRTC1_4BPP:                //!< SRGB_PVRTC_4BPPV1
            PixelFormat = ePVRTPF_PVRTCI_4bpp_RGB;
            bpp_inv = 2;
            break;
        case nw::eft2::EFT_TEXTURE_FORMAT_UNORM_PVRTC1_ALPHA_2BPP:         //!< RGBA_PVRTC_2BPPV1
        case nw::eft2::EFT_TEXTURE_FORMAT_SRGB_PVRTC1_ALPHA_2BPP:          //!< SRGBA_PVRTC_2BPPV1
            PixelFormat = ePVRTPF_PVRTCI_2bpp_RGBA;
            bpp_inv = 4;
            break;
        case nw::eft2::EFT_TEXTURE_FORMAT_UNORM_PVRTC1_ALPHA_4BPP:         //!< RGBA_PVRTC_4BPPV1
        case nw::eft2::EFT_TEXTURE_FORMAT_SRGB_PVRTC1_ALPHA_4BPP:          //!< SRGBA_PVRTC_4BPPV1
            PixelFormat = ePVRTPF_PVRTCI_4bpp_RGBA;
            bpp_inv = 2;
            break;
        case nw::eft2::EFT_TEXTURE_FORMAT_UNORM_PVRTC2_ALPHA_2BPP:         //!< RGBA_PVRTC_2BPPV2
        case nw::eft2::EFT_TEXTURE_FORMAT_SRGB_PVRTC2_ALPHA_2BPP:          //!< SRGBA_PVRTC_2BPPV2
            PixelFormat = ePVRTPF_PVRTCII_2bpp;
            bpp_inv = 4;
            break;
        case nw::eft2::EFT_TEXTURE_FORMAT_UNORM_PVRTC2_ALPHA_4BPP:         //!< RGBA_PVRTC_4BPPV2
        case nw::eft2::EFT_TEXTURE_FORMAT_SRGB_PVRTC2_ALPHA_4BPP:           //!< SRGBA_PVRTC_4BPPV2
            PixelFormat = ePVRTPF_PVRTCII_4bpp;
            bpp_inv = 2;
            break;
    }

    // カラースペース値の変換
    EPVRTColourSpace ColourSpace;
    switch(resTexture->format)
    {
        case nw::eft2::EFT_TEXTURE_FORMAT_SRGB_PVRTC1_2BPP:                //!< SRGB_PVRTC_2BPPV
        case nw::eft2::EFT_TEXTURE_FORMAT_SRGB_PVRTC1_4BPP:                //!< SRGB_PVRTC_4BPPV1
        case nw::eft2::EFT_TEXTURE_FORMAT_SRGB_PVRTC1_ALPHA_2BPP:          //!< SRGBA_PVRTC_2BPPV1
        case nw::eft2::EFT_TEXTURE_FORMAT_SRGB_PVRTC1_ALPHA_4BPP:          //!< SRGBA_PVRTC_4BPPV1
        case nw::eft2::EFT_TEXTURE_FORMAT_SRGB_PVRTC2_ALPHA_2BPP:          //!< SRGBA_PVRTC_2BPPV2
        case nw::eft2::EFT_TEXTURE_FORMAT_SRGB_PVRTC2_ALPHA_4BPP:           //!< SRGBA_PVRTC_4BPPV2
            ColourSpace = EPVRTColourSpace::ePVRTCSpacesRGB;
            break;
        default:
            ColourSpace = EPVRTColourSpace::ePVRTCSpacelRGB;
    }

    // 展開後の配列を、構造体を使わず自前でサイズを計算して確保
    PVRTuint32 levelSize = resTexture->width * resTexture->height * resTexture->depth * 4;
    PVRTuint32 totalSize = 0;
    for(u32 i = 0; i < resTexture->mipLevel; ++i)
    {
        totalSize += levelSize;
        levelSize /= 4;
    }
    void* pDecompressedData = resHeap->Alloc(totalSize);

    //Get the dimensions for the current MIP level.
    PVRTuint32 uiMIPWidth = resTexture->width;
    PVRTuint32 uiMIPHeight = resTexture->height;

    //Setup temporary variables.
    PVRTuint8* pTempDecompData = (PVRTuint8*)pDecompressedData;
    PVRTuint8* pTempCompData = (PVRTuint8*)pTextureData;

    //Check if it's 2bpp.
    bool bIs2bppPVRTC = ColourSpace!=ePVRTCSpacesRGB && (PixelFormat==ePVRTPF_PVRTCI_2bpp_RGB || PixelFormat==ePVRTPF_PVRTCI_2bpp_RGBA);

    //Decompress all the MIP levels.
    for (PVRTuint32 uiMIPMap=0;uiMIPMap<(PVRTuint32)resTexture->mipLevel;++uiMIPMap)
    {
        // Get the face offset. Varies per MIP level.
        PVRTuint32 decompressedFaceOffset = uiMIPWidth * uiMIPHeight * 4;

        // 2bpp/4bppは全チャンネルを含めたbit per pixelだったので、ピクセル数を4か2で割るだけでオフセットが求まる
        PVRTuint32 compressedFaceOffset = uiMIPWidth * uiMIPHeight / bpp_inv;

        // キューブマップは別関数で処理するからここでは常に1枚ずつで良いはず
        for (PVRTuint32 uiFace=0;uiFace<1;++uiFace)
        {
            //Decompress the texture data.
            PVRTDecompressPVRTC(pTempCompData,bIs2bppPVRTC?1:0,uiMIPWidth,uiMIPHeight,pTempDecompData);

            //Move forward through the pointers.
            pTempDecompData+=decompressedFaceOffset;
            pTempCompData+=compressedFaceOffset;
        }

        //Work out the current MIP dimensions.
        uiMIPWidth=PVRT_MAX(1,uiMIPWidth>>1);
        uiMIPHeight=PVRT_MAX(1,uiMIPHeight>>1);
    }

    GLuint texFormat = GL_RGBA;
    GLuint texTarget = GL_TEXTURE_2D;

    nw::eft2::TextureInterface texInterface;

    if (resTexture->depth > 1 )  texTarget = GL_TEXTURE_2D_ARRAY;

    glGenTextures( 1, &(texInterface) );
    glBindTexture( texTarget, texInterface );
    glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );

    texture->Initialize(texInterface, id, resTexture);

    uiMIPWidth = resTexture->width;
    uiMIPHeight = resTexture->height;
    pTempDecompData = (PVRTuint8*)pDecompressedData;

    // ミップレベルごとにテクスチャをロード
    for (PVRTuint32 uiMIPMap=0;uiMIPMap<(PVRTuint32)resTexture->mipLevel;++uiMIPMap)
    {
        // Get the face offset. Varies per MIP level.
        PVRTuint32 decompressedFaceOffset = uiMIPWidth * uiMIPHeight * 4;
        glTexImage2D( texTarget, uiMIPMap, texFormat, uiMIPWidth, uiMIPHeight, 0, texFormat, GL_UNSIGNED_BYTE, pTempDecompData );
        //Move forward through the pointers.
        pTempDecompData+=decompressedFaceOffset;

        uiMIPWidth=PVRT_MAX(1,uiMIPWidth>>1);
        uiMIPHeight=PVRT_MAX(1,uiMIPHeight>>1);
    }

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    resHeap->Free(pDecompressedData);
    return true;
}

//------------------------------------------------------------------------------
//  PVRテクスチャを破棄します
//------------------------------------------------------------------------------
static bool PvrTextureResourceFinalizier( nw::eft2::TextureResource* texture )
{
    EFT_UNUSED_VARIABLE( texture );
    return true;
}

//------------------------------------------------------------------------------
//  PvrTextureTranslation をセットアップします。
//------------------------------------------------------------------------------
void SetupPvrTextureTrasnlationGL( nw::eft2::System* system )
{
    system->SetupTextureTranslation( PvrTextureResourceInitializer, PvrTextureResourceFinalizier );
}



