﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/demo.h>
#include <nw/dev.h>
#include <nw/gfnd.h>
#include <nw/math.h>
#include <nw/ut.h>

#include <nw/eft/eft2_System.h>
#include <nw/eft/eftut2_Heap.h>

#include <Camera.h>
#include <System.h>
#include <DrawParam.h>
#include <Texture.h>


#if defined(NW_PLATFORM_CAFE)
#include <cafe/hio.h>
#if defined(NW_USE_CPU_PROFILER)
#include <cafe/profiler.h>
#endif
#endif

//------------------------------------------------------------------------------
//      Viewer クラス
//------------------------------------------------------------------------------
class Viewer
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタ
    //---------------------------------------------------------------------------
    Viewer()
    {
        mEftSystem                  = NULL;
        mHandle.Invalidate();
        mEmitterSetIndex            = 0;
        mEffectBinaryResource       = NULL;
        mVwrCamera                  = NULL;
        mNwAllocator                = NULL;
        mEnableDisplayList          = true;
        mDisplayWindow              = false;
        mFrameRate                  = 60.f;
        mBackgroundColor.Set( 0.0f, 0.0f, 0.0f, 0.0f );
        mBackgroundTextureRes       = NULL;
        mPlaySpeed                  = 1.0f;
        mGridType                   = 0;
        mIsLinearMode               = true;
        mDoDebugDraw                = true;

        InitDemoValue();
    };

    enum SWAP_INTERVAL
    {
        SWAP_INTERVAL_60FPS = 1,
        SWAP_INTERVAL_30FPS,
        SWAP_INTERVAL_20FPS,
        SWAP_INTERVAL_15FPS
    };

    //---------------------------------------------------------------------------
    //! @brief        デモごとの変数設定
    //---------------------------------------------------------------------------
    void InitDemoValue();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタ
    //---------------------------------------------------------------------------
    ~Viewer() {};

    //char mArgs[256];

    //---------------------------------------------------------------------------
    //! @brief        エフェクトデモの初期化処理
    //---------------------------------------------------------------------------
    void Initialize( nw::ut::MemoryAllocator* allocator, s32 screenWidth, s32 screenHeight );

    //---------------------------------------------------------------------------
    //! @brief        エフェクトデモの終了処理
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief        エフェクトデモの計算処理
    //---------------------------------------------------------------------------
    void ProcCalc( nw::dev::Pad* pad, nw::eftdemo::DrawParam& drawParam  );

    //---------------------------------------------------------------------------
    //! @brief        エフェクトの描画処理
    //---------------------------------------------------------------------------
    void ProcDraw( nw::eftdemo::DrawParam&          drawParam,
                   nw::eftdemo::FrameBuffer*        currentFrameBuffer,
                   nw::eftdemo::FrameBufferTexture* frameBufferCopy,
                   nw::eftdemo::DepthBufferTexture* depthBufferCopy );

    //---------------------------------------------------------------------------
    //! @brief        モデルの描画処理
    //---------------------------------------------------------------------------
//  void ProcModelDraw( nw::eftdemo::DrawParam& drawParam, nw::eftdemo::SimpleShader::ViewId id = nw::eftdemo::SimpleShader::VIEW_ID_CAMERA );

    //---------------------------------------------------------------------------
    //! @brief        デバッグ文字描画
    //---------------------------------------------------------------------------
    void DebugDraw( nw::dev::DevTextWriter* writer );

    //---------------------------------------------------------------------------
    //! @brief        Heapを取得
    //---------------------------------------------------------------------------
    nw::eftut2::NwHeap* GetHeap() { return &mEftHeap; }

    //---------------------------------------------------------------------------
    //! @brief        カメラをセットする
    //---------------------------------------------------------------------------
    void SetCamera( nw::eftdemo::Camera* camera ) { mVwrCamera = camera; }

    //---------------------------------------------------------------------------
    //! @brief        プロジェクションマトリクスをセットする
    //---------------------------------------------------------------------------
    void SetProjectionMatrix( nw::math::MTX44* projMatrix ) { mProjectionMatrix = projMatrix; }

#ifdef NW_PLATFORM_CAFE
    //---------------------------------------------------------------------------
    //! @brief        デプスバッファテクスチャをセットする
    //---------------------------------------------------------------------------
    void SetDepthBufferTexture( const GX2Texture* depthBufferTexture )
    {
        mEftSystem->SetDepthTexture( const_cast<GX2Texture*>( depthBufferTexture ) );
    }
    //---------------------------------------------------------------------------
    //! @brief        フレームバッファテクスチャをセットする
    //---------------------------------------------------------------------------
    void SetFrameBufferTexture( const GX2Texture* frameBufferTexture )
    {
        mEftSystem->SetFrameBufferTexture( const_cast<GX2Texture*>( frameBufferTexture ) );
    }
#endif
#ifdef NW_PLATFORM_WIN32
    //---------------------------------------------------------------------------
    //! @brief        デプスバッファテクスチャをセットする
    //---------------------------------------------------------------------------
    void SetDepthBufferTexture( GLuint depthBufferTexture )
    {
        mEftSystem->SetDepthTexture( depthBufferTexture );
    }
    //---------------------------------------------------------------------------
    //! @brief        フレームバッファテクスチャをセットする
    //---------------------------------------------------------------------------
    void SetFrameBufferTexture( GLuint frameBufferTexture )
    {
        mEftSystem->SetFrameBufferTexture( frameBufferTexture );
    }
#endif

    //---------------------------------------------------------------------------
    //! @brief        フレームレートを設定します。
    //---------------------------------------------------------------------------
    void SetFrameRate( nw::eftdemo::INIT_SETTINGS_FRAME_RATE frameRate )
    {
        switch( frameRate )
        {
        case nw::eftdemo::INIT_SETTINGS_FRAME_RATE_60:
            mFrameRate = 60.f;
            break;
        case nw::eftdemo::INIT_SETTINGS_FRAME_RATE_30:
            mFrameRate = 30.f;
            break;
        case nw::eftdemo::INIT_SETTINGS_FRAME_RATE_20:
            mFrameRate = 20.f;
            break;
        case nw::eftdemo::INIT_SETTINGS_FRAME_RATE_15:
            mFrameRate = 15.f;
            break;
        }
    }

    void SetFrameRate( f32 frameRate ) { mFrameRate = frameRate; }

    //---------------------------------------------------------------------------
    //! @brief        フレームレートを取得します。
    //---------------------------------------------------------------------------
    void GetFrameRate( f32* frameRate ) const { *frameRate = mFrameRate; }

    f32 GetFovy() const { return mFovy; }

    //---------------------------------------------------------------------------
    //! @brief        描画すべきエミッタが存在するか。
    //---------------------------------------------------------------------------
    bool IsHasRenderingEmitter( u8 groupID, u32 drawPath ) { return mEftSystem->IsHasRenderingEmitter( groupID, drawPath ); }

    //---------------------------------------------------------------------------
    //! @brief        シェーダタイプを切り替える。
    //---------------------------------------------------------------------------
    void SetShaderType( nw::eft2::ShaderType type ) { return mEftSystem->SetShaderType( type ); }

    //---------------------------------------------------------------------------
    //! @brief        背景カラーを設定します。
    //---------------------------------------------------------------------------
    void SetBackgroundColor( nw::math::VEC4 color ) { mBackgroundColor = color; }

    //---------------------------------------------------------------------------
    //! @brief        カメラを設定します。
    //---------------------------------------------------------------------------
    void SetCameraSettings(
        f32 posX, f32 posY, f32 posZ,
        f32 lookAtX, f32 lookAtY, f32 lookAtZ,
        f32 rotation,
        int projType,
        f32 aspect,
        f32 fov,
        f32 height,
        f32 znear, f32 zfar );

    //---------------------------------------------------------------------------
    //! @brief        背景カラーを設定します。
    //---------------------------------------------------------------------------
    const nw::math::VEC4 GetBackgroundColor() const { return mBackgroundColor; }

    //---------------------------------------------------------------------------
    //! @brief        背景画像を設定します。
    //---------------------------------------------------------------------------
    void SetBackgroundImage( u32 width, u32 height, const u8* pixels )
    {
        mBackgroundTexture.SetImage( mNwAllocator, width, height, pixels );
    }

    //---------------------------------------------------------------------------
    //! @brief        背景画像を設定します。
    //---------------------------------------------------------------------------
    void SetBackgroundImageRes( const void* resource )
    {
        mBackgroundTextureRes = resource;
    }

    //---------------------------------------------------------------------------
    //! @brief        背景画像を取得します。
    //---------------------------------------------------------------------------
    nw::eftdemo::Texture& GetBackgroundTexture() { return mBackgroundTexture; }

    //---------------------------------------------------------------------------
    //! @brief        背景画像リソースを取得します。
    //---------------------------------------------------------------------------
    const void* GetBackgroundTextureRes() { return mBackgroundTextureRes; }

    //---------------------------------------------------------------------------
    //! @brief        エフェクトシステムを取得します。
    //---------------------------------------------------------------------------
    nw::eft2::System* GetEffectSystem() { return mEftSystem; }

    //---------------------------------------------------------------------------
    //! @brief        ディスプレイリスト設定の有効/無効
    //---------------------------------------------------------------------------
    bool EnabledDisplayList() { return mEnableDisplayList; }

    //---------------------------------------------------------------------------
    //! @brief        ウィンドウの有効/無効
    //---------------------------------------------------------------------------
    bool DisplayWindow() { return mDisplayWindow; }

    //---------------------------------------------------------------------------
    //! @brief        再生速度を設定します。
    //!
    //! @param[in]    playSpeed 再生速度。
    //---------------------------------------------------------------------------
    void SetPlaySpeed( float playSpeed) { mPlaySpeed = playSpeed; }

    //---------------------------------------------------------------------------
    //! @brief        再生速度を取得します。
    //!
    //! @return       再生速度を返します。
    //---------------------------------------------------------------------------
    float GetPlaySpeed() { return mPlaySpeed; }

    //---------------------------------------------------------------------------
    //! @brief        グリッドタイプを設定します。
    //!
    //! @param[in]    gridType グリッドタイプ。
    //---------------------------------------------------------------------------
    void SetGridType( int gridType ) { mGridType = gridType; }

    //---------------------------------------------------------------------------
    //! @brief        グリッドタイプを取得します。
    //!
    //! @return       グリッドタイプを返します。
    //---------------------------------------------------------------------------
    int GetGridType() { return mGridType; }

    //---------------------------------------------------------------------------
    //! @brief        リニアモードの有効/無効を設定します。
    //!
    //! @param[in]    isLinearMode リニアモードの有効/無効。
    //---------------------------------------------------------------------------
    void SetLinearMode( bool isLinearMode ) { mIsLinearMode = isLinearMode; }

    //---------------------------------------------------------------------------
    //! @brief        リニアモードの有効/無効を取得します。
    //!
    //! @return       リニアモードの有効/無効を返します。
    //---------------------------------------------------------------------------
    bool GetLinearMode() { return mIsLinearMode; }

    //---------------------------------------------------------------------------
    //! @brief        デバッグ描画の有効/無効を取得します。
    //!
    //! @return       デバッグ描画の有効/無効を返します。
    //---------------------------------------------------------------------------
    bool DoDebugDraw() { return mDoDebugDraw; }

    //---------------------------------------------------------------------------
    //! @brief        垂直同期する数を取得します。
    //!
    //! @return       垂直同期する数。
    //---------------------------------------------------------------------------
    u32 GetSwapInterval();

private:
    //---------------------------------------------------------------------------
    //! @brief        ファイルをロードします
    //---------------------------------------------------------------------------
    void* LoadBinary( const char* fileName, s32 alignment )
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path      = fileName;
        loadArg.allocator = mNwAllocator;
        loadArg.alignment = alignment;
        u8* pPtcl = fileSystem->Load( loadArg );

        return pPtcl;
    }

    //---------------------------------------------------------------------------
    //! @brief        ロードされたファイルを破棄します。
    //---------------------------------------------------------------------------
    void UnLoadBinary( const char* fileName, void* binary, s32 alignment )
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path      = fileName;
        loadArg.allocator = mNwAllocator;
        loadArg.alignment = alignment;

        fileSystem->Unload( loadArg, static_cast<u8*>( binary ) );
    }


private:
    nw::eftut2::NwHeap                  mEftHeap;                       //!< エフェクトヒープクラス
    nw::eftut2::NwHeap                  mEftDynamicHeap;                //!< エフェクトダイナミックヒープクラス
    nw::eft2::System*                   mEftSystem;                     //!< エフェクトシステム
    nw::eft2::Handle                    mHandle;                        //!< エフェクトハンドル
    s32                                 mEmitterSetIndex;               //!< エミッタセットID
    void*                               mEffectBinaryResource;          //!< エフェクトバイナリリソース
    f32                                 mFrameRate;                     //!< フレームレート
    nw::math::VEC4                      mBackgroundColor;               //!< 背景カラー
    nw::eftdemo::BackgroundTexture      mBackgroundTexture;             //!< 背景テクスチャ
    const void*                         mBackgroundTextureRes;          //!< 背景テクスチャリソース
    nw::eftdemo::Camera*                mVwrCamera;                     //!< カメラ
    nw::math::MTX44*                    mProjectionMatrix;              //!< プロジェクション
    f32                                 mFovy;                          //!< fovy
    nw::ut::MemoryAllocator*            mNwAllocator;                   //!< アロケータ―
    bool                                mEnableDisplayList;             //!< ディスプレイリスト設定の有効/無効
    bool                                mDisplayWindow;                 //!< ログウインドウの表示/非表示
    float                               mPlaySpeed;                     //!< 再生速度
    int                                 mGridType;                      //!< グリッドタイプ
    bool                                mIsLinearMode;                  //!< リニアモードの有効/無効
    bool                                mDoDebugDraw;                   //!< デバッグ描画の有効/無効
};
