﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include <nn/fontll/fontll_ScalableFontEngine.h>
#include "../Include/DebugFontTextureCache.h"

namespace {

bool IsPos2(uint32_t n)
{
    return (((n - 1) & n) == 0);
}

}

void DebugFontTextureCache::InitializeArg::SetDefault()
{
    textureCacheWidth = 1024;
    textureCacheHeight = 1024;
    allocateFunction = NULL;
    pUserDataForAllocateFunction = NULL;
    pFontData = NULL;
    workMemorySize = WorkMemorySizeDefault;
    noPlotWorkMemorySize = WorkMemorySizeNoPlotDefault;
    glyphNodeCountMax = GlyphNodeCountMaxDefault;
    fontIndex = 0;
    fontSize = 40;
}

DebugFontTextureCache::DebugFontTextureCache()
: mFontEngine(NULL)
, mFontEngineNoPlot(NULL)
, mTextureBitMap(NULL)
, mTextureCacheWidth(0)
, mTextureCacheHeight(0)
, mLineCurrentPos(0)
, m_FontSize(40)
, m_FontHeight(50)
, m_FontAscent(40)
, m_FontBaselinePos(40)
, m_FontLineFeed(50)
{
}

DebugFontTextureCache::~DebugFontTextureCache()
{
}

void DebugFontTextureCache::Initialize(const InitializeArg& arg)
{
    // 引数のチェック
    if ( ! IsPos2(arg.textureCacheWidth))
    {
        NN_SDK_ASSERT(false, "arg.width[%u] is not power of 2.", arg.textureCacheWidth);
        return;
    }
    if (arg.allocateFunction == NULL)
    {
        NN_SDK_ASSERT(false, "arg.allocateFunction must be set.");
        return;
    }
    if (arg.pFontData == NULL)
    {
        NN_SDK_ASSERT(false, "arg.pFontData must be set.");
        return;
    }

    mTextureCacheWidth = arg.textureCacheWidth;
    mTextureCacheHeight = arg.textureCacheHeight;
    mLineCurrentPos = 0;

    // ステート構造体の確保
    {
        mFontEngine = static_cast<nn::fontll::ScalableFontEngine*>(arg.allocateFunction(sizeof(nn::fontll::ScalableFontEngine), 4, arg.pUserDataForAllocateFunction));
        new(mFontEngine)nn::fontll::ScalableFontEngine();

        mFontEngineNoPlot = static_cast<nn::fontll::ScalableFontEngine*>(arg.allocateFunction(sizeof(nn::fontll::ScalableFontEngine), 4, arg.pUserDataForAllocateFunction));
        new(mFontEngineNoPlot)nn::fontll::ScalableFontEngine();
    }

    // ステート構造体の初期化
    mFontEngine->Initialize(arg.allocateFunction(arg.workMemorySize, 4, arg.pUserDataForAllocateFunction), static_cast<uint32_t>(arg.workMemorySize));
    AssertFsError("Initialize");
    // このステート構造体ではプロットは行わないので、ワークエリアは固定の小さい領域にする。フォントを複数扱う場合は消費されるサイズはその数の倍数必要。

    mFontEngineNoPlot->Initialize(arg.allocateFunction(arg.noPlotWorkMemorySize, 4, arg.pUserDataForAllocateFunction), static_cast<uint32_t>(arg.noPlotWorkMemorySize));
    AssertFsErrorNoPlot("Initialize");

    // フォントの登録
    {
        const void* fontData = arg.pFontData;
        {
            mFontEngine->LoadFont(mFontNameBuf, fontData, arg.fontIndex, FontNameLengthMax);
            AssertFsError("LoadFont");

            mFontEngine->SetFont(mFontNameBuf);
            AssertFsError("SetFont");

            mFontEngine->SetBoldWeight(0);
            AssertFsError("SetBoldWeight");

            mFontEngine->SetFlags(nn::fontll::ScalableFontEngine::Flags_NoEffect);
            AssertFsError("SetFlags");
        }
        {
            mFontEngineNoPlot->LoadFont(mFontNameBuf, fontData, arg.fontIndex, FontNameLengthMax);
            AssertFsErrorNoPlot("LoadFont");

            mFontEngineNoPlot->SetFont(mFontNameBuf);
            AssertFsErrorNoPlot("SetFont");

            mFontEngineNoPlot->SetBoldWeight(0);
            AssertFsErrorNoPlot("SetBoldWeight");

            mFontEngineNoPlot->SetFlags(nn::fontll::ScalableFontEngine::Flags_NoEffect);
            AssertFsErrorNoPlot("SetFlags");
        }

        // フォントのアセントと高さを求めておく;
        nn::fontll::Metrics metrics;
        mFontEngine->GetFontMetrics(&metrics);
        AssertFsError("GetFontMetrics");

        mFontMetrics.boundingBoxAscentRatio = static_cast<float>(metrics.fontBoundingBox.yMax) / metrics.metricsResolution;
        mFontMetrics.boundingBoxHeightRatio = static_cast<float>(metrics.fontBoundingBox.yMax - metrics.fontBoundingBox.yMin) / metrics.metricsResolution;
        mFontMetrics.ascent_ratio = static_cast<float>(metrics.os2WinAscent) / metrics.metricsResolution;
        mFontMetrics.height_ratio = static_cast<float>(metrics.os2WinAscent + metrics.os2WinDescent) / metrics.metricsResolution;
    }

    // 高さが最大のものにベースラインを合わせる
    {
        float maxBoundingBoxAscentRatio = 0.0f;
        float maxBoundingBoxHeightRatio = 0.0f;
        float maxAscentRatio = 0.0f;
        float maxHeightRatio = 0.0f;
        if (maxBoundingBoxHeightRatio < mFontMetrics.boundingBoxHeightRatio)
        {
            maxBoundingBoxAscentRatio = mFontMetrics.boundingBoxAscentRatio;
            maxBoundingBoxHeightRatio = mFontMetrics.boundingBoxHeightRatio;
            maxAscentRatio = mFontMetrics.ascent_ratio;
            maxHeightRatio = mFontMetrics.height_ratio;
        }
        mFontMetrics.boundingBoxAscentRatio = maxBoundingBoxAscentRatio;
        mFontMetrics.boundingBoxHeightRatio = maxBoundingBoxHeightRatio;
        mFontMetrics.ascent_ratio = maxAscentRatio;
        mFontMetrics.height_ratio = maxHeightRatio;
    }

    const size_t TextureSize = mTextureCacheWidth * mTextureCacheHeight * 1;

    // テクスチャ書き換えのためのバッファを初期化
    mTextureBitMap = static_cast<uint8_t*>(arg.allocateFunction(TextureSize, 4, arg.pUserDataForAllocateFunction));
    std::memset(mTextureBitMap, 0x00, TextureSize);

    mGlyphTreeMap.Initialize(arg.allocateFunction, arg.pUserDataForAllocateFunction, arg.glyphNodeCountMax);

    m_FontSize = arg.fontSize;
    {
        const DebugFontTextureCache::FontMetrics& metrics = GetFontMetrics();
        // ResFontの表示に近くなるように、切り捨てではなく四捨五入する
        m_FontAscent = static_cast<int>(metrics.ascent_ratio * m_FontSize + 0.5f);
        m_FontHeight = static_cast<int>(metrics.height_ratio * m_FontSize + 0.5f);
        m_FontBaselinePos = static_cast<int>(metrics.boundingBoxAscentRatio * m_FontSize + 0.5f);
    }
    m_FontLineFeed = m_FontHeight;
}// NOLINT(impl/function_size)

void DebugFontTextureCache::Finalize(nn::FreeFunctionWithUserData freeFunction, void* pUserDataForFreeFunction)
{
    void* work = reinterpret_cast<void*>(mFontEngine->GetPointerToWorkBuffer());
    void* noPlotWorks;
    noPlotWorks = reinterpret_cast<void*>(mFontEngineNoPlot->GetPointerToWorkBuffer());

    mFontEngine->Finalize();
    AssertFsError("Finalize");

    mFontEngineNoPlot->Finalize();
    AssertFsErrorNoPlot("Finalize");

    freeFunction( mFontEngine, pUserDataForFreeFunction );
    freeFunction( mFontEngineNoPlot, pUserDataForFreeFunction );
    freeFunction( work, pUserDataForFreeFunction );
    freeFunction( noPlotWorks, pUserDataForFreeFunction );
    freeFunction(mTextureBitMap, pUserDataForFreeFunction);
    mGlyphTreeMap.Finalize(freeFunction, pUserDataForFreeFunction);
}

void DebugFontTextureCache::RegisterGlyphs(const uint16_t *str)
{
    const uint16_t *cur = str;
    while (*cur)
    {
        RegisterGlyph(*cur);
        cur++;
    }
}

bool DebugFontTextureCache::RegisterGlyph(uint16_t code)
{
    if (!IsGlyphExistInFont(code))
    {
        // 文字が存在しなかった場合は登録を行わずに抜ける。
        return false;
    }

    GlyphNode* node = mGlyphTreeMap.Find(code, static_cast<uint16_t>(m_FontSize));

    if (node)
    {
        // 見つかった -> 既に登録されている文字
        node->SetRequested(true);

        // 既に登録されている文字の場合、IsPlottingがtrueのときだけまだ描画できない
        return node->IsPlotting();
    }
    else
    {
        // 見つからない -> まだ登録されていない文字
        node = mGlyphTreeMap.Insert(code, static_cast<uint16_t>(m_FontSize));
        if (node)
        {
            node->SetRequested(true);
            node->SetPlotting(true);

            // 未プロットリストに入れる
            mNeedPlotGlyphList.push_back(*node);
            return true;
        }
        else
        {
            // 平衡二分木のノードに空きがない。
            if (EraseNotInFontGlyphs() > 0) {
                // フォントにない文字のノードを削除し、空きができたら登録を行う。(次は登録できるはず)
                return RegisterGlyph(code);
            } else {
                // 空きができなかったら、文字の登録を行わずに抜ける。
                NN_LOG("nn::font::TextureCache: Node number max over. Please increase InitializeArg::glyphNodeCountMax.\n");
                return false;
            }
        }
    }
}

void DebugFontTextureCache::UpdateTextureCache()
{
    // 結局テクスチャキャッシュ内の文字の配置はUpdateTextureCache内で行わなければならない。
    // 文字の横幅を取る処理が重く、AcquireGlyphmap() で取ってきてその場で配置したほうがトータルのコストが小さいため。
    GlyphList::iterator step_it = mNeedPlotGlyphList.begin();
    GlyphList::iterator end_it = mNeedPlotGlyphList.end();
    for (;step_it != end_it;)
    {
        GlyphList::iterator it = step_it;
        ++step_it;
        mNeedPlotGlyphList.erase(it);

        // まず、文字がフォントにあるかどうかを調べる。ここで調べているのは、RegisterGlyphでやると時間がかかり、
        // メインスレッドの遅延に繋がるため。UpdateTextureCacheの中であれば、プロットの処理に紛れる
        if ( ! mFontEngine->CheckGlyphExist(it->GetCode())) {
            // ない場合、削除は行わない。(削除してしまうとその文字のRegisterGlyphが常にtrueになってしまうため。)
            // フォントにない文字フラグを立て、フォントにない文字リストに入れておく。
            it->SetNotInFont(true);
            mNotInFontGlyphList.push_back(*it);
            continue;
        }

        // フォントサイズの設定
        mFontEngine->SetScale(it->GetSize() << 16, 0, 0, it->GetSize() << 16);
        AssertFsError("SetScale");

        // このフォントのアセントを求める
        int font_ascent = static_cast<int>(mFontMetrics.boundingBoxAscentRatio * it->GetSize() + 0.5f);

        // GRAYMAP8固定にする。そのままA8のテクスチャにコピーできるという利点がある。
        // GRAYMAP8の場合は、ScGlyphMapのメンバbplとwidthは同じ値になる。
        nn::fontll::GlyphMap *map = mFontEngine->AcquireGlyphmap(it->GetCode(), nn::fontll::FormatGrayMap8);
        AssertFsError("AcquireGlyphmap");

        it->SetLineKind(GlyphNode::CalculateLineKind(static_cast<uint16_t>(map->height + font_ascent - map->hiY)));

        // 文字をプロットするラインを決める
        LineInfo* matched_line = NULL;
        for (uint32_t i = 0; i < mLineCurrentPos; i++)
        {
            if (mLineInfos[i].kind == it->GetLineKind() && (static_cast<uint32_t>(mLineInfos[i].currentX + map->width + GlyphPadding) < mTextureCacheWidth))
            {
                matched_line = &mLineInfos[i];
                break;
            }
        }
        // マッチするラインがなかった場合
        if ( ! matched_line)
        {
            if (mLineCurrentPos >= TextureCacheLineCountMax)
            {
                // ラインの最大数を超えた。エラー扱い
                NN_LOG("TextureCache: line num[%d] exceeds\n", mLineCurrentPos);
                mNeedEraseGlyphList.push_back(*it);
                mFontEngine->ReleasesGlyph(map);
                AssertFsError("ReleasesGlyph");
                continue;
            }

            // 新しいラインを作成する
            matched_line = CreateNewLineImpl(static_cast<LineHeight>(it->GetLineKind()));

            if (matched_line == NULL)
            {
                // 新しいラインを作成できなかった。
                // 現在プロットされているものを消して、その上にプロットする必要がある。

                // 消去可能な文字を探す
                GlyphNode* node = NULL;
                node = FindAndReserveEraseGlyph(it->GetLineKind(), map->width);
                if (node)
                {
                    // 元々あったグリフの領域に上書きする
                    GlyphLineList* list = &mLineInfos[node->GetLineNo()].list;
                    it->SetCachePosX(node->GetCachePosX());
                    it->SetCachePosY(node->GetCachePosY());
                    it->SetLineNo(node->GetLineNo());
                    list->insert(list->iterator_to(*node), *it);
                }
                else
                {
                    // 入るスペースがないので抜ける
                    NN_LOG("TextureCache: no erasable glyph. charCode = 0x%x lineHeight = %d  size=%d\n", it->GetCode(), it->GetLineKind(), it->GetSize());
                    mNeedEraseGlyphList.push_back(*it);
                    mFontEngine->ReleasesGlyph(map);
                    AssertFsError("ReleasesGlyph");
                    continue;
                }
            }
        }

        if (matched_line)
        {
            it->SetCachePosX(matched_line->currentX);
            it->SetCachePosY(matched_line->y);
            it->SetLineNo(matched_line->no);
            matched_line->list.push_back(*it);
            matched_line->currentX += map->width + GlyphPadding;
        }

        int offset_from_top = font_ascent - map->hiY;
        int render_origin_y = it->GetCachePosY() + offset_from_top;

        it->SetGlyphWidth(map->width);
        it->SetGlyphHeight(static_cast<uint16_t>(map->height + offset_from_top));
        it->SetAdvanceX(map->idX);
        it->SetLeftOffset(map->loX);

        uint32_t copy_size = map->width + 2;

        uint8_t* pTextureCache = mTextureBitMap;

        // 前の文字が描画されている場合があるので、文字の上のスペースをクリアする
        for (int l = it->GetCachePosY(); l < render_origin_y; l++) {
            // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
            uint8_t* line_start = &pTextureCache[(mTextureCacheHeight - l - 1) * mTextureCacheWidth + it->GetCachePosX()];
            std::memset(line_start, 0x00, copy_size);
        }

        int copy_start_y = std::max(0, - render_origin_y);
        {
            for (int l = copy_start_y; l < map->height; l++) {
                // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
                uint8_t* line_start = &pTextureCache[(mTextureCacheHeight - (l + render_origin_y) - 1) * mTextureCacheWidth + it->GetCachePosX()];

                // 文字をコピーして左右の1ピクセルずつを0でクリアする
                std::memcpy(line_start + 1, &map->bits[l * map->width], map->width);
                line_start[0] = 0;
                line_start[map->width + 1] = 0;
            }
        }
        // heightがそのラインの高さより小さい場合は、最大2ドット0でクリアする
        uint32_t line_height = it->GetLineKind();
        for (uint32_t l = map->height; l < static_cast<uint32_t>(map->height + 2) && l < line_height; l++)
        {
            // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
            uint8_t* line_start = &pTextureCache[(mTextureCacheHeight - (l + render_origin_y) - 1) * mTextureCacheWidth + it->GetCachePosX()];
            std::memset(line_start, 0, copy_size);
        }

        mFontEngine->ReleasesGlyph(map);
        AssertFsError("ReleasesGlyph");
    }
}// NOLINT(impl/function_size)

DebugFontTextureCache::LineInfo* DebugFontTextureCache::CreateNewLineImpl(LineHeight lineHeight)
{
    uint32_t next_line_y;
    if (mLineCurrentPos == 0)
    {
        next_line_y = 0;
    }
    else
    {
        next_line_y = mLineInfos[mLineCurrentPos - 1].y + mLineInfos[mLineCurrentPos - 1].kind + GlyphPadding;
    }
    uint32_t endY = next_line_y + static_cast<uint32_t>(lineHeight);

    if (endY > mTextureCacheHeight)
    {
        // 全てのラインが埋まっていて、新たにラインを作ることができない。
        return NULL;
    }
    else
    {
        // 新しいラインを作成できる
        LineInfo* new_line = &mLineInfos[mLineCurrentPos];
        new_line->currentX = 0;
        new_line->y = static_cast<uint16_t>(next_line_y);
        new_line->kind = static_cast<uint8_t>(lineHeight);
        new_line->no = static_cast<uint8_t>(mLineCurrentPos);

        mLineCurrentPos++;
        return new_line;
    }
}

GlyphNode* DebugFontTextureCache::FindAndReserveEraseGlyph(uint8_t lineKind, uint16_t glyphWidth)
{
    for (uint32_t l = 0; l < mLineCurrentPos; l++)
    {
        LineInfo* lineInfo = &mLineInfos[l];
        if (lineInfo->kind == lineKind)
        {
            // 種類が一致するラインを見つけた、消去可能な文字がないか探す
            GlyphLineList::iterator it = lineInfo->list.begin();
            GlyphLineList::iterator begin_it = lineInfo->list.begin();
            GlyphLineList::iterator end_it = lineInfo->list.end();
            GlyphLineList::iterator mostleft_erasable_node = it;
            uint32_t erasable_num = 0;
            for (;it != end_it; ++it)
            {
                if (it->IsErasable())
                {
                    uint32_t erasable_width;
                    if (erasable_num == 0)
                    {
                        mostleft_erasable_node = it;
                        erasable_width = it->GetGlyphWidth();
                    }
                    else
                    {
                        erasable_width = it->GetGlyphWidth() + it->GetCachePosX() - mostleft_erasable_node->GetCachePosX();
                    }
                    int mostleft_move_val = 0;
                    if (mostleft_erasable_node != begin_it)
                    {
                        // mostleft_erasable_nodeが最初のノードでなければ、左側の空きスペースを使うことができる
                        GlyphLineList::iterator mostleft_prev_it = mostleft_erasable_node;
                        do {
                            --mostleft_prev_it;
                        } while (mostleft_prev_it->IsErased() && begin_it != mostleft_prev_it);

                        mostleft_move_val = mostleft_erasable_node->GetCachePosX() - (mostleft_prev_it->GetCachePosX() + mostleft_prev_it->GetGlyphWidth() + GlyphPadding);
                        if (mostleft_move_val > 0)
                        {
                            erasable_width += mostleft_move_val;
                        }
                    }
                    if (erasable_width >= glyphWidth)
                    {
                        // 消去可能な幅が必要な文字幅を超えた
                        // もう一度消去されることがないように消去フラグを立て、消去予約済みリストに入れる
                        GlyphLineList::iterator erasable_it = mostleft_erasable_node;
                        GlyphLineList::iterator erasable_end_it = it;
                        ++erasable_end_it;
                        for (; erasable_it != erasable_end_it; ++erasable_it)
                        {
                            if ( ! erasable_it->IsErased())
                            {
                                erasable_it->SetErased(true);
                                mNeedEraseGlyphList.push_back(*erasable_it);
                            }
                        }
                        if (mostleft_move_val > 0)
                        {
                            // mostleft_erasable_nodeの左側の空きスペースを使っている場合はCachePosXを動かす
                            mostleft_erasable_node->SetCachePosX(static_cast<uint16_t>(mostleft_erasable_node->GetCachePosX() - mostleft_move_val));
                        }
                        return &(*mostleft_erasable_node);
                    }
                    erasable_num += 1;
                }
                else
                {
                    erasable_num = 0;
                }
            }
        }
    }
    return NULL;
}

void DebugFontTextureCache::CompleteTextureCache()
{
    // 消去予約済みリストに入っているノードを破棄する
    // 消去予約済みリストに入るのは、新しい文字を表示する必要があって削除する場合と、
    // プロットすることができなかったために削除する場合の二通りある。
    // 前者はIsErasedがtrueになり、後者はfalseになるので区別できる。後者はラインの
    // リストから削除する必要がない。
    GlyphList::iterator step_it = mNeedEraseGlyphList.begin();
    GlyphList::iterator end_it = mNeedEraseGlyphList.end();
    for (;step_it != end_it;)
    {
        GlyphList::iterator it = step_it;
        ++step_it;
        mNeedEraseGlyphList.erase(it);
        if (it->IsErased())
        {
            GlyphLineList& list = mLineInfos[it->GetLineNo()].list;
            list.erase(list.iterator_to(*it));
            // グリフが消えたことによりラインの右側が空く可能性があるので、currentXを設定する
            if (list.size() > 0)
            {
                mLineInfos[it->GetLineNo()].currentX = list.back().GetCachePosX() + list.back().GetGlyphWidth() + GlyphPadding;
            }
            else
            {
                mLineInfos[it->GetLineNo()].currentX = 0;
            }
        }
        mGlyphTreeMap.Erase(it->GetCode(), it->GetSize());
    }

    mGlyphTreeMap.UpdateFlagsForCompleteTextureCache();
}

bool DebugFontTextureCache::IsGlyphExistInFont(uint16_t code)
{
    bool existing = mFontEngineNoPlot->CheckGlyphExist(code);
    if (existing)
    {
        return true;
    }
    return false;
}

const DebugFontTextureCache::FontMetrics& DebugFontTextureCache::GetFontMetrics()
{
    return mFontMetrics;
}

void DebugFontTextureCache::AssertFsError(const char* api_name)
{
    NN_UNUSED(api_name);
    NN_SDK_ASSERT(mFontEngine->GetError() == nn::fontll::Success, "%s error: %d", api_name, mFontEngine->GetError());
}

void DebugFontTextureCache::AssertFsErrorNoPlot(const char* api_name)
{
    NN_UNUSED(api_name);
    NN_SDK_ASSERT(mFontEngineNoPlot->GetError() == nn::fontll::Success, "%s error: %d", api_name, mFontEngineNoPlot->GetError());
}

uint32_t DebugFontTextureCache::EraseNotInFontGlyphs()
{
    uint32_t count = 0;
    GlyphList::iterator step_it = mNotInFontGlyphList.begin();
    GlyphList::iterator end_it = mNotInFontGlyphList.end();
    for (;step_it != end_it;)
    {
        GlyphList::iterator it = step_it;
        ++step_it;
        // 現在使われている文字については削除しない
        if ( ! it->IsRequestedOrKeeped()) {
            mNotInFontGlyphList.erase(it);
            mGlyphTreeMap.Erase(it->GetCode(), it->GetSize());
            count++;
        }
    }
    return count;
}
